<?php
if (!defined('ABSPATH')) {
    exit;
}

class NexlifyDesk_Admin {
    public static function init() {
        add_action('admin_menu', array(__CLASS__, 'register_admin_menu'));
        add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_admin_assets'));
        add_action('admin_notices', array(__CLASS__, 'display_admin_notices'));
        add_action('admin_init', array(__CLASS__, 'handle_settings_save'));
        add_action('admin_post_nexlifydesk_save_settings', array(__CLASS__, 'save_settings'));
        add_action('admin_post_nopriv_nexlifydesk_save_settings', array(__CLASS__, 'save_settings'));
        add_action('admin_post_nexlifydesk_save_agent_position', array(__CLASS__, 'handle_save_agent_position'));
        add_action('admin_post_nexlifydesk_delete_agent_position', array(__CLASS__, 'handle_delete_agent_position'));
        add_action('admin_post_nexlifydesk_save_agent_assignments', array(__CLASS__, 'handle_save_agent_assignments'));
        add_action('admin_post_nexlifydesk_save_imap_settings', array(__CLASS__, 'save_imap_settings'));
        add_action('admin_init', ['NexlifyDesk_Admin', 'maybe_encrypt_existing_passwords']);
        add_action('wp_ajax_nexlifydesk_refresh_ticket_list', array(__CLASS__, 'ajax_refresh_ticket_list'));
        add_action('wp_ajax_nexlifydesk_mark_ticket_read', array(__CLASS__, 'ajax_mark_ticket_read'));
        add_action('wp_ajax_nexlifydesk_clear_rate_limit', array(__CLASS__, 'ajax_clear_rate_limit'));
        add_action('wp_ajax_nexlifydesk_check_rate_limit', array(__CLASS__, 'ajax_check_rate_limit'));
        add_action('admin_post_nexlifydesk_deauth_google', array(__CLASS__, 'handle_deauth_google'));
        add_action('admin_post_nexlifydesk_deauth_aws', array(__CLASS__, 'handle_deauth_aws'));
        add_action('admin_post_nexlifydesk_deauth_custom', array(__CLASS__, 'handle_deauth_custom'));
        add_action('admin_post_nexlifydesk_deauth_all', array(__CLASS__, 'handle_deauth_all'));

        register_setting('nexlifydesk_settings', 'nexlifydesk_imap_settings', [
            'type' => 'array',
            'sanitize_callback' => array(__CLASS__, 'sanitize_imap_settings'),
        ]);
    }

    /**
     * Sanitize IMAP settings input
     */
    public static function sanitize_imap_settings($input) {
        
        $new_input = get_option('nexlifydesk_imap_settings', []);

        $new_input['enabled'] = isset($input['enabled']) ? 1 : 0;
        $new_input['provider'] = sanitize_text_field($input['provider'] ?? 'custom');
        
        $fetch_interval = absint($input['fetch_interval'] ?? 5);
        $allowed_intervals = [2, 5, 10, 15];
        $new_input['fetch_interval'] = in_array($fetch_interval, $allowed_intervals) ? $fetch_interval : 5;
        
        $is_ssl_enabled = function_exists('nexlifydesk_check_ssl_enabled') ? nexlifydesk_check_ssl_enabled() : true;
        
        $provider = $new_input['provider'] ?? 'custom';
        
        if ($provider === 'aws') {
            $is_ssl_enabled = true;
        }
        
        $new_input['protocol'] = sanitize_text_field($input['protocol'] ?? 'imap');
        $new_input['host'] = sanitize_text_field($input['host'] ?? '');
        $new_input['port'] = absint($input['port'] ?? 993);
        $new_input['encryption'] = sanitize_text_field($input['encryption'] ?? 'ssl');
        $new_input['username'] = sanitize_text_field($input['username'] ?? '');
        
        if (!empty($input['password']) && $input['password'] !== '••••••••••••••••') {
            $plain = sanitize_text_field($input['password']);
            $new_input['password'] = nexlifydesk_encrypt($plain);
        } elseif (!empty($input['password_preserved'])) {
            $new_input['password'] = $new_input['password'] ?? '';
        } else {
            $new_input['password'] = '';
        }

        $new_input['google_client_id'] = sanitize_text_field($input['google_client_id'] ?? '');
        
        if (!empty($input['google_client_secret']) && $input['google_client_secret'] !== '••••••••••••••••') {
            $plain = sanitize_text_field($input['google_client_secret']);
            $new_input['google_client_secret'] = nexlifydesk_encrypt($plain);
        } elseif (!empty($input['google_client_secret_preserved'])) {
            $new_input['google_client_secret'] = $new_input['google_client_secret'] ?? '';
        } else {
            $new_input['google_client_secret'] = '';
        }
        
        if (isset($input['google_access_token'])) {
            $new_input['google_access_token'] = sanitize_text_field($input['google_access_token']);
        }
        if (isset($input['google_refresh_token'])) {
            $new_input['google_refresh_token'] = sanitize_text_field($input['google_refresh_token']);
        }
        if (isset($input['google_account_email'])) {
            $new_input['google_account_email'] = sanitize_email($input['google_account_email']);
        }

        $new_input['aws_region'] = sanitize_text_field($input['aws_region'] ?? '');
        $new_input['aws_organization_id'] = sanitize_text_field($input['aws_organization_id'] ?? '');
        $new_input['aws_email'] = sanitize_email($input['aws_email'] ?? '');
        
        if (!empty($input['aws_password']) && $input['aws_password'] !== '••••••••••••••••') {
            $plain = sanitize_text_field($input['aws_password']);
            $new_input['aws_password'] = nexlifydesk_encrypt($plain);
        } elseif (!empty($input['aws_password_preserved'])) {
            $new_input['aws_password'] = $new_input['aws_password'] ?? '';
        } else {
            $new_input['aws_password'] = '';
        }
        
        $new_input['aws_access_key_id'] = sanitize_text_field($input['aws_access_key_id'] ?? '');
        
        if (!empty($input['aws_secret_access_key']) && $input['aws_secret_access_key'] !== '••••••••••••••••') {
            $plain = sanitize_text_field($input['aws_secret_access_key']);
            $new_input['aws_secret_access_key'] = nexlifydesk_encrypt($plain);
        } elseif (!empty($input['aws_secret_access_key_preserved'])) {
            $new_input['aws_secret_access_key'] = $new_input['aws_secret_access_key'] ?? '';
        } else {
            $new_input['aws_secret_access_key'] = '';
        }

        $new_input['delete_emails_after_fetch'] = isset($input['delete_emails_after_fetch']) ? 1 : 0;
        $new_input['block_admin_emails'] = isset($input['block_admin_emails']) ? 1 : 0;
        $new_input['block_notification_subjects'] = isset($input['block_notification_subjects']) ? 1 : 0;
        $new_input['block_marketing_emails'] = isset($input['block_marketing_emails']) ? 1 : 0;
        $new_input['blocked_emails'] = sanitize_textarea_field($input['blocked_emails'] ?? '');
        $new_input['blocked_domains'] = sanitize_textarea_field($input['blocked_domains'] ?? '');
        $new_input['spam_url_filtering'] = isset($input['spam_url_filtering']) ? 1 : 0;
        $new_input['max_links_per_email'] = absint($input['max_links_per_email'] ?? 3);
        $new_input['blocked_keywords'] = sanitize_textarea_field($input['blocked_keywords'] ?? '');
        $new_input['aws_integration_type'] = sanitize_text_field($input['aws_integration_type'] ?? 'imap');
        if ($new_input['aws_integration_type'] === 'lambda') {
            $new_input['aws_integration_type'] = 'imap';
        }
        $new_input['lambda_webhook_secret'] = sanitize_text_field($input['lambda_webhook_secret'] ?? '');

        return $new_input;
    }

    /**
     * Save IMAP settings from the IMAP Auth settings page
     */
    public static function save_imap_settings() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission.', 'nexlifydesk'));
        }
        check_admin_referer('nexlifydesk_save_imap_settings', 'nexlifydesk_imap_settings_nonce');
        
        $old_settings = get_option('nexlifydesk_imap_settings', []);
        
        $settings = array(
            'enabled' => isset($_POST['enabled']) ? 1 : 0,
            'provider' => sanitize_text_field(isset($_POST['provider']) ? wp_unslash($_POST['provider']) : ''),
            'host' => sanitize_text_field(isset($_POST['host']) ? wp_unslash($_POST['host']) : ''),
            'port' => absint(isset($_POST['port']) ? wp_unslash($_POST['port']) : 993),
            'encryption' => sanitize_text_field(isset($_POST['encryption']) ? wp_unslash($_POST['encryption']) : ''),
            'username' => sanitize_text_field(isset($_POST['username']) ? wp_unslash($_POST['username']) : ''),
            'protocol' => sanitize_text_field(isset($_POST['protocol']) ? wp_unslash($_POST['protocol']) : 'imap'),
        );
        
        $password_input = isset($_POST['password']) ? sanitize_text_field(wp_unslash($_POST['password'])) : '';
        $password_preserved = isset($_POST['password_preserved']) ? sanitize_text_field(wp_unslash($_POST['password_preserved'])) : '';
        
        if (!empty($password_input)) {
            $settings['password'] = nexlifydesk_encrypt($password_input);
        } elseif (!empty($password_preserved)) {
            $settings['password'] = $old_settings['password'] ?? '';
        } else {
            $settings['password'] = '';
        }
        
        $aws_password_input = isset($_POST['aws_password']) ? sanitize_text_field(wp_unslash($_POST['aws_password'])) : '';
        $aws_password_preserved = isset($_POST['aws_password_preserved']) ? sanitize_text_field(wp_unslash($_POST['aws_password_preserved'])) : '';
        
        if (!empty($aws_password_input)) {
            $settings['aws_password'] = nexlifydesk_encrypt($aws_password_input);
        } elseif (!empty($aws_password_preserved)) {
            $settings['aws_password'] = $old_settings['aws_password'] ?? '';
        } else {
            $settings['aws_password'] = '';
        }
        
        $aws_secret_input = isset($_POST['aws_secret_access_key']) ? sanitize_text_field(wp_unslash($_POST['aws_secret_access_key'])) : '';
        $aws_secret_preserved = isset($_POST['aws_secret_access_key_preserved']) ? sanitize_text_field(wp_unslash($_POST['aws_secret_access_key_preserved'])) : '';
        
        if (!empty($aws_secret_input)) {
            $settings['aws_secret_access_key'] = nexlifydesk_encrypt($aws_secret_input);
        } elseif (!empty($aws_secret_preserved)) {
            $settings['aws_secret_access_key'] = $old_settings['aws_secret_access_key'] ?? '';
        } else {
            $settings['aws_secret_access_key'] = '';
        }
        
        $google_secret_input = isset($_POST['google_client_secret']) ? sanitize_text_field(wp_unslash($_POST['google_client_secret'])) : '';
        $google_secret_preserved = isset($_POST['google_client_secret_preserved']) ? sanitize_text_field(wp_unslash($_POST['google_client_secret_preserved'])) : '';
        
        if (!empty($google_secret_input)) {
            $settings['google_client_secret'] = nexlifydesk_encrypt($google_secret_input);
        } elseif (!empty($google_secret_preserved)) {
            $settings['google_client_secret'] = $old_settings['google_client_secret'] ?? '';
        } else {
            $settings['google_client_secret'] = '';
        }
        
        $settings['aws_integration_type'] = sanitize_text_field(isset($_POST['aws_integration_type']) ? wp_unslash($_POST['aws_integration_type']) : 'imap');
        if ($settings['aws_integration_type'] === 'lambda') {
            $settings['aws_integration_type'] = 'imap';
        }
        $settings['lambda_webhook_secret'] = sanitize_text_field(isset($_POST['lambda_webhook_secret']) ? wp_unslash($_POST['lambda_webhook_secret']) : '');
        $settings = array_merge($old_settings, $settings);
        
        update_option('nexlifydesk_imap_settings', $settings);
        wp_redirect(add_query_arg(array('page' => 'nexlifydesk_imap_auth', 'settings-updated' => 'true'), admin_url('admin.php')));
        exit;
    }

    /**
     * Handle Google de-authentication
     */
    public static function handle_deauth_google() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission.', 'nexlifydesk'));
        }
        
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_key($_GET['_wpnonce']), 'nexlifydesk_deauth_google')) {
            wp_die(esc_html__('Security check failed.', 'nexlifydesk'));
        }
        
        $settings = get_option('nexlifydesk_imap_settings', []);
        
        unset($settings['google_access_token']);
        unset($settings['google_refresh_token']);
        unset($settings['google_client_id']);
        unset($settings['google_client_secret']);
        unset($settings['google_auth_status']);
        unset($settings['google_account_email']);
        unset($settings['google_token_expires_at']);
        unset($settings['google_fetch_start_time']);
        
        $settings['google_access_token'] = '';
        $settings['google_refresh_token'] = '';
        $settings['google_client_id'] = '';
        $settings['google_client_secret'] = '';
        $settings['google_auth_status'] = '';
        $settings['google_account_email'] = '';
        $settings['google_token_expires_at'] = '';
        $settings['google_fetch_start_time'] = '';
        
        update_option('nexlifydesk_imap_settings', $settings);
        
        delete_option('nexlifydesk_processed_emails');
        
        wp_redirect(add_query_arg(array(
            'page' => 'nexlifydesk_imap_auth', 
            'deauth' => 'google_success',
            '_wpnonce' => wp_create_nonce('nexlifydesk_imap_auth')
        ), admin_url('admin.php')));
        exit;
    }

    /**
     * Handle AWS de-authentication
     */
    public static function handle_deauth_aws() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission.', 'nexlifydesk'));
        }
        
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_key($_GET['_wpnonce']), 'nexlifydesk_deauth_aws')) {
            wp_die(esc_html__('Security check failed.', 'nexlifydesk'));
        }
        
        $settings = get_option('nexlifydesk_imap_settings', []);
        
        unset($settings['aws_region']);
        unset($settings['aws_organization_id']);
        unset($settings['aws_email']);
        unset($settings['aws_password']); 
        unset($settings['aws_access_key_id']);
        unset($settings['aws_secret_access_key']); 
        unset($settings['aws_fetch_start_time']);
        
        $settings['aws_region'] = '';
        $settings['aws_organization_id'] = '';
        $settings['aws_email'] = '';
        $settings['aws_password'] = '';
        $settings['aws_access_key_id'] = '';
        $settings['aws_secret_access_key'] = '';
        $settings['aws_fetch_start_time'] = '';
        
        update_option('nexlifydesk_imap_settings', $settings);
        
        delete_option('nexlifydesk_aws_processed_emails');
        
        wp_redirect(add_query_arg(array(
            'page' => 'nexlifydesk_imap_auth', 
            'deauth' => 'aws_success',
            '_wpnonce' => wp_create_nonce('nexlifydesk_imap_auth')
        ), admin_url('admin.php')));
        exit;
    }

    /**
     * Handle Custom IMAP/POP3 de-authentication
     */
    public static function handle_deauth_custom() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission.', 'nexlifydesk'));
        }
        
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_key($_GET['_wpnonce']), 'nexlifydesk_deauth_custom')) {
            wp_die(esc_html__('Security check failed.', 'nexlifydesk'));
        }
        
        $settings = get_option('nexlifydesk_imap_settings', []);
        
        unset($settings['host']);
        unset($settings['port']);
        unset($settings['username']);
        unset($settings['password']);
        unset($settings['encryption']);
        unset($settings['protocol']);
        
        $settings['host'] = '';
        $settings['port'] = 993;
        $settings['username'] = '';
        $settings['password'] = '';
        $settings['encryption'] = 'ssl';
        $settings['protocol'] = 'imap';
        
        update_option('nexlifydesk_imap_settings', $settings);
        
        wp_redirect(add_query_arg(array(
            'page' => 'nexlifydesk_imap_auth', 
            'deauth' => 'custom_success',
            '_wpnonce' => wp_create_nonce('nexlifydesk_imap_auth')
        ), admin_url('admin.php')));
        exit;
    }

    /**
     * Handle complete de-authentication (all providers)
     */
    public static function handle_deauth_all() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission.', 'nexlifydesk'));
        }
        
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_key($_GET['_wpnonce']), 'nexlifydesk_deauth_all')) {
            wp_die(esc_html__('Security check failed.', 'nexlifydesk'));
        }
        
        $current_settings = get_option('nexlifydesk_imap_settings', []);
        $credential_keys = [
            // Custom IMAP/POP3 credentials
            'host', 'port', 'username', 'password', 'encryption', 'protocol',
            
            // Google credentials
            'google_client_id', 'google_client_secret', 'google_access_token', 
            'google_refresh_token', 'google_auth_status', 'google_account_email',
            'google_token_expires_at', 'google_fetch_start_time',
            
            // AWS credentials
            'aws_region', 'aws_organization_id', 'aws_email', 'aws_password',
            'aws_access_key_id', 'aws_secret_access_key', 'aws_fetch_start_time'
        ];
        
        foreach ($credential_keys as $key) {
            unset($current_settings[$key]);
        }
        
        $default_settings = array(
            'enabled' => 0,
            'provider' => 'custom',
            'fetch_interval' => 5,
            'delete_emails_after_fetch' => 1,
            
            // Custom IMAP/POP3 settings
            'host' => '',
            'port' => 993,
            'username' => '',
            'password' => '',
            'encryption' => 'ssl',
            'protocol' => 'imap',
            
            // Google settings
            'google_client_id' => '',
            'google_client_secret' => '',
            'google_access_token' => '',
            'google_refresh_token' => '',
            'google_auth_status' => '',
            'google_account_email' => '',
            'google_token_expires_at' => '',
            'google_fetch_start_time' => '',
            
            // AWS settings
            'aws_region' => '',
            'aws_organization_id' => '',
            'aws_email' => '',
            'aws_password' => '',
            'aws_access_key_id' => '',
            'aws_secret_access_key' => '',
            'aws_fetch_start_time' => '',
            
            // Spam protection settings (preserve these)
            'block_admin_emails' => 1,
            'block_notification_subjects' => 1,
            'block_marketing_emails' => 1,
            'blocked_emails' => '',
            'blocked_domains' => '',
            'spam_url_filtering' => 1,
            'max_links_per_email' => 3,
            'blocked_keywords' => '',
        );
        
        $spam_settings = [
            'block_admin_emails', 'block_notification_subjects', 'block_marketing_emails',
            'blocked_emails', 'blocked_domains', 'spam_url_filtering', 'max_links_per_email', 'blocked_keywords'
        ];
        
        foreach ($spam_settings as $setting) {
            if (isset($current_settings[$setting])) {
                $default_settings[$setting] = $current_settings[$setting];
            }
        }
        
        $final_settings = array_merge($current_settings, $default_settings);
        
        update_option('nexlifydesk_imap_settings', $final_settings);
        
        delete_option('nexlifydesk_processed_emails');
        delete_option('nexlifydesk_aws_processed_emails');
        
        wp_redirect(add_query_arg(array(
            'page' => 'nexlifydesk_imap_auth', 
            'deauth' => 'all_success',
            '_wpnonce' => wp_create_nonce('nexlifydesk_imap_auth')
        ), admin_url('admin.php')));
        exit;
    }

    public static function maybe_encrypt_existing_passwords() {
        $settings = get_option('nexlifydesk_imap_settings', []);
        $updated = false;

        $is_encrypted = function($value) {
            if (empty($value)) return false;
            return nexlifydesk_is_encrypted($value);
        };

        // Custom IMAP/POP3 password
        if (!empty($settings['password']) && !$is_encrypted($settings['password'])) {
            $settings['password'] = nexlifydesk_encrypt($settings['password']);
            $updated = true;
        }
        // AWS password
        if (!empty($settings['aws_password']) && !$is_encrypted($settings['aws_password'])) {
            $settings['aws_password'] = nexlifydesk_encrypt($settings['aws_password']);
            $updated = true;
        }
        // Google client secret
        if (!empty($settings['google_client_secret']) && !$is_encrypted($settings['google_client_secret'])) {
            $settings['google_client_secret'] = nexlifydesk_encrypt($settings['google_client_secret']);
            $updated = true;
        }
        // AWS secret access key
        if (!empty($settings['aws_secret_access_key']) && !$is_encrypted($settings['aws_secret_access_key'])) {
            $settings['aws_secret_access_key'] = nexlifydesk_encrypt($settings['aws_secret_access_key']);
            $updated = true;
        }
        if ($updated) {
            update_option('nexlifydesk_imap_settings', $settings);
        }
    }

    public static function register_admin_menu() {
        // Main menu
        $hook = add_menu_page(
            __('NexlifyDesk Tickets', 'nexlifydesk'),
            __('NexlifyDesk', 'nexlifydesk'),
            'nexlifydesk_manage_tickets',
            'nexlifydesk_tickets',
            array(__CLASS__, 'render_ticket_list_page'),
            NEXLIFYDESK_PLUGIN_URL . 'assets/images/dashboard-icon.png',
            25
        );

        // Submenu: Tickets
        add_submenu_page(
            'nexlifydesk_tickets',
            __('All Tickets', 'nexlifydesk'),
            __('All Tickets', 'nexlifydesk'),
            'nexlifydesk_manage_tickets',
            'nexlifydesk_tickets',
            array(__CLASS__, 'render_ticket_list_page')
        );

        // Submenu: Categories
        add_submenu_page(
            'nexlifydesk_tickets',
            __('Ticket Categories', 'nexlifydesk'),
            __('Categories', 'nexlifydesk'),
            'nexlifydesk_manage_categories',
            'nexlifydesk_categories',
            array(__CLASS__, 'render_categories_page')
        );

        // Submenu: Settings
        add_submenu_page(
            'nexlifydesk_tickets',
            __('Settings', 'nexlifydesk'),
            __('Settings', 'nexlifydesk'),
            'manage_options',
            'nexlifydesk_settings',
            array(__CLASS__, 'render_settings_page')
        );

        // Submenu: Reports
        add_submenu_page(
            'nexlifydesk_tickets',
            __('Reports', 'nexlifydesk'),
            __('Reports', 'nexlifydesk'),
            'nexlifydesk_view_reports',
            'nexlifydesk_reports',
            array(__CLASS__, 'render_reports_page')
        );

        // Submenu: Order History
        add_submenu_page(
            'nexlifydesk_tickets',
            __('Order History', 'nexlifydesk'),
            __('Order History', 'nexlifydesk'),
            'nexlifydesk_manage_tickets',
            'nexlifydesk_order_history',
            'nexlifydesk_render_order_history_page'
        );

        // Submenu: Agent Positions
         add_submenu_page(
            'nexlifydesk_tickets',
            __('Agent Positions', 'nexlifydesk'),
            __('Agent Positions', 'nexlifydesk'),
            'manage_options',
            'nexlifydesk_agent_positions',
            array(__CLASS__, 'render_agent_positions_page')
        );

        // Submenu: Email Templates
         add_submenu_page(
            'nexlifydesk_tickets',
            __('Email Templates', 'nexlifydesk'),
            __('Email Templates', 'nexlifydesk'),
            'manage_options',
            'nexlifydesk_email_templates',
            array(__CLASS__, 'render_email_templates_page')
        );

        // Add Support submenu
        add_submenu_page(
            'nexlifydesk_tickets',
            __('Support', 'nexlifydesk'),
            __('Support', 'nexlifydesk'),
            'manage_options',
            'nexlifydesk_support',
            array('NexlifyDesk_Admin', 'render_support_page')
        );

        // Add IMAP Auth submenu
        add_submenu_page(
            'nexlifydesk_tickets',
            __('IMAP Auth', 'nexlifydesk'),
            __('IMAP Auth', 'nexlifydesk'),
            'manage_options',
            'nexlifydesk_imap_auth',
            array(__CLASS__, 'render_imap_auth_page')
        );
    
    }

    public static function render_imap_auth_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Email Piping (IMAP Auth)', 'nexlifydesk'); ?></h1>
            
            <?php
            if (!extension_loaded('imap')) {
                echo '<div class="notice notice-error"><p><strong>' . esc_html__('Warning:', 'nexlifydesk') . '</strong> ' . 
                     esc_html__('IMAP extension is not installed on this server. Email piping will not work for Custom IMAP/POP3 and AWS providers. Please contact your hosting provider to enable the PHP IMAP extension.', 'nexlifydesk') . 
                     '</p></div>';
            }
            ?>
            
            <?php include NEXLIFYDESK_PLUGIN_DIR . 'templates/admin/imap-auth.php'; ?>
        </div>
        <?php
    }

    /**
     * AJAX handler for refreshing ticket list
     */
    public static function ajax_refresh_ticket_list() {
        if (!current_user_can('nexlifydesk_manage_tickets')) {
            wp_die('Unauthorized');
        }
        
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');
        
        $last_refresh = isset($_POST['last_refresh']) ? intval($_POST['last_refresh']) / 1000 : 0;
        
        if (class_exists('NexlifyDesk_Tickets')) {
            $tickets = NexlifyDesk_Tickets::get_tickets_for_refresh($last_refresh);
            
            wp_send_json_success(array(
                'tickets' => $tickets,
                'timestamp' => time()
            ));
        }
        
        wp_send_json_error('Tickets system not available');
    }
    
    /**
     * AJAX handler for marking ticket as read
     */
    public static function ajax_mark_ticket_read() {
        if (!current_user_can('nexlifydesk_manage_tickets')) {
            wp_die('Unauthorized');
        }
        
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');
        
        $ticket_id = isset($_POST['ticket_id']) ? intval($_POST['ticket_id']) : 0;
        
        if ($ticket_id && class_exists('NexlifyDesk_Tickets')) {
            NexlifyDesk_Tickets::mark_ticket_read($ticket_id, get_current_user_id());
            wp_send_json_success('Ticket marked as read');
        }
        
        wp_send_json_error('Failed to mark ticket as read');
    }
    
    /**
     * AJAX handler to clear rate limit for a specific email address
     */
    public static function ajax_clear_rate_limit() {
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('You do not have permission to perform this action.', 'nexlifydesk'));
        }

        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'nexlifydesk-ajax-nonce')) {
            wp_send_json_error(__('Nonce verification failed.', 'nexlifydesk'));
        }

        // Get email address
        $email_address = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        
        if (empty($email_address) || !is_email($email_address)) {
            wp_send_json_error(__('Please enter a valid email address.', 'nexlifydesk'));
        }

        // Check if user exists
        $user = get_user_by('email', $email_address);
        $user_id = $user ? $user->ID : 0;

        // Get current status before clearing
        $status = NexlifyDesk_Rate_Limiter::get_rate_limit_status($user_id, $email_address);
        
        if (!$status['is_limited']) {
            wp_send_json_error(__('This email address is not currently rate limited.', 'nexlifydesk'));
        }

        // Clear the rate limit
        $result = NexlifyDesk_Rate_Limiter::clear_rate_limit($user_id, $email_address);

        if ($result) {
            $message = sprintf(
                /* translators: %s: email address */
                __('Rate limit cleared for %s. They can now send emails again.', 'nexlifydesk'),
                esc_html($email_address)
            );
            wp_send_json_success($message);
        } else {
            wp_send_json_error(__('Failed to clear rate limit. Please try again.', 'nexlifydesk'));
        }
    }

    /**
     * AJAX handler to check rate limit status for a specific email address
     */
    public static function ajax_check_rate_limit() {
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('You do not have permission to perform this action.', 'nexlifydesk'));
        }

        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'nexlifydesk-ajax-nonce')) {
            wp_send_json_error(__('Nonce verification failed.', 'nexlifydesk'));
        }

        // Get email address
        $email_address = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        
        if (empty($email_address) || !is_email($email_address)) {
            wp_send_json_error(__('Please enter a valid email address.', 'nexlifydesk'));
        }

        // Check if user exists
        $user = get_user_by('email', $email_address);
        $user_id = $user ? $user->ID : 0;

        // Get rate limit status
        $status = NexlifyDesk_Rate_Limiter::get_rate_limit_status($user_id, $email_address);

        if ($status['is_limited']) {
            $time_remaining = NexlifyDesk_Rate_Limiter::format_time_remaining($status['time_until_reset']);
            $message = sprintf(
                /* translators: 1: email address, 2: current count, 3: max allowed, 4: time remaining */
                __('Rate limit status for %1$s: %2$d/%3$d emails sent. Rate limit will reset in %4$s.', 'nexlifydesk'),
                esc_html($email_address),
                $status['current_count'],
                $status['max_allowed'],
                $time_remaining
            );
        } else {
            $message = sprintf(
                /* translators: 1: email address, 2: current count, 3: max allowed */
                __('Rate limit status for %1$s: %2$d/%3$d emails sent. No rate limit active.', 'nexlifydesk'),
                esc_html($email_address),
                $status['current_count'],
                $status['max_allowed']
            );
        }

        wp_send_json_success($message);
    }
    
    /**
     * Get the ticket page URL from settings
     * 
     * @return string The ticket page URL or empty string if not set
     */
    public static function get_ticket_page_url() {
        $settings = get_option('nexlifydesk_settings', array());
        $ticket_page_id = isset($settings['ticket_page_id']) ? (int)$settings['ticket_page_id'] : 0;
        
        if ($ticket_page_id > 0) {
            $ticket_page_url = get_permalink($ticket_page_id);
            return $ticket_page_url ? $ticket_page_url : '';
        }
        
        return '';
    }
    
    public static function render_support_page() {
        if (isset($_POST['submit']) && isset($_POST['nexlify_support_nonce'])) {
            if (wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nexlify_support_nonce'])), 'nexlify_support_form')) {
                $subject = isset($_POST['support_subject']) ? sanitize_text_field(wp_unslash($_POST['support_subject'])) : '';
                $message = isset($_POST['support_message']) ? sanitize_textarea_field(wp_unslash($_POST['support_message'])) : '';
                $admin_email = get_option('admin_email');

                if (!empty($subject) && !empty($message)) {
                    $email_data = array(
                        'plugin_name'    => 'NexlifyDesk',
                        'plugin_version' => NEXLIFYDESK_VERSION,
                        'subject'        => $subject,
                        'message'        => $message,
                        'customer_email' => $admin_email,
                        'site_url'       => home_url(),
                    );
                    
                    if (function_exists('nexlifydesk_send_support_email')) {
                        $sent = nexlifydesk_send_support_email($email_data);
                        if ($sent) {
                            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Support request sent successfully!', 'nexlifydesk') . '</p></div>';
                        } else {
                            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Failed to send support request. Please ensure your WordPress site can send emails.', 'nexlifydesk') . '</p></div>';
                        }
                    } else {
                        echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Error: Support email function not found.', 'nexlifydesk') . '</p></div>';
                    }
                } else {
                     echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Subject and Message are required fields.', 'nexlifydesk') . '</p></div>';
                }
            } else {
                echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Security check failed. Please try again.', 'nexlifydesk') . '</p></div>';
            }
        }

        if (function_exists('nexlifydesk_display_support_form')) {
            nexlifydesk_display_support_form('NexlifyDesk', NEXLIFYDESK_VERSION);
        } else {
            echo '<div class="wrap"><h1>' . esc_html__('Support', 'nexlifydesk') . '</h1><p>' . esc_html__('Error: Support form display function not found.', 'nexlifydesk') . '</p></div>';
        }
    }

    public static function enqueue_admin_assets($hook) {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Only reading $_GET for asset loading, not processing sensitive data.
        $page = isset($_GET['page']) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Only reading $_GET for asset loading, not processing sensitive data.
        $ticket_id = isset($_GET['ticket_id']) ? sanitize_text_field( wp_unslash( $_GET['ticket_id'] ) ) : '';

        if (
            ($page && strpos($page, 'nexlifydesk') === 0) ||
            (strpos($hook, 'nexlifydesk') !== false)
        ) {
            wp_enqueue_style(
                'nexlifydesk-admin',
                NEXLIFYDESK_PLUGIN_URL . 'assets/css/nexlifydesk-admin.css',
                array(),
                NEXLIFYDESK_VERSION
            );
            wp_enqueue_script(
                'nexlifydesk-admin',
                NEXLIFYDESK_PLUGIN_URL . 'assets/js/nexlifydesk.js',
                array('jquery'),
                NEXLIFYDESK_VERSION,
                true
            );
            
            // Enqueue page-specific scripts
            if ($page === 'nexlifydesk_tickets') {
                wp_enqueue_script(
                    'nexlifydesk-admin-ticket-list',
                    NEXLIFYDESK_PLUGIN_URL . 'assets/js/admin-ticket-list.js',
                    array('jquery', 'nexlifydesk-admin'),
                    NEXLIFYDESK_VERSION,
                    true
                );
            }

            $available_capabilities = array(
                NEXLIFYDESK_CAP_VIEW_ALL_TICKETS => __('View All Tickets', 'nexlifydesk'),
                NEXLIFYDESK_CAP_ASSIGN_TICKETS => __('Assign Tickets', 'nexlifydesk'),
                NEXLIFYDESK_CAP_MANAGE_CATEGORIES => __('Manage Categories', 'nexlifydesk'),
                NEXLIFYDESK_CAP_VIEW_REPORTS => __('View Reports', 'nexlifydesk'),
            );

            wp_localize_script(
                'nexlifydesk-admin',
                'nexlifydesk_admin_vars',
                array(
                    'ajaxurl' => admin_url('admin-ajax.php'),
                    'nonce' => wp_create_nonce('nexlifydesk-ajax-nonce'),
                    'position_nonce' => wp_create_nonce('nexlifydesk_save_agent_position'),
                    'admin_post_url' => admin_url('admin-post.php'),
                    'available_capabilities' => $available_capabilities,
                    'position_name_label' => __('Position Name:', 'nexlifydesk'),
                    'position_slug_label' => __('Position Slug (unique):', 'nexlifydesk'),
                    'assign_capabilities_label' => __('Assign Capabilities:', 'nexlifydesk'),
                    'save_text' => __('Save', 'nexlifydesk'),
                    'cancel_text' => __('Cancel', 'nexlifydesk'),
                    'add_new_text' => __('Add New', 'nexlifydesk'),
                    'adding_text' => __('Adding...', 'nexlifydesk'),
                    'add_category_text' => __('Add Category', 'nexlifydesk'),
                    'error_occurred_text' => __('An error occurred. Please try again.', 'nexlifydesk'),
                    'required_fields_text' => __('Please fill in all required fields.', 'nexlifydesk'),
                    'delete_confirm' => __('Are you sure you want to delete this position?', 'nexlifydesk'),
                    'loading_tickets_text' => __('Loading tickets...', 'nexlifydesk'),
                    'no_tickets_found_text' => __('No tickets found.', 'nexlifydesk'),
                    'error_loading_tickets_text' => __('Error loading tickets:', 'nexlifydesk'),
                    'ajax_error_loading_tickets_text' => __('AJAX Error loading tickets.', 'nexlifydesk'),
                    'ticket_id_header' => __('Ticket ID', 'nexlifydesk'),
                    'subject_header' => __('Subject', 'nexlifydesk'),
                    'status_header' => __('Status', 'nexlifydesk'),
                    'created_header' => __('Created', 'nexlifydesk'),
                    'actions_header' => __('Actions', 'nexlifydesk'),
                    'view_text' => __('View', 'nexlifydesk'),
                    'assigned_to_header' => __('Assigned To', 'nexlifydesk'),
                    'agent_assigned_text' => __('Ticket assigned successfully!', 'nexlifydesk'),
                    'template_update_nonce' => wp_create_nonce('nexlifydesk_update_templates'),
                    'template_dismiss_nonce' => wp_create_nonce('nexlifydesk_dismiss_notice'),
                    'template_update_confirm' => __('This will clear your current email templates and use the new professional template files. You can customize them later in Email Templates settings. Continue?', 'nexlifydesk'),
                    'template_update_success' => __('Email templates updated successfully! Check the Email Templates page to see the changes.', 'nexlifydesk'),
                    'template_update_error' => __('Error updating templates. Please try again.', 'nexlifydesk'),
                )
            );
            
            if ($page === 'nexlifydesk_reports' && class_exists('NexlifyDesk_Reports')) {
                $stats = NexlifyDesk_Reports::get_dashboard_stats();
                wp_localize_script(
                    'nexlifydesk-admin',
                    'nexlifydesk_reports_data',
                    array(
                        'status_data' => array_values($stats['status_breakdown']),
                        'status_labels' => array_keys($stats['status_breakdown']),
                        'priority_data' => array_values($stats['priority_breakdown']),
                        'priority_labels' => array_keys($stats['priority_breakdown']),
                        'monthly_data' => $stats['monthly_data'],
                    )
                );
            }
            
            if ($page === 'nexlifydesk_imap_auth' || $page === 'nexlifydesk_imap_settings') {
                $is_ssl_enabled = function_exists('nexlifydesk_check_ssl_enabled') ? nexlifydesk_check_ssl_enabled() : true;
                
                if (defined('NEXLIFYDESK_FORCE_SSL_ENABLED') && NEXLIFYDESK_FORCE_SSL_ENABLED) {
                    $is_ssl_enabled = true;
                }
                
                wp_add_inline_script(
                    'nexlifydesk-admin',
                    'window.nexlifydesk_ssl_enabled = ' . wp_json_encode($is_ssl_enabled) . ';' .
                    'window.nexlifydesk_fetch_emails_nonce = ' . wp_json_encode(wp_create_nonce('nexlifydesk_fetch_emails_now')) . ';' .
                    'window.nexlifydesk_aws_test_nonce = ' . wp_json_encode(wp_create_nonce('nexlifydesk_aws_test')) . ';' .
                    'window.nexlifydesk_google_test_nonce = ' . wp_json_encode(wp_create_nonce('nexlifydesk-ajax-nonce')) . ';' .
                    'window.nexlifydesk_custom_test_nonce = ' . wp_json_encode(wp_create_nonce('nexlifydesk-ajax-nonce')) . ';' .
                    'window.nexlifydesk_generate_key_nonce = ' . wp_json_encode(wp_create_nonce('nexlifydesk_generate_key')) . ';',
                    'before'
                );
            }
            
            if ($page === 'nexlifydesk_tickets') {
                wp_add_inline_script(
                    'nexlifydesk-admin',
                    'window.nexlifydesk_tickets_list_texts = ' . wp_json_encode(array(
                        'no_tickets_found' => __('No tickets found.', 'nexlifydesk'),
                        'loading_tickets' => __('Loading tickets...', 'nexlifydesk'),
                        'error_loading_tickets' => __('Error loading tickets.', 'nexlifydesk'),
                        'ticket_id' => __('Ticket ID', 'nexlifydesk'),
                        'subject' => __('Subject', 'nexlifydesk'),
                        'status' => __('Status', 'nexlifydesk'),
                        'assigned_to' => __('Assigned To', 'nexlifydesk'),
                        'created' => __('Created', 'nexlifydesk'),
                        'actions' => __('Actions', 'nexlifydesk'),
                        'view' => __('View', 'nexlifydesk'),
                    )) . ';',
                    'before'
                );
            }
        }
    }

    public static function render_ticket_list_page() {
    if (!current_user_can('nexlifydesk_manage_tickets')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'nexlifydesk'));
    }

    $ticket_id = 0;
    
    if (isset($_GET['ticket_id']) && !isset($_POST['action'])) {
        $ticket_id = absint($_GET['ticket_id']);
        
        if (!current_user_can('nexlifydesk_view_all_tickets')) {
            $ticket = NexlifyDesk_Tickets::get_ticket($ticket_id);
            $current_user = wp_get_current_user();
            $is_agent = in_array('nexlifydesk_agent', $current_user->roles) && !in_array('administrator', $current_user->roles);
            
            if (!$ticket) {
                wp_die(esc_html__('Ticket not found.', 'nexlifydesk'));
            }
            
            if ($is_agent) {
                $can_view_all = NexlifyDesk_Users::agent_can('nexlifydesk_view_all_tickets', get_current_user_id());
                if (!$can_view_all && (int)$ticket->assigned_to !== get_current_user_id()) {
                    wp_die(esc_html__('You do not have permission to view this ticket.', 'nexlifydesk'));
                }
            } elseif (!$is_agent && (int)$ticket->user_id !== get_current_user_id()) {
                wp_die(esc_html__('You do not have permission to view this ticket.', 'nexlifydesk'));
            }
        }
        
        if ($ticket_id > 0) {
            $ticket = NexlifyDesk_Tickets::get_ticket($ticket_id);
            if (!$ticket) {
                wp_die(esc_html__('Ticket not found.', 'nexlifydesk'));
            }
        }
    }
    
    if (isset($_POST['action']) || isset($_GET['s']) || isset($_GET['status'])) {
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'nexlifydesk_tickets_action')) {
            wp_die(esc_html__('Security check failed. Please try again.', 'nexlifydesk'));
        }
    }

    ?>
    <div class="wrap">
        <?php if ($ticket_id > 0) : ?>
            <?php
            $ticket = NexlifyDesk_Tickets::get_ticket($ticket_id);
            if ($ticket) {
                if (method_exists('NexlifyDesk_Tickets', 'mark_ticket_read')) {
                    NexlifyDesk_Tickets::mark_ticket_read($ticket_id);
                }
                include NEXLIFYDESK_PLUGIN_DIR . 'templates/admin/ticket-single.php';
            } else {
                echo '<p>' . esc_html__('Ticket not found.', 'nexlifydesk') . '</p>';
            }
            ?>
        <?php else : ?>
            <?php include NEXLIFYDESK_PLUGIN_DIR . 'templates/admin/tickets-list.php'; ?>
        <?php endif; ?>
    </div>
    <?php
}

    public static function render_categories_page() {

        if (!NexlifyDesk_Users::agent_can('nexlifydesk_manage_categories')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'nexlifydesk'));
        }

        $cache_key = 'nexlifydesk_categories_admin_page';
        $categories = wp_cache_get($cache_key);

        if (false === $categories) {
            global $wpdb;
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
            $categories = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT * FROM `{$wpdb->prefix}nexlifydesk_categories` WHERE is_active = %d", 
                    1
                )
            );
            wp_cache_set($cache_key, $categories, '', 600);
        }
    ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Ticket Categories', 'nexlifydesk'); ?> <a href="#add-category" class="page-title-action"><?php esc_html_e('Add New', 'nexlifydesk'); ?></a></h1>
            <form id="nexlifydesk-category-form" method="post" action="" style="display: none;">
                <?php wp_nonce_field('nexlifydesk_save_category', 'nexlifydesk_category_nonce'); ?>
                <table class="form-table">
                    <tr>
                        <th><label for="category_name"><?php esc_html_e('Category Name', 'nexlifydesk'); ?></label></th>
                        <td><input type="text" name="category_name" id="category_name" class="regular-text" required></td>
                    </tr>
                    <tr>
                        <th><label for="category_description"><?php esc_html_e('Description', 'nexlifydesk'); ?></label></th>
                        <td><textarea name="category_description" id="category_description" rows="4" cols="50"></textarea></td>
                    </tr>
                </table>
                <p class="submit">
                    <input type="submit" name="submit_category" class="button button-primary" value="<?php esc_html_e('Add Category', 'nexlifydesk'); ?>">
                </p>
            </form>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php esc_html_e('Name', 'nexlifydesk'); ?></th>
                        <th><?php esc_html_e('Slug', 'nexlifydesk'); ?></th>
                        <th><?php esc_html_e('Description', 'nexlifydesk'); ?></th>
                        <th><?php esc_html_e('Actions', 'nexlifydesk'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($categories as $category): ?>
                        <tr>
                            <td><?php echo esc_html($category->name); ?></td>
                            <td><?php echo esc_html($category->slug); ?></td>
                            <td><?php echo esc_html($category->description); ?></td>
                            <td>
                                <a href="#" class="delete-category" data-id="<?php echo esc_attr($category->id); ?>"><?php esc_html_e('Delete', 'nexlifydesk'); ?></a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    public static function render_settings_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('NexlifyDesk Settings', 'nexlifydesk'); ?></h1>
            <?php include NEXLIFYDESK_PLUGIN_DIR . 'templates/admin/settings.php'; ?>
        </div>
        <?php
    }

    public static function render_reports_page() {

        if (!NexlifyDesk_Users::agent_can('nexlifydesk_view_reports')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'nexlifydesk'));
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('NexlifyDesk Reports', 'nexlifydesk'); ?></h1>
            <?php include NEXLIFYDESK_PLUGIN_DIR . 'templates/admin/reports.php'; ?>
        </div>
        <?php
    }

    public static function render_agent_positions_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('NexlifyDesk Agent Positions', 'nexlifydesk'); ?></h1>
            <?php settings_errors('nexlifydesk_agent_positions'); ?>
            <?php self::display_agent_positions_management(); ?>
        </div>
        <?php
    }

    private static function handle_agent_positions_save() {
    if (isset($_POST['add_position']) && current_user_can('manage_options')) {
        if (
            !isset($_POST['nexlifydesk_agent_position_nonce']) || 
            !wp_verify_nonce(
                sanitize_text_field(wp_unslash($_POST['nexlifydesk_agent_position_nonce'])),
                'nexlifydesk_save_agent_position'
            )
        ) {
            wp_die(esc_html__('Security check failed.', 'nexlifydesk'));
        }

        $position_name = isset($_POST['position_name']) 
            ? sanitize_text_field(wp_unslash($_POST['position_name'])) 
            : '';

        $position_slug = isset($_POST['position_slug']) 
            ? sanitize_title(wp_unslash($_POST['position_slug']))
            : '';

        $position_capabilities = isset($_POST['position_capabilities']) 
            ? array_map('sanitize_text_field', wp_unslash((array)$_POST['position_capabilities']))
            : array();

        if (empty($position_name) || empty($position_slug)) {
            add_settings_error(
                'nexlifydesk_agent_positions',
                'empty_fields',
                esc_html__('Position Name and Slug are required.', 'nexlifydesk'),
                'error'
            );
            return;
        }

        $positions = get_option('nexlifydesk_agent_positions', array());

        if (isset($positions[$position_slug])) {
            add_settings_error('nexlifydesk_agent_positions', 'duplicate_slug', __('A position with this slug already exists.', 'nexlifydesk'), 'error');
            return;
        }

        $positions[$position_slug] = array(
            'name' => $position_name,
            'capabilities' => $position_capabilities,
        );

        update_option('nexlifydesk_agent_positions', $positions);

        add_settings_error('nexlifydesk_agent_positions', 'position_added', __('Agent position added successfully.', 'nexlifydesk'), 'success');

        $redirect_url = add_query_arg(array(
            'page' => 'nexlifydesk_agent_positions',
            'settings-updated' => 'true'
        ), admin_url('admin.php'));
        wp_redirect($redirect_url);
        exit;
    }

    if (isset($_POST['delete_position'], $_POST['position_slug']) && current_user_can('manage_options')) {
        $nonce = isset($_POST['nexlifydesk_agent_position_nonce']) 
            ? sanitize_text_field(wp_unslash($_POST['nexlifydesk_agent_position_nonce']))
            : '';
        
        if (!wp_verify_nonce($nonce, 'nexlifydesk_save_agent_position')) {
            wp_die(esc_html__('Security check failed.', 'nexlifydesk'));
        }

        $slug = sanitize_title(wp_unslash($_POST['position_slug']));
        
        $positions = get_option('nexlifydesk_agent_positions', array());
        if (isset($positions[$slug])) {
            unset($positions[$slug]);
            update_option('nexlifydesk_agent_positions', $positions);
            add_settings_error(
                'nexlifydesk_agent_positions', 
                'position_deleted', 
                esc_html__('Agent position deleted.', 'nexlifydesk'), 
                'success'
            );
        } else {
            add_settings_error(
                'nexlifydesk_agent_positions', 
                'position_not_found', 
                esc_html__('Position not found.', 'nexlifydesk'), 
                'error'
            );
        }

        $redirect_url = add_query_arg(array(
            'page' => 'nexlifydesk_agent_positions',
            'settings-updated' => 'true'
        ), admin_url('admin.php'));
        wp_redirect($redirect_url);
        exit;
    }

    if (isset($_POST['edit_position'], $_POST['original_slug']) && current_user_can('manage_options')) {
        $nonce = isset($_POST['nexlifydesk_agent_position_nonce']) 
            ? sanitize_text_field(wp_unslash($_POST['nexlifydesk_agent_position_nonce']))
            : '';
        
        if (!wp_verify_nonce($nonce, 'nexlifydesk_save_agent_position')) {
            wp_die(esc_html__('Security check failed.', 'nexlifydesk'));
        }

        $original_slug = sanitize_title(wp_unslash($_POST['original_slug']));
        $position_name = sanitize_text_field(wp_unslash($_POST['position_name']));
        $position_slug = sanitize_title(wp_unslash($_POST['position_slug']));
        $position_capabilities = isset($_POST['position_capabilities']) 
            ? array_map('sanitize_text_field', wp_unslash((array)$_POST['position_capabilities']))
            : array();

        $positions = get_option('nexlifydesk_agent_positions', array());

        if ($original_slug !== $position_slug && isset($positions[$original_slug])) {
            unset($positions[$original_slug]);
        }

        $positions[$position_slug] = array(
            'name' => $position_name,
            'capabilities' => $position_capabilities,
        );

        update_option('nexlifydesk_agent_positions', $positions);
        add_settings_error('nexlifydesk_agent_positions', 'position_edited', __('Agent position updated.', 'nexlifydesk'), 'success');

        $redirect_url = add_query_arg(array(
            'page' => 'nexlifydesk_agent_positions',
            'settings-updated' => 'true'
        ), admin_url('admin.php'));
        wp_redirect($redirect_url);
        exit;
    }
}

    public static function handle_save_agent_position() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'nexlifydesk'));
        }

        if (!isset($_POST['nexlifydesk_agent_position_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nexlifydesk_agent_position_nonce'])), 'nexlifydesk_save_agent_position')) {
            wp_die(esc_html__('Security check failed.', 'nexlifydesk'));
        }

        $positions = get_option('nexlifydesk_agent_positions', array());

        if (isset($_POST['add_position'])) {
            $position_name = isset($_POST['position_name']) ? sanitize_text_field(wp_unslash($_POST['position_name'])) : '';
            $position_slug = isset($_POST['position_slug']) ? sanitize_title(wp_unslash($_POST['position_slug'])) : '';
            $position_capabilities = isset($_POST['position_capabilities']) ? array_map('sanitize_text_field', wp_unslash((array)$_POST['position_capabilities'])) : array();

            if (empty($position_name) || empty($position_slug)) {
                add_settings_error('nexlifydesk_agent_positions', 'empty_fields', __('Position Name and Slug are required.', 'nexlifydesk'), 'error');
            } elseif (isset($positions[$position_slug])) {
                add_settings_error('nexlifydesk_agent_positions', 'duplicate_slug', __('A position with this slug already exists.', 'nexlifydesk'), 'error');
            } else {
                $positions[$position_slug] = array(
                    'name' => $position_name,
                    'capabilities' => $position_capabilities,
                );
                update_option('nexlifydesk_agent_positions', $positions);
                add_settings_error('nexlifydesk_agent_positions', 'position_added', __('Agent position added successfully.', 'nexlifydesk'), 'success');
            }
        } elseif (isset($_POST['edit_position'])) {
            $position_slug = isset($_POST['position_slug']) ? sanitize_title(wp_unslash($_POST['position_slug'])) : '';
            $position_name = isset($_POST['position_name']) ? sanitize_text_field(wp_unslash($_POST['position_name'])) : '';
            $position_slug = isset($_POST['position_slug']) ? sanitize_title(wp_unslash($_POST['position_slug'])) : '';
            $position_capabilities = isset($_POST['position_capabilities']) ? array_map('sanitize_text_field', wp_unslash((array)$_POST['position_capabilities'])) : array();

            if (isset($positions[$original_slug])) {
                unset($positions[$original_slug]);
            }
            $positions[$position_slug] = array(
                'name' => $position_name,
                'capabilities' => $position_capabilities,
            );
            update_option('nexlifydesk_agent_positions', $positions);
            add_settings_error('nexlifydesk_agent_positions', 'position_edited', __('Agent position updated.', 'nexlifydesk'), 'success');
        }

        wp_redirect(add_query_arg(array(
            'page' => 'nexlifydesk_agent_positions',
            'settings-updated' => 'true'
        ), admin_url('admin.php')));
        exit;
    }

    public static function handle_delete_agent_position() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'nexlifydesk'));
        }

        if (!isset($_POST['nexlifydesk_agent_position_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nexlifydesk_agent_position_nonce'])), 'nexlifydesk_save_agent_position')) {
            wp_die(esc_html__('Security check failed.', 'nexlifydesk'));
        }

        $slug = isset($_POST['position_slug']) ? sanitize_title(wp_unslash($_POST['position_slug'])) : '';
        $positions = get_option('nexlifydesk_agent_positions', array());
        if (isset($positions[$slug])) {
            unset($positions[$slug]);
            update_option('nexlifydesk_agent_positions', $positions);
            add_settings_error('nexlifydesk_agent_positions', 'position_deleted', __('Agent position deleted.', 'nexlifydesk'), 'success');
        } else {
            add_settings_error('nexlifydesk_agent_positions', 'position_not_found', __('Position not found.', 'nexlifydesk'), 'error');
        }

        wp_redirect(add_query_arg(array(
            'page' => 'nexlifydesk_agent_positions',
            'settings-updated' => 'true'
        ), admin_url('admin.php')));
        exit;
    }

    private static function display_agent_positions_management() {
        $positions = get_option('nexlifydesk_agent_positions', array());
        $available_capabilities = array(
            NEXLIFYDESK_CAP_VIEW_ALL_TICKETS => __('View All Tickets', 'nexlifydesk'),
            NEXLIFYDESK_CAP_ASSIGN_TICKETS => __('Assign Tickets', 'nexlifydesk'),
            NEXLIFYDESK_CAP_MANAGE_CATEGORIES => __('Manage Categories', 'nexlifydesk'),
            NEXLIFYDESK_CAP_VIEW_REPORTS => __('View Reports', 'nexlifydesk'),
        );

        echo '<h3>' . esc_html__('Existing Positions', 'nexlifydesk') . '</h3>';
        if (!empty($positions)) {
            echo '<table class="wp-list-table widefat fixed striped">';
            echo '<thead><tr><th>' . esc_html__('Name', 'nexlifydesk') . '</th><th>' . esc_html__('Slug', 'nexlifydesk') . '</th><th>' . esc_html__('Capabilities', 'nexlifydesk') . '</th><th>' . esc_html__('Actions', 'nexlifydesk') . '</th></tr></thead>';
            echo '<tbody>';
            foreach ($positions as $slug => $position) {
                $assigned_caps = isset($position['capabilities']) ? implode(',', $position['capabilities']) : '';
                echo '<tr data-capabilities="' . esc_attr($assigned_caps) . '">';
                echo '<td><strong>' . esc_html($position['name']) . '</strong></td>';
                echo '<td>' . esc_html($slug) . '</td>';
                echo '<td>';
                if (!empty($position['capabilities'])) {
                    $cap_names = array_map(function($cap_slug) use ($available_capabilities) {
                        return $available_capabilities[$cap_slug] ?? $cap_slug;
                    }, $position['capabilities']);
                    echo esc_html(implode(', ', $cap_names));
                } else {
                    esc_html_e('None', 'nexlifydesk');
                }
                echo '</td>';
                echo '<td>';
                echo '<button class="button button-small edit-position" data-slug="' . esc_attr($slug) . '">' . esc_html__('Edit', 'nexlifydesk') . '</button> ';
                echo '<button class="button button-small button-danger delete-position" data-slug="' . esc_attr($slug) . '">' . esc_html__('Delete', 'nexlifydesk') . '</button>';
                echo '</td>';
                echo '</tr>';
            }
            echo '</tbody>';
            echo '</table>';
        } else {
            echo '<p>' . esc_html__('No agent positions defined yet.', 'nexlifydesk') . '</p>';
        }

        echo '<h3>' . esc_html__('Add New Position', 'nexlifydesk') . '</h3>';
        echo '<h3>' . esc_html__('Add New Position', 'nexlifydesk') . '</h3>';
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
        echo '<input type="hidden" name="action" value="nexlifydesk_save_agent_position">';
        wp_nonce_field('nexlifydesk_save_agent_position', 'nexlifydesk_agent_position_nonce');
        echo '<table class="form-table">';
        echo '<tr><th><label for="position_name">' . esc_html__('Position Name:', 'nexlifydesk') . '</label></th><td><input type="text" name="position_name" id="position_name" class="regular-text" required></td></tr>';
        echo '<tr><th><label for="position_slug">' . esc_html__('Position Slug (unique):', 'nexlifydesk') . '</label></th><td><input type="text" name="position_slug" id="position_slug" class="regular-text" required></td></tr>';
        echo '<tr><th><strong>' . esc_html__('Assign Capabilities:', 'nexlifydesk') . '</strong></th><td><ul>';
        foreach ($available_capabilities as $cap_slug => $cap_name) {
            echo '<li><label><input type="checkbox" name="position_capabilities[]" value="' . esc_attr($cap_slug) . '"> ' . esc_html($cap_name) . '</label></li>';
        }
        echo '</ul></td></tr>';
        echo '</table>';
        submit_button(__('Add Position', 'nexlifydesk'), 'primary', 'add_position', false);
        echo '</form>';

        echo '<hr>';
        echo '<h3>' . esc_html__('Assign Positions to Agents', 'nexlifydesk') . '</h3>';
        self::display_agent_assignment_management($positions);
    }

    private static function display_agent_assignment_management($positions) {
        settings_errors('nexlifydesk_agent_assignments');

        $agents = get_users(array('role' => 'nexlifydesk_agent', 'orderby' => 'display_name'));

        if (empty($agents)) {
            echo '<p>' . esc_html__('No NexlifyDesk Agents found.', 'nexlifydesk') . '</p>';
            return;
        }

        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
        echo '<input type="hidden" name="action" value="nexlifydesk_save_agent_assignments">';
        wp_nonce_field('nexlifydesk_save_agent_assignments', 'nexlifydesk_agent_assignments_nonce');

        echo '<table class="wp-list-table widefat fixed striped">';
        echo '<thead><tr><th>' . esc_html__('Agent Name', 'nexlifydesk') . '</th><th>' . esc_html__('Current Position', 'nexlifydesk') . '</th><th>' . esc_html__('Assign Position', 'nexlifydesk') . '</th></tr></thead>';
        echo '<tbody>';

        foreach ($agents as $agent) {
            $current_position_slug = get_user_meta($agent->ID, 'nexlifydesk_agent_position', true);
            $current_position_name = $positions[$current_position_slug]['name'] ?? __('None', 'nexlifydesk');

            echo '<tr>';
            echo '<td>' . esc_html($agent->display_name) . '</td>';
            echo '<td>' . esc_html($current_position_name) . '</td>';
            echo '<td>';
            echo '<select name="agent_positions[' . esc_attr($agent->ID) . ']">';
            echo '<option value="">' . esc_html__('None', 'nexlifydesk') . '</option>';
            foreach ($positions as $slug => $position) {
                echo '<option value="' . esc_attr($slug) . '" ' . selected($current_position_slug, $slug, false) . '>' . esc_html($position['name']),
                '</option>';
            }
            echo '</select>';
            echo '</td>';
            echo '</tr>';
        }

        echo '</tbody>';
        echo '</table>';
        submit_button(__('Save Agent Assignments', 'nexlifydesk'), 'primary', 'save_assignments', false);
        echo '</form>';
    }

    private static function handle_agent_assignments_save() {

        if (isset($_POST['save_assignments']) && current_user_can('manage_options')) {
            if (
                !isset($_POST['nexlifydesk_agent_assignments_nonce']) || 
                !wp_verify_nonce(
                    sanitize_text_field(wp_unslash($_POST['nexlifydesk_agent_assignments_nonce'])),
                    'nexlifydesk_save_agent_assignments'
                )
            ) {
                wp_die(esc_html__('Security check failed.', 'nexlifydesk'));
            }

            $agent_positions = isset($_POST['agent_positions']) 
                ? array_map('sanitize_text_field', wp_unslash((array)$_POST['agent_positions'])) 
                : array();

            if (!empty($agent_positions)) {
                foreach ($agent_positions as $user_id => $position_slug) {
                    $user_id = absint($user_id);
                    $position_slug = sanitize_text_field($position_slug);

                    $user = get_userdata($user_id);
                    if ($user && in_array('nexlifydesk_agent', $user->roles)) {
                        update_user_meta($user_id, 'nexlifydesk_agent_position', $position_slug);
                    }
                }
                add_settings_error('nexlifydesk_agent_assignments', 'assignments_saved', __('Agent assignments saved successfully.', 'nexlifydesk'), 'success');
            } else {
                 add_settings_error('nexlifydesk_agent_assignments', 'no_assignments', __('No agent assignments to save.', 'nexlifydesk'), 'warning');
            }


            $redirect_url = add_query_arg(array(
                'page' => 'nexlifydesk_agent_positions',
                'settings-updated' => 'true'
            ), admin_url('admin.php'));
            if (function_exists('ob_end_clean')) { @ob_end_clean(); }
            wp_redirect($redirect_url);
            exit;
        }
    }

    public static function handle_save_agent_assignments() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'nexlifydesk'));
        }

        if (!isset($_POST['nexlifydesk_agent_assignments_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nexlifydesk_agent_assignments_nonce'])), 'nexlifydesk_save_agent_assignments')) {
            wp_die(esc_html__('Security check failed.', 'nexlifydesk'));
        }

        if (isset($_POST['save_assignments'])) {
            $agent_positions = isset($_POST['agent_positions']) ? array_map('sanitize_text_field', wp_unslash((array)$_POST['agent_positions'])) : array();

            if (!empty($agent_positions)) {
                foreach ($agent_positions as $user_id => $position_slug) {
                    $user_id = absint($user_id);
                    $position_slug = sanitize_text_field($position_slug);

                    $user = get_userdata($user_id);
                    if ($user && in_array('nexlifydesk_agent', $user->roles)) {
                        update_user_meta($user_id, 'nexlifydesk_agent_position', $position_slug);
                    }
                }
                add_settings_error('nexlifydesk_agent_assignments', 'assignments_saved', __('Agent assignments saved successfully.', 'nexlifydesk'), 'success');
            } else {
                add_settings_error('nexlifydesk_agent_assignments', 'no_assignments', __('No agent assignments to save.', 'nexlifydesk'), 'warning');
            }
        }

        wp_redirect(add_query_arg(array(
            'page' => 'nexlifydesk_agent_positions',
            'settings-updated' => 'true'
        ), admin_url('admin.php')));
        exit;
    }

    public static function display_admin_notices() {
        if (isset($_GET['settings-updated']) && $_GET['settings-updated'] === 'true') {
            ?>
            <div class="notice notice-success is-dismissible">
                <p><?php esc_html_e('Settings saved successfully!', 'nexlifydesk'); ?></p>
            </div>
            <?php
        }
    
        if (isset($_GET['category_added']) && $_GET['category_added'] === 'true') {
            $nonce = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : '';
            
            if (!wp_verify_nonce($nonce, 'nexlifydesk_category_nonce')) {
                return;
            }   
            ?>
            <div class="notice notice-success is-dismissible">
                <p><?php esc_html_e('Category added successfully!', 'nexlifydesk'); ?></p>
            </div>
            <?php
        }
    }

    public static function handle_settings_save() {
        if (isset($_POST['nexlifydesk_settings_save']) && check_admin_referer('nexlifydesk_save_settings', 'nexlifydesk_settings_nonce')) {
            $settings = array(
                'email_notifications' => isset($_POST['email_notifications']) ? 1 : 0,
                'admin_email_notifications' => isset($_POST['admin_email_notifications']) ? 1 : 0,
                'disable_email_notifications_for_email_tickets' => isset($_POST['disable_email_notifications_for_email_tickets']) ? 1 : 0,
                'default_priority' => isset($_POST['default_priority']) ? sanitize_text_field(wp_unslash($_POST['default_priority'])) : '',
                'auto_assign' => isset($_POST['auto_assign']) ? 1 : 0,
                'auto_assign_to_admin' => isset($_POST['auto_assign_to_admin']) ? 1 : 0,
                'allowed_file_types' => isset($_POST['allowed_file_types']) ? self::validate_file_types(sanitize_text_field(wp_unslash($_POST['allowed_file_types']))) : 'jpg,jpeg,png,pdf',
                'max_file_size' => isset($_POST['max_file_size']) ? absint($_POST['max_file_size']) : 0,
                'default_category' => isset($_POST['default_category']) ? absint($_POST['default_category']) : 0,
                'sla_response_time' => isset($_POST['sla_response_time']) ? absint($_POST['sla_response_time']) : 0,
                'ticket_page_id' => isset($_POST['ticket_page_id']) && is_array($_POST['ticket_page_id']) ? array_map('absint', wp_unslash($_POST['ticket_page_id'])) : array(),
                'ticket_form_page_id' => isset($_POST['ticket_form_page_id']) && is_array($_POST['ticket_form_page_id']) ? array_map('absint', wp_unslash($_POST['ticket_form_page_id'])) : array(),
                'ticket_id_prefix' => isset($_POST['ticket_id_prefix']) ? sanitize_text_field(wp_unslash($_POST['ticket_id_prefix'])) : '',
                'ticket_id_start' => isset($_POST['ticket_id_start']) ? absint($_POST['ticket_id_start']) : 0,
                'status_change_notification' => isset($_POST['status_change_notification']) ? 1 : 0,
                'keep_data_on_uninstall' => isset($_POST['keep_data_on_uninstall']) ? 1 : 0,
                'check_duplicates' => isset($_POST['check_duplicates']) ? 1 : 0,
                'duplicate_threshold' => isset($_POST['duplicate_threshold']) ? absint($_POST['duplicate_threshold']) : 80,
            );

            update_option('nexlifydesk_settings', $settings);
            wp_redirect(add_query_arg(array('page' => 'nexlifydesk_settings', 'settings-updated' => 'true'), admin_url('admin.php')));
            exit;
        }

        if (isset($_POST['submit_category']) && check_admin_referer('nexlifydesk_save_category', 'nexlifydesk_category_nonce')) {
            global $wpdb;

            if (empty($_POST['category_name'])) {
                add_action('admin_notices', function() {
                    echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Category name is required.', 'nexlifydesk') . '</p></div>';
                });
                return;
            }

            $category_name = sanitize_text_field(wp_unslash($_POST['category_name']));
            $category_description = isset($_POST['category_description']) ? sanitize_textarea_field(wp_unslash($_POST['category_description'])) : '';
            $slug = sanitize_title($category_name);

            $table_name = NexlifyDesk_Database::get_table('categories');
            if (empty($table_name) || strpos($table_name, $wpdb->prefix) !== 0) {
                add_action('admin_notices', function() {
                    echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('Invalid database table. Please contact support.', 'nexlifydesk') . '</p></div>';
                });
                return;
            }

            $cache_key = 'nexlifydesk_category_slug_check_admin_add_' . md5($slug . '_' . get_current_user_id());
            wp_cache_delete($cache_key);

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is safe and controlled, query is prepared, no caching needed for real-time category existence check to prevent race conditions
            $existing = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT id, is_active FROM `" . esc_sql($table_name) . "` WHERE slug = %s",
                    $slug
                )
            );

            if ($existing) {
                if ($existing->is_active == 1) {
                    add_action('admin_notices', function() {
                        echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__('A category with this name already exists.', 'nexlifydesk') . '</p></div>';
                    });
                    return;
                } else {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
                    $result = $wpdb->update(
                        $table_name,
                        array(
                            'name' => $category_name,
                            'description' => $category_description,
                            'is_active' => 1
                        ),
                        array('id' => $existing->id),
                        array('%s', '%s', '%d'),
                        array('%d')
                    );

                    if ($result === false) {
                        add_action('admin_notices', function() use ($wpdb) {
                            echo '<div class="notice notice-error is-dismissible"><p>' . 
                                sprintf(
                                    /* translators: %s: Database error message */
                                    esc_html__('Failed to reactivate category: %s', 'nexlifydesk'),
                                    esc_html($wpdb->last_error ?: 'Unknown database error')
                                ) . 
                            '</p></div>';
                        });
                        return;
                    }

                    self::clear_all_category_caches($existing->id);

                    wp_redirect(add_query_arg(array(
                        'page' => 'nexlifydesk_categories',
                        'category_reactivated' => 'true'
                    ), admin_url('admin.php')));
                    exit;
                }
            }

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled, query is prepared
            $result = $wpdb->insert(
                $table_name,
                array(
                    'name' => $category_name,
                    'slug' => $slug,
                    'description' => $category_description,
                    'is_active' => 1
                ),
                array('%s', '%s', '%s', '%d')
            );

            if ($result === false) {
                if (defined('DOING_AJAX') && DOING_AJAX) {
                    wp_send_json_error(__('Failed to add category.', 'nexlifydesk'));
                }
                add_action('admin_notices', function() use ($wpdb) {
                    echo '<div class="notice notice-error is-dismissible"><p>' . 
                        sprintf(
                            /* translators: %s: Database error message */
                            esc_html__('Failed to add category: %s', 'nexlifydesk'),
                            esc_html($wpdb->last_error)
                        ) . 
                    '</p></div>';
                });
                return;
            }

            wp_cache_delete($cache_key);
            
            self::clear_all_category_caches($wpdb->insert_id);

            if (defined('DOING_AJAX') && DOING_AJAX) {
                wp_send_json_success(array('message' => __('Category added successfully!', 'nexlifydesk')));
            }

            wp_redirect(add_query_arg(array(
                'page' => 'nexlifydesk_categories',
                'category_added' => 'true'
            ), admin_url('admin.php')));
            exit;
        }
    }

    private static function validate_file_types($file_types) {
        $allowed_types = array(
            'jpg', 'jpeg', 'png', 'pdf'
        );
        
        $file_types = strtolower(trim($file_types));
        if (empty($file_types)) {
            return 'jpg,jpeg,png,pdf';
        }
        
        $types = array_map('trim', explode(',', $file_types));
        $valid_types = array();
        
        foreach ($types as $type) {
            $type = ltrim($type, '.');
            if (in_array($type, $allowed_types, true)) {
                $valid_types[] = $type;
            }
        }
        
        return !empty($valid_types) ? implode(',', array_unique($valid_types)) : 'jpg,jpeg,png,pdf';
    }

    /**
     * Clear all category-related caches globally
     * 
     * @param int $category_id Optional specific category ID
     */
    public static function clear_all_category_caches($category_id = null) {
        wp_cache_delete('nexlifydesk_categories');
        wp_cache_delete('nexlifydesk_categories_admin_page');
        wp_cache_flush_group('nexlifydesk_categories');
        
        if ($category_id) {
            wp_cache_delete('nexlifydesk_category_' . $category_id);
        }
        
        global $wp_object_cache;
        if (is_object($wp_object_cache) && method_exists($wp_object_cache, 'cache')) {
            foreach ($wp_object_cache->cache as $group => $cache_group) {
                if (is_array($cache_group)) {
                    foreach ($cache_group as $key => $value) {
                        if (strpos($key, 'nexlifydesk_category_slug_check_') === 0) {
                            wp_cache_delete($key, $group);
                        }
                    }
                }
            }
        }
        
        if (function_exists('wp_cache_flush')) {
            wp_cache_flush();
        }
    }

    /**
     * Get ticket count for a category
     * 
     * @param int $category_id
     * @return int Number of tickets in this category
     */
    public static function get_category_ticket_count($category_id) {
        global $wpdb;
        
        $cache_key = 'nexlifydesk_category_ticket_count_' . $category_id;
        $count = wp_cache_get($cache_key);
        
        if (false === $count) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Safe query for counting
            $count = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->prefix}nexlifydesk_tickets WHERE category_id = %d",
                    $category_id
                )
            );
            wp_cache_set($cache_key, $count, '', 300);
        }
        
        return (int) $count;
    }

    /**
     * Get all categories for reassignment dropdown
     * 
     * @param int $exclude_category_id Category ID to exclude
     * @return array Available categories
     */
    public static function get_categories_for_reassignment($exclude_category_id) {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is safe and controlled, query is prepared, no caching needed for real-time category existence check to prevent race conditions
        $categories = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT id, name FROM {$wpdb->prefix}nexlifydesk_categories 
                 WHERE is_active = 1 AND id != %d 
                 ORDER BY name ASC",
                $exclude_category_id
            )
        );
        
        return $categories;
    }

    public static function render_email_templates_page() {
        $templates = get_option('nexlifydesk_email_templates', array(
            'new_ticket' => '',
            'new_reply' => '',
            'status_changed' => '',
            'sla_breach' => '',
            'email_auto_response' => '',
        ));

        if (isset($_POST['nexlifydesk_save_email_templates']) && check_admin_referer('nexlifydesk_save_email_templates')) {
            $templates = [
                'new_ticket' => isset($_POST['new_ticket']) ? wp_kses_post(wp_unslash($_POST['new_ticket'])) : '',
                'new_reply' => isset($_POST['new_reply']) ? wp_kses_post(wp_unslash($_POST['new_reply'])) : '',
                'status_changed' => isset($_POST['status_changed']) ? wp_kses_post(wp_unslash($_POST['status_changed'])) : '',
                'sla_breach' => isset($_POST['sla_breach']) ? wp_kses_post(wp_unslash($_POST['sla_breach'])) : '',
                'email_auto_response' => isset($_POST['email_auto_response']) ? wp_kses_post(wp_unslash($_POST['email_auto_response'])) : '',
            ];
                update_option('nexlifydesk_email_templates', $templates);
                
                echo '<div class="notice notice-success is-dismissible"><p>' 
                    . esc_html__('Email templates saved!', 'nexlifydesk') 
                    . '</p></div>';
            }
            ?>
        <div class="wrap">
            <h1><?php esc_html_e('NexlifyDesk Email Templates', 'nexlifydesk'); ?></h1>
            
            <div class="notice notice-info">
                <p>
                    <strong><?php esc_html_e('Template System Information:', 'nexlifydesk'); ?></strong><br>
                    <?php esc_html_e('When fields below are empty, the system uses default template files from ', 'nexlifydesk'); ?>
                    <code>templates/emails/</code> <?php esc_html_e('directory. Add content here only if you want to customize the default templates.', 'nexlifydesk'); ?>
                </p>
            </div>
            
            <form method="post">
                <?php wp_nonce_field('nexlifydesk_save_email_templates'); ?>
                
                <h2><?php esc_html_e('New Ticket', 'nexlifydesk'); ?></h2>
                <?php if (empty($templates['new_ticket'])): ?>
                    <p class="description" style="color: #0073aa; margin-bottom: 10px;">
                        <strong><?php esc_html_e('Currently using:', 'nexlifydesk'); ?></strong> 
                        <code>templates/emails/new_ticket.php</code> 
                        <?php esc_html_e('(Default template with proper styling)', 'nexlifydesk'); ?>
                    </p>
                <?php endif; ?>
                <?php wp_editor(
                    $templates['new_ticket'],
                    'new_ticket',
                    array(
                        'textarea_name' => 'new_ticket',
                        'textarea_rows' => 8,
                        'media_buttons' => false,
                        'tinymce' => false,
                        'quicktags' => true,
                        'default_editor' => 'html',
                    )
                ); ?>
                <button type="button" class="button preview-email-template" data-editor="new_ticket" style="margin-top:5px;">
                    <?php esc_html_e('Preview', 'nexlifydesk'); ?>
                </button>
                <div id="preview-new_ticket" class="nexlifydesk-email-preview" style="border:1px solid #ddd; margin-top:10px; padding:10px; display:none;"></div>
                
                <h2><?php esc_html_e('New Reply', 'nexlifydesk'); ?></h2>
                <?php if (empty($templates['new_reply'])): ?>
                    <p class="description" style="color: #0073aa; margin-bottom: 10px;">
                        <strong><?php esc_html_e('Currently using:', 'nexlifydesk'); ?></strong> 
                        <code>templates/emails/new_reply.php</code> 
                        <?php esc_html_e('(Default template with proper styling)', 'nexlifydesk'); ?>
                    </p>
                <?php endif; ?>
                <?php wp_editor(
                    $templates['new_reply'],
                    'new_reply',
                    array(
                        'textarea_name' => 'new_reply',
                        'textarea_rows' => 8,
                        'media_buttons' => false,
                        'tinymce' => false,
                        'quicktags' => true,
                        'default_editor' => 'html',
                    )
                ); ?>
                <button type="button" class="button preview-email-template" data-editor="new_reply" style="margin-top:5px;">
                    <?php esc_html_e('Preview', 'nexlifydesk'); ?>
                </button>
                <div id="preview-new_reply" class="nexlifydesk-email-preview" style="border:1px solid #ddd; margin-top:10px; padding:10px; display:none;"></div>
                
                <h2><?php esc_html_e('Status Changed', 'nexlifydesk'); ?></h2>
                <?php if (empty($templates['status_changed'])): ?>
                    <p class="description" style="color: #0073aa; margin-bottom: 10px;">
                        <strong><?php esc_html_e('Currently using:', 'nexlifydesk'); ?></strong> 
                        <code>templates/emails/status_changed.php</code> 
                        <?php esc_html_e('(Default template with proper styling)', 'nexlifydesk'); ?>
                    </p>
                <?php endif; ?>
                <?php wp_editor(
                    $templates['status_changed'],
                    'status_changed',
                    array(
                        'textarea_name' => 'status_changed',
                        'textarea_rows' => 8,
                        'media_buttons' => false,
                        'tinymce' => false,
                        'quicktags' => true,
                        'default_editor' => 'html',
                    )
                ); ?>
                <button type="button" class="button preview-email-template" data-editor="status_changed" style="margin-top:5px;">
                    <?php esc_html_e('Preview', 'nexlifydesk'); ?>
                </button>
                <div id="preview-status_changed" class="nexlifydesk-email-preview" style="border:1px solid #ddd; margin-top:10px; padding:10px; display:none;"></div>
                
                <h2><?php esc_html_e('Email Auto-Response (For Email Channel)', 'nexlifydesk'); ?></h2>
                <p class="description"><?php esc_html_e('This is the automatic response sent to customers when they create a ticket via email. Leave empty to use the default message.', 'nexlifydesk'); ?></p>
                <?php 
                $auto_response_content = isset($templates['email_auto_response']) ? $templates['email_auto_response'] : '';
                if (empty($auto_response_content)) {
                    $auto_response_content = '<p>Hello {customer_name},</p>

<p>Thank you for contacting us. We have received your support request and have assigned it ticket ID <strong>#{ticket_id}</strong>.</p>

<p><strong>Subject:</strong> {subject}</p>

<p>Our support team will review your request and get back to you as soon as possible. You can reference this ticket ID in any future correspondence.</p>

<p>Best regards,<br>
{site_name} Support Team<br>
<a href="{site_url}">{site_url}</a></p>';
                }
                wp_editor(
                    $auto_response_content,
                    'email_auto_response',
                    array(
                        'textarea_name' => 'email_auto_response',
                        'textarea_rows' => 10,
                        'media_buttons' => false,
                        'tinymce' => false,
                        'quicktags' => true,
                        'default_editor' => 'html',
                    )
                ); ?>
                <button type="button" class="button preview-email-template" data-editor="email_auto_response" style="margin-top:5px;">
                    <?php esc_html_e('Preview', 'nexlifydesk'); ?>
                </button>
                <div id="preview-email_auto_response" class="nexlifydesk-email-preview" style="border:1px solid #ddd; margin-top:10px; padding:10px; display:none;"></div>
                
                <h2><?php esc_html_e('SLA Breach', 'nexlifydesk'); ?></h2>
                <?php if (empty($templates['sla_breach'])): ?>
                    <p class="description" style="color: #0073aa; margin-bottom: 10px;">
                        <strong><?php esc_html_e('Currently using:', 'nexlifydesk'); ?></strong> 
                        <code>templates/emails/sla_breach.php</code> 
                        <?php esc_html_e('(Default template with proper styling)', 'nexlifydesk'); ?>
                    </p>
                <?php endif; ?>
                <?php wp_editor(
                    $templates['sla_breach'],
                    'sla_breach',
                    array(
                        'textarea_name' => 'sla_breach',
                        'textarea_rows' => 8,
                        'media_buttons' => false,
                        'tinymce' => false,
                        'quicktags' => true,
                        'default_editor' => 'html',
                    )
                ); ?>
                <button type="button" class="button preview-email-template" data-editor="sla_breach" style="margin-top:5px;">
                    <?php esc_html_e('Preview', 'nexlifydesk'); ?>
                </button>
                <div id="preview-sla_breach" class="nexlifydesk-email-preview" style="border:1px solid #ddd; margin-top:10px; padding:10px; display:none;"></div>
                
                <p>
                    <input type="submit" name="nexlifydesk_save_email_templates" class="button button-primary" value="<?php esc_attr_e('Save Templates', 'nexlifydesk'); ?>">
                </p>
            </form>
            <div class="nexlifydesk-email-placeholders" style="margin-top:2em;">
                <h2><?php esc_html_e('Available Placeholders', 'nexlifydesk'); ?></h2>
                <ul>
                    <li><code>{ticket_id}</code> – <?php esc_html_e('Ticket ID', 'nexlifydesk'); ?></li>
                    <li><code>{subject}</code> – <?php esc_html_e('Ticket Subject', 'nexlifydesk'); ?></li>
                    <li><code>{message}</code> – <?php esc_html_e('Ticket Message', 'nexlifydesk'); ?></li>
                    <li><code>{user_name}</code> – <?php esc_html_e('User Name', 'nexlifydesk'); ?></li>
                    <li><code>{user_email}</code> – <?php esc_html_e('User Email', 'nexlifydesk'); ?></li>
                    <li><code>{status}</code> – <?php esc_html_e('Ticket Status', 'nexlifydesk'); ?></li>
                    <li><code>{priority}</code> – <?php esc_html_e('Ticket Priority', 'nexlifydesk'); ?></li>
                    <li><code>{category}</code> – <?php esc_html_e('Ticket Category', 'nexlifydesk'); ?></li>
                    <li><code>{created_at}</code> – <?php esc_html_e('Created Date/Time', 'nexlifydesk'); ?></li>
                    <li><code>{updated_at}</code> – <?php esc_html_e('Last Updated Date/Time', 'nexlifydesk'); ?></li>
                    <li><code>{reply_message}</code> – <?php esc_html_e('Reply Message (for reply emails)', 'nexlifydesk'); ?></li>
                    <li><code>{reply_user_name}</code> – <?php esc_html_e('Reply Author Name (for reply emails)', 'nexlifydesk'); ?></li>
                    <li><code>{ticket_url}</code> – <?php esc_html_e('Direct link to the ticket conversation', 'nexlifydesk'); ?></li>
                    <li><code>{ticket_admin_url}</code> – <?php esc_html_e('Direct link to the ticket in the admin area (for agents/admins)', 'nexlifydesk'); ?></li>
                </ul>
                
                <h3><?php esc_html_e('Auto-Response Specific Placeholders', 'nexlifydesk'); ?></h3>
                <p class="description"><?php esc_html_e('These placeholders are specifically for the Email Auto-Response template:', 'nexlifydesk'); ?></p>
                <ul>
                    <li><code>{customer_name}</code> – <?php esc_html_e('Customer Name (extracted from email)', 'nexlifydesk'); ?></li>
                    <li><code>{site_name}</code> – <?php esc_html_e('Your Website Name', 'nexlifydesk'); ?></li>
                    <li><code>{site_url}</code> – <?php esc_html_e('Your Website URL', 'nexlifydesk'); ?></li>
                    <li><code>{admin_email}</code> – <?php esc_html_e('Admin Email Address', 'nexlifydesk'); ?></li>
                </ul>
                
                <p><?php esc_html_e('Copy and paste these placeholders into your templates. They will be replaced with real ticket data.', 'nexlifydesk'); ?></p>
            </div>
        </div>
        <?php
    }

    public static function show_data_retention_notice() {
        $settings = get_option('nexlifydesk_settings', array());
        $keep_data = isset($settings['keep_data_on_uninstall']) ? $settings['keep_data_on_uninstall'] : 1;
        
        if (!$keep_data) {
            ?>
            <div class="notice notice-warning is-dismissible">
                <p>
                    <strong><?php esc_html_e('NexlifyDesk Data Warning:', 'nexlifydesk'); ?></strong> 
                    <?php esc_html_e('Data retention is currently disabled. All tickets and customer data will be permanently deleted if you uninstall the plugin.', 'nexlifydesk'); ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=nexlifydesk_settings')); ?>" class="button button-small" style="margin-left: 10px;">
                        <?php esc_html_e('Review Settings', 'nexlifydesk'); ?>
                    </a>
                </p>
            </div>
            <?php
        }
    }
}

add_action('admin_notices', array('NexlifyDesk_Admin', 'show_data_retention_notice'));

add_filter('wp_editor_settings', function($settings, $editor_id) {
    if (in_array($editor_id, array('new_ticket', 'new_reply', 'status_changed', 'sla_breach'))) {
        $settings['tinymce'] = false;
        $settings['quicktags'] = true;
    }
    return $settings;
}, 10, 2);

function nexlifydesk_render_order_history_page() {
    if (!function_exists('wc_get_orders')) {
        wp_die(esc_html__('WooCommerce is required for this feature.', 'nexlifydesk'));
    }

    if (!current_user_can('nexlifydesk_manage_tickets')) {
        wp_die(esc_html__('You do not have permission to view this page.', 'nexlifydesk'));
    }

    $search_order_id = '';
        if (
            isset($_GET['nexlifydesk_order_search_nonce']) &&
            wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['nexlifydesk_order_search_nonce'])), 'nexlifydesk_order_search')
        ) {
            $search_order_id = (isset($_GET['order_id']) && $_GET['order_id'] !== '') ? absint($_GET['order_id']) : '';
        }

    $order = null;

    if ($search_order_id) {
        $args = array(
            'limit' => 1,
            'include' => array($search_order_id),
            'return' => 'ids',
        );
        $order_ids = wc_get_orders($args);
        if (!empty($order_ids)) {
            $order = wc_get_order($order_ids[0]);
        }
    }

    echo '<div class="wrap"><h1>' . esc_html__('Order Search', 'nexlifydesk') . '</h1>';

    echo '<form method="get" style="margin-bottom:20px;">';
    echo '<input type="hidden" name="page" value="nexlifydesk_order_history" />';
    wp_nonce_field('nexlifydesk_order_search', 'nexlifydesk_order_search_nonce');
    echo '<input type="text" name="order_id" placeholder="' . esc_attr__('Enter Order ID', 'nexlifydesk') . '" value="' . esc_attr($search_order_id) . '" style="margin-right:10px;" />';
    if (function_exists('submit_button')) {
        submit_button(__('Search', 'nexlifydesk'), 'secondary', '', false);
    } else {
        echo '<input type="submit" class="button button-secondary" value="' . esc_attr__('Search', 'nexlifydesk') . '" />';
    }
    echo ' <a href="' . esc_url(admin_url('admin.php?page=nexlifydesk_order_history')) . '" class="button">' . esc_html__('Reset', 'nexlifydesk') . '</a>';
    echo '</form>';

    if ($search_order_id && !$order) {
        echo '<p>' . esc_html__('No order found with ID #', 'nexlifydesk') . esc_html($search_order_id) . '</p>';
    } elseif ($order) {
        echo '<table class="widefat"><thead><tr>
            <th>' . esc_html__('Order', 'nexlifydesk') . '</th>
            <th>' . esc_html__('Customer', 'nexlifydesk') . '</th>
            <th>' . esc_html__('Email', 'nexlifydesk') . '</th>
            <th>' . esc_html__('Phone', 'nexlifydesk') . '</th>
            <th>' . esc_html__('Total', 'nexlifydesk') . '</th>
            <th>' . esc_html__('Status', 'nexlifydesk') . '</th>
            <th>' . esc_html__('Date', 'nexlifydesk') . '</th>
            <th>' . esc_html__('Items', 'nexlifydesk') . '</th>
        </tr></thead><tbody>';

        $items = array();
        foreach ($order->get_items() as $item) {
            $items[] = esc_html($item->get_name()) . ' x' . esc_html($item->get_quantity());
        }
        echo '<tr>';
        echo '<td>#' . esc_html($order->get_id()) . '</td>';
        echo '<td>' . esc_html($order->get_billing_first_name() . ' ' . $order->get_billing_last_name()) . '</td>';
        echo '<td>' . esc_html($order->get_billing_email()) . '</td>';
        echo '<td>' . esc_html($order->get_billing_phone()) . '</td>';
        echo '<td>' . wp_kses_post($order->get_formatted_order_total()) . '</td>';
        echo '<td>' . esc_html(wc_get_order_status_name($order->get_status())) . '</td>';
        echo '<td>' . esc_html($order->get_date_created() ? $order->get_date_created()->date('Y-m-d H:i') : '') . '</td>';
        echo '<td>' . wp_kses_post(implode('<br>', array_map('esc_html', $items))) . '</td>';
        echo '</tr>';
        echo '</tbody></table>';
    } else {
        echo '<p>' . esc_html__('Please enter an Order ID to search.', 'nexlifydesk') . '</p>';
    }

    echo '</div>';
}