<?php

if (!defined('ABSPATH')) exit;

if (!function_exists('nexlifydesk_decode_email_subject')) {
    require_once dirname(__FILE__) . '/../../includes/helpers.php';
}

/**
 * Handles the callback from Google, exchanges the code for tokens, and stores them.
 */
function nexlifydesk_google_handle_oauth_callback() {
    // phpcs:disable WordPress.Security.NonceVerification.Recommended -- OAuth callback from Google, uses state parameter for verification
    if (
        !current_user_can('manage_options') ||
        !isset($_GET['action']) ||
        $_GET['action'] !== 'nexlifydesk_google_oauth_callback' ||
        !isset($_GET['code'])
    ) {
        wp_die('Security check failed.');
    }

    if (isset($_GET['state'])) {
        $state = sanitize_text_field(wp_unslash($_GET['state']));
        $stored_state = get_transient('nexlifydesk_google_oauth_state');
        
        if (!$stored_state || $state !== $stored_state) {
            delete_transient('nexlifydesk_google_oauth_state');
            wp_die('Security check failed: Invalid state parameter.');
        }
        
        delete_transient('nexlifydesk_google_oauth_state');
    } else {
        wp_die('Security check failed: Missing state parameter.');
    }

    $settings = get_option('nexlifydesk_imap_settings', []);
    $client_id = $settings['google_client_id'] ?? '';
    
    $client_secret = !empty($settings['google_client_secret']) ? nexlifydesk_decrypt($settings['google_client_secret']) : '';
    
    $redirect_uri = admin_url('admin.php?action=nexlifydesk_google_oauth_callback');
    $code = sanitize_text_field(wp_unslash($_GET['code']));
    // phpcs:enable WordPress.Security.NonceVerification.Recommended

    if (empty($client_id) || empty($client_secret)) {
        wp_redirect(admin_url('admin.php?page=nexlifydesk_imap_auth&auth=failed'));
        exit;
    }

    $response = wp_remote_post('https://oauth2.googleapis.com/token', [
        'body' => [
            'client_id' => $client_id,
            'client_secret' => $client_secret,
            'code' => $code,
            'grant_type' => 'authorization_code',
            'redirect_uri' => $redirect_uri
        ]
    ]);

    if (is_wp_error($response)) {
        wp_redirect(admin_url('admin.php?page=nexlifydesk_imap_auth&auth=failed'));
        exit;
    }

    $body = json_decode(wp_remote_retrieve_body($response), true);

    if (isset($body['access_token'])) {
        $settings['google_access_token'] = $body['access_token'];
        $settings['google_refresh_token'] = $body['refresh_token'];
        $settings['google_token_expires_at'] = time() + (int)$body['expires_in'];
        $settings['google_auth_status'] = 'success';
        
        $user_info_response = wp_remote_get('https://www.googleapis.com/oauth2/v2/userinfo', [
            'headers' => ['Authorization' => 'Bearer ' . $body['access_token']]
        ]);
        
        if (!is_wp_error($user_info_response) && wp_remote_retrieve_response_code($user_info_response) === 200) {
            $user_info = json_decode(wp_remote_retrieve_body($user_info_response), true);
            if (isset($user_info['email'])) {
                $settings['google_account_email'] = $user_info['email'];
            }
        }
        
        $settings['google_fetch_start_time'] = time();
        
        update_option('nexlifydesk_imap_settings', $settings);

        wp_redirect(admin_url('admin.php?page=nexlifydesk_imap_auth&auth=success&_wpnonce=' . wp_create_nonce('nexlifydesk_imap_auth')));
    } else {
        $error_msg = $body['error_description'] ?? $body['error'] ?? 'Unknown error';
        
        $settings['google_auth_status'] = 'failed: ' . $error_msg;
        update_option('nexlifydesk_imap_settings', $settings);
        
        wp_redirect(admin_url('admin.php?page=nexlifydesk_imap_auth&auth=failed&_wpnonce=' . wp_create_nonce('nexlifydesk_imap_auth')));
    }
    exit;
}

add_action('admin_action_nexlifydesk_google_oauth_callback', 'nexlifydesk_google_handle_oauth_callback');

/**
 * Handle the Google OAuth initialization
 */
function nexlifydesk_google_handle_oauth_init() {
    if (!current_user_can('manage_options')) {
        wp_die('Insufficient permissions.');
    }

    // Verify nonce
    if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_key($_GET['_wpnonce']), 'google-auth-init')) {
        wp_die('Security check failed.');
    }

    $settings = get_option('nexlifydesk_imap_settings', []);
    $client_id = $settings['google_client_id'] ?? '';

    if (empty($client_id)) {
        wp_die('Google Client ID is not set. Please configure it in the settings.');
    }

    $state = wp_generate_password(32, false);
    set_transient('nexlifydesk_google_oauth_state', $state, 600);

    $redirect_uri = admin_url('admin.php?action=nexlifydesk_google_oauth_callback');
    
    $auth_url = 'https://accounts.google.com/o/oauth2/v2/auth?' . http_build_query([
        'client_id' => $client_id,
        'redirect_uri' => $redirect_uri,
        'response_type' => 'code',
        'scope' => 'https://www.googleapis.com/auth/gmail.modify https://www.googleapis.com/auth/userinfo.email',
        'access_type' => 'offline', 
        'prompt' => 'consent select_account',
        'state' => $state
    ]);

    wp_redirect($auth_url);
    exit;
}

add_action('admin_action_nexlifydesk_google_auth_init', 'nexlifydesk_google_handle_oauth_init');

/**
 * Fetches and processes emails from Google using the Gmail API.
 */
function nexlifydesk_fetch_google_emails() {
    $settings = get_option('nexlifydesk_imap_settings', []);
    $delete_emails = isset($settings['delete_emails_after_fetch']) ? $settings['delete_emails_after_fetch'] : 1;

    if (empty($settings['google_client_id']) || empty($settings['google_client_secret']) || empty($settings['google_refresh_token'])) {
        return array('error' => 'Google credentials not configured. Please authorize with Google first.');
    }

    $access_token = nexlifydesk_get_google_access_token();
    if (!$access_token) {
        return array('error' => 'Failed to obtain Google access token. Please re-authorize with Google.');
    }

    $fetch_start_time = $settings['google_fetch_start_time'] ?? null;
    
    if (!$fetch_start_time) {
        $fetch_start_time = time();
        $settings['google_fetch_start_time'] = $fetch_start_time;
        update_option('nexlifydesk_imap_settings', $settings);
    }
    
    $start_date = gmdate('Y/m/d', $fetch_start_time);
    
    $search_query = "is:unread after:{$start_date}";
    
    $processed_emails = get_option('nexlifydesk_processed_emails', []);
    $new_processed = [];
    
    $list_url = 'https://www.googleapis.com/gmail/v1/users/me/messages?' . http_build_query([
        'q' => $search_query,
        'maxResults' => 50 
    ]);
    
    $response = wp_remote_get($list_url, [
        'headers' => ['Authorization' => 'Bearer ' . $access_token]
    ]);

    if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
        return array('error' => 'Failed to connect to Gmail API. Please check your Google authorization.');
    }

    $list_body = json_decode(wp_remote_retrieve_body($response), true);
    if (empty($list_body['messages'])) {
        return array('success' => true, 'message' => 'No new emails found in Gmail.', 'count' => 0);
    }

    $message_count = count($list_body['messages']);
    $tickets_created = 0;
    $replies_added = 0;

    foreach ($list_body['messages'] as $message_item) {
        $message_id = $message_item['id'];
        
        if (in_array($message_id, $processed_emails)) {
            continue;
        }
        
        $new_processed[] = $message_id;
        
        $message_url = "https://www.googleapis.com/gmail/v1/users/me/messages/{$message_id}?format=full";
        $msg_response = wp_remote_get($message_url, [
            'headers' => ['Authorization' => 'Bearer ' . $access_token]
        ]);
        
        $msg_body = json_decode(wp_remote_retrieve_body($msg_response), true);

        $internal_date = isset($msg_body['internalDate']) ? intval($msg_body['internalDate']) / 1000 : 0;
        
        $day_start = strtotime(gmdate('Y-m-d', $fetch_start_time));
        
        if ($internal_date < $day_start) {
            continue;
        }
        
        $headers = $msg_body['payload']['headers'];
        $from_header = array_values(array_filter($headers, fn($h) => $h['name'] === 'From'))[0]['value'] ?? '';
        $subject_header = array_values(array_filter($headers, fn($h) => $h['name'] === 'Subject'))[0]['value'] ?? '(No Subject)';

        if (function_exists('nexlifydesk_decode_email_subject')) {
            $subject_header = nexlifydesk_decode_email_subject($subject_header);
        }

        $parsed_from = function_exists('nexlifydesk_parse_email_from') ? 
            nexlifydesk_parse_email_from($from_header) : ['name' => '', 'email' => $from_header];
        
        $email_address = $parsed_from['email'];
        $sender_name = $parsed_from['name'];
        
        if (empty($email_address)) {
            $email_address = $from_header;
            if (preg_match('/<(.+?)>/', $from_header, $matches)) {
                $email_address = $matches[1];
            }
        }
        
        if (!filter_var($email_address, FILTER_VALIDATE_EMAIL)) {
            continue; 
        }
        
        $message_content = nexlifydesk_get_gmail_body($msg_body['payload']);
        $original_length = strlen($message_content);
        
        if (function_exists('nexlifydesk_extract_clean_email_content')) {
            $message_content = nexlifydesk_extract_clean_email_content($message_content, $email_address);
        } elseif (function_exists('nexlifydesk_strip_email_thread')) {
            $message_content = nexlifydesk_strip_email_thread($message_content, $email_address);
        }
        $stripped_length = strlen($message_content);
        
        if ($stripped_length < ($original_length * 0.1) && $original_length > 100) {
            if (function_exists('nexlifydesk_strip_email_thread')) {
                $fallback_message = nexlifydesk_strip_email_thread($message_content, $email_address);
                if (strlen($fallback_message) > $stripped_length) {
                    $message_content = $fallback_message;
                }
            }
        }

        if (function_exists('nexlifydesk_should_block_email') && 
            nexlifydesk_should_block_email($email_address, $subject_header, $message_content, $settings)) {
             if ($delete_emails) {
                 nexlifydesk_google_mark_as_read_and_delete($message_id, $access_token);
             } else {
                 nexlifydesk_google_mark_as_read($message_id, $access_token);
             }
             continue;
        }

        $user = get_user_by('email', $email_address);
        $user_id = $user ? $user->ID : 0;
        
        if (class_exists('NexlifyDesk_Rate_Limiter') && NexlifyDesk_Rate_Limiter::is_rate_limited($user_id, $email_address)) {
            
            if ($delete_emails) {
                nexlifydesk_google_mark_as_read_and_delete($message_id, $access_token);
            } else {
                nexlifydesk_google_mark_as_read($message_id, $access_token);
            }
            continue;
        }
        
        $ticket_id_from_subject = function_exists('nexlifydesk_extract_ticket_id_from_subject') ? 
            nexlifydesk_extract_ticket_id_from_subject($subject_header) : null;
        $existing_ticket = null;
        
        if ($ticket_id_from_subject && function_exists('nexlifydesk_get_ticket_by_ticket_id')) {
            $existing_ticket = nexlifydesk_get_ticket_by_ticket_id($ticket_id_from_subject);
        }
        
        if (!$existing_ticket) {
            $cleaned_subject = nexlifydesk_clean_email_subject($subject_header);
            $ticket_data = [
                'user_id' => $user_id, 
                'subject' => $cleaned_subject, 
                'message' => $message_content,
                'email' => $email_address,
                'source' => 'email'
            ];
            
            if (function_exists('nexlifydesk_check_email_duplicate')) {
                $existing_ticket = nexlifydesk_check_email_duplicate($ticket_data);
            }
            
            if (!$existing_ticket) {
                $existing_ticket = NexlifyDesk_Tickets::check_for_duplicate_ticket($ticket_data);
            }
        }

        if ($existing_ticket) {
            $reply_data = [
                'ticket_id' => $existing_ticket->id, 
                'user_id' => $user_id, 
                'message' => $message_content,
                'source' => 'email'  
            ];
            
            if (!$user_id && (!empty($sender_name) || !empty($email_address))) {
                $sender_info = [];
                if (!empty($sender_name)) {
                    $sender_info[] = "Name: {$sender_name}";
                }
                if (!empty($email_address)) {
                    $sender_info[] = "Email: {$email_address}";
                }
                if (!empty($sender_info)) {
                    $reply_data['message'] = "[Customer Details]\n" . implode("\n", $sender_info) . "\n\n[Reply]\n" . $message_content;
                }
            }
            
            $reply_result = NexlifyDesk_Tickets::add_reply($reply_data);
            if ($reply_result) {
                $replies_added++;
            }
        } else {
            $cleaned_subject = nexlifydesk_clean_email_subject($subject_header);
            $ticket_data = [
                'user_id' => $user_id, 
                'subject' => $cleaned_subject, 
                'message' => $message_content,
                'source' => 'email', 
                'email' => $email_address 
            ];
            
            if (!$user_id && (!empty($sender_name) || !empty($email_address))) {
                $sender_info = [];
                if (!empty($sender_name)) {
                    $sender_info[] = "Name: {$sender_name}";
                }
                if (!empty($email_address)) {
                    $sender_info[] = "Email: {$email_address}";
                }
                if (!empty($sender_info)) {
                    $ticket_data['message'] = "[Customer Details]\n" . implode("\n", $sender_info) . "\n\n[Message]\n" . $message_content;
                }
            }
            
            $ticket = NexlifyDesk_Tickets::create_ticket($ticket_data);
            if ($ticket && !is_wp_error($ticket)) {
                $tickets_created++;
            }
        }
        
        if ($delete_emails) {
            nexlifydesk_google_mark_as_read_and_delete($message_id, $access_token);
        } else {
            nexlifydesk_google_mark_as_read($message_id, $access_token);
        }
    }
    
    $all_processed = array_merge($processed_emails, $new_processed);
    if (count($all_processed) > 500) {
        $all_processed = array_slice($all_processed, -500);
    }
    update_option('nexlifydesk_processed_emails', $all_processed);
    
    return array(
        'success' => true, 
        'message' => sprintf('Processed %d emails. Created %d tickets, added %d replies.', 
                           $message_count, $tickets_created, $replies_added),
        'count' => $message_count,
        'tickets_created' => $tickets_created,
        'replies_added' => $replies_added
    );
}


/**
 * Gets a valid Google access token, refreshing it if it's expired.
 *
 * @return string|false The access token or false on failure.
 */
function nexlifydesk_get_google_access_token() {
    $settings = get_option('nexlifydesk_imap_settings', []);
    
    if (empty($settings['google_refresh_token'])) {
        return false;
    }
    
    if (!empty($settings['google_access_token']) && time() < ($settings['google_token_expires_at'] ?? 0) - 30) {
        return $settings['google_access_token'];
    }
    
    $client_secret = !empty($settings['google_client_secret']) ? nexlifydesk_decrypt($settings['google_client_secret']) : '';
    
    $response = wp_remote_post('https://oauth2.googleapis.com/token', [
        'body' => [
            'client_id' => $settings['google_client_id'],
            'client_secret' => $client_secret,
            'refresh_token' => $settings['google_refresh_token'],
            'grant_type' => 'refresh_token',
        ]
    ]);
    
    if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
        return false;
    }
    
    $body = json_decode(wp_remote_retrieve_body($response), true);
    
    if (isset($body['access_token'])) {
        $settings['google_access_token'] = $body['access_token'];
        $settings['google_token_expires_at'] = time() + (int)$body['expires_in'];
        update_option('nexlifydesk_imap_settings', $settings);
        return $body['access_token'];
    }

    return false;
}

/**
 *
 * @param array $payload The payload from the Gmail API message.
 * @return string The email body content.
 */
function nexlifydesk_get_gmail_body($payload) {
    $body = '';
    if (isset($payload['parts'])) {
        foreach ($payload['parts'] as $part) {
            if ($part['mimeType'] == 'text/plain' && isset($part['body']['data'])) {
                $body = base64_decode(strtr($part['body']['data'], '-_', '+/'));
                break;
            }
        }
        if (empty($body)) {
             foreach ($payload['parts'] as $part) {
                if ($part['mimeType'] == 'text/html' && isset($part['body']['data'])) {
                    $body = base64_decode(strtr($part['body']['data'], '-_', '+/'));
                    break;
                }
            }
        }
    } elseif (isset($payload['body']['data'])) {
        $body = base64_decode(strtr($payload['body']['data'], '-_', '+/'));
    }
    return $body;
}

/**
 * 
 * @param string $message_id The ID of the message to modify.
 * @param string $access_token A valid access token.
 */
function nexlifydesk_google_mark_as_read($message_id, $access_token) {
    $modify_url = "https://www.googleapis.com/gmail/v1/users/me/messages/{$message_id}/modify";
    wp_remote_post($modify_url, [
        'method'  => 'POST',
        'headers' => ['Authorization' => 'Bearer ' . $access_token, 'Content-Type' => 'application/json'],
        'body'    => json_encode(['removeLabelIds' => ['UNREAD']])
    ]);
}

/**
 * 
 * @param string $message_id The ID of the message to modify.
 * @param string $access_token A valid access token.
 */
function nexlifydesk_google_mark_as_read_and_delete($message_id, $access_token) {
    $modify_url = "https://www.googleapis.com/gmail/v1/users/me/messages/{$message_id}/modify";
    wp_remote_post($modify_url, [
        'method'  => 'POST',
        'headers' => ['Authorization' => 'Bearer ' . $access_token, 'Content-Type' => 'application/json'],
        'body'    => json_encode(['removeLabelIds' => ['UNREAD']])
    ]);

    $trash_url = "https://www.googleapis.com/gmail/v1/users/me/messages/{$message_id}/trash";
    wp_remote_post($trash_url, [
        'method'  => 'POST',
        'headers' => ['Authorization' => 'Bearer ' . $access_token]
    ]);
}

/**
 * Resets the Gmail fetch start time to current time.
 * Useful for testing or if you want to start fresh without old emails.
 */
function nexlifydesk_reset_gmail_fetch_start_time() {
    $settings = get_option('nexlifydesk_imap_settings', []);
    $settings['google_fetch_start_time'] = time();
    update_option('nexlifydesk_imap_settings', $settings);
    
    return true;
}