<?php

if (!defined('ABSPATH')) exit;

if (!function_exists('nexlifydesk_decode_email_subject')) {
    require_once dirname(__FILE__) . '/../../includes/helpers.php';
}

/**
 * Fetches and processes emails from AWS WorkMail using IMAP
 */
function nexlifydesk_fetch_aws_emails() {
    if (!extension_loaded('imap') || !function_exists('imap_open')) {
        
        add_action('admin_notices', function() {
            if (current_user_can('manage_options')) {
                echo '<div class="notice notice-error"><p><strong>NexlifyDesk:</strong> ' . 
                     esc_html__('IMAP extension is not properly installed or configured on this server. AWS WorkMail email piping is not available. Please contact your hosting provider to enable the PHP IMAP extension with all required functions.', 'nexlifydesk') . 
                     '</p></div>';
            }
        });
        
        return;
    }

    $is_ssl_enabled = function_exists('nexlifydesk_check_ssl_enabled') ? 
        nexlifydesk_check_ssl_enabled() : 
        (
            is_ssl() || 
            (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ||
            (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https') ||
            (isset($_SERVER['HTTP_X_FORWARDED_SSL']) && $_SERVER['HTTP_X_FORWARDED_SSL'] === 'on') ||
            (isset($_SERVER['REQUEST_SCHEME']) && $_SERVER['REQUEST_SCHEME'] === 'https') ||
            ((wp_parse_url(home_url(), ) === 'https'))
        );
    
    if (!$is_ssl_enabled) {
        return;
    }

    $settings = get_option('nexlifydesk_imap_settings', []);
    
    $integration_type = $settings['aws_integration_type'] ?? 'imap';
    if ($integration_type === 'lambda') {
        $integration_type = 'imap';
        $settings['aws_integration_type'] = 'imap';
        update_option('nexlifydesk_imap_settings', $settings);
    }
    
    $region = $settings['aws_region'] ?? 'us-east-1';
    $organization_id = $settings['aws_organization_id'] ?? '';
    $email = $settings['aws_email'] ?? '';
    $delete_emails = isset($settings['delete_emails_after_fetch']) ? $settings['delete_emails_after_fetch'] : 1;
    
    $password = !empty($settings['aws_password']) ? nexlifydesk_decrypt($settings['aws_password']) : '';
    
    if (empty($organization_id) || empty($email) || empty($password)) {
        return array('error' => 'AWS WorkMail credentials not configured. Please configure Organization ID, Email, and Password.');
    }
    
    $imap_host = "imap.mail.{$region}.awsapps.com";
    $imap_port = 993;
    $encryption = 'ssl';
    
    $mailbox = "{{$imap_host}:{$imap_port}/imap/{$encryption}}INBOX";
    
    if (function_exists('imap_errors')) {
        imap_errors();
    }
    if (function_exists('imap_alerts')) {
        imap_alerts();
    }
    
    $inbox = @imap_open($mailbox, $email, $password, OP_SILENT);
    if (!$inbox) {
        return array('error' => 'Failed to connect to AWS WorkMail IMAP server. Please check your credentials and region.');
    }
    
    $fetch_start_time = $settings['aws_fetch_start_time'] ?? null;
    
    if (!$fetch_start_time) {
        $fetch_start_time = time();
        $settings['aws_fetch_start_time'] = $fetch_start_time;
        update_option('nexlifydesk_imap_settings', $settings);
    }
    
    $search_date = gmdate('d-M-Y', $fetch_start_time);
    $search_criteria = "UNSEEN SINCE \"{$search_date}\"";
    
    $emails = imap_search($inbox, $search_criteria);
    
    if (!$emails) {
        $emails = imap_search($inbox, "UNSEEN");
        
        if (!$emails) {
            $total_msgs = imap_num_msg($inbox);
            $manually_found = [];
            for ($i = 1; $i <= $total_msgs; $i++) {
                $overview = imap_fetch_overview($inbox, $i, 0);
                if (!empty($overview) && isset($overview[0])) {
                    $msg = $overview[0];
                    if (!isset($msg->seen) || $msg->seen == 0) {
                        $manually_found[] = $i;
                    }
                }
            }
            if (!empty($manually_found)) {
                $emails = $manually_found;
            }
        }
    }
    
    if (!$emails) {
        imap_close($inbox);
        return array('success' => true, 'message' => 'No new emails found in AWS WorkMail.', 'count' => 0);
    }
    
    $processed_emails = get_option('nexlifydesk_aws_processed_emails', []);
    $new_processed = [];
    $tickets_created = 0;
    $replies_added = 0;
    $emails_processed = 0;
    
    foreach ($emails as $email_number) {
        $uid = imap_uid($inbox, $email_number);
        
        if (in_array($uid, $processed_emails)) {
            continue;
        }
        
        $emails_processed++;
        $new_processed[] = $uid;
        
        $overview = imap_fetch_overview($inbox, $email_number, 0)[0];
        $header = imap_headerinfo($inbox, $email_number);
        
        $from = $header->fromaddress ?? $overview->from;
        $subject = $overview->subject ?? '(No Subject)';
        
        if (function_exists('nexlifydesk_decode_email_subject')) {
            $subject = nexlifydesk_decode_email_subject($subject);
        }
        
        $parsed_from = function_exists('nexlifydesk_parse_email_from') ? 
            nexlifydesk_parse_email_from($from) : ['name' => '', 'email' => $from];
        
        $email_address = $parsed_from['email'];
        $sender_name = $parsed_from['name'];
        
        if (empty($email_address)) {
            $email_address = $from;
            if (preg_match('/<(.+?)>/', $from, $matches)) {
                $email_address = $matches[1];
            }
        }
        
        $message = imap_fetchbody($inbox, $email_number, 1.1);
        if (empty($message)) {
            $message = imap_fetchbody($inbox, $email_number, 1);
        }
        
        $original_length = strlen($message);
        if (function_exists('nexlifydesk_extract_clean_email_content')) {
            $message = nexlifydesk_extract_clean_email_content($message, $email_address);
        } elseif (function_exists('nexlifydesk_strip_email_thread')) {
            $message = nexlifydesk_strip_email_thread($message, $email_address);
        }
        $stripped_length = strlen($message);
        
        if ($stripped_length < ($original_length * 0.1) && $original_length > 100) {
            if (function_exists('nexlifydesk_strip_email_thread')) {
                $fallback_message = nexlifydesk_strip_email_thread($message, $email_address);
                if (strlen($fallback_message) > $stripped_length) {
                    $message = $fallback_message;
                }
            }
        }
        
        if (!filter_var($email_address, FILTER_VALIDATE_EMAIL)) {
            imap_setflag_full($inbox, $email_number, "\\Seen");
            continue;
        }
        
        if (function_exists('nexlifydesk_should_block_email') && 
            nexlifydesk_should_block_email($email_address, $subject, $message, $settings)) {
            imap_setflag_full($inbox, $email_number, "\\Seen");
            if ($delete_emails) {
                imap_delete($inbox, $email_number);
            }
            continue;
        }
        
        $user = get_user_by('email', $email_address);
        $user_id = $user ? $user->ID : 0;
        
        if (class_exists('NexlifyDesk_Rate_Limiter') && NexlifyDesk_Rate_Limiter::is_rate_limited($user_id, $email_address)) {
            imap_setflag_full($inbox, $email_number, "\\Seen");
            if ($delete_emails) {
                imap_delete($inbox, $email_number);
            }
            continue;
        }
        
        $ticket_id_from_subject = function_exists('nexlifydesk_extract_ticket_id_from_subject') ? 
            nexlifydesk_extract_ticket_id_from_subject($subject) : null;
        $existing_ticket = null;
        
        if ($ticket_id_from_subject && function_exists('nexlifydesk_get_ticket_by_ticket_id')) {
            $existing_ticket = nexlifydesk_get_ticket_by_ticket_id($ticket_id_from_subject);
        }
        
        if (!$existing_ticket) {
            $cleaned_subject = nexlifydesk_clean_email_subject($subject);
            $ticket_data = [
                'user_id' => $user_id, 
                'subject' => $cleaned_subject, 
                'message' => $message,
                'email' => $email_address, 
                'source' => 'email'
            ];
            
            if (function_exists('nexlifydesk_check_email_duplicate')) {
                $existing_ticket = nexlifydesk_check_email_duplicate($ticket_data);
            } else {
                $existing_ticket = NexlifyDesk_Tickets::check_for_duplicate_ticket($ticket_data);
            }
        }
        
        if ($existing_ticket) {
            $reply_data = [
                'ticket_id' => $existing_ticket->id, 
                'user_id' => $user_id, 
                'message' => $message,
                'source' => 'email'
            ];
            
            if (!$user_id && (!empty($sender_name) || !empty($email_address))) {
                $sender_info = [];
                if (!empty($sender_name)) {
                    $sender_info[] = "Name: {$sender_name}";
                }
                if (!empty($email_address)) {
                    $sender_info[] = "Email: {$email_address}";
                }
                if (!empty($sender_info)) {
                    $reply_data['message'] = "[Customer Details]\n" . implode("\n", $sender_info) . "\n\n[Reply]\n" . $message;
                }
            }
            
            $reply_result = NexlifyDesk_Tickets::add_reply($reply_data);
            if ($reply_result) {
                $replies_added++;
            }
        } else {
            $cleaned_subject = nexlifydesk_clean_email_subject($subject);
            $ticket_data = [
                'user_id' => $user_id, 
                'subject' => $cleaned_subject, 
                'message' => $message,
                'source' => 'email',
                'email' => $email_address 
            ];
            
            if (!$user_id && (!empty($sender_name) || !empty($email_address))) {
                $sender_info = [];
                if (!empty($sender_name)) {
                    $sender_info[] = "Name: {$sender_name}";
                }
                if (!empty($email_address)) {
                    $sender_info[] = "Email: {$email_address}";
                }
                if (!empty($sender_info)) {
                    $ticket_data['message'] = "[Customer Details]\n" . implode("\n", $sender_info) . "\n\n[Message]\n" . $message;
                }
            }
            
            $ticket = NexlifyDesk_Tickets::create_ticket($ticket_data);
        }
        
        imap_setflag_full($inbox, $email_number, "\\Seen");
        if ($delete_emails) {
            imap_delete($inbox, $email_number);
        }
    }
    
    if ($delete_emails) {
        imap_expunge($inbox);
    }
    imap_close($inbox);
    
    $all_processed = array_merge($processed_emails, $new_processed);
    if (count($all_processed) > 500) {
        $all_processed = array_slice($all_processed, -500);
    }
    update_option('nexlifydesk_aws_processed_emails', $all_processed);
    
    return array(
        'success' => true, 
        'message' => sprintf('Processed %d emails. Created %d tickets, added %d replies.', 
                           $emails_processed, $tickets_created, $replies_added),
        'count' => $emails_processed,
        'tickets_created' => $tickets_created,
        'replies_added' => $replies_added
    );
}

/**
 * Reset AWS fetch start time to current time
 */
function nexlifydesk_reset_aws_fetch_start_time() {
    $settings = get_option('nexlifydesk_imap_settings', []);
    $settings['aws_fetch_start_time'] = time();
    update_option('nexlifydesk_imap_settings', $settings);
}

/**
 * Based on WP Mail SMTP Pro structure but adapted for NexlifyDesk
 */
class NexlifyDesk_AWS_Auth {
    
    private $settings;
    private $region;
    private $access_key;
    private $secret_key;
    
    public function __construct() {
        $this->settings = get_option('nexlifydesk_imap_settings', []);
        $this->region = $this->settings['aws_region'] ?? 'us-east-1';
        $this->access_key = $this->settings['aws_access_key_id'] ?? ''; 
        $this->secret_key = !empty($this->settings['aws_secret_access_key']) ? nexlifydesk_decrypt($this->settings['aws_secret_access_key']) : '';  
    }
    
    /**
     * Uses centralized SSL detection from helpers.php
     */
    private function check_ssl_enabled() {
        if (function_exists('nexlifydesk_check_ssl_enabled')) {
            return nexlifydesk_check_ssl_enabled();
        }
        
        $home_url_scheme = wp_parse_url(home_url(), PHP_URL_SCHEME);
        if ($home_url_scheme !== 'https') {
            return false;
        }
        
        if (!is_ssl()) {
            return false;
        }
        
        return true;
    }
    
    public function is_connection_ready() {
        $is_ssl_enabled = $this->check_ssl_enabled();
        
        if (!$is_ssl_enabled) {
            return false;
        }
        
        $has_credentials = !empty($this->access_key) && !empty($this->secret_key) && !empty($this->region);
        
        return $has_credentials;
    }
    
    /**
     * Get available AWS regions for WorkMail
     */
    public static function get_regions() {
        return [
            'us-east-1' => 'US East (N. Virginia)',
            'us-west-2' => 'US West (Oregon)',
            'eu-west-1' => 'Europe (Ireland)',
            'eu-west-2' => 'Europe (London)',
            'eu-west-3' => 'Europe (Paris)',
            'eu-central-1' => 'Europe (Frankfurt)',
            'ca-central-1' => 'Canada (Central)',
            'ap-south-1' => 'Asia Pacific (Mumbai)',
            'ap-northeast-1' => 'Asia Pacific (Tokyo)',
            'ap-northeast-2' => 'Asia Pacific (Seoul)',
            'ap-southeast-1' => 'Asia Pacific (Singapore)',
            'ap-southeast-2' => 'Asia Pacific (Sydney)',
            'sa-east-1' => 'South America (São Paulo)',
        ];
    }
    
    /**
     * Test AWS WorkMail IMAP connection
     */
    public function test_imap_connection() {
        if (!extension_loaded('imap') || !function_exists('imap_open')) {
            return new WP_Error('imap_not_available', 'IMAP extension is not properly installed or configured on this server. Please contact your hosting provider to enable the PHP IMAP extension with all required functions.');
        }
        
        $is_ssl_enabled = $this->check_ssl_enabled();
        
        if (!$is_ssl_enabled) {
            return new WP_Error('ssl_required', 'SSL certificate is required for AWS WorkMail integration. Please ensure your site is running on HTTPS.');
        }
        
        $organization_id = $this->settings['aws_organization_id'] ?? '';
        $email = $this->settings['aws_email'] ?? '';
        
        $password = !empty($this->settings['aws_password']) ? nexlifydesk_decrypt($this->settings['aws_password']) : '';
        
        if (empty($organization_id) || empty($email) || empty($password)) {
            return new WP_Error('missing_credentials', 'Missing AWS WorkMail credentials (Organization ID, Email, Password required for IMAP)');
        }
        
        $imap_host = "imap.mail.{$this->region}.awsapps.com";
        $mailbox = "{{$imap_host}:993/imap/ssl}INBOX";
        
        if (function_exists('imap_errors')) {
            imap_errors();
        }
        if (function_exists('imap_alerts')) {
            imap_alerts();
        }
        
        $inbox = @imap_open($mailbox, $email, $password, OP_READONLY | OP_SILENT);
        
        if (!$inbox) {
            $error = 'Unknown connection error';
            
            if (function_exists('imap_last_error')) {
                $last_error = imap_last_error();
                if ($last_error) {
                    $error = $last_error;
                }
            }
            
            return new WP_Error('connection_failed', "AWS WorkMail IMAP connection failed: {$error}");
        }
        
        imap_close($inbox);
        return true;
    }
    
    /**
     * Validate AWS SES credentials by making a test API call
     */
    public function validate_ses_credentials() {
        if (!$this->is_connection_ready()) {
            return new WP_Error('missing_credentials', 'Missing AWS SES credentials');
        }
        
        $endpoint = "https://email.{$this->region}.amazonaws.com/";
        $timestamp = gmdate('Ymd\THis\Z');
        $date = gmdate('Ymd');
        
        $headers = [
            'Content-Type' => 'application/x-amz-json-1.0',
            'X-Amz-Target' => 'SimpleEmailService.GetSendQuota',
            'X-Amz-Date' => $timestamp,
        ];
        
        $canonical_request = $this->create_canonical_request('POST', '/', '', $headers, '{}');
        $string_to_sign = $this->create_string_to_sign($timestamp, $date, $canonical_request);
        $signature = $this->calculate_signature($string_to_sign, $date);
        
        $headers['Authorization'] = $this->create_authorization_header($signature, $timestamp, $date);
        
        $response = wp_remote_post($endpoint, [
            'headers' => $headers,
            'body' => '{}',
            'timeout' => 30,
        ]);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        if ($code === 200) {
            return true;
        }
        
        return new WP_Error('invalid_credentials', 'AWS SES credentials validation failed');
    }
    
    /**
     * Create canonical request for AWS signature
     */
    private function create_canonical_request($method, $uri, $query, $headers, $payload) {
        $canonical_headers = '';
        $signed_headers = '';
        
        ksort($headers);
        foreach ($headers as $key => $value) {
            $canonical_headers .= strtolower($key) . ':' . trim($value) . "\n";
            $signed_headers .= strtolower($key) . ';';
        }
        $signed_headers = rtrim($signed_headers, ';');
        
        return implode("\n", [
            $method,
            $uri,
            $query,
            $canonical_headers,
            $signed_headers,
            hash('sha256', $payload)
        ]);
    }
    
    /**
     * Create string to sign for AWS signature
     */
    private function create_string_to_sign($timestamp, $date, $canonical_request) {
        $algorithm = 'AWS4-HMAC-SHA256';
        $credential_scope = "{$date}/{$this->region}/ses/aws4_request";
        
        return implode("\n", [
            $algorithm,
            $timestamp,
            $credential_scope,
            hash('sha256', $canonical_request)
        ]);
    }
    
    /**
     * Calculate AWS signature
     */
    private function calculate_signature($string_to_sign, $date) {
        $k_date = hash_hmac('sha256', $date, 'AWS4' . $this->secret_key, true);
        $k_region = hash_hmac('sha256', $this->region, $k_date, true);
        $k_service = hash_hmac('sha256', 'ses', $k_region, true);
        $k_signing = hash_hmac('sha256', 'aws4_request', $k_service, true);
        
        return hash_hmac('sha256', $string_to_sign, $k_signing);
    }
    
    /**
     * Create authorization header
     */
    private function create_authorization_header($signature, $timestamp, $date) {
        $algorithm = 'AWS4-HMAC-SHA256';
        $credential = "{$this->access_key}/{$date}/{$this->region}/ses/aws4_request";
        $signed_headers = 'content-type;x-amz-date;x-amz-target';
        
        return "{$algorithm} Credential={$credential}, SignedHeaders={$signed_headers}, Signature={$signature}";
    }
}