<?php
if (!defined('ABSPATH')) { require_once dirname(__DIR__, 3) . '/../../../../wp-load.php'; }

if (defined('NEXLIFYDESK_PLUGIN_DIR')) {
    require_once NEXLIFYDESK_PLUGIN_DIR . 'includes/helpers.php';
    require_once NEXLIFYDESK_PLUGIN_DIR . 'email-source/providers/google/google-handler.php';
    require_once NEXLIFYDESK_PLUGIN_DIR . 'email-source/providers/aws-ses/aws-handler.php';
}

function nexlifydesk_fetch_emails() {
    $settings = get_option('nexlifydesk_imap_settings', array());
    if (empty($settings['enabled'])) return;

    $provider = isset($settings['provider']) ? $settings['provider'] : 'custom';

    if (($provider === 'custom' || $provider === 'aws') && !extension_loaded('imap')) {
        add_action('admin_notices', function() {
            if (current_user_can('manage_options')) {
                echo '<div class="notice notice-error"><p><strong>NexlifyDesk:</strong> ' . 
                     esc_html__('IMAP extension is not installed on this server. Email piping is not available for Custom IMAP/POP3 and AWS providers. Please contact your hosting provider to enable the PHP IMAP extension.', 'nexlifydesk') . 
                     '</p></div>';
            }
        });
        return;
    }

    if ($provider === 'google') {
        if (function_exists('nexlifydesk_fetch_google_emails')) {
            nexlifydesk_fetch_google_emails();
        }
    } elseif ($provider === 'aws') {
        if (function_exists('nexlifydesk_fetch_aws_emails')) {
            nexlifydesk_fetch_aws_emails();
        }
    } elseif ($provider === 'custom') {
        nexlifydesk_fetch_custom_emails();
    }
}

/**
 * Extract ticket ID from email subject line
 * 
 * @param string $subject The email subject line
 * @return string|null The ticket ID if found, null otherwise
 */
function nexlifydesk_extract_ticket_id_from_subject($subject) {
    if (preg_match('/#(T\d+)/', $subject, $matches)) {
        return $matches[1];
    }
    return null;
}

/**
 * Gets ticket by ticket_id string (e.g., "T1102")
 *
 * @param string $ticket_id The ticket ID string
 * @return object|null The ticket object if found, null otherwise
 */
function nexlifydesk_get_ticket_by_ticket_id($ticket_id) {
    return NexlifyDesk_Tickets::get_ticket_by_ticket_id($ticket_id);
}

/**
 *
 * @param string $subject The email subject line
 * @return string The cleaned subject
 */
function nexlifydesk_clean_email_subject($subject) {
    $subject = preg_replace('/^(Re|RE|Fw|FW|Fwd|FWD|Fw|fw):\s*/i', '', $subject);
    
    $subject = trim($subject);
    
    $subject = preg_replace('/\[#T\d+\]\s*/', '', $subject);
    
    return $subject;
}

/**
 * @param array $ticket_data Ticket data including user_id, subject, message
 * @return object|null Existing ticket if found, null otherwise
 */
function nexlifydesk_check_email_duplicate($ticket_data) {
    global $wpdb;
    
    $user_id = isset($ticket_data['user_id']) ? absint($ticket_data['user_id']) : 0;
    $subject = sanitize_text_field($ticket_data['subject']);
    $email = isset($ticket_data['email']) ? sanitize_email($ticket_data['email']) : '';

    $table_name = $wpdb->prefix . 'nexlifydesk_tickets';

    if ($user_id == 0 && !empty($email)) {
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
        $existing_ticket = $wpdb->get_row($wpdb->prepare(
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is constructed from $wpdb->prefix and is safe
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is constructed from $wpdb->prefix and is safe
            "
            SELECT t.* FROM `{$table_name}` t
            LEFT JOIN `{$wpdb->prefix}postmeta` pm ON pm.post_id = t.id AND pm.meta_key = 'customer_email'
            WHERE t.user_id = 0
            AND pm.meta_value = %s
            AND t.status IN ('open', 'pending', 'in_progress')
            ORDER BY t.created_at DESC 
            LIMIT 1
            ",
            $email
        ));
        
        if (!$existing_ticket) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $existing_ticket = $wpdb->get_row($wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is constructed from $wpdb->prefix and is safe
                "
                SELECT t.* FROM `{$table_name}` t
                WHERE t.user_id = 0
                AND t.message LIKE %s
                AND t.status IN ('open', 'pending', 'in_progress')
                ORDER BY t.created_at DESC 
                LIMIT 1
                ",
                '%Email: ' . $email . '%'
            ));
        }
        
        return $existing_ticket;
    }
    
    if ($user_id > 0) {
        
        $cache_key = 'nexlifydesk_email_registered_user_duplicate_' . intval($user_id) . '_' . md5($subject);
        $existing_ticket = wp_cache_get($cache_key, 'nexlifydesk_tickets');
        
        if (false === $existing_ticket) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $existing_ticket = $wpdb->get_row($wpdb->prepare(
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is constructed from $wpdb->prefix and is safe
                "
                SELECT * FROM `{$table_name}` 
                WHERE user_id = %d 
                AND subject = %s 
                AND status IN ('open', 'pending', 'in_progress')
                AND created_at > DATE_SUB(NOW(), INTERVAL 30 DAY)
                ORDER BY created_at DESC 
                LIMIT 1
                ",
                $user_id,
                $subject
            ));
            //phpcs:enable
            
            if ($existing_ticket) {
                wp_cache_set($cache_key, $existing_ticket, 'nexlifydesk_tickets', 300);
            }
        }
        
        return $existing_ticket;
    }

    return null;
}

function nexlifydesk_fetch_custom_emails() {
    if (!extension_loaded('imap')) {
        add_action('admin_notices', function() {
            if (current_user_can('manage_options')) {
                echo '<div class="notice notice-error"><p><strong>NexlifyDesk:</strong> ' . 
                     esc_html__('IMAP extension is not installed on this server. Custom IMAP/POP3 email piping is not available. Please contact your hosting provider to enable the PHP IMAP extension.', 'nexlifydesk') . 
                     '</p></div>';
            }
        });
        return array('error' => 'IMAP extension is not available on this server.');
    }

    $settings = get_option('nexlifydesk_imap_settings', array());

    $host = $settings['host'] ?? '';
    $port = $settings['port'] ?? '';
    $encryption = $settings['encryption'] ?? '';
    $username = $settings['username'] ?? '';
    $password = nexlifydesk_get_safe_password($settings['password'] ?? '');
    $protocol = isset($settings['protocol']) ? $settings['protocol'] : 'imap';
    $delete_emails = isset($settings['delete_emails_after_fetch']) ? $settings['delete_emails_after_fetch'] : 1;
    
    if (empty($host) || empty($port) || empty($username) || empty($password)) {
        return array('error' => 'Custom IMAP/POP3 credentials not configured. Please configure Host, Port, Username, and Password.');
    }

    $tickets_created = 0;
    $replies_added = 0;
    $emails_processed = 0;

    if ($protocol === 'imap') {
        return nexlifydesk_process_imap_emails($host, $port, $encryption, $username, $password, $delete_emails);
    } elseif ($protocol === 'pop3') {
        return nexlifydesk_process_pop3_emails($host, $port, $encryption, $username, $password, $delete_emails);
    } else {
        return array('error' => 'Unsupported protocol. Please use IMAP or POP3.');
    }
}

function nexlifydesk_process_imap_emails($host, $port, $encryption, $username, $password, $delete_emails) {
    $mailbox = "{" . $host . ":" . $port . "/imap/" . $encryption . "}INBOX";
    
    $inbox = @imap_open($mailbox, $username, $password);
    if (!$inbox) {
        return array('error' => 'Failed to connect to IMAP server. Please check your credentials and server settings.');
    }
    
    $emails = imap_search($inbox, 'UNSEEN');
    if (!$emails) {
        imap_close($inbox);
        return array('success' => true, 'message' => 'No new emails found on IMAP server.', 'count' => 0);
    }
    
    $tickets_created = 0;
    $replies_added = 0;
    $emails_processed = count($emails);
    
    foreach ($emails as $email_number) {
        $overview = imap_fetch_overview($inbox, $email_number, 0)[0];
        
        $message = nexlifydesk_extract_email_body($inbox, $email_number);
        $subject = isset($overview->subject) ? $overview->subject : '';
        $from = isset($overview->from) ? $overview->from : '';
        $date = isset($overview->date) ? $overview->date : '';
        
        if (function_exists('nexlifydesk_decode_email_subject')) {
            $subject = nexlifydesk_decode_email_subject($subject);
        } else {
            $subject = nexlifydesk_decode_email_content($subject);
        }
        $from = nexlifydesk_decode_email_content($from);
        
        $parsed_from = function_exists('nexlifydesk_parse_email_from') ? 
            nexlifydesk_parse_email_from($from) : ['name' => '', 'email' => $from];
        
        $email_address = $parsed_from['email'];
        $sender_name = $parsed_from['name'];

        $is_admin_or_agent = function_exists('nexlifydesk_is_admin_or_agent_email') && nexlifydesk_is_admin_or_agent_email($email_address);
        
        $original_length = strlen($message);
        if ($is_admin_or_agent) {
            // Keep original message intact for admin/agent
        } else {
            if (function_exists('nexlifydesk_extract_clean_email_content')) {
                $message = nexlifydesk_extract_clean_email_content($message, $email_address);
            } else {
                $message = nexlifydesk_strip_email_thread($message, $email_address);
            }
        }
        $stripped_length = strlen($message);
        
        if (!$is_admin_or_agent && $stripped_length < ($original_length * 0.1) && $original_length > 100) {
            if (function_exists('nexlifydesk_strip_email_thread')) {
                $fallback_message = nexlifydesk_strip_email_thread($message, $email_address);
                if (strlen($fallback_message) > $stripped_length) {
                    $message = $fallback_message;
                }
            }
        }
        
        if (empty($email_address)) {
            $email_address = $from;
            if (preg_match('/<(.+?)>/', $from, $matches)) {
                $email_address = $matches[1];
            }
        }
        
        if (!filter_var($email_address, FILTER_VALIDATE_EMAIL)) {
            imap_setflag_full($inbox, $email_number, "\\Seen");
            continue;
        }

        $settings = get_option('nexlifydesk_imap_settings', array());
        if (function_exists('nexlifydesk_should_block_email') && 
            nexlifydesk_should_block_email($email_address, $subject, $message, $settings)) {
            imap_setflag_full($inbox, $email_number, "\\Seen");
            if ($delete_emails) {
                imap_delete($inbox, $email_number);
            }
            continue;
        }

        $user = get_user_by('email', $email_address);
        $user_id = $user ? $user->ID : 0;

        if (class_exists('NexlifyDesk_Rate_Limiter') && NexlifyDesk_Rate_Limiter::is_rate_limited($user_id, $email_address)) {
            
            if ($delete_emails) {
                imap_delete($inbox, $email_number);
            }
            continue;
        }

        $ticket_id_from_subject = function_exists('nexlifydesk_extract_ticket_id_from_subject') ? nexlifydesk_extract_ticket_id_from_subject($subject) : null;
        $existing_ticket = null;
        
        if ($ticket_id_from_subject && function_exists('nexlifydesk_get_ticket_by_ticket_id')) {
            $existing_ticket = nexlifydesk_get_ticket_by_ticket_id($ticket_id_from_subject);
        }
        
        if (!$existing_ticket) {
            $cleaned_subject = nexlifydesk_clean_email_subject($subject);
            
            if (function_exists('nexlifydesk_check_email_duplicate')) {
                $duplicate_data = array(
                    'user_id' => $user_id, 
                    'subject' => $cleaned_subject, 
                    'message' => $message,
                    'email' => $email_address,
                    'source' => 'email'
                );
                $existing_ticket = nexlifydesk_check_email_duplicate($duplicate_data);
            } else {
                $ticket_data = array(
                    'user_id' => $user_id, 
                    'subject' => $cleaned_subject, 
                    'message' => $message,
                    'email' => $email_address,
                    'source' => 'email'
                );
                $existing_ticket = NexlifyDesk_Tickets::check_for_duplicate_ticket($ticket_data);
            }
        }

        if ($existing_ticket) {
            $reply_data = array(
                'ticket_id' => $existing_ticket->id, 
                'user_id' => $user_id, 
                'message' => $message,
                'source' => 'email'
            );
            
            if (!$user_id && (!empty($sender_name) || !empty($email_address))) {
                $sender_info = [];
                if (!empty($sender_name)) {
                    $sender_info[] = "Name: {$sender_name}";
                }
                if (!empty($email_address)) {
                    $sender_info[] = "Email: {$email_address}";
                }
                if (!empty($sender_info)) {
                    $reply_data['message'] = "[Customer Details]\n" . implode("\n", $sender_info) . "\n\n[Reply]\n" . $message;
                }
            }
            
            NexlifyDesk_Tickets::add_reply($reply_data);
            $replies_added++;
        } else {
            $cleaned_subject = nexlifydesk_clean_email_subject($subject);
            
            $ticket_data = array(
                'user_id' => $user_id, 
                'subject' => $cleaned_subject, 
                'message' => $message,
                'source' => 'email',
                'email' => $email_address
            );
            
            if (!$user_id && (!empty($sender_name) || !empty($email_address))) {
                $sender_info = [];
                if (!empty($sender_name)) {
                    $sender_info[] = "Name: {$sender_name}";
                }
                if (!empty($email_address)) {
                    $sender_info[] = "Email: {$email_address}";
                }
                if (!empty($sender_info)) {
                    $ticket_data['message'] = "[Customer Details]\n" . implode("\n", $sender_info) . "\n\n[Message]\n" . $message;
                }
            }
            
            $new_ticket_id = NexlifyDesk_Tickets::create_ticket($ticket_data);
            if ($new_ticket_id && !is_wp_error($new_ticket_id)) {
                $tickets_created++;
            }
            if (!$user_id && !empty($email_address) && $new_ticket_id && !is_wp_error($new_ticket_id)) {
                update_post_meta($new_ticket_id, 'customer_email', $email_address);
            }
        }

        imap_setflag_full($inbox, $email_number, "\\Seen");
        if ($delete_emails) {
            imap_delete($inbox, $email_number);
        }
    }
    
    if ($delete_emails) {
        imap_expunge($inbox);
    }
    
    imap_close($inbox);
    
    return array(
        'success' => true,
        'message' => "Successfully processed {$emails_processed} emails. Created {$tickets_created} tickets and added {$replies_added} replies.",
        'tickets_created' => $tickets_created,
        'replies_added' => $replies_added,
        'emails_processed' => $emails_processed
    );
}

function nexlifydesk_process_pop3_emails($host, $port, $encryption, $username, $password, $delete_emails) {
    $mailbox = "{" . $host . ":" . $port . "/pop3/" . $encryption . "}INBOX";
    
    $inbox = @imap_open($mailbox, $username, $password);
    if (!$inbox) {
        return array('error' => 'Failed to connect to POP3 server. Please check your credentials and server settings.');
    }
    
    $num_msgs = imap_num_msg($inbox);
    $tickets_created = 0;
    $replies_added = 0;
    $emails_processed = $num_msgs;
    
    for ($i = 1; $i <= $num_msgs; $i++) {
        $header = imap_headerinfo($inbox, $i);
        $overview = imap_fetch_overview($inbox, $i, 0)[0];
        
        $message = nexlifydesk_extract_email_body($inbox, $i);
        $subject = isset($overview->subject) ? $overview->subject : (isset($header->subject) ? $header->subject : '');
        $from = isset($overview->from) ? $overview->from : (isset($header->from) ? $header->from : '');
        $date = isset($overview->date) ? $overview->date : (isset($header->date) ? $header->date : '');
        
        if (function_exists('nexlifydesk_decode_email_subject')) {
            $subject = nexlifydesk_decode_email_subject($subject);
        } else {
            $subject = nexlifydesk_decode_email_content($subject);
        }
        $from = nexlifydesk_decode_email_content($from);
        
        $parsed_from = function_exists('nexlifydesk_parse_email_from') ? 
            nexlifydesk_parse_email_from($from) : ['name' => '', 'email' => $from];
        
        $email_address = $parsed_from['email'];
        $sender_name = $parsed_from['name'];
        
        $is_admin_or_agent = function_exists('nexlifydesk_is_admin_or_agent_email') && nexlifydesk_is_admin_or_agent_email($email_address);
        
        $original_length = strlen($message);
        if ($is_admin_or_agent) {
            // Keep original message intact for admin/agent
        } else {
            if (function_exists('nexlifydesk_extract_clean_email_content')) {
                $message = nexlifydesk_extract_clean_email_content($message, $email_address);
            } else {
                $message = nexlifydesk_strip_email_thread($message, $email_address);
            }
        }
        $stripped_length = strlen($message);
        
        if (!$is_admin_or_agent && $stripped_length < ($original_length * 0.1) && $original_length > 100) {
            if (function_exists('nexlifydesk_strip_email_thread')) {
                $fallback_message = nexlifydesk_strip_email_thread($message, $email_address);
                if (strlen($fallback_message) > $stripped_length) {
                    $message = $fallback_message;
                }
            }
        }
        
        if (empty($email_address)) {
            $email_address = $from;
            if (preg_match('/<(.+?)>/', $from, $matches)) {
                $email_address = $matches[1];
            }
        }
        
        if (!filter_var($email_address, FILTER_VALIDATE_EMAIL)) {
            if ($delete_emails) {
                imap_delete($inbox, $i);
            }
            continue;
        }

        $settings = get_option('nexlifydesk_imap_settings', array());
        if (function_exists('nexlifydesk_should_block_email') && 
            nexlifydesk_should_block_email($email_address, $subject, $message, $settings)) {
            if ($delete_emails) {
                imap_delete($inbox, $i);
            }
            continue;
        }

        $user = get_user_by('email', $email_address);
        $user_id = $user ? $user->ID : 0;

        if (class_exists('NexlifyDesk_Rate_Limiter') && NexlifyDesk_Rate_Limiter::is_rate_limited($user_id, $email_address)) {
            
            if ($delete_emails) {
                imap_delete($inbox, $i);
            }
            continue;
        }

        $ticket_id_from_subject = function_exists('nexlifydesk_extract_ticket_id_from_subject') ? nexlifydesk_extract_ticket_id_from_subject($subject) : null;
        $existing_ticket = null;
        
        if ($ticket_id_from_subject && function_exists('nexlifydesk_get_ticket_by_ticket_id')) {
            $existing_ticket = nexlifydesk_get_ticket_by_ticket_id($ticket_id_from_subject);
        }
        
        if (!$existing_ticket) {
            $cleaned_subject = nexlifydesk_clean_email_subject($subject);
            
            if (function_exists('nexlifydesk_check_email_duplicate')) {
                $duplicate_data = array(
                    'user_id' => $user_id, 
                    'subject' => $cleaned_subject, 
                    'message' => $message,
                    'email' => $email_address,
                    'source' => 'email'
                );
                $existing_ticket = nexlifydesk_check_email_duplicate($duplicate_data);
            } else {
                $ticket_data = array(
                    'user_id' => $user_id, 
                    'subject' => $cleaned_subject, 
                    'message' => $message,
                    'email' => $email_address,
                    'source' => 'email'
                );
                $existing_ticket = NexlifyDesk_Tickets::check_for_duplicate_ticket($ticket_data);
            }
        }

        if ($existing_ticket) {
            $reply_data = array(
                'ticket_id' => $existing_ticket->id, 
                'user_id' => $user_id, 
                'message' => $message,
                'source' => 'email'
            );
            
            if (!$user_id && (!empty($sender_name) || !empty($email_address))) {
                $sender_info = [];
                if (!empty($sender_name)) {
                    $sender_info[] = "Name: {$sender_name}";
                }
                if (!empty($email_address)) {
                    $sender_info[] = "Email: {$email_address}";
                }
                if (!empty($sender_info)) {
                    $reply_data['message'] = "[Customer Details]\n" . implode("\n", $sender_info) . "\n\n[Reply]\n" . $message;
                }
            }
            
            NexlifyDesk_Tickets::add_reply($reply_data);
            $replies_added++;
        } else {
            $cleaned_subject = nexlifydesk_clean_email_subject($subject);
            
            $ticket_data = array(
                'user_id' => $user_id, 
                'subject' => $cleaned_subject, 
                'message' => $message,
                'source' => 'email',
                'email' => $email_address
            );
            
            if (!$user_id && (!empty($sender_name) || !empty($email_address))) {
                $sender_info = [];
                if (!empty($sender_name)) {
                    $sender_info[] = "Name: {$sender_name}";
                }
                if (!empty($email_address)) {
                    $sender_info[] = "Email: {$email_address}";
                }
                if (!empty($sender_info)) {
                    $ticket_data['message'] = "[Customer Details]\n" . implode("\n", $sender_info) . "\n\n[Message]\n" . $message;
                }
            }
            
            $new_ticket_id = NexlifyDesk_Tickets::create_ticket($ticket_data);
            if ($new_ticket_id && !is_wp_error($new_ticket_id)) {
                $tickets_created++;
            }
        }
        if ($delete_emails) {
            imap_delete($inbox, $i);
        }
    }
    
    if ($delete_emails) {
        imap_expunge($inbox);
    }
    
    imap_close($inbox);
    
    return array(
        'success' => true,
        'message' => "Successfully processed {$emails_processed} emails. Created {$tickets_created} tickets and added {$replies_added} replies.",
        'tickets_created' => $tickets_created,
        'replies_added' => $replies_added,
        'emails_processed' => $emails_processed
    );
}

/**
 * @param string $email_address The email address to check
 * @return bool True if email is from admin/agent, false otherwise
 */
function nexlifydesk_is_admin_or_agent_email($email_address) {
    if (empty($email_address) || !filter_var($email_address, FILTER_VALIDATE_EMAIL)) {
        return false;
    }
    
    if (preg_match('/<(.+?)>/', $email_address, $matches)) {
        $email_address = $matches[1];
    }
    $email_address = trim($email_address);
    
    $user = get_user_by('email', $email_address);
    if (!$user) {
        return false;
    }
    
    $is_admin = in_array('administrator', $user->roles);
    $is_agent = in_array('nexlifydesk_agent', $user->roles);
    
    $result = $is_admin || $is_agent;
    
    return $result;
}

/**
 * 
 * @param string $email_content The full email content
 * @param string $from_email Optional email address to check if sender is admin/agent
 * @return string Clean reply content without thread history
 */
function nexlifydesk_strip_email_thread($email_content, $from_email = '') {
    if (empty($email_content)) {
        return $email_content;
    }
    if (!empty($from_email) && nexlifydesk_is_admin_or_agent_email($from_email)) {
        return $email_content;
    }
    
    $body = str_replace(["\r\n", "\r"], "\n", $email_content);
    if (preg_match('/^[A-Za-z0-9+\/=\s]+$/', $body) && base64_decode($body, true) !== false) {
        $decoded_base64 = base64_decode($body);
        if (strlen($decoded_base64) > strlen($body) * 0.5) {
            $body = $decoded_base64;
        }
    }
    if (strpos($body, '=') !== false) {
        if (preg_match('/=\n/', $body) || preg_match('/=[0-9A-F]{2}/', $body)) {
            $body = quoted_printable_decode($body);
        }
    }
    if (strpos($body, '=?') !== false) {
        $body = preg_replace_callback('/=\?([^?]*)\?([BQ])\?([^?]*)\?=/', function($matches) {
            $charset = $matches[1];
            $encoding = strtoupper($matches[2]);
            $text = $matches[3];
            if ($encoding === 'B') {
                $decoded = base64_decode($text);
            } elseif ($encoding === 'Q') {
                $decoded = quoted_printable_decode(str_replace('_', ' ', $text));
            } else {
                return $matches[0];
            }
            if (strcasecmp($charset, 'UTF-8') !== 0) {
                $decoded = mb_convert_encoding($decoded, 'UTF-8', $charset);
            }
            return $decoded;
        }, $body);
    }
    if (!mb_check_encoding($body, 'UTF-8')) {
        $body = mb_convert_encoding($body, 'UTF-8', 'auto');
    }
    $is_html = (stripos($body, '<html') !== false || stripos($body, '<body') !== false || strpos($body, '<') !== false);
    if ($is_html) {
        if (function_exists('mb_convert_encoding')) {
            $body = mb_convert_encoding($body, 'HTML-ENTITIES', 'UTF-8');
        }
        $body = preg_replace('/<br\s*\/?>(?i)/', "\n", $body);
        $body = preg_replace('/<\/p>(?i)/', "\n\n", $body);
        $body = preg_replace('/<\/div>(?i)/', "\n", $body);
        $body = preg_replace('/<\/h[1-6]>(?i)/', "\n\n", $body);
        $body = preg_replace('/<\/li>(?i)/', "\n", $body);
        $body = preg_replace('/<hr\s*\/?>(?i)/', "\n---\n", $body);
        $body = wp_strip_all_tags($body);
        $body = html_entity_decode($body, ENT_QUOTES, 'UTF-8');
    }
    $body = preg_replace('/\n{3,}/u', "\n\n", $body);
    $header_pattern = '/^On[\s\S]+?wrote:\s*$/mus';
    $reply_separator_regexes = [
        $header_pattern,
        '/^Am[\s\S]+?schrieb:\s*$/mus',
        '/^Le[\s\S]+?écrit\s*:/mus',
        '/^Il[\s\S]+?scritto\s*:/mus',
        '/^El[\s\S]+?escribió\s*:/mus',
        '/^Em[\s\S]+?escreveu\s*:/mus',
        '/^At[\s\S]+?wrote:\s*$/mus',
        '/^From:[\s\S]+?$/mus',
        '/^Sent:[\s\S]+?$/mus',
        '/^Date:[\s\S]+?$/mus',
        '/^To:[\s\S]+?$/mus',
        '/^Subject:[\s\S]+?$/mus',
        '/^Reply above this line$/mus',
        '/^Please reply above this line$/mus',
        '/^Do not reply below this line$/mus',
        '/^-----Original Message-----$/mus',
        '/^-{2,}\s*On[\s\S]+wrote\s*-{2,}$/mus',
        '/=+$/mus',
        '/-{10,}/u',
        '/_{10,}/u',
    ];
    $separator_pos = false;
    foreach ($reply_separator_regexes as $regex) {
        if (preg_match($regex, $body, $match, PREG_OFFSET_CAPTURE)) {
            $separator_pos = $match[0][1];
            break;
        }
    }
    if ($separator_pos !== false) {
        $body = substr($body, 0, $separator_pos);
    }
    $body = preg_replace('/^\s*(>|\|).*/m', '', $body);
    $body = preg_replace('/\n{3,}/u', "\n\n", $body);
    $clean_content = trim($body);
    if (empty($clean_content)) {
        $clean_content = '[Customer replied via email]';
    }
    return $clean_content;
}

/**
 * Safe password retrieval with strict encryption enforcement
 * Only returns decrypted passwords, never plain text
 */
function nexlifydesk_get_safe_password($encrypted_password) {
    if (empty($encrypted_password)) {
        return '';
    }
    
    if (nexlifydesk_is_encrypted($encrypted_password)) {
        $decrypted = nexlifydesk_decrypt($encrypted_password);
        if ($decrypted !== $encrypted_password) {
            return $decrypted;
        }
    }
    
    return '';
}

/**
 * @param resource $inbox IMAP connection resource
 * @param int $message_number Message number
 * @return string Extracted email body content
 */
function nexlifydesk_extract_email_body($inbox, $message_number) {
    $structure = imap_fetchstructure($inbox, $message_number);
    $message = '';
    $is_html = false;
    
    if (isset($structure->parts) && is_array($structure->parts)) {
        for ($i = 0; $i < count($structure->parts); $i++) {
            $part = $structure->parts[$i];
            $part_number = $i + 1;
            
            if ($part->type == 0) { 
                $body = imap_fetchbody($inbox, $message_number, $part_number);
                
                if (isset($part->encoding)) {
                    switch ($part->encoding) {
                        case 3: // BASE64
                            $body = base64_decode($body);
                            break;
                        case 4:
                            $body = quoted_printable_decode($body);
                            break;
                    }
                }
                
                if (isset($part->subtype)) {
                    if (strtoupper($part->subtype) == 'PLAIN') {
                        $message = $body;
                        $is_html = false;
                        break;
                    } elseif (strtoupper($part->subtype) == 'HTML' && empty($message)) {
                        $message = $body;
                        $is_html = true;
                    }
                }
            }
        }
    } else {
        $message = imap_fetchbody($inbox, $message_number, 1);
        if (empty($message)) {
            $message = imap_body($inbox, $message_number);
        }
        
        if (isset($structure->encoding)) {
            switch ($structure->encoding) {
                case 3: // BASE64
                    $message = base64_decode($message);
                    break;
                case 4:
                    $message = quoted_printable_decode($message);
                    break;
            }
        }
        
        $message = nexlifydesk_decode_email_content($message);
        
        if (isset($structure->subtype) && strtoupper($structure->subtype) == 'HTML') {
            $is_html = true;
        }
    }
    
    if (empty($message)) {
        $message = imap_fetchbody($inbox, $message_number, 1.1);
        if (empty($message)) {
            $message = imap_fetchbody($inbox, $message_number, 1);
        }
        if (empty($message)) {
            $message = imap_body($inbox, $message_number);
        }
    }
    
    if ($is_html && !empty($message)) {
        $message = nexlifydesk_html_to_plain_text($message);
    } elseif (!$is_html && !empty($message) && nexlifydesk_is_html_content($message)) {
        $message = nexlifydesk_html_to_plain_text($message);
    }
    
    return $message;
}

/**
 * 
 * @param string $html_content The HTML content to convert
 * @return string Plain text version of the HTML
 */
function nexlifydesk_html_to_plain_text($html_content) {
    $html_content = preg_replace('/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/mi', '', $html_content);
    $html_content = preg_replace('/<style\b[^<]*(?:(?!<\/style>)<[^<]*)*<\/style>/mi', '', $html_content);
    $html_content = preg_replace('/<div[^>]*style="[^"]*max-height:\s*0[^"]*"[^>]*>.*?<\/div>/si', '', $html_content);
    $html_content = preg_replace('/<br\s*\/?>/i', "\n", $html_content);
    $html_content = preg_replace('/<\/p>/i', "\n\n", $html_content);
    $html_content = preg_replace('/<\/div>/i', "\n", $html_content);
    $html_content = preg_replace('/<\/h[1-6]>/i', "\n\n", $html_content);
    $html_content = preg_replace('/<\/tr>/i', "\n", $html_content);
    $html_content = preg_replace('/<\/td>/i', "\t", $html_content);
    $html_content = preg_replace('/<\/li>/i', "\n", $html_content);
    $html_content = preg_replace('/<a\s[^>]*href\s*=\s*["\']([^"\']*)["\'][^>]*>([^<]*)<\/a>/i', '$2 ($1)', $html_content);
    $html_content = preg_replace('/\n\s*\n/', "\n\n", $html_content);
    $html_content = preg_replace('/[ \t]+/', ' ', $html_content);
    $html_content = trim($html_content);
    $html_content = preg_replace('/Support Ticket Created.*?(?=\[Customer Details\]|\[Message\]|$)/s', '', $html_content);
    $html_content = preg_replace('/Your ticket #T\d+ has been created.*?(?=\[Customer Details\]|\[Message\]|$)/s', '', $html_content);
    $html_content = preg_replace('/Hello [^,]+,.*?(?=\[Customer Details\]|\[Message\]|$)/s', '', $html_content);
    $html_content = preg_replace('/Thank you for contacting.*?(?=\[Customer Details\]|\[Message\]|$)/s', '', $html_content);
    $html_content = preg_replace('/Your ticket ID is:.*?(?=\[Customer Details\]|\[Message\]|$)/s', '', $html_content);
    $html_content = preg_replace('/Ticket Summary.*?(?=\[Customer Details\]|\[Message\]|$)/s', '', $html_content);
    $html_content = preg_replace('/Subject:.*?Submitted:.*?(?=\[Customer Details\]|\[Message\]|$)/s', '', $html_content);
    $html_content = preg_replace('/Here is the message we received from you:.*?(?=\[Customer Details\]|\[Message\]|$)/s', '', $html_content);
    $html_content = preg_replace('/View Ticket Online.*?(?=\[Customer Details\]|\[Message\]|$)/s', '', $html_content);
    $html_content = preg_replace('/P\.S\. You can also.*?(?=\[Customer Details\]|\[Message\]|$)/s', '', $html_content);
    $html_content = preg_replace('/© \d{4}.*?All rights reserved\./s', '', $html_content);
    $html_content = preg_replace('/\n\s*\n/', "\n\n", $html_content);
    $html_content = trim($html_content);
    
    return $html_content;
}

/**
 * 
 * @param string $content The email content to analyze
 * @return bool True if content appears to be HTML, false otherwise
 */
function nexlifydesk_is_html_content($content) {
    $html_patterns = [
        '/<html[^>]*>/i',
        '/<body[^>]*>/i',
        '/<div[^>]*>/i',
        '/<table[^>]*>/i',
        '/<p[^>]*>/i',
        '/<br\s*\/?>/i',
        '/<title[^>]*>/i',
        '/<h[1-6][^>]*>/i',
        '/<a\s[^>]*href/i',
        '/<img\s[^>]*src/i'
    ];
    
    $html_count = 0;
    foreach ($html_patterns as $pattern) {
        if (preg_match($pattern, $content)) {
            $html_count++;
        }
    }

    return $html_count >= 2;
}