(function($) {
    'use strict';

    window.nexlifydesk_vars = window.nexlifydesk_vars || window.nexlifydesk_admin_vars || {};

    let isPluginValid = true;

    function disablePluginFunctionality(message) {
        $('form#nexlifydesk-new-ticket, form#nexlifydesk-reply-form, form#nexlifydesk-category-form, form#nexlifydesk-tickets-filter').each(function() {
            $(this).find('input, textarea, select, button').prop('disabled', true);
            $(this).prepend('<div class="nexlifydesk-error integrity-error" style="color: red; margin-bottom: 10px;">' + message + '</div>');
        });

        $('.delete-category, .ticket-status-select, .ticket-agent-select, .delete-position, .edit-position, .page-title-action').off('click change').css('opacity', '0.5').css('cursor', 'not-allowed');

        $('#nexlifydesk-message, .nexlifydesk-form-messages').addClass('error').text(message).show();
    }

    $(document).ready(function() {
        
        $('.page-title-action').on('click', function(e) {
            e.preventDefault();
            var $form = $('#nexlifydesk-category-form');
            var $button = $(this);
            
            $form.slideToggle(300, function() {
                if ($form.is(':visible')) {
                    var cancelText = (typeof nexlifydesk_admin_vars !== 'undefined' && nexlifydesk_admin_vars.cancel_text) || 
                                   (typeof nexlifydesk_vars !== 'undefined' && nexlifydesk_vars.cancel_text) || 'Cancel';
                    $button.text(cancelText);
                    $form.find('#category_name').focus();
                    
                    $form.find('.nexlifydesk-form-messages').removeClass('success error notice').empty();
                    
                    if ($form.find('#category_name').val() || $form.find('#category_description').val()) {
                        $form[0].reset();
                    }
                } else {
                    var addNewText = (typeof nexlifydesk_admin_vars !== 'undefined' && nexlifydesk_admin_vars.add_new_text) || 
                                   (typeof nexlifydesk_vars !== 'undefined' && nexlifydesk_vars.add_new_text) || 'Add New';
                    $button.text(addNewText);
                }
            });
        });

        $('.delete-category').on('click', function(e) {
            e.preventDefault();
            if (!isPluginValid) return; 
            
            var categoryId = $(this).data('id');
            var $button = $(this);
            
            if (!confirm('Are you sure you want to delete this category? This action cannot be undone.')) {
                return;
            }
            
            $.ajax({
                url: nexlifydesk_admin_vars.ajaxurl,
                type: 'POST',
                data: {
                    action: 'nexlifydesk_delete_category',
                    nonce: nexlifydesk_admin_vars.nonce,
                    category_id: categoryId
                },
                success: function(response) {
                    if (response.success) {
                        alert(response.data);
                        location.reload();
                    } else {
                        if (response.data && typeof response.data === 'object' && response.data.type === 'reassignment_required') {
                            showReassignmentModal(categoryId, response.data);
                        } else {
                            alert(response.data || 'Failed to delete category.');
                        }
                    }
                },
                error: function() {
                    alert('An error occurred while deleting the category.');
                }
            });
        });

        function showReassignmentModal(categoryId, data) {
            var modalHtml = '<div id="category-reassignment-modal" style="position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.7); z-index: 10000; display: flex; align-items: center; justify-content: center;">';
            modalHtml += '<div style="background: white; padding: 30px; border-radius: 8px; max-width: 500px; width: 90%;">';
            modalHtml += '<h2 style="margin-top: 0; color: #d63638;">⚠️ Warning: Category Deletion</h2>';
            modalHtml += '<p><strong>' + data.message + '</strong></p>';
            modalHtml += '<p style="color: #666; margin: 15px 0;">This action cannot be undone. All tickets will be moved to the selected category.</p>';
            modalHtml += '<div style="margin: 20px 0;">';
            modalHtml += '<label for="reassign-category" style="display: block; margin-bottom: 8px; font-weight: bold;">Select destination category:</label>';
            modalHtml += '<select id="reassign-category" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;">';
            modalHtml += '<option value="">-- Select Category --</option>';
            
            if (data.available_categories) {
                data.available_categories.forEach(function(category) {
                    modalHtml += '<option value="' + category.id + '">' + category.name + '</option>';
                });
            }
            
            modalHtml += '</select>';
            modalHtml += '</div>';
            modalHtml += '<div style="text-align: right; margin-top: 25px;">';
            modalHtml += '<button id="cancel-reassignment" style="margin-right: 10px; padding: 8px 16px; border: 1px solid #ddd; background: #f7f7f7; border-radius: 4px; cursor: pointer;">Cancel</button>';
            modalHtml += '<button id="confirm-reassignment" style="padding: 8px 16px; background: #d63638; color: white; border: none; border-radius: 4px; cursor: pointer;">Delete Category & Reassign Tickets</button>';
            modalHtml += '</div>';
            modalHtml += '</div>';
            modalHtml += '</div>';
            
            $('body').append(modalHtml);
            
            $('#cancel-reassignment, #category-reassignment-modal').on('click', function(e) {
                if (e.target === this) {
                    $('#category-reassignment-modal').remove();
                }
            });
            
            $('#confirm-reassignment').on('click', function() {
                var reassignTo = $('#reassign-category').val();
                if (!reassignTo) {
                    alert('Please select a category for reassignment.');
                    return;
                }
                
                $.ajax({
                    url: nexlifydesk_admin_vars.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'nexlifydesk_delete_category',
                        nonce: nexlifydesk_admin_vars.nonce,
                        category_id: categoryId,
                        reassign_to: reassignTo
                    },
                    success: function(response) {
                        $('#category-reassignment-modal').remove();
                        if (response.success) {
                            alert(response.data);
                            location.reload();
                        } else {
                            alert(response.data || 'Failed to delete category.');
                        }
                    },
                    error: function() {
                        $('#category-reassignment-modal').remove();
                        alert('An error occurred while deleting the category.');
                    }
                });
            });
        }

        $('#nexlifydesk-tickets-filter').on('submit', function(e) {
            if (!isPluginValid) return;
            var bulkAction = $('select[name="bulk_action"]').val();
            if (bulkAction && !confirm(nexlifydesk_admin_vars.bulk_action_confirm)) {
                e.preventDefault();
            }
        });

        $('#nexlifydesk-new-ticket').on('submit', function(e) {
            e.preventDefault();
            
            if (!isPluginValid) {
                return;
            }
            var $fileInput = $('#ticket-attachments');
            var files = $fileInput[0] ? $fileInput[0].files : [];
            var maxSize = (typeof nexlifydesk_vars.max_file_size !== 'undefined' ? nexlifydesk_vars.max_file_size : 2) * 1024 * 1024;
            var allowedTypes = (typeof nexlifydesk_vars.allowed_file_types !== 'undefined' ? nexlifydesk_vars.allowed_file_types : 'jpg,jpeg,png,pdf').split(',');
            var $wrapper = $fileInput.closest('.file-input-wrapper');
            var $error = $wrapper.find('.nexlifydesk-file-error');
            if ($error.length === 0) {
                $error = $('<div class="nexlifydesk-file-error" style="color:red;margin-top:5px;"></div>');
                $wrapper.append($error);
            }
            $error.text('');
            
            var totalSize = 0;
            for (var i = 0; i < files.length; i++) {
                var file = files[i];
                var fileExt = file.name.split('.').pop().toLowerCase();
                totalSize += file.size;
                
                if (file.size > maxSize) {
                    $error.text('File "' + file.name + '" is too large. Maximum size is ' + (maxSize / 1024 / 1024) + 'MB.');
                    $fileInput.val('');
                    return;
                }
                if (allowedTypes.indexOf(fileExt) === -1) {
                    $error.text('File type "' + fileExt + '" is not allowed. Allowed: ' + allowedTypes.join(', '));
                    $fileInput.val('');
                    return;
                }
            }
            
            var maxTotalSize = maxSize * 10;
            if (totalSize > maxTotalSize * 0.8) {
                var totalSizeMB = (totalSize / 1024 / 1024).toFixed(1);
                var maxTotalSizeMB = (maxTotalSize / 1024 / 1024).toFixed(1);
                if (totalSize > maxTotalSize) {
                    $error.text('Total file size (' + totalSizeMB + 'MB) exceeds limit of ' + maxTotalSizeMB + 'MB. Please reduce file sizes.');
                    $fileInput.val('');
                    return;
                } else {
                    $error.css('color', 'orange').text('Warning: Large file size (' + totalSizeMB + 'MB). Upload may take longer.');
                }
            }
            var form = $(this);
            var formData = new FormData(form[0]);
            
            formData.append('current_url', window.location.href);
            
            var submitBtn = $('#submit-ticket-btn');
            var buttonText = submitBtn.find('.button-text');
            var buttonSpinner = submitBtn.find('.button-spinner');
            
            submitBtn.prop('disabled', true);
            buttonText.hide();
            buttonSpinner.show();
            
            $.ajax({
                url: nexlifydesk_vars.ajaxurl,
                type: 'POST',
                data: formData,
                processData: false, 
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        if (response.data.redirect) {
                            window.location.href = response.data.redirect;
                            return;
                        }
                        
                        $('#nexlifydesk-message')
                            .removeClass('error')
                            .addClass('success')
                            .text(response.data.message || nexlifydesk_vars.ticket_submitted_text)
                            .show();
                        
                        form[0].reset();
                    } else {
                        var errorMsg = response.data || nexlifydesk_vars.error_message;
                        $('#nexlifydesk-message')
                            .removeClass('success')
                            .addClass('error')
                            .text(errorMsg)
                            .show();
                        if ($error.length) {
                            $error.text(errorMsg);
                        }
                    }
                },
                error: function(xhr, status, error) {
                    var errorMsg = nexlifydesk_vars.error_message || 'An error occurred. Please try again.';
                    
                    if (xhr.status === 413) {
                        errorMsg = 'Request too large. Please reduce file sizes or number of files and try again.';
                    } else if (xhr.status === 504 || xhr.status === 502) {
                        errorMsg = 'Server timeout. Please try again or contact support if the issue persists.';
                    } else if (xhr.status === 500) {
                        errorMsg = 'Server error. Please try again or contact support if the issue persists.';
                    }
                    
                    $('#nexlifydesk-message')
                        .removeClass('success')
                        .addClass('error')
                        .text(errorMsg)
                        .show();
                },
                complete: function() {
                    submitBtn.prop('disabled', false);
                    buttonText.show();
                    buttonSpinner.hide();
                }
            });
        });

        $('#nexlifydesk-category-form').on('submit', function(e) {
            e.preventDefault();
            if (!isPluginValid) return;

            var $form = $(this);
            var $submitButton = $form.find('input[type="submit"]');
            var $categoryName = $form.find('#category_name');
            var $categoryDescription = $form.find('#category_description');
            var vars = (typeof nexlifydesk_admin_vars !== 'undefined') ? nexlifydesk_admin_vars : 
                      (typeof nexlifydesk_vars !== 'undefined') ? nexlifydesk_vars : {};
            
            var $messageContainer = $form.find('.nexlifydesk-form-messages');
            if ($messageContainer.length === 0) {
                $messageContainer = $('<div class="nexlifydesk-form-messages"></div>').prependTo($form);
            }
            $messageContainer.removeClass('success error notice').empty();

            var categoryName = $categoryName.val().trim();
            if (!categoryName) {
                $messageContainer.addClass('error').text(vars.required_fields_text || 'Category name is required.');
                $categoryName.focus();
                return;
            }

            $submitButton.prop('disabled', true).val(vars.adding_text || 'Adding...');
            $categoryName.prop('disabled', true);
            $categoryDescription.prop('disabled', true);
            $form.addClass('nexlifydesk-form-loading');

            var formData = {
                action: 'nexlifydesk_add_category',
                nexlifydesk_category_nonce: $form.find('input[name="nexlifydesk_category_nonce"]').val(),
                category_name: categoryName,
                category_description: $categoryDescription.val().trim(),
                submit_category: true
            };

            $.ajax({
                url: nexlifydesk_admin_vars.ajaxurl, 
                type: 'POST',
                data: formData,
                timeout: 10000, 
                success: function(response) {
                    if (typeof response === 'string' && response.trim().startsWith('<!DOCTYPE html>')) {
                        location.reload();
                        return;
                    }
                    
                    if (response && response.success) {
                        $messageContainer.addClass('success').html('<strong>Success!</strong> ' + (response.data.message || 'Category added successfully.'));
                        
                        $form[0].reset();
                        
                        setTimeout(function() {
                            $form.slideUp(300, function() {
                                location.reload();
                            });
                        }, 1500);
                    } else {
                        var errorMessage = 'Failed to add category.';
                        if (response && response.data) {
                            errorMessage = typeof response.data === 'string' ? response.data : (response.data.message || errorMessage);
                        }
                        $messageContainer.addClass('error').html('<strong>Error:</strong> ' + errorMessage);
                        $categoryName.focus();
                    }
                },
                error: function(xhr, status, error) {
                    var errorText = vars.error_occurred_text || 'An error occurred. Please try again.';
                    if (status === 'timeout') {
                        errorText = 'Request timed out. Please try again.';
                    } else if (xhr.status === 403) {
                        errorText = 'Access denied. Please refresh the page and try again.';
                    }
                    $messageContainer.addClass('error').html('<strong>Error:</strong> ' + errorText);
                    $categoryName.focus();
                },
                complete: function() {
                    $submitButton.prop('disabled', false).val(vars.add_category_text || 'Add Category');
                    $categoryName.prop('disabled', false);
                    $categoryDescription.prop('disabled', false);
                    $form.removeClass('nexlifydesk-form-loading');
                }
            });
        });

        $(document).on('submit', '#nexlifydesk-reply-form', function(e) {
            e.preventDefault();
            if (!isPluginValid) return;

            var $form = $(this);
            var $submitButton = $form.find('button[type="submit"]');
            var $messageContainer = $form.find('.nexlifydesk-form-messages');

            if ($messageContainer.length === 0) {
                $messageContainer = $('<div class="nexlifydesk-form-messages"></div>').prependTo($form);
            }

            $messageContainer.empty();

            var isAdminSingle = $('.nexlifydesk-admin-single-ticket-ui').length > 0;
            var originalText = $submitButton.text();
            var sendingText = (typeof nexlifydesk_vars.sending_text !== 'undefined') ? nexlifydesk_vars.sending_text : 'Sending...';
            var submittingText = (typeof nexlifydesk_vars.submitting_text !== 'undefined') ? nexlifydesk_vars.submitting_text : 'Submitting...';

            if (isAdminSingle) {
                $submitButton.prop('disabled', true).text(sendingText);
            } else {
                $submitButton.prop('disabled', true).text(submittingText);
            }

            var formData = new FormData(this);
            formData.append('action', 'nexlifydesk_add_reply');
            formData.append('nonce', $form.find('input[name="nonce"]').val());

            if ($('body').hasClass('wp-admin') || isAdminSingle) {
                formData.append('context', 'admin');
            } else {
                formData.append('context', 'frontend');
            }

            var $fileInput = $form.find('input[type="file"][name^="attachments"]');
            var files = $fileInput[0] ? $fileInput[0].files : [];
            var maxSize = (typeof nexlifydesk_vars.max_file_size !== 'undefined' ? nexlifydesk_vars.max_file_size : 2) * 1024 * 1024;
            var allowedTypes = (typeof nexlifydesk_vars.allowed_file_types !== 'undefined' ? nexlifydesk_vars.allowed_file_types : 'jpg,jpeg,png,pdf').split(',');
            var $error = $form.find('.nexlifydesk-file-error');
            if ($error.length === 0) {
                $error = $('<div class="nexlifydesk-file-error" style="color:red;margin-top:5px;"></div>');
                $fileInput.after($error);
            }
            $error.text('');
            for (var i = 0; i < files.length; i++) {
                var file = files[i];
                var fileExt = file.name.split('.').pop().toLowerCase();
                if (file.size > maxSize) {
                    $error.text('File "' + file.name + '" is too large. Maximum size is ' + (maxSize / 1024 / 1024) + 'MB.');
                    $fileInput.val('');
                    $submitButton.prop('disabled', false).text(originalText);
                    return;
                }
                if (allowedTypes.indexOf(fileExt) === -1) {
                    $error.text('File type "' + fileExt + '" is not allowed. Allowed: ' + allowedTypes.join(', '));
                    $fileInput.val('');
                    $submitButton.prop('disabled', false).text(originalText);
                    return;
                }
            }
            
            $.ajax({
                url: nexlifydesk_vars.ajaxurl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        $messageContainer.removeClass('error').addClass('success').text(response.data.message);
                        if (response.data.reply_html) {
                            var $repliesList = $('.nexlifydesk-frontend-container .conversation, .nexlifydesk-frontend-container #nexlifydesk-replies-list, .nexlifydesk-admin-single-ticket-ui .conversation-history').first();
                            if ($repliesList.length) {
                                var $newReply = $(response.data.reply_html);
                                $newReply.hide();
                                $repliesList.append($newReply);
                                $newReply.addClass('message-newly-added');
                                $newReply[0].offsetHeight;
                                $newReply.fadeIn(200);
                                $('html, body').animate({
                                    scrollTop: $repliesList.children().last().offset().top - 100
                                }, 500);
                            }
                        }
                        $form[0].reset();
                    } else {
                        $messageContainer.removeClass('success').addClass('error').text(response.data);
                        if (response.data.includes('closed')) {
                            if (nexlifydesk_vars.ticket_form_url) {
                                $messageContainer.append(
                                    '<p><a href="' + nexlifydesk_vars.ticket_form_url + '" class="nexlifydesk-button">' +
                                    nexlifydesk_vars.create_new_ticket_text + '</a></p>'
                                );
                            }
                        }
                    }
                },
                error: function(xhr, status, error) {
                    $messageContainer.removeClass('success').addClass('error').text(nexlifydesk_vars.error_occurred_text);
                },
                complete: function() {
                    $submitButton.prop('disabled', false).text(originalText);
                }
            });
        });

        $(document).on('submit', '#nexlifydesk-note-form', function(e) {
            e.preventDefault();
            if (!isPluginValid) return;

            var $form = $(this);
            var $submitButton = $form.find('button[type="submit"]');
            var $messageContainer = $form.find('.nexlifydesk-form-messages');

            if ($messageContainer.length === 0) {
                $messageContainer = $('<div class="nexlifydesk-form-messages"></div>').prependTo($form);
            }

            $messageContainer.empty();

            var originalText = $submitButton.text();
            var addingText = (typeof nexlifydesk_vars.adding_text !== 'undefined') ? nexlifydesk_vars.adding_text : 'Adding...';

            $submitButton.prop('disabled', true).text(addingText);

            var formData = new FormData(this);
            formData.append('action', 'nexlifydesk_add_note');
            formData.append('nonce', $form.find('input[name="nonce"]').val());

            $.ajax({
                url: nexlifydesk_vars.ajaxurl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        $messageContainer.removeClass('error').addClass('success').text(response.data.message || 'Internal note added successfully!');
                        $form[0].reset();
                        if (response.data.note_html) {
                            var $notesList = $('.nexlifydesk-frontend-container .conversation, .nexlifydesk-frontend-container #nexlifydesk-replies-list, .nexlifydesk-admin-single-ticket-ui .conversation-history').first();
                            if ($notesList.length) {
                                var $newNote = $(response.data.note_html);
                                $newNote.hide();
                                $notesList.append($newNote);
                                $newNote.addClass('message-newly-added');
                                $newNote[0].offsetHeight;
                                $newNote.fadeIn(200, function() {
                                    $('html, body').animate({
                                        scrollTop: $newNote.offset().top - 100
                                    }, 500);
                                });
                            }
                        }
                        setTimeout(function() {
                            $messageContainer.removeClass('success').empty();
                        }, 1500);
                    } else {
                        $messageContainer.removeClass('success').addClass('error').text(response.data || 'An error occurred while adding the note.');
                    }
                },
                error: function(xhr, status, error) {
                    $messageContainer.removeClass('success').addClass('error').text(nexlifydesk_vars.error_occurred_text);
                },
                complete: function() {
                    $submitButton.prop('disabled', false).text(originalText);
                }
            });
        });

        $(document).on('change', 'input[type="file"][name^="attachments"]', function() {
            if (!isPluginValid) return;

            var files = this.files;
            var maxSize = nexlifydesk_vars.max_file_size * 1024 * 1024;
            var allowedTypes = nexlifydesk_vars.allowed_file_types.split(',');
            var $input = $(this);
            var $wrapper = $input.closest('.file-input-wrapper').length ? $input.closest('.file-input-wrapper') : $input.parent();
            var $fileList = $wrapper.find('.nexlifydesk-file-list');
            var $error = $wrapper.find('.nexlifydesk-file-error');

            if ($fileList.length === 0) {
                $fileList = $input.siblings('.nexlifydesk-file-list');
                if ($fileList.length === 0) {
                    $fileList = $('<ul class="nexlifydesk-file-list" style="margin:8px 0 0 0;padding:0;list-style:none;font-size:14px;"></ul>');
                    $input.after($fileList);
                }
            }
            if ($error.length === 0) {
                $error = $input.siblings('.nexlifydesk-file-error');
                if ($error.length === 0) {
                    $error = $('<div class="nexlifydesk-file-error" style="color:red;margin-top:5px;"></div>');
                    $fileList.after($error);
                }
            }

            $fileList.empty();
            $error.text('');

            var hasError = false;
            for (var i = 0; i < files.length; i++) {
                var file = files[i];
                var fileExt = file.name.split('.').pop().toLowerCase();
                if (file.size > maxSize) {
                    $error.text('File "' + file.name + '" is too large. Maximum size is ' + nexlifydesk_vars.max_file_size + 'MB.');
                    $input.val('');
                    hasError = true;
                    break;
                }
                if (allowedTypes.indexOf(fileExt) === -1) {
                    $error.text('File type "' + fileExt + '" is not allowed. Allowed: ' + allowedTypes.join(', '));
                    $input.val('');
                    hasError = true;
                    break;
                }
            }
            if (!hasError && files.length > 0) {
                for (var j = 0; j < files.length; j++) {
                    $fileList.append('<li>' + files[j].name + '</li>');
                }
            }
        });

        $(document).on('change', '.ticket-status-select', function() {
            if (!isPluginValid) return;
        
            var $select = $(this);
            var ticketId = $select.data('ticket-id');
            var newStatus = $select.val();
        
            if (!ticketId) {
                return;
            }
        
            $.ajax({
                url: nexlifydesk_admin_vars.ajaxurl,
                type: 'POST',
                data: {
                    action: 'nexlifydesk_update_status',
                    nonce: nexlifydesk_admin_vars.nonce,
                    ticket_id: ticketId,
                    status: newStatus
                },
                success: function(response) {
                    if (response.success) {
                        var $statusCell = $select.closest('td').length ? $select.closest('td') : $select.parent();
                        var statusText = (typeof nexlifydesk_admin_vars !== 'undefined' && nexlifydesk_admin_vars.status_updated_text)
                            ? nexlifydesk_admin_vars.status_updated_text
                            : (typeof nexlifydesk_vars !== 'undefined' && nexlifydesk_vars.status_updated_text)
                                ? nexlifydesk_vars.status_updated_text
                                : 'Status updated!';

                        var $notice = $('<div class="nexlifydesk-notice success">' +
                            '<span class="dashicons dashicons-yes"></span> ' +
                            statusText +
                            '</div>');

                        $statusCell.append($notice);

                        $select.css('border-color', '#46b450').css('box-shadow', '0 0 2px #46b450');

                        setTimeout(function() {
                            $notice.fadeOut(function() { $(this).remove(); });
                            $select.css('border-color', '').css('box-shadow', '');
                        }, 3000);
                    } else {
                        alert('Error updating status: ' + response.data);
                    }
                },
                error: function(xhr, status, error) {
                    alert('Error updating ticket status: ' + error);
                    $select.val($select.data('original-value'));
                },
                beforeSend: function() {
                    $select.data('original-value', $select.val());
                }
            });
        });

        $(document).on('change', '.ticket-agent-select', function() {
            if (!isPluginValid) return;

            var $select = $(this);
            var ticketId = $select.data('ticket-id');
            var newAgentId = $select.val();

            if (!ticketId) return;

            $select.prop('disabled', true);

            $.ajax({
                url: nexlifydesk_admin_vars.ajaxurl, 
                type: 'POST',
                data: {
                    action: 'nexlifydesk_admin_assign_ticket',
                    nonce: nexlifydesk_admin_vars.nonce,
                    ticket_id: ticketId,
                    user_id: newAgentId
                },
                success: function(response) {
                    if (response.success) {
                        var $notice = $('<div class="nexlifydesk-notice success">' +
                            '<span class="dashicons dashicons-yes"></span> ' +
                            (typeof nexlifydesk_admin_vars !== 'undefined' && nexlifydesk_admin_vars.agent_assigned_text
                                ? nexlifydesk_admin_vars.agent_assigned_text
                                : 'Ticket assigned successfully!') +
                            '</div>');
                        $select.after($notice);
                        setTimeout(function() {
                            $notice.fadeOut(function() { $(this).remove(); });
                        }, 3000);
                    } else {
                        alert('Error assigning agent: ' + response.data);
                    }
                },
                error: function() {
                    alert('Error assigning ticket.');
                },
                complete: function() {
                    $select.prop('disabled', false);
                }
            });
        });

        function fetchAdminTickets() {
            if (!isPluginValid) return;

            var $tableContainer = $('#nexlifydesk-tickets-table');
            var $filterForm = $('#nexlifydesk-tickets-filter');
            var formData = $filterForm.serialize();

            $tableContainer.html('<tr><td colspan="5">' + nexlifydesk_admin_vars.loading_tickets_text + '</td></tr>');

            $.ajax({
                url: nexlifydesk_admin_vars.ajaxurl,
                type: 'GET',
                data: formData + '&action=nexlifydesk_admin_get_tickets&nonce=' + nexlifydesk_admin_vars.nonce,
                success: function(response) {
                    if (response.success) {
                        var tickets = response.data;
                        var tableHtml = '<table><thead><tr>' +
                            '<th>' + nexlifydesk_admin_vars.ticket_id_header + '</th>' +
                            '<th>' + nexlifydesk_admin_vars.subject_header + '</th>' +
                            '<th>' + nexlifydesk_admin_vars.status_header + '</th>' +
                            '<th>' + nexlifydesk_admin_vars.assigned_to_header + '</th>' +
                            '<th>' + nexlifydesk_admin_vars.created_header + '</th>' +
                            '<th>' + nexlifydesk_admin_vars.actions_header + '</th>' +
                            '</tr></thead><tbody>';

                        if (tickets.length > 0) {
                            $.each(tickets, function(index, ticket) {
                                tableHtml += '<tr>';
                                tableHtml += '<td>' + ticket.ticket_id + '</td>';
                                tableHtml += '<td>' + ticket.subject + '</td>';
                                tableHtml += '<td class="status-' + ticket.status + '">' + ticket.status.charAt(0).toUpperCase() + ticket.status.slice(1) + '</td>';
                                tableHtml += '<td>' + (ticket.assigned_to_display_name ? ticket.assigned_to_display_name : 'Unassigned') + '</td>';
                                tableHtml += '<td>' + ticket.created_at + '</td>';
                                tableHtml += '<td><a href="?page=nexlifydesk_tickets&ticket_id=' + ticket.id + '">' + nexlifydesk_admin_vars.view_text + '</a></td>';
                                tableHtml += '</tr>';
                            });
                        } else {
                            tableHtml += '<tr><td colspan="5">' + nexlifydesk_admin_vars.no_tickets_found_text + '</td></tr>';
                        }

                        tableHtml += '</tbody></table>';
                        $tableContainer.html(tableHtml);

                    } else {
                        $tableContainer.html('<tr><td colspan="5">' + nexlifydesk_admin_vars.error_loading_tickets_text + ' ' + response.data + '</td></tr>');
                    }
                },
                error: function(xhr, status, error) {
                    $tableContainer.html('<tr><td colspan="5">' + nexlifydesk_admin_vars.ajax_error_loading_tickets_text + '</td></tr>');
                }
            });
        }

        if ($('#nexlifydesk-tickets-table').length) {
            if (isPluginValid) {
                fetchAdminTickets();
            }
        }

        $('#nexlifydesk-tickets-filter').on('submit', function(e) {
            e.preventDefault();
            if (isPluginValid) {
                fetchAdminTickets();
            }
        });

        $(document).on('click', '.delete-position', function(e) {
            e.preventDefault();
            if (!isPluginValid) return;
            var slug = $(this).data('slug');
            if (!confirm('Are you sure you want to delete this position?')) return;

            var $form = $('<form method="post" action="' + nexlifydesk_admin_vars.admin_post_url + '" style="display:none;"></form>');
            $form.append('<input type="hidden" name="action" value="nexlifydesk_delete_agent_position">');
            $form.append('<input type="hidden" name="delete_position" value="1">');
            $form.append('<input type="hidden" name="position_slug" value="' + slug + '">');
            $form.append('<input type="hidden" name="nexlifydesk_agent_position_nonce" value="' + nexlifydesk_admin_vars.position_nonce + '">');
            $('body').append($form);
            $form.submit();
        });

        $(document).on('click', '.edit-position', function(e) {
            e.preventDefault();
            if (!isPluginValid) return;
            var $row = $(this).closest('tr');
            var slug = $(this).data('slug');
            var name = $row.find('td').eq(0).text().trim();
            var assignedCaps = $row.data('capabilities') ? $row.data('capabilities').toString().split(',') : [];

            if ($row.next().hasClass('edit-position-row')) return;

            var formHtml = '<tr class="edit-position-row"><td colspan="4">';
            formHtml += '<form class="nexlifydesk-edit-position-form" method="post" action="' + nexlifydesk_vars.admin_post_url + '">';
            formHtml += '<input type="hidden" name="action" value="nexlifydesk_save_agent_position">';
            formHtml += '<input type="hidden" name="edit_position" value="1">';
            formHtml += '<input type="hidden" name="original_slug" value="' + slug + '">';
            formHtml += '<input type="hidden" name="nexlifydesk_agent_position_nonce" value="' + nexlifydesk_admin_vars.position_nonce + '">';
            formHtml += '<label>' + nexlifydesk_admin_vars.position_name_label + ' <input type="text" name="position_name" value="' + name + '" required></label> ';
            formHtml += '<label>' + nexlifydesk_admin_vars.position_slug_label + ' <input type="text" name="position_slug" value="' + slug + '" required></label> ';
            formHtml += '<br><strong>' + nexlifydesk_admin_vars.assign_capabilities_label + '</strong><br>';
            $.each(nexlifydesk_admin_vars.available_capabilities, function(capSlug, capName) {
                var checked = (assignedCaps.indexOf(capSlug) !== -1) ? 'checked' : '';
                formHtml += '<label style="margin-right:15px;"><input type="checkbox" name="position_capabilities[]" value="' + capSlug + '" ' + checked + '> ' + capName + '</label>';
            });
            formHtml += '<br><button type="submit" class="button button-primary">' + nexlifydesk_admin_vars.save_text + '</button> ';
            formHtml += '<button type="button" class="button cancel-edit-position">' + nexlifydesk_admin_vars.cancel_text + '</button>';
            formHtml += '</form></td></tr>';

            $row.after(formHtml);
        });

         $(document).on('click', '.cancel-edit-position', function() {
            $(this).closest('tr.edit-position-row').remove();
        });

        $('#reassign-orphaned-tickets').on('click', function(e) {
            e.preventDefault();
            if (!isPluginValid) return;
            
            var $button = $(this);
            var originalText = $button.text();
            
            $button.prop('disabled', true).text('Reassigning...');
            
            $.ajax({
                url: nexlifydesk_admin_vars.ajaxurl,
                type: 'POST',
                data: {
                    action: 'nexlifydesk_reassign_orphaned_tickets',
                    nonce: nexlifydesk_admin_vars.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert(response.data);
                        location.reload();
                    } else {
                        alert('Error: ' + response.data);
                    }
                },
                error: function() {
                    alert('Error occurred while reassigning tickets.');
                },
                complete: function() {
                    $button.prop('disabled', false).text(originalText);
                }
            });
        });
    });

})(jQuery);

jQuery(document).ready(function($) {
    initializeReportsCharts();
    
    $('#keep_data_on_uninstall').on('change', function() {
        if ($(this).is(':checked')) {
            $('#data-deletion-warning').slideUp();
        } else {
            $('#data-deletion-warning').slideDown();
        }
    });
    
    if (!$('#keep_data_on_uninstall').is(':checked')) {
        $('#data-deletion-warning').show();
    }
    
    $('.preview-email-template').on('click', function() {
        var editorId = $(this).data('editor');
        var content = $('#' + editorId).val();
        content = content
            .replace(/{user_name}/g, 'John Doe')
            .replace(/{reply_user_name}/g, 'Support Agent')
            .replace(/{ticket_id}/g, '12345')
            .replace(/{subject}/g, 'Sample Subject')
            .replace(/{reply_message}/g, 'This is a sample reply message.')
            .replace(/{updated_at}/g, '2025-06-19 12:00 PM');
        $('#preview-' + editorId).html(content).show();
    });
    
    const urlParams = new URLSearchParams(window.location.search);
    const ticketId = urlParams.get('ticket_id');
    
    if (ticketId && /^\d+$/.test(ticketId) && window.location.href.indexOf('ticket_id=') !== -1) {
        if (window.location.pathname.includes('support') || window.location.pathname.includes('ticket')) {
            // No Function Needed
        }
    }
    
    $(document).on('click', '.tab-link', function(e) {
        e.preventDefault();
        var tab = $(this).data('tab');
        $('.tab-link').removeClass('active');
        $(this).addClass('active');
        $('.tab-content').hide();
        $('#' + tab).show();
    });
});

function drawPieChart(ctx, data, labels, colors) {
    const total = data.reduce((a, b) => a + b, 0);
    let currentAngle = -Math.PI / 2;
    
    const centerX = ctx.canvas.width / 2;
    const centerY = ctx.canvas.height / 2;
    const radius = Math.min(centerX, centerY) - 20;
    
    data.forEach((value, index) => {
        const sliceAngle = (value / total) * 2 * Math.PI;
        
        ctx.beginPath();
        ctx.moveTo(centerX, centerY);
        ctx.arc(centerX, centerY, radius, currentAngle, currentAngle + sliceAngle);
        ctx.closePath();
        ctx.fillStyle = colors[index] || '#999';
        ctx.fill();
        ctx.strokeStyle = '#fff';
        ctx.lineWidth = 2;
        ctx.stroke();
        
        currentAngle += sliceAngle;
    });
}

function drawLineChart(ctx, data) {
    const padding = 40;
    const chartWidth = ctx.canvas.width - padding * 2;
    const chartHeight = ctx.canvas.height - padding * 2;
    ctx.clearRect(0, 0, ctx.canvas.width, ctx.canvas.height);
    ctx.strokeStyle = '#e0e0e0';
    ctx.lineWidth = 1;
    
    for (let i = 0; i <= 30; i += 5) {
        const x = padding + (i / 30) * chartWidth;
        ctx.beginPath();
        ctx.moveTo(x, padding);
        ctx.lineTo(x, padding + chartHeight);
        ctx.stroke();
    }
    
    if (data.length > 0) {
        ctx.strokeStyle = '#3498db';
        ctx.lineWidth = 3;
        ctx.beginPath();
        
        data.forEach((point, index) => {
            const x = padding + (index / (data.length - 1)) * chartWidth;
            const y = padding + chartHeight - (point.tickets / Math.max(...data.map(d => d.tickets)) * chartHeight);
            
            if (index === 0) {
                ctx.moveTo(x, y);
            } else {
                ctx.lineTo(x, y);
            }
        });
        
        ctx.stroke();
        ctx.fillStyle = '#3498db';
        data.forEach((point, index) => {
            const x = padding + (index / (data.length - 1)) * chartWidth;
            const y = padding + chartHeight - (point.tickets / Math.max(...data.map(d => d.tickets)) * chartHeight);
            
            ctx.beginPath();
            ctx.arc(x, y, 4, 0, 2 * Math.PI);
            ctx.fill();
        });
    }
}

function initializeReportsCharts() {
    if (typeof nexlifydesk_reports_data === 'undefined') {
        return;
    }
    
    const data = nexlifydesk_reports_data;

    const statusColorMap = {
        'closed': '#95a5a6',      // gray
        'in_progress': '#3498db', // blue
        'resolved': '#27ae60',    // green
        'open': '#f39c12',        // orange
    };
    const statusColors = data.status_labels.map(label => statusColorMap[label] || '#999');
    const statusCtx = document.getElementById('statusChart');
    if (statusCtx) {
        drawPieChart(statusCtx.getContext('2d'), data.status_data, data.status_labels, statusColors);
    }

    const priorityColorMap = {
        'low': '#27ae60',     // green
        'medium': '#f39c12',  // orange
        'high': '#e67e22',    // dark orange
        'urgent': '#e74c3c',  // red        
    };
    const priorityColors = data.priority_labels.map(label => priorityColorMap[label] || '#999');

    const priorityCtx = document.getElementById('priorityChart');
    if (priorityCtx) {
        drawPieChart(priorityCtx.getContext('2d'), data.priority_data, data.priority_labels, priorityColors);
    }

    const monthlyCtx = document.getElementById('monthlyChart');
    if (monthlyCtx) {
        drawLineChart(monthlyCtx.getContext('2d'), data.monthly_data);
    }
}

if (document.readyState === 'complete' || document.readyState === 'interactive') {
    initializeReportsCharts();
}

(function($) {
    'use strict';
    
    let ticketsData = [];
    
    function formatRelativeTime(dateString) {
        const now = new Date();
        const date = new Date(dateString);
        const diffInSeconds = Math.floor((now - date) / 1000);
        
        if (diffInSeconds < 60) {
            return 'just now';
        } else if (diffInSeconds < 3600) {
            const minutes = Math.floor(diffInSeconds / 60);
            return `${minutes} ${minutes === 1 ? 'minute' : 'minutes'} ago`;
        } else if (diffInSeconds < 86400) {
            const hours = Math.floor(diffInSeconds / 3600);
            return `${hours} ${hours === 1 ? 'hour' : 'hours'} ago`;
        } else if (diffInSeconds < 604800) {
            const days = Math.floor(diffInSeconds / 86400);
            return `${days} ${days === 1 ? 'day' : 'days'} ago`;
        } else {
            return date.toLocaleDateString();
        }
    }
    
    function getInitials(name) {
        if (!name) return 'NA';
        return name.split(' ')
            .map(word => word.charAt(0))
            .join('')
            .substring(0, 2)
            .toUpperCase();
    }
    
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    function truncateText(text, maxLength) {
        if (!text || text.length <= maxLength) return text;
        return text.substring(0, maxLength) + '...';
    }
    
    function loadTickets(status = 'all', priority = 'all', search = '') {
        if (!window.nexlifydesk_admin_vars) return;

        $('#nexlifydesk-ticket-grid').html(
            '<div class="loading-tickets"><span class="spinner is-active"></span>' +
            nexlifydesk_admin_vars.loading_tickets_text + '</div>'
        );

        $.ajax({
            url: nexlifydesk_admin_vars.ajaxurl,
            type: 'POST',
            data: {
                action: 'nexlifydesk_admin_get_tickets',
                status: status,
                priority: priority,
                search: search,
                nonce: nexlifydesk_admin_vars.nonce
            },
            success: function(response) {
                if (response.success && response.data) {
                    renderTickets(response.data);
                } else {
                    var errorMsg = response.data || nexlifydesk_admin_vars.no_tickets_found_text;
                    $('#nexlifydesk-ticket-grid').html('<p>' + errorMsg + '</p>');
                }
            },
            error: function() {
                var errorMsg = nexlifydesk_admin_vars.ajax_error_loading_tickets_text;
                $('#nexlifydesk-ticket-grid').html('<p style="color: red;">' + errorMsg + '</p>');
            }
        });
    }
    
    function renderTickets(tickets) {
        const $grid = $('#nexlifydesk-ticket-grid');

        if (!tickets || tickets.length === 0) {
            $grid.html('<div class="no-tickets">' + nexlifydesk_admin_vars.no_tickets_found_text + '</div>');
            return;
        }

        $grid.empty();

        $.each(tickets, function(index, ticket) {
            $grid.append(createTicketCard(ticket));
        });
    }

    function createTicketCard(ticket) {
        const assigneeName = ticket.assigned_to_display_name || 'Unassigned';
        const assigneeInitials = getInitials(assigneeName);
        const statusClass = 'status-' + ticket.status.replace('_', '-');
        const priorityClass = 'priority-' + ticket.priority;

        return $('<div class="ticket-card">')
            .data('id', ticket.id)
            .append(
                $('<div class="ticket-header">').append(
                    $('<div class="ticket-id">').text('#' + ticket.ticket_id)
                )
            )
            .append(
                $('<div class="ticket-title">').text(ticket.subject)
            )
            .append(
                $('<div class="ticket-description">').text(
                    truncateText(ticket.content, 100)
                )
            )
            .append(
                $('<div class="ticket-meta">')
                    .append(
                        $('<span class="status-badge ' + statusClass + '">').text(
                            ticket.status.charAt(0).toUpperCase() +
                            ticket.status.slice(1).replace('_', ' ')
                        )
                    )
                    .append(
                        $('<span class="priority-badge ' + priorityClass + '">').text(
                            ticket.priority.charAt(0).toUpperCase() + ticket.priority.slice(1)
                        )
                    )
            )
            .append(
                $('<div class="ticket-footer">')
                    .append(
                        $('<div class="assignee">')
                            .append(
                                $('<div class="avatar">').text(assigneeInitials)
                            )
                            .append(
                                $('<span>').text(assigneeName)
                            )
                    )
                    .append(
                        $('<div>').text(formatRelativeTime(ticket.created_at))
                    )
            )
            .on('click', function() {
                window.location.href = '?page=nexlifydesk_tickets&ticket_id=' + ticket.id;
            });
    }
    
    $(function() {
        if ($('.nexlifydesk-admin-ticket-list-ui').length) {

            let searchTimeout;
            $('#ticket-search').on('input', function() {
                clearTimeout(searchTimeout);
                let searchValue = $(this).val();
                if (searchValue.startsWith('#')) {
                    searchValue = searchValue.substring(1);
                }
                searchValue = searchValue.trim();

                searchTimeout = setTimeout(function() {
                    const status = $('#ticket-status-filter').val();
                    const priority = $('#ticket-priority-filter').val();
                    loadTickets(status, priority, searchValue);
                }, 500);
            });

            $('#ticket-status-filter').on('change', function() {
                const status = $(this).val();
                const priority = $('#ticket-priority-filter').val();
                let search = $('#ticket-search').val();
                if (search.startsWith('#')) {
                    search = search.substring(1);
                }
                search = search.trim();
                loadTickets(status, priority, search);
            });

            $('#ticket-priority-filter').on('change', function() {
                const status = $('#ticket-status-filter').val();
                const priority = $(this).val();
                let search = $('#ticket-search').val();
                if (search.startsWith('#')) {
                    search = search.substring(1);
                }
                search = search.trim();
                loadTickets(status, priority, search);
            });

            $('#new-ticket-btn').on('click', function() {
                const ticketFormUrl = nexlifydesk_admin_vars.ticket_form_url ||
                    '?page=nexlifydesk_tickets&action=new';
                window.location.href = ticketFormUrl;
            });
        }
    });

})(jQuery);

function initializeReportsPageUI() {
    if (!document.querySelector('.nexlifydesk-reports-page-ui')) {
        return;
    }

    document.querySelectorAll('.nexlifydesk-reports-page-ui .agent-name').forEach(function(element) {
        const name = element.textContent.trim();
        if (name) {
            const initials = getInitialsFromName(name);
            const avatarElement = document.createElement('div');
            avatarElement.className = 'agent-avatar';
            avatarElement.textContent = initials;
            element.prepend(avatarElement);
        }
    });

    document.querySelectorAll('.nexlifydesk-reports-page-ui .response-rate-value').forEach(function(element) {
        const rateText = element.textContent;
        const rate = parseInt(rateText, 10);
        if (!isNaN(rate)) {
            const barElement = document.createElement('div');
            barElement.className = 'rate-bar';
            const fillElement = document.createElement('div');
            fillElement.className = 'rate-fill';
            fillElement.style.width = rate + '%';
            barElement.appendChild(fillElement);
            
            const container = document.createElement('div');
            container.className = 'response-rate';
            container.appendChild(barElement);
            container.appendChild(element.cloneNode(true));
            
            element.parentNode.replaceChild(container, element);
        }
    });

    document.querySelectorAll('.nexlifydesk-reports-page-ui .activity-item').forEach(function(element) {
        const type = element.querySelector('.activity-icon').classList.contains('status-new') ? 'new' : 'reply';
        const iconElement = document.createElement('span');
        iconElement.className = 'dashicons ' + (type === 'new' ? 'dashicons-plus-alt' : 'dashicons-format-chat');
        element.querySelector('.activity-icon').appendChild(iconElement);
    });

    const metricIcons = [
        'dashicons-tickets-alt',      // Total Tickets
        'dashicons-clock',            // Active Tickets
        'dashicons-yes-alt',          // Closed Tickets
        'dashicons-performance'       // Avg Response Time
    ];

    document.querySelectorAll('.nexlifydesk-reports-page-ui .metric-card').forEach(function(element, index) {
        if (index < metricIcons.length) {
            const iconElement = document.createElement('div');
            iconElement.className = 'metric-icon';
            const dashicon = document.createElement('span');
            dashicon.className = 'dashicons ' + metricIcons[index];
            iconElement.appendChild(dashicon);
            element.prepend(iconElement);
        }
    });
}

function getInitialsFromName(name) {
    return name.split(' ')
        .map(part => part[0])
        .join('')
        .toUpperCase()
        .substring(0, 2);
}

jQuery(document).ready(function($) {
    initializeReportsPageUI();
});

if (document.readyState === 'complete' || document.readyState === 'interactive') {
    initializeReportsPageUI();
}

jQuery(document).ready(function($) {
    var $purgeBtn = $('#nexlifydesk_purge_data');
    if ($purgeBtn.length) {
        $purgeBtn.off('click').on('click', function() {
            var button = $(this);
            var resultDiv = $('#nexlifydesk_purge_data_result');
            resultDiv.html('');

            var ajaxurl = (typeof nexlifydesk_vars !== 'undefined' && nexlifydesk_vars.ajaxurl) ? nexlifydesk_vars.ajaxurl : (typeof ajaxurl !== 'undefined' ? ajaxurl : window.ajaxurl);
            var nonce = (typeof nexlifydesk_admin_vars !== 'undefined' && nexlifydesk_admin_vars.nonce) ? nexlifydesk_admin_vars.nonce : '';

            if (confirm('Are you sure you want to purge old data? This action cannot be undone.')) {
                button.prop('disabled', true).text('Purging...');
                $.post(ajaxurl, {
                    action: 'nexlifydesk_purge_data',
                    _ajax_nonce: nonce
                }, function(response) {
                    button.prop('disabled', false).text('Purge Data');
                    if (response && response.success) {
                        resultDiv.html(response.data ? response.data : 'Success.');
                    } else {
                        resultDiv.html('<span style="color: red;">' + (response && response.data ? response.data : 'Error purging data. Please try again.') + '</span>');
                    }
                }).fail(function(xhr, status, error) {
                    button.prop('disabled', false).text('Purge Data');
                    resultDiv.html('<span style="color: red;">Error purging data. Please try again.<br>JS: ' + error + '</span>');
                });
            }
        });
    }
});

jQuery(document).ready(function($) {
    var $clearRateLimitBtn = $('#nexlifydesk_clear_rate_limit');
    var $checkRateLimitBtn = $('#nexlifydesk_check_rate_limit');
    
    if ($clearRateLimitBtn.length) {
        $clearRateLimitBtn.off('click').on('click', function() {
            var button = $(this);
            var resultDiv = $('#nexlifydesk_clear_rate_limit_result');
            var emailInput = $('#nexlifydesk_clear_rate_limit_email');
            var email = emailInput.val().trim();
            
            resultDiv.html('');
            
            if (!email || !isValidEmail(email)) {
                resultDiv.html('<span style="color: red;">Please enter a valid email address.</span>');
                return;
            }
            
            var ajaxurl = (typeof nexlifydesk_vars !== 'undefined' && nexlifydesk_vars.ajaxurl) ? nexlifydesk_vars.ajaxurl : (typeof ajaxurl !== 'undefined' ? ajaxurl : window.ajaxurl);
            var nonce = (typeof nexlifydesk_admin_vars !== 'undefined' && nexlifydesk_admin_vars.nonce) ? nexlifydesk_admin_vars.nonce : '';
            
            button.prop('disabled', true).text('Clearing...');
            $.post(ajaxurl, {
                action: 'nexlifydesk_clear_rate_limit',
                email: email,
                nonce: nonce
            }, function(response) {
                button.prop('disabled', false).text('Clear Rate Limit');
                if (response && response.success) {
                    resultDiv.html('<span style="color: green;">' + response.data + '</span>');
                    emailInput.val('');
                } else {
                    resultDiv.html('<span style="color: red;">' + (response && response.data ? response.data : 'Error clearing rate limit. Please try again.') + '</span>');
                }
            }).fail(function(xhr, status, error) {
                button.prop('disabled', false).text('Clear Rate Limit');
                resultDiv.html('<span style="color: red;">Error clearing rate limit. Please try again.</span>');
            });
        });
    }
    
    if ($checkRateLimitBtn.length) {
        $checkRateLimitBtn.off('click').on('click', function() {
            var button = $(this);
            var resultDiv = $('#nexlifydesk_check_rate_limit_result');
            var emailInput = $('#nexlifydesk_check_rate_limit_email');
            var email = emailInput.val().trim();
            
            resultDiv.html('');
            
            if (!email || !isValidEmail(email)) {
                resultDiv.html('<span style="color: red;">Please enter a valid email address.</span>');
                return;
            }
            
            var ajaxurl = (typeof nexlifydesk_vars !== 'undefined' && nexlifydesk_vars.ajaxurl) ? nexlifydesk_vars.ajaxurl : (typeof ajaxurl !== 'undefined' ? ajaxurl : window.ajaxurl);
            var nonce = (typeof nexlifydesk_admin_vars !== 'undefined' && nexlifydesk_admin_vars.nonce) ? nexlifydesk_admin_vars.nonce : '';
            
            button.prop('disabled', true).text('Checking...');
            $.post(ajaxurl, {
                action: 'nexlifydesk_check_rate_limit',
                email: email,
                nonce: nonce
            }, function(response) {
                button.prop('disabled', false).text('Check Status');
                if (response && response.success) {
                    resultDiv.html('<span style="color: blue;">' + response.data + '</span>');
                } else {
                    resultDiv.html('<span style="color: red;">' + (response && response.data ? response.data : 'Error checking rate limit. Please try again.') + '</span>');
                }
            }).fail(function(xhr, status, error) {
                button.prop('disabled', false).text('Check Status');
                resultDiv.html('<span style="color: red;">Error checking rate limit. Please try again.</span>');
            });
        });
    }
    
    function isValidEmail(email) {
        var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
});

jQuery(document).ready(function($) {
    $('#reassign-orphaned-tickets-btn').off('click').on('click', function(e){
        e.preventDefault();
        if (typeof nexlifydesk_admin_vars === 'undefined') return;
        if(confirm('Are you sure you want to reassign all orphaned tickets?')) {
            $.post(nexlifydesk_admin_vars.ajaxurl, {
                action: 'nexlifydesk_reassign_orphaned_tickets',
                nonce: nexlifydesk_admin_vars.nonce
            }, function(response) {
                if(response.success) {
                    alert('Orphaned tickets reassigned successfully!');
                    location.reload();
                } else {
                    alert(response.data || 'Failed to reassign orphaned tickets.');
                }
            });
        }
    });
});

jQuery(document).ready(function($){
    if (typeof ajaxurl === 'undefined') {
        var ajaxurl = window.nexlifydesk_admin_vars ? window.nexlifydesk_admin_vars.ajaxurl : '';
    }
    
    function toggleProviderSettings() {
        var provider = $('#provider-select').val();
        $('.provider-settings').hide();
        $('#' + provider + '-settings').show();
        
        if (provider === 'aws') {
            var awsIntegrationType = $('#aws_integration_type').val() || 'imap';
            if (awsIntegrationType === 'imap') {
                $('#aws-imap-settings').show();
                $('#aws-lambda-settings').hide();
            } else {
                $('#aws-imap-settings').hide();
                $('#aws-lambda-settings').show();
            }
        } else {
            $('#aws-imap-settings').hide();
            $('#aws-lambda-settings').hide();
        }
        
        var sslEnabled = window.nexlifydesk_ssl_enabled || false;
        
        if (!sslEnabled && provider === 'aws') {
            // alert('Note: SSL is recommended for AWS WorkMail for security purposes.');
        }
    }
    
    if ($('#provider-select').length) {
        toggleProviderSettings();
        $('#provider-select').on('change', toggleProviderSettings);
    }

    $('#aws_integration_type').on('change', function() {
        const integrationType = $(this).val();
        
        if (integrationType === 'imap') {
            $('#aws-imap-settings').show();
            $('#aws-lambda-settings').hide();
        } else if (integrationType === 'lambda') {
            $('#aws-imap-settings').hide();
            $('#aws-lambda-settings').show();
        }
    });
    
    $('#nexlifydesk-fetch-emails-now').on('click', function(){
        var $btn = $(this);
        var $status = $('#nexlifydesk-fetch-emails-status');
        $btn.prop('disabled', true);
        $status.text('Fetching...');
        
        $.post(ajaxurl, {
            action: 'nexlifydesk_fetch_emails_now',
            _ajax_nonce: window.nexlifydesk_fetch_emails_nonce || ''
        }, function(response){
            $btn.prop('disabled', false);
            if(response.success){
                $status.text('Fetch complete!');
            }else{
                $status.text(response.data || 'Fetch failed.');
            }
        }).fail(function(){
            $btn.prop('disabled', false);
            $status.text('Fetch failed.');
        });
    });
    
    $('#test-aws-connection').on('click', function(){
        var $btn = $(this);
        var $result = $('#aws-connection-result');
        
        $btn.prop('disabled', true).text('Testing...');
        $result.html('<p style="color: #666;">Connecting to AWS WorkMail...</p>');
        
        $.post(ajaxurl, {
            action: 'nexlifydesk_test_aws_connection',
            nonce: window.nexlifydesk_aws_test_nonce || '',
            region: $('#aws_region').val(),
            organization_id: $('#aws_organization_id').val(),
            email: $('#aws_email').val(),
            password: $('#aws_password').val(),
            access_key_id: $('#aws_access_key_id').val(),
            secret_access_key: $('#aws_secret_access_key').val()
        }).done(function(response) {
            $btn.prop('disabled', false).text('Test AWS Connection');
            if (response.success) {
                $result.html('<p style="color: green;">✅ ' + response.data.message + '</p>');
            } else {
                $result.html('<p style="color: red;">❌ ' + response.data.message + '</p>');
            }
        }).fail(function() {
            $btn.prop('disabled', false).text('Test AWS Connection');
            $result.html('<p style="color: red;">❌ Connection test failed.</p>');
        });
    });

    $('#test-aws-fetch-emails').on('click', function(){
        var $btn = $(this);
        var $result = $('#aws-fetch-result');
        
        $btn.prop('disabled', true).text('Fetching Emails...');
        $result.html('<p style="color: #666;">Attempting to fetch emails from AWS WorkMail...</p>');
        
        $.post(ajaxurl, {
            action: 'nexlifydesk_manual_fetch_emails',
            nonce: window.nexlifydesk_aws_test_nonce || '',
            region: $('#aws_region').val(),
            organization_id: $('#aws_organization_id').val(),
            email: $('#aws_email').val(),
            password: $('#aws_password').val(),
            access_key_id: $('#aws_access_key_id').val(),
            secret_access_key: $('#aws_secret_access_key').val()
        }).done(function(response) {
            $btn.prop('disabled', false).text('Test Email Fetch');
            if (response.success) {
                var message = response.data.message.replace(/\n/g, '<br>');
                $result.html('<div style="color: green; font-family: monospace; white-space: pre-wrap; background: #f0f9ff; padding: 10px; border-left: 4px solid #0087be;">📧 ' + message + '</div>');
            } else {
                $result.html('<p style="color: red;">❌ ' + response.data.message + '</p>');
            }
        }).fail(function() {
            $btn.prop('disabled', false).text('Test Email Fetch');
            $result.html('<p style="color: red;">❌ Email fetch test failed.</p>');
        });
    });

    $('#aws-diagnostics').on('click', function(){
        var $btn = $(this);
        $btn.data('diagnostics-bound', true);
        
        var $result = $('#aws-diagnostics-result');
        
        if ($result.length === 0) {
            alert('Error: Diagnostics result container not found. Please refresh the page.');
            return;
        }
        
        if (!window.nexlifydesk_aws_test_nonce) {
            $result.html('<p style="color: red;">❌ Security nonce not available. Please refresh the page.</p>');
            return;
        }
        
        $btn.prop('disabled', true).text('Running Diagnostics...');
        $result.html('<p style="color: #666;">Running system diagnostics...</p>');
        
        $.post(ajaxurl, {
            action: 'nexlifydesk_aws_diagnostics',
            nonce: window.nexlifydesk_aws_test_nonce || ''
        }).done(function(response) {
            $btn.prop('disabled', false).text('System Diagnostics');
            if (response.success && response.data.diagnostics) {
                var html = '<div style="background: #f5f5f5; padding: 15px; border-radius: 5px; margin-top: 10px;">';
                html += '<h4 style="margin-top: 0;">🔍 System Diagnostics Report</h4>';
                
                Object.keys(response.data.diagnostics).forEach(function(key) {
                    var diag = response.data.diagnostics[key];
                    var statusIcon = '';
                    var statusColor = '';
                    
                    switch(diag.status) {
                        case 'OK':
                            statusIcon = '✅';
                            statusColor = 'green';
                            break;
                        case 'FAILED':
                            statusIcon = '❌';
                            statusColor = 'red';
                            break;
                        case 'WARNING':
                            statusIcon = '⚠️';
                            statusColor = 'orange';
                            break;
                        default:
                            statusIcon = 'ℹ️';
                            statusColor = 'blue';
                    }
                    
                    html += '<div style="margin: 8px 0; padding: 8px; background: white; border-radius: 3px;">';
                    html += '<strong style="color: ' + statusColor + ';">' + statusIcon + ' ' + key.replace(/_/g, ' ').toUpperCase() + ':</strong> ';
                    html += '<span style="color: #333;">' + diag.message + '</span>';
                    html += '</div>';
                });
                
                html += '<div style="margin-top: 15px; padding: 10px; background: #e3f2fd; border-radius: 3px; font-size: 12px; color: #1976d2;">';
                html += '<strong>💡 Troubleshooting Tips:</strong><br>';
                html += '• If IMAP extension is missing, contact your hosting provider<br>';
                html += '• SSL/HTTPS is required for AWS WorkMail<br>';
                html += '• Ensure your server can connect to AWS (check firewall/network)<br>';
                html += '• Verify your AWS WorkMail credentials are correct';
                html += '</div>';
                html += '</div>';
                
                $result.html(html);
            } else {
                $result.html('<p style="color: red;">❌ Diagnostics failed to run.</p>');
            }
        }).fail(function() {
            $btn.prop('disabled', false).text('System Diagnostics');
            $result.html('<p style="color: red;">❌ Diagnostics request failed.</p>');
        });
    });

    $(document).on('click', '#aws-diagnostics', function(e) {
        if ($(this).data('diagnostics-bound')) {
            return;
        }
        
        var $btn = $(this);
        var $result = $('#aws-diagnostics-result');
        
        if ($result.length === 0) {
            alert('Error: Diagnostics result container not found. Please refresh the page.');
            return;
        }
        
        if (!window.nexlifydesk_aws_test_nonce) {
            $result.html('<p style="color: red;">❌ Security nonce not available. Please refresh the page.</p>');
            return;
        }
        
        $btn.prop('disabled', true).text('Running Diagnostics...');
        $result.html('<p style="color: #666;">Running system diagnostics...</p>');
        
        $.post(ajaxurl, {
            action: 'nexlifydesk_aws_diagnostics',
            nonce: window.nexlifydesk_aws_test_nonce || ''
        }).done(function(response) {
            $btn.prop('disabled', false).text('System Diagnostics');
            if (response.success && response.data.diagnostics) {
                var html = '<div style="background: #f5f5f5; padding: 15px; border-radius: 5px; margin-top: 10px;">';
                html += '<h4 style="margin-top: 0;">🔍 System Diagnostics Report</h4>';
                
                Object.keys(response.data.diagnostics).forEach(function(key) {
                    var diag = response.data.diagnostics[key];
                    var statusIcon = '';
                    var statusColor = '';
                    
                    switch(diag.status) {
                        case 'OK':
                            statusIcon = '✅';
                            statusColor = 'green';
                            break;
                        case 'FAILED':
                            statusIcon = '❌';
                            statusColor = 'red';
                            break;
                        case 'WARNING':
                            statusIcon = '⚠️';
                            statusColor = 'orange';
                            break;
                        default:
                            statusIcon = 'ℹ️';
                            statusColor = 'blue';
                    }
                    
                    html += '<div style="margin: 8px 0; padding: 8px; background: white; border-radius: 3px;">';
                    html += '<strong style="color: ' + statusColor + ';">' + statusIcon + ' ' + key.replace(/_/g, ' ').toUpperCase() + ':</strong> ';
                    html += '<span style="color: #333;">' + diag.message + '</span>';
                    html += '</div>';
                });
                
                html += '<div style="margin-top: 15px; padding: 10px; background: #e3f2fd; border-radius: 3px; font-size: 12px; color: #1976d2;">';
                html += '<strong>💡 Troubleshooting Tips:</strong><br>';
                html += '• If IMAP extension is missing, contact your hosting provider<br>';
                html += '• SSL/HTTPS is required for AWS WorkMail<br>';
                html += '• Ensure your server can connect to AWS (check firewall/network)<br>';
                html += '• Verify your AWS WorkMail credentials are correct';
                html += '</div>';
                html += '</div>';
                
                $result.html(html);
            } else {
                $result.html('<p style="color: red;">❌ Diagnostics failed to run.</p>');
            }
        }).fail(function() {
            $btn.prop('disabled', false).text('System Diagnostics');
            $result.html('<p style="color: red;">❌ Diagnostics request failed.</p>');
        });
    });

    $('#test-google-connection').on('click', function(){
        var $btn = $(this);
        var $result = $('#google-connection-result');
        
        $btn.prop('disabled', true).text('Testing...');
        $result.html('<p style="color: #666;">Testing Google connection...</p>');
        
        $.post(ajaxurl, {
            action: 'nexlifydesk_test_google_connection',
            nonce: window.nexlifydesk_google_test_nonce || '',
            client_id: $('#google_client_id').val(),
            client_secret: $('#google_client_secret').val()
        }).done(function(response) {
            $btn.prop('disabled', false).text('Test Google Connection');
            if (response.success) {
                $result.html('<p style="color: green;">✅ ' + response.data.message + '</p>');
            } else {
                $result.html('<p style="color: red;">❌ ' + response.data.message + '</p>');
            }
        }).fail(function() {
            $btn.prop('disabled', false).text('Test Google Connection');
            $result.html('<p style="color: red;">❌ Google connection test failed.</p>');
        });
    });

    $('#test-google-fetch-emails').on('click', function(){
        var $btn = $(this);
        var $result = $('#google-fetch-result');
        
        $btn.prop('disabled', true).text('Fetching Emails...');
        $result.html('<p style="color: #666;">Attempting to fetch emails from Google...</p>');
        
        $.post(ajaxurl, {
            action: 'nexlifydesk_manual_fetch_google_emails',
            nonce: window.nexlifydesk_google_test_nonce || ''
        }).done(function(response) {
            $btn.prop('disabled', false).text('Test Email Fetch');
            if (response.success) {
                var message = response.data.message.replace(/\n/g, '<br>');
                $result.html('<div style="color: green; font-family: monospace; white-space: pre-wrap; background: #f0f9ff; padding: 10px; border-left: 4px solid #0087be;">📧 ' + message + '</div>');
            } else {
                $result.html('<p style="color: red;">❌ ' + response.data.message + '</p>');
            }
        }).fail(function() {
            $btn.prop('disabled', false).text('Test Email Fetch');
            $result.html('<p style="color: red;">❌ Google email fetch test failed.</p>');
        });
    });

    $('#test-custom-connection').on('click', function(){
        var $btn = $(this);
        var $result = $('#custom-connection-result');
        
        $btn.prop('disabled', true).text('Testing...');
        $result.html('<p style="color: #666;">Testing custom IMAP/POP3 connection...</p>');
        
        $.post(ajaxurl, {
            action: 'nexlifydesk_test_custom_connection',
            nonce: window.nexlifydesk_custom_test_nonce || '',
            host: $('#host').val(),
            port: $('#port').val(),
            username: $('#username').val(),
            password: $('#password').val(),
            encryption: $('#encryption').val(),
            protocol: $('#protocol').val()
        }).done(function(response) {
            $btn.prop('disabled', false).text('Test Custom Connection');
            if (response.success) {
                $result.html('<p style="color: green;">✅ ' + response.data.message + '</p>');
            } else {
                $result.html('<p style="color: red;">❌ ' + response.data.message + '</p>');
            }
        }).fail(function() {
            $btn.prop('disabled', false).text('Test Custom Connection');
            $result.html('<p style="color: red;">❌ Custom connection test failed.</p>');
        });
    });

    $('#test-custom-fetch-emails').on('click', function(){
        var $btn = $(this);
        var $result = $('#custom-fetch-result');
        
        $btn.prop('disabled', true).text('Fetching Emails...');
        $result.html('<p style="color: #666;">Attempting to fetch emails from custom server...</p>');
        
        $.post(ajaxurl, {
            action: 'nexlifydesk_manual_fetch_emails',
            nonce: window.nexlifydesk_custom_test_nonce || '',
            provider: 'custom'
        }).done(function(response) {
            $btn.prop('disabled', false).text('Test Email Fetch');
            if (response.success) {
                var message = response.data.message.replace(/\n/g, '<br>');
                $result.html('<div style="color: green; font-family: monospace; white-space: pre-wrap; background: #f0f9ff; padding: 10px; border-left: 4px solid #0087be;">📧 ' + message + '</div>');
            } else {
                $result.html('<p style="color: red;">❌ ' + response.data.message + '</p>');
            }
        }).fail(function() {
            $btn.prop('disabled', false).text('Test Email Fetch');
            $result.html('<p style="color: red;">❌ Custom email fetch test failed.</p>');
        });
    });
});

jQuery(document).ready(function($){
    $('#edit-custom-password').on('click', function() {
        const passwordField = $('#password');
        const button = $(this);
        const hiddenField = $('#password-preserved-flag');
        
        passwordField.prop('disabled', false)
                    .prop('readonly', false)
                    .css('background-color', '#fff')
                    .css('border', '1px solid #8c8f94')
                    .css('cursor', 'text')
                    .attr('placeholder', 'Enter new password')
                    .val('')
                    .focus();
        
        hiddenField.remove();
        
        button.text('Cancel').off('click').on('click', function() {
            passwordField.prop('disabled', true)
                        .prop('readonly', true)
                        .css('background-color', '#f7f7f7')
                        .css('border', '1px solid #ddd')
                        .css('cursor', 'not-allowed')
                        .attr('placeholder', '')
                        .val('••••••••••••••••');
            
            passwordField.after('<input type="hidden" name="nexlifydesk_imap_settings[password_preserved]" value="1" id="password-preserved-flag">');
            
            button.text('Edit Password').off('click').on('click', arguments.callee);
        });
    });
    
    $('#edit-aws-password').on('click', function() {
        const passwordField = $('#aws_password');
        const button = $(this);
        const hiddenField = $('#aws-password-preserved-flag');
        
        passwordField.prop('disabled', false)
                    .prop('readonly', false)
                    .css('background-color', '#fff')
                    .css('border', '1px solid #8c8f94')
                    .css('cursor', 'text')
                    .attr('placeholder', 'Enter new password')
                    .val('')
                    .focus();
        
        hiddenField.remove();
        
        button.text('Cancel').off('click').on('click', function() {
            passwordField.prop('disabled', true)
                        .prop('readonly', true)
                        .css('background-color', '#f7f7f7')
                        .css('border', '1px solid #ddd')
                        .css('cursor', 'not-allowed')
                        .attr('placeholder', '')
                        .val('••••••••••••••••');
            
            passwordField.after('<input type="hidden" name="nexlifydesk_imap_settings[aws_password_preserved]" value="1" id="aws-password-preserved-flag">');
            
            button.text('Edit Password').off('click').on('click', arguments.callee);
        });
    });
    
    $('#edit-aws-secret-key').on('click', function() {
        const passwordField = $('#aws_secret_access_key');
        const button = $(this);
        const hiddenField = $('#aws-secret-key-preserved-flag');
        
        passwordField.prop('disabled', false)
                    .prop('readonly', false)
                    .css('background-color', '#fff')
                    .css('border', '1px solid #8c8f94')
                    .css('cursor', 'text')
                    .attr('placeholder', 'Enter new secret key')
                    .val('')
                    .focus();
        
        hiddenField.remove();
        
        button.text('Cancel').off('click').on('click', function() {
            passwordField.prop('disabled', true)
                        .prop('readonly', true)
                        .css('background-color', '#f7f7f7')
                        .css('border', '1px solid #ddd')
                        .css('cursor', 'not-allowed')
                        .attr('placeholder', '')
                        .val('••••••••••••••••');
            
            passwordField.after('<input type="hidden" name="nexlifydesk_imap_settings[aws_secret_access_key_preserved]" value="1" id="aws-secret-key-preserved-flag">');
            
            button.text('Edit Secret Key').off('click').on('click', arguments.callee);
        });
    });
    
    $('#edit-google-secret').on('click', function() {
        const passwordField = $('#google_client_secret');
        const button = $(this);
        const hiddenField = $('#google-secret-preserved-flag');
        
        passwordField.prop('disabled', false)
                    .prop('readonly', false)
                    .css('background-color', '#fff')
                    .css('border', '1px solid #8c8f94')
                    .css('cursor', 'text')
                    .attr('placeholder', 'Enter new client secret')
                    .val('')
                    .focus();
        
        hiddenField.remove();
        
        button.text('Cancel').off('click').on('click', function() {
            passwordField.prop('disabled', true)
                        .prop('readonly', true)
                        .css('background-color', '#f7f7f7')
                        .css('border', '1px solid #ddd')
                        .css('cursor', 'not-allowed')
                        .attr('placeholder', '')
                        .val('••••••••••••••••');
            
            passwordField.after('<input type="hidden" name="nexlifydesk_imap_settings[google_client_secret_preserved]" value="1" id="google-secret-preserved-flag">');
            
            button.text('Edit Secret').off('click').on('click', arguments.callee);
        });
    });
    
    $('#aws_integration_type').on('change', function() {
        const integrationType = $(this).val();
        
        if (integrationType === 'lambda') {
            alert('Lambda integration is coming soon! Please use IMAP integration for now.');
            $(this).val('imap');
            $('#aws-imap-settings').show();
            $('#aws-lambda-settings').hide();
        } else {
            $('#aws-imap-settings').show();
            $('#aws-lambda-settings').hide();
        }
    });
    
    $(document).ready(function() {
        $('#aws_integration_type option[value="lambda"]').prop('disabled', true);
    });
    
    $('#regenerate-webhook-secret').on('click', function() {
        const newSecret = generateRandomString(32);
        $('#lambda_webhook_secret').val(newSecret);
    });
    
    $('#copy-webhook-url').on('click', function() {
        const webhookUrl = $(this).prev('input').val();
        navigator.clipboard.writeText(webhookUrl).then(function() {
            alert('Webhook URL copied to clipboard!');
        });
    });
    
    $('#test-lambda-webhook').on('click', function() {
        const button = $(this);
        const resultDiv = $('#lambda-webhook-result');
        
        button.prop('disabled', true).text('Testing...');
        resultDiv.html('<p>Testing webhook connection...</p>');
        
        const testData = {
            from: 'test@example.com',
            to: 'support@yoursite.com',
            subject: 'Test Email from Lambda',
            body: 'This is a test email to verify the Lambda webhook integration.',
            message_id: 'test-' + Date.now(),
            timestamp: Math.floor(Date.now() / 1000)
        };
        
        $.ajax({
            url: $('#copy-webhook-url').prev('input').val(),
            method: 'POST',
            contentType: 'application/json',
            headers: {
                'X-NexlifyDesk-Secret': $('#lambda_webhook_secret').val()
            },
            data: JSON.stringify(testData),
            success: function(response) {
                resultDiv.html('<div style="color: green;"><strong>✓ Webhook test successful!</strong><br>Response: ' + JSON.stringify(response) + '</div>');
            },
            error: function(xhr, status, error) {
                resultDiv.html('<div style="color: red;"><strong>✗ Webhook test failed!</strong><br>Error: ' + error + '<br>Status: ' + xhr.status + '</div>');
            },
            complete: function() {
                button.prop('disabled', false).text('Test Webhook');
            }
        });
    });
    
    function generateRandomString(length) {
        const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
        let result = '';
        for (let i = 0; i < length; i++) {
            result += chars.charAt(Math.floor(Math.random() * chars.length));
        }
        return result;
    }
    
    $('#provider-select').on('change', function() {
        $('input[type="password"]').each(function() {
            const field = $(this);
            const fieldId = field.attr('id');
            const editButton = $('#edit-' + fieldId.replace('_', '-'));
            
            if (editButton.length && field.val() === '••••••••••••••••') {
                field.prop('disabled', true)
                    .prop('readonly', true)
                    .css('background-color', '#f7f7f7')
                    .css('border', '1px solid #ddd')
                    .css('cursor', 'not-allowed');
                
                editButton.text(editButton.text().replace('Cancel', 'Edit'));
            }
        });
    });
});

jQuery(document).ready(function($) {
    $('#generate-encryption-key').on('click', function() {
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'nexlifydesk_generate_encryption_key',
                nonce: window.nexlifydesk_generate_key_nonce || ''
            },
            success: function(response) {
                if (response.success) {
                    $('#encryption-key-output').val(response.data.key);
                    $('#generated-key-display').show();
                } else {
                    alert('Error generating key: ' + response.data);
                }
            },
            error: function() {
                alert('Error generating encryption key. Please try again.');
            }
        });
    });
    
    $('#copy-encryption-key').on('click', function() {
        const keyField = document.getElementById('encryption-key-output');
        keyField.select();
        keyField.setSelectionRange(0, 99999);
        
        navigator.clipboard.writeText(keyField.value).then(function() {
            alert('Encryption key copied to clipboard!');
        }, function(err) {
            document.execCommand('copy');
            alert('Encryption key copied to clipboard!');
        });
    });
});

function nexlifydesk_update_templates() {
    if (typeof nexlifydesk_admin_vars === 'undefined') {
        alert('Configuration error. Please refresh the page.');
        return;
    }
    
    if (confirm(nexlifydesk_admin_vars.template_update_confirm)) {
        var xhr = new XMLHttpRequest();
        xhr.open('POST', nexlifydesk_admin_vars.ajaxurl, true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
        xhr.onreadystatechange = function() {
            if (xhr.readyState === 4) {
                if (xhr.status === 200) {
                    document.getElementById('nexlifydesk-template-update-notice').style.display = 'none';
                    alert(nexlifydesk_admin_vars.template_update_success);
                } else {
                    alert(nexlifydesk_admin_vars.template_update_error);
                }
            }
        };
        xhr.send('action=nexlifydesk_update_email_templates&nonce=' + nexlifydesk_admin_vars.template_update_nonce);
    }
}

function nexlifydesk_dismiss_notice() {
    if (typeof nexlifydesk_admin_vars === 'undefined') {
        document.getElementById('nexlifydesk-template-update-notice').style.display = 'none';
        return;
    }
    
    var xhr = new XMLHttpRequest();
    xhr.open('POST', nexlifydesk_admin_vars.ajaxurl, true);
    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
    xhr.onreadystatechange = function() {
        if (xhr.readyState === 4) {
            document.getElementById('nexlifydesk-template-update-notice').style.display = 'none';
        }
    };
    xhr.send('action=nexlifydesk_dismiss_template_notice&nonce=' + nexlifydesk_admin_vars.template_dismiss_nonce);
}

window.nexlifydesk_update_templates = nexlifydesk_update_templates;
window.nexlifydesk_dismiss_notice = nexlifydesk_dismiss_notice;