jQuery(document).ready(function($) {
    
    let refreshInterval;
    let lastRefreshTime = Date.now();
    
    if (typeof nexlifydesk_admin_vars === 'undefined') {
        return;
    }
    
    if (!$('#nexlifydesk-ticket-list').length) {
        return;
    }
    
    $('#manual-refresh-btn').on('click', function() {
        refreshTicketList();
    });
    
    $('#select-all-tickets, #header-select-all').on('change', function() {
        const isChecked = $(this).is(':checked');
        $('.ticket-checkbox').prop('checked', isChecked);
        updateBulkActionButtons();
    });

    $(document).on('change', '.ticket-checkbox', function() {
        const totalCheckboxes = $('.ticket-checkbox').length;
        const checkedCheckboxes = $('.ticket-checkbox:checked').length;
        
        $('#select-all-tickets, #header-select-all').prop('checked', checkedCheckboxes === totalCheckboxes);
        updateBulkActionButtons();
    });

    function updateBulkActionButtons() {
        const checkedCount = $('.ticket-checkbox:checked').length;
        const hasSelection = checkedCount > 0;
        
        $('#bulk-action-select, #apply-bulk-action').prop('disabled', !hasSelection);
        
        if (hasSelection) {
            $('#bulk-action-select').removeClass('disabled');
        } else {
            $('#bulk-action-select').addClass('disabled').val('');
        }
    }

    $('#apply-bulk-action').on('click', function() {
        const action = $('#bulk-action-select').val();
        const selectedTickets = $('.ticket-checkbox:checked').map(function() {
            return $(this).val();
        }).get();

        if (!action || selectedTickets.length === 0) {
            return;
        }

        switch(action) {
            case 'assign':
                $('#bulk-assign-modal').show();
                break;
            case 'status':
                $('#bulk-status-modal').show();
                break;
            case 'priority':
                showBulkPriorityModal();
                break;
            case 'delete':
                showBulkDeleteModal();
                break;
        }
    });

    $('#cancel-bulk-assign, #cancel-bulk-status, #cancel-bulk-priority, #cancel-bulk-delete').on('click', function() {
        $('.bulk-modal').hide();
    });

    $('#confirm-bulk-assign').on('click', function() {
        const agentId = $('#bulk-assign-agent').val();
        const selectedTickets = $('.ticket-checkbox:checked').map(function() {
            return $(this).val();
        }).get();
        
        performBulkAction('assign', selectedTickets, { agent_id: agentId });
        $('#bulk-assign-modal').hide();
    });

    $('#confirm-bulk-status').on('click', function() {
        const status = $('#bulk-status-select').val();
        const selectedTickets = $('.ticket-checkbox:checked').map(function() {
            return $(this).val();
        }).get();
        
        performBulkAction('status', selectedTickets, { status: status });
        $('#bulk-status-modal').hide();
    });

    $('#confirm-bulk-priority').on('click', function() {
        const priority = $('#bulk-priority-select').val();
        const selectedTickets = $('.ticket-checkbox:checked').map(function() {
            return $(this).val();
        }).get();
        
        performBulkAction('priority', selectedTickets, { priority: priority });
        $('#bulk-priority-modal').hide();
    });

    function showBulkPriorityModal() {
        $('#bulk-priority-modal').show();
    }

    function showBulkDeleteModal() {
        const selectedTickets = $('.ticket-checkbox:checked').map(function() {
            return $(this).val();
        }).get();
        
        if (!confirm('Are you sure you want to permanently delete the selected tickets? This action cannot be undone!')) {
            return;
        }
        
        $('#selected-tickets-count').text(selectedTickets.length);
        $('#bulk-delete-modal').show();
    }

    $('#confirm-bulk-delete').on('click', function() {
        const selectedTickets = $('.ticket-checkbox:checked').map(function() {
            return $(this).val();
        }).get();
        
        performBulkAction('delete', selectedTickets);
        $('#bulk-delete-modal').hide();
    });

    function performBulkAction(action, ticketIds, data = {}) {
        $.ajax({
            url: nexlifydesk_admin_vars.ajaxurl,
            type: 'POST',
            data: {
                action: 'nexlifydesk_bulk_action',
                bulk_action: action,
                ticket_ids: ticketIds,
                ...data,
                nonce: $('#nexlifydesk-nonce').val()
            },
            beforeSend: function() {
                $('#apply-bulk-action').prop('disabled', true).text('Processing...');
            },
            success: function(response) {
                if (response.success) {
                    if (action === 'delete') {
                        ticketIds.forEach(function(ticketId) {
                            $('.ticket-row[data-ticket-id="' + ticketId + '"]').fadeOut(function() {
                                $(this).remove();
                            });
                        });
                        
                        $('.ticket-checkbox, .select-all-checkbox').prop('checked', false);
                        updateBulkActionButtons();
                        
                        alert('Selected tickets have been deleted successfully.');
                    } else {
                        location.reload();
                    }
                } else {
                    alert('Error: ' + (response.data || 'Unknown error occurred'));
                }
            },
            error: function(xhr, status, error) {
                alert('An error occurred while processing the bulk action.');
            },
            complete: function() {
                $('#apply-bulk-action').prop('disabled', false).text('Apply');
            }
        });
    }

    $(document).on('click', '.bulk-modal', function(e) {
        if (e.target === this) {
            $(this).hide();
        }
    });

    $('.ticket-checkbox').on('click', function(e) {
        e.stopPropagation();
    });

    $(document).on('click', '.ticket-row', function(e) {
        if (!$(e.target).is('input[type="checkbox"]') && !$(e.target).closest('.row-checkbox').length) {
            const ticketId = $(this).data('ticket-id');
            const $row = $(this);
            
            if ($row.hasClass('unread')) {
                markTicketAsRead(ticketId, $row);
            }
            
            window.location.href = 'admin.php?page=nexlifydesk_tickets&ticket_id=' + ticketId;
        }
    });
    
    $(document).on('click', '.ticket-link', function(e) {
        const $row = $(this).closest('.ticket-row');
        const ticketId = $row.data('ticket-id');
        
        if ($row.hasClass('unread')) {
            markTicketAsRead(ticketId, $row);
        }
    });
    
    function markTicketAsRead(ticketId, $row) {
        $row.removeClass('unread');
        $row.find('.ticket-title').removeClass('unread-title');
        $row.find('.unread-dot').remove();
        $row.attr('data-is-unread', '0');
        
        $.ajax({
            url: nexlifydesk_admin_vars.ajaxurl,
            type: 'POST',
            data: {
                action: 'nexlifydesk_mark_ticket_read',
                ticket_id: ticketId,
                nonce: $('#nexlifydesk-nonce').val()
            }
        });
    }

     $('#ticket-status-filter, #ticket-priority-filter').on('change', function() {
         filterTickets();
    });

    setTimeout(function() {
        filterTickets();
    }, 500);

    function filterTickets() {
        const searchTerm = $('#ticket-search').val().toLowerCase();
        const statusFilter = $('#ticket-status-filter').val();
        const priorityFilter = $('#ticket-priority-filter').val();
        
        let visibleCount = 0;

        $('.ticket-row').each(function() {
            const $row = $(this);
            let showRow = true;

            if (searchTerm && searchTerm.length > 0) {
                const ticketText = $row.text().toLowerCase();
                if (!ticketText.includes(searchTerm)) {
                    showRow = false;
                }
            }

            if (statusFilter && statusFilter !== 'all') {
                const rowStatus = $row.data('status');
                if (rowStatus !== statusFilter) {
                    showRow = false;
                }
            }

            if (priorityFilter && priorityFilter !== 'all') {
                const rowPriority = $row.data('priority');
                if (rowPriority !== priorityFilter) {
                    showRow = false;
                }
            }

            if (showRow) {
                $row.show();
                visibleCount++;
            } else {
                $row.hide();
            }
        });
        
        $('.no-tickets-filtered').remove();
        if (visibleCount === 0) {
            $('.ticket-list-body').append('<div class="no-tickets-filtered" style="text-align: center; padding: 20px; color: #666;">No tickets match the current filters.</div>');
        }
    }
    
    function startAutoRefresh() {
        if (!$('#nexlifydesk-ticket-list').length) {
            return;
        }
        
        if (refreshInterval) {
            clearInterval(refreshInterval);
        }
        
        const getRefreshInterval = () => {
            if (document.hidden) {
                const timeHidden = Date.now() - lastActiveTime;
                if (timeHidden > 600000) { // 10 minutes
                    return 300000; 
                } else {
                    return 60000; // 1 minute when recently hidden
                }
            } else {
                return 30000; // 30 seconds when active
            }
        };
        
        const scheduleNext = () => {
            const interval = getRefreshInterval();
            refreshInterval = setTimeout(() => {
                refreshTicketList();
                scheduleNext(); 
            }, interval);
        };
        
        scheduleNext();
    }
    
    function stopAutoRefresh() {
        if (refreshInterval) {
            clearTimeout(refreshInterval);
            refreshInterval = null;
        }
    }
    
    function refreshTicketList() {
        if (!$('#nexlifydesk-ticket-list').length) {
            return;
        }
        
        const nonce = $('#nexlifydesk-nonce').val();
        if (!nonce) {
            location.reload();
            return;
        }
        
        $.ajax({
            url: nexlifydesk_admin_vars.ajaxurl,
            type: 'POST',
            data: {
                action: 'nexlifydesk_refresh_ticket_list',
                nonce: nonce,
                last_refresh: lastRefreshTime
            },
            timeout: 30000,
            success: function(response) {
                if (response.success && response.data && response.data.tickets) {
                     if (typeof response.data === 'string') {
                         location.reload();
                         return;
                     }
                    if (response.data.new_nonce) {
                        $('#nexlifydesk-nonce').val(response.data.new_nonce);
                    }
                    
                    updateTicketList(response.data.tickets);
                    lastRefreshTime = Date.now();
                    
                    setTimeout(function() {
                        filterTickets();
                    }, 100);
                }
            },
            error: function(xhr, status, error) {
                if (xhr.status === 403 || 
                    (xhr.responseText && xhr.responseText.includes('nonce')) ||
                    (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.code === 'nonce_expired')) {
                    location.reload();
                    return;
                }
                
                if (status === 'timeout') {
                    $header.text(originalText + ' (Timeout)');
                    setTimeout(() => {
                        $header.text(originalText);
                        updatePageTitle();
                    }, 3000);
                    return;
                }
                
                $header.text(originalText + ' (Error)');
                setTimeout(() => {
                    $header.text(originalText);
                    updatePageTitle();
                }, 3000);
            }
        });
    }
    
    function updateTicketList(tickets) {
        if (!tickets || !Array.isArray(tickets)) {
            $('#nexlifydesk-ticket-grid').html('<p>No tickets found.</p>');
            return;
        }
        
        const selectedTickets = [];
        $('.ticket-checkbox:checked').each(function() {
            selectedTickets.push($(this).val());
        });
        
        const $ticketListBody = $('.ticket-list-body');
        $ticketListBody.empty();
        
        tickets.forEach(function(ticket) {
            const ticketHtml = createTicketRowHtml(ticket);
            if (ticketHtml) {
                $ticketListBody.append(ticketHtml);
            }
        });
        
        selectedTickets.forEach(function(ticketId) {
            $('.ticket-checkbox[value="' + ticketId + '"]').prop('checked', true);
        });
        
        updateBulkActionButtons();
        
        sortTicketList();
        
        updatePageTitle();
    }
    
    function updateTicketRow($row, ticket) {
        const isUnread = ticket.is_unread == 1;
        const lastReplyTime = ticket.last_reply_time || ticket.created_at;
        
        $row.toggleClass('unread', isUnread);
        $row.attr('data-last-reply', new Date(lastReplyTime).getTime() / 1000);
        $row.attr('data-is-unread', isUnread ? '1' : '0');
        
        const $unreadDot = $row.find('.unread-dot');
        if (isUnread && $unreadDot.length === 0) {
            $row.find('.ticket-link').prepend('<span class="unread-dot"></span>');
        } else if (!isUnread && $unreadDot.length > 0) {
            $unreadDot.remove();
        }
        
        $row.find('.ticket-title').toggleClass('unread-title', isUnread);
        $row.find('.ticket-title').text(ticket.subject || 'No Subject');
        $row.find('.ticket-preview').text((ticket.message || '').substring(0, 100) + '...');
        
        const $statusBadge = $row.find('.status-badge');
        $statusBadge.attr('class', 'status-badge status-' + ticket.status);
        $statusBadge.text(ticket.status.charAt(0).toUpperCase() + ticket.status.slice(1).replace('_', ' '));
        
        const $priorityBadge = $row.find('.priority-badge');
        $priorityBadge.attr('class', 'priority-badge priority-' + ticket.priority);
        $priorityBadge.text(ticket.priority.charAt(0).toUpperCase() + ticket.priority.slice(1));
        
        const $assigneeDiv = $row.find('.row-assignee');
        if (ticket.assigned_to_display_name) {
            $assigneeDiv.html(`
                <div class="assignee-info">
                    <div class="avatar">${ticket.assigned_to_display_name.charAt(0)}</div>
                    <span>${escapeHtml(ticket.assigned_to_display_name)}</span>
                </div>
            `);
        } else {
            $assigneeDiv.html('<span class="unassigned">Unassigned</span>');
        }
        
        $row.find('.date-time').text(formatDate(lastReplyTime));
        $row.find('.time-ago').text(getTimeAgo(lastReplyTime));
    }
    
function createTicketRowHtml(ticket) {
    if (!ticket || !ticket.id) {
        return '';
    }
    const isUnread = ticket.is_unread == 1;
    const lastReplyTime = ticket.last_reply_time || ticket.created_at || new Date().toISOString();
    const customerName = escapeHtml(ticket.user_name || 'Guest');
    const customerEmail = escapeHtml(ticket.user_email || 'N/A');
    const assignedName = escapeHtml(ticket.assigned_to_display_name || '');
    const subject = escapeHtml(ticket.subject || 'No Subject');
    const preview = escapeHtml((ticket.message || '').substring(0, 100));
    const status = ticket.status || 'open';
    const priority = ticket.priority || 'medium';
    const ticketId = ticket.ticket_id || ticket.id;

    return `
        <div class="ticket-row ${isUnread ? 'unread' : ''}"
             data-ticket-id="${ticket.id}"
             data-last-reply="${new Date(lastReplyTime).getTime() / 1000}"
             data-is-unread="${isUnread ? '1' : '0'}"
             data-status="${status}"
             data-priority="${priority}">
            <div class="row-checkbox">
                <input type="checkbox" class="ticket-checkbox" value="${ticket.id}">
            </div>
            <div class="row-subject">
                <a href="admin.php?page=nexlifydesk_tickets&ticket_id=${ticket.id}" class="ticket-link">
                    ${isUnread ? '<span class="unread-dot"></span>' : ''}
                    <span class="ticket-id">#${ticketId}</span>
                    <span class="ticket-title ${isUnread ? 'unread-title' : ''}">${subject}</span>
                </a>
                <div class="ticket-preview">${preview}...</div>
            </div>
            <div class="row-customer">
                <span class="customer-name">${customerName}</span>
                <span class="customer-email">${customerEmail}</span>
            </div>
            <div class="row-status">
                <span class="status-badge status-${status}">
                    ${status.charAt(0).toUpperCase() + status.slice(1).replace('_', ' ')}
                </span>
            </div>
            <div class="row-priority">
                <span class="priority-badge priority-${priority}">
                    ${priority.charAt(0).toUpperCase() + priority.slice(1)}
                </span>
            </div>
            <div class="row-assignee">
                ${assignedName ? `
                    <div class="assignee-info">
                        <div class="avatar">${assignedName.charAt(0)}</div>
                        <span>${assignedName}</span>
                    </div>
                ` : '<span class="unassigned">Unassigned</span>'}
            </div>
            <div class="row-created">
                <span class="date-time">${formatDate(ticket.created_at)}</span>
                <span class="time-ago">${getTimeAgo(ticket.created_at)}</span>
            </div>
            <div class="row-updated">
                <span class="date-time">${formatDate(ticket.last_reply_time || lastReplyTime)}</span>
                <span class="time-ago">${getTimeAgo(ticket.last_reply_time || lastReplyTime)}</span>
            </div>
        </div>
    `;
}
    
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    function formatDate(dateString) {
        const date = new Date(dateString);
        const options = { month: 'short', day: 'numeric', year: 'numeric' };
        return date.toLocaleDateString('en-US', options);
    }
    
    function sortTicketList() {
        const $ticketListBody = $('.ticket-list-body');
        const $tickets = $('.ticket-row').detach();
        
        $tickets.sort(function(a, b) {
            const aUnread = $(a).data('is-unread') === '1';
            const bUnread = $(b).data('is-unread') === '1';
            const aTime = parseInt($(a).data('last-reply'));
            const bTime = parseInt($(b).data('last-reply'));
            
            if (aUnread && !bUnread) return -1;
            if (!aUnread && bUnread) return 1;
            
            return bTime - aTime;
        });
        
        $ticketListBody.append($tickets);
    }
    
    function getTimeAgo(dateString) {
        const now = new Date();
        const date = new Date(dateString);
        const diffInSeconds = Math.floor((now - date) / 1000);
        
        if (diffInSeconds < 60) return 'Just now';
        if (diffInSeconds < 3600) return Math.floor(diffInSeconds / 60) + 'm ago';
        if (diffInSeconds < 86400) return Math.floor(diffInSeconds / 3600) + 'h ago';
        return Math.floor(diffInSeconds / 86400) + 'd ago';
    }
    
    function updatePageTitle() {
        const unreadCount = $('.ticket-row.unread').length;
        const $titleElement = $('title');
        const $headerElement = $('.nexlifydesk-admin-ticket-list-ui .header h1');
        
        if (unreadCount > 0) {
            if ($titleElement.length && !$titleElement.text().includes('(')) {
                $titleElement.text('(' + unreadCount + ') ' + $titleElement.text());
            }
            
            if ($headerElement.length && !$headerElement.text().includes('(')) {
                $headerElement.text($headerElement.text() + ' (' + unreadCount + ' unread)');
            }
        } else {
            if ($titleElement.length) {
                $titleElement.text($titleElement.text().replace(/^\(\d+\)\s+/, ''));
            }
            if ($headerElement.length) {
                $headerElement.text($headerElement.text().replace(/\s+\(\d+\s+unread\)$/, ''));
            }
        }
    }
    
    let lastActiveTime = Date.now();
    let wasHidden = false;
    
    document.addEventListener('visibilitychange', function() {
        if (document.hidden) {
            wasHidden = true;
            lastActiveTime = Date.now();
        } else {
            if (wasHidden) {
                const timeHidden = Date.now() - lastActiveTime;
                
                const $header = $('.nexlifydesk-admin-ticket-list-ui .header h1');
                let originalText = $header.text().replace(/\s+\(.*?\)$/, '');
                $header.text(originalText + ' (Refreshing after tab switch...)');
                
                if (timeHidden > 120000) {
                    refreshTicketList();
                } else {
                    setTimeout(() => {
                        refreshTicketList();
                    }, 1000);
                }
                
                startAutoRefresh();
                wasHidden = false;
                
                setTimeout(() => {
                    $header.text(originalText);
                    updatePageTitle();
                }, 3000);
            }
        }
    });
    
    window.addEventListener('focus', function() {
        if (wasHidden || !refreshInterval) {
            startAutoRefresh();
            refreshTicketList();
        }
    });
    
    window.addEventListener('blur', function() {
        lastActiveTime = Date.now();
    });
    
    window.addEventListener('beforeunload', function() {
        stopAutoRefresh();
    });
    
    window.addEventListener('pageshow', function(event) {
        if (event.persisted) {
            startAutoRefresh();
            refreshTicketList();
        }
    });
    
    window.addEventListener('pagehide', function() {
        lastActiveTime = Date.now();
    });
    
    setInterval(function() {
        const now = Date.now();
        const timeSinceLastActive = now - lastActiveTime;
        
        if (!document.hidden || timeSinceLastActive < 600000) {
            if (!refreshInterval) {
                startAutoRefresh();
            }
        }
        
        if (document.hidden && timeSinceLastActive > 600000) {
            if (refreshInterval) {
                clearInterval(refreshInterval);
                refreshInterval = setInterval(function() {
                    refreshTicketList();
                }, 300000); 
            }
        }
    }, 60000);
    
    sortTicketList();
    updatePageTitle();
});