<?php
/**
 * Plugin Name: NexButton
 * Description: Add stylish, customizable buttons to Elementor with various hover effects and animations.
 * Version: 1.5
 * Author: wpcoder75
 * Author URI: https://github.com/asikwp75
 * Text Domain: nex-button
 * Elementor tested up to: 3.13.2
 * PHP: 7.4
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

class NexButtonPlugin {
    const VERSION = '1.5';
    const MINIMUM_ELEMENTOR_VERSION = '3.13.2';
    const MINIMUM_PHP_VERSION = '7.4';

    private static $_instance = null;

    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    public function __construct() {
        add_action( 'plugins_loaded', array( $this, 'init' ) );
    }

    public function admin_notice_minimum_php_version() {
        if ( isset( $_GET['activate'] ) ) {
            unset( $_GET['activate'] );
        }

        $message = sprintf(
            esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'nex-button' ),
            '<strong>' . esc_html__( 'NexButton', 'nex-button' ) . '</strong>',
            '<strong>' . esc_html__( 'PHP', 'nex-button' ) . '</strong>',
            self::MINIMUM_PHP_VERSION
        );
        printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
    }

    public function admin_notice_minimum_elementor_version() {
        if ( isset( $_GET['activate'] ) ) {
            unset( $_GET['activate'] );
        }

        $message = sprintf(
            esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'nex-button' ),
            '<strong>' . esc_html__( 'NexButton', 'nex-button' ) . '</strong>',
            '<strong>' . esc_html__( 'Elementor', 'nex-button' ) . '</strong>',
            self::MINIMUM_ELEMENTOR_VERSION
        );
        printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
    }

    public function admin_notice_missing_main_plugin() {
        if ( isset( $_GET['activate'] ) ) {
            unset( $_GET['activate'] );
        }

        $message = sprintf(
            esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'nex-button' ),
            '<strong>' . esc_html__( 'NexButton', 'nex-button' ) . '</strong>',
            '<strong>' . esc_html__( 'Elementor', 'nex-button' ) . '</strong>'
        );
        printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
    }

    public function init() {
        load_plugin_textdomain( 'nex-button', false, plugin_dir_path( __FILE__ ) . '/languages' );

        if ( ! did_action( 'elementor/loaded' ) ) {
            add_action( 'admin_notices', array( $this, 'admin_notice_missing_main_plugin' ) );
            return;
        }

        if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
            add_action( 'admin_notices', array( $this, 'admin_notice_minimum_elementor_version' ) );
            return;
        }

        if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
            add_action( 'admin_notices', array( $this, 'admin_notice_minimum_php_version' ) );
            return;
        }

        add_action( 'elementor/elements/categories_registered', array( $this, 'add_elementor_category' ) );
        add_action( 'elementor/widgets/register', array( $this, 'register_widgets' ) );
        add_action( 'elementor/frontend/after_enqueue_styles', array( $this, 'enqueue_styles' ) );
    }

    public function add_elementor_category( $elements_manager ) {
        $elements_manager->add_category(
            'nexbutton-category',
            [
                'title' => esc_html__( 'NexButton', 'nex-button' ),
                'icon' => 'eicon-cog',
            ],
            1
        );
    }

    public function register_widgets( $widgets_manager ) {
        require_once plugin_dir_path( __FILE__ ) . 'widgets/advanced-button/advanced-button.php';
        $widgets_manager->register( new \NexButton\Widgets\NexButton_Widget() );
    }

    public function enqueue_styles() {
        wp_enqueue_style( 'nexbutton-styles', plugins_url( 'css/advanced-button.css', __FILE__ ), array(), '1.0.0', false );
    }
}

NexButtonPlugin::instance();
