<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Generates a page URL based on the given route, arguments, and namespace.
 *
 * @param string $route The route to generate the URL for.
 * @param array $args Optional. Additional arguments to append to the URL.
 * @param bool $namespace Optional. Whether to include the namespace in the URL.
 * @return string The generated page URL.
 */
if ( ! function_exists( 'neuro_generate_page_url' ) ) {
	function neuro_generate_page_url( $route, $args = array(), $namespace = false ) {
		if ( $namespace === false ) {
			$namespace = NEUROGENESIS_PAGE_ID;
		}
		$url_params = array(
			'page'  => $namespace,
			'route' => $route,
		);

		// Add nonce for security verification
		$url_params['_wpnonce'] = wp_create_nonce( 'neuro_navigation_action' );

		if ( ! empty( $args ) ) {
			$url_params = array_merge( $url_params, $args );
		}

		return add_query_arg(
			$url_params,
			admin_url( 'admin.php' )
		);
	}
}

if ( ! function_exists( 'neuro_active_page' ) ) {
	/**
	 * Checks if the current page matches the given route.
	 * Note: This function reads GET parameters for navigation purposes only.
	 * Nonce verification is required for security to prevent CSRF attacks.
	 *
	 * @param string $route The route to match against.
	 * @param bool $namespace Optional. Whether to include the namespace in the route. Default is false.
	 * @return bool True if the current page matches the route, false otherwise.
	 */
	function neuro_active_page( $route, $namespace = false ) {
		if ( $namespace === false ) {
			$namespace = NEUROGENESIS_PAGE_ID;
		}

		// Verify nonce for security before accessing GET parameters
		if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'neuro_navigation_action' ) ) {
			return false;
		}

		// Safe navigation parameter access - these are URL structure parameters, not form submissions
		$page          = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';
		$current_route = isset( $_GET['route'] ) ? sanitize_text_field( wp_unslash( $_GET['route'] ) ) : '';

		return $page === $namespace && $current_route === $route;
	}
}

/**
 * Includes a file from the mango framework.
 *
 * @param string $file_path The path to the file to be included.
 * @return void
 */
if ( ! function_exists( 'neuro_require' ) ) {

	function neuro_require( $file_path ) {
		require_once( NEUROGENESIS_PATH . $file_path );
	}
}

if ( ! function_exists( 'neuro_url' ) ) {
	/**
	 * Generates a URL for the plugin.
	 *
	 * @param string $path The path to append to the base URL.
	 * @return string The generated URL.
	 */
	function neuro_url( $path = '' ) {
		return NEUROGENESIS_URL . $path;
	}
}

// function neuro_path( $path = '' )
if ( ! function_exists( 'neuro_path' ) ) {
	/**
	 * Generates a path for the plugin.
	 *
	 * @param string $path The path to append to the base path.
	 * @return string The generated path.
	 */
	function neuro_path( $path = '' ) {
		return NEUROGENESIS_PATH . $path;
	}
}
