<?php
namespace Neurogenesis\Core\Form;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class FormFields {
	public $fields;
	private $only           = array();
	private $nonce_verified = false;

	public function __construct( $fields, $only = array() ) {
		$this->fields = $fields;
		$this->only   = $only;
	}

	public function get_values() {
		// Verify nonce for security - sanitize input first
		$nonce = sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ?? '' ) );
		if ( ! wp_verify_nonce( $nonce, 'form_fields_action' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'neurogenesis-styler' ) );
		}

		$this->nonce_verified = true;

		$values = array();
		foreach ( $this->fields as $field_name => $field ) {
			if ( ! empty( $this->only ) && ! in_array( $field_name, $this->only ) ) {
				continue;
			}
			$values[ $field_name ] = $this->get_post_value( $field_name );
		}
		return $values;
	}

	public function get_group_values() {
		// Verify nonce for security - sanitize input first
		$nonce = sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ?? '' ) );
		if ( ! wp_verify_nonce( $nonce, 'form_fields_action' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'neurogenesis-styler' ) );
		}

		$this->nonce_verified = true;

		$values = array();
		foreach ( $this->fields as $field_name => $field ) {
			if ( ! empty( $this->only ) && ! in_array( $field_name, $this->only ) ) {
				continue;
			}

			$values[ $field_name ] = array();
			foreach ( $field as $subfield_name => $subfield ) {
				$values[ $field_name ][ $subfield ] = $this->get_array_post_value( $field_name, $subfield );
			}
		}
		return $values;
	}

	public function get_array_post_value( $name, $key ) {
		// Ensure this method is only called after nonce verification
		if ( ! $this->nonce_verified ) {
			wp_die( esc_html__( 'Security check failed. Nonce verification required.', 'neurogenesis-styler' ) );
		}

		// Verify nonce again for additional security
		$nonce = sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ?? '' ) );
		if ( ! wp_verify_nonce( $nonce, 'form_fields_action' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'neurogenesis-styler' ) );
		}

		if ( ! isset( $_POST[ $name ][ $key ] ) ) {
			return null;
		}

		$value = sanitize_text_field( wp_unslash( $_POST[ $name ][ $key ] ) );
		return $this->filter_field_value( $value );
	}

	public function get_post_value( $name ) {
		// Ensure this method is only called after nonce verification
		if ( ! $this->nonce_verified ) {
			wp_die( esc_html__( 'Security check failed. Nonce verification required.', 'neurogenesis-styler' ) );
		}

		// Verify nonce again for additional security
		$nonce = sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ?? '' ) );
		if ( ! wp_verify_nonce( $nonce, 'form_fields_action' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'neurogenesis-styler' ) );
		}

		if ( ! isset( $_POST[ $name ] ) ) {
			return null;
		}

		$value = sanitize_text_field( wp_unslash( $_POST[ $name ] ) );
		return $this->filter_field_value( $value );
	}

	private function filter_field_value( $field_value ) {
		if ( is_array( $field_value ) ) {
			return array_map( array( $this, 'sanitize_field_value' ), $field_value );
		}
		return $this->sanitize_field_value( $field_value );
	}

	private function sanitize_field_value( $value ) {
		// Apply appropriate sanitization based on the value type
		if ( is_string( $value ) ) {
			return sanitize_text_field( $value );
		} elseif ( is_numeric( $value ) ) {
			return is_float( $value ) ? (float) $value : (int) $value;
		} elseif ( is_bool( $value ) ) {
			return (bool) $value;
		}

		// For other types, return as-is but ensure it's safe
		return $value;
	}
}
