<?php

namespace Neurogenesis\Core\Form;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class FieldGenerator {
	public static function input( $atts ) {
		$atts = shortcode_atts(
			array(
				'name'       => '',
				'value'      => null,
				'input_type' => 'text',
				'label'      => '',
				'class'      => self::get_css_class( $atts ),
				'required'   => false,
				'readonly'   => false,
			),
			$atts
		);

		return InputField::generate( $atts['name'], $atts['value'], $atts['input_type'], $atts['label'], $atts['required'], $atts['readonly'] );
	}

	public static function text( $atts ) {
		return self::input( $atts );
	}

	public static function textarea( $atts ) {
		$atts = shortcode_atts(
			array(
				'name'     => '',
				'value'    => null,
				'label'    => '',
				'required' => false,
				'class'    => self::get_css_class( $atts ),
				'info'     => false,
			),
			$atts
		);

		return TextareaField::generate( $atts['name'], $atts['value'], $atts['label'], $atts['required'], $atts['info'] );
	}

	public static function select( $atts ) {
		$atts = shortcode_atts(
			array(
				'name'     => '',
				'value'    => null,
				'options'  => array(),
				'label'    => '',
				'required' => false,
				'class'    => self::get_css_class( $atts ),
			),
			$atts
		);

		return SelectField::generate( $atts['name'], $atts['value'], $atts['options'], $atts['label'], $atts['required'] );
	}

	public static function yes_no( $atts ) {
		$atts = shortcode_atts(
			array(
				'name'     => '',
				'value'    => null,
				'label'    => '',
				'required' => false,
				'class'    => self::get_css_class( $atts ),
				'info'     => '',
			),
			$atts
		);

		return YesNoField::generate( $atts['name'], $atts['value'], $atts['label'], $atts['required'] );
	}

	public static function get_css_class( $atts ) {
		$css_class = isset( $atts['class'] ) ? $atts['class'] : '';
		$css_class = 'form-control ' . $css_class;
		return $css_class;
	}

	public static function action( $action, $record_id = null ) {
		ob_start();
		?>
			<input type="hidden" name="action" value="<?php echo esc_attr( $action ); ?>"/>
			<?php wp_nonce_field( $action ); ?>
			
			<?php if ( $record_id !== null ) : ?>            
				<input type="hidden" name="record-id" value="<?php echo esc_attr( $record_id ); ?>"/>
			<?php endif; ?>
		<?php
		return ob_get_clean();
	}

	public static function form_buttons(
		$save_label = 'Save',
		$cancel_label = 'Reset',
		$section_return_url = null,
		$reset = false
	) {
		ob_start();
		?>
			<div class="neuro-form-buttons">                
				<input type="submit" class="btn btn-save btn-primary" value="<?php echo esc_attr( $save_label ); ?>">
				<?php if ( $section_return_url !== null ) : ?>
					<a href="<?php echo esc_url( neuro_generate_page_url( $section_return_url ) ); ?>" class="btn btn-cancel btn-secondary"><?php echo esc_html( $cancel_label ); ?></a>
				<?php endif; ?>

				<?php if ( $reset ) : ?>
					<input type="reset" class="btn btn-cancel btn-secondary" value="<?php echo esc_attr( $cancel_label ); ?>">
				<?php endif; ?>
			</div>
		<?php
		return ob_get_clean();
	}

	// prompt
	public static function prompt( $atts ) {
		$atts = shortcode_atts(
			array(
				'name'     => '',
				'value'    => null,
				'label'    => '',
				'required' => false,
				'class'    => self::get_css_class( $atts ),
			),
			$atts
		);

		$textarea_user_atts = array(
			'name'  => $atts['name'] . '[user]',
			'value' => isset( $atts['value']['user'] ) ? $atts['value']['user'] : '',
			'label' => __( 'User', 'neurogenesis-styler' ),
		);

		$textarea_system_atts = array(
			'name'  => $atts['name'] . '[system]',
			'value' => isset( $atts['value']['system'] ) ? $atts['value']['system'] : '',
			'label' => __( 'System', 'neurogenesis-styler' ),
			'info'  => '<textarea name="display" class="form-control" disabled="disabled"></textarea>',
		);

		ob_start();
		?>
			<div class="neuro-prompt-section">
				<label><?php echo esc_html( $atts['label'] ); ?></label>
				<div class="neuro-prompt-field">
					<div class="neuro-prompt-field-user">
						<?php echo wp_kses_post( self::textarea( $textarea_user_atts ) ); ?>
					</div>
					<div class="neuro-prompt-field-system">                    
						<label>
							<input 
								type="checkbox" 
								name="<?php echo esc_attr( $atts['name'] ); ?>[system_enabled]" 
								value="1" 
								<?php checked( isset( $atts['value']['system_enabled'] ) && $atts['value']['system_enabled'] == 1 ); ?> />
							<?php esc_html_e( 'Use custom system prompt.', 'neurogenesis-styler' ); ?>
						</label>

						<?php echo wp_kses_post( self::textarea( $textarea_system_atts ) ); ?>                        
					</div>
				</div>
			</div>
		<?php
		return ob_get_clean();
	}

	public static function fields( $fields ) {
		$html = '';
		if ( ! is_array( $fields ) ) {
			return $html;
		}

		foreach ( $fields as $field_name => $field ) {
			$field_type     = isset( $field['type'] ) ? $field['type'] : 'input';
			$field['name']  = $field_name;
			$field['value'] = isset( $field['value'] ) ? self::esc( $field['value'] ) : null;

			$html .= self::$field_type( $field );
		}
		return $html;
	}

	public static function esc( $value ) {
		return is_array( $value ) ? array_map( 'neuro_escape_value', $value ) : neuro_escape_value( $value );
	}

	public static function json_fields( $json_fields ) {
		return self::get_fields_from_json( $json_fields );
	}

	public static function get_fields_from_json( $json ) {
		$string = html_entity_decode( neuro_escape_value( $json ) );
		$fields = json_decode( $string, true );
		$html   = '';
		// error_log(print_r($string, true));

		if ( empty( $fields ) ) {
			return $html;
		}

		foreach ( $fields as $key => $data ) {
			$tag         = self::get_field_tag( $key );
			$field_type  = self::get_field_type( $tag );
			$data['tag'] = $tag;

			$html .= self::$field_type( $key, $data );
		}

		return $html;
	}

	public static function get_field_tag( $field ) {
		$tmp = explode( '_', $field );
		return $tmp[0];
	}

	public static function get_field_type( $tag ) {
		switch ( $tag ) {
			case 'img':
				$field_type = 'json_image';
				break;
			default:
				$field_type = 'json_text';
		}

		return $field_type;
	}

	public static function json_text( $key, $data, $atts = array() ) {
		$id_atts = shortcode_atts(
			array(
				'name'     => $key,
				'value'    => $key,
				'label'    => __( 'Field ID', 'neurogenesis-styler' ),
				'readonly' => true,
				'class'    => self::get_css_class( $atts ),
				'info'     => false,
			),
			$atts
		);

		$description_atts = shortcode_atts(
			array(
				'name'     => $key . '[description]',
				'value'    => $data['description'],
				'label'    => __( 'Field description', 'neurogenesis-styler' ),
				'required' => false,
				'class'    => self::get_css_class( $atts ),
				'info'     => false,
			),
			$atts
		);

		$placeholder_atts = shortcode_atts(
			array(
				'name'     => $key . '[placeholder]',
				'value'    => $data['placeholder'],
				'label'    => __( 'Field default content', 'neurogenesis-styler' ),
				'required' => false,
				'class'    => self::get_css_class( $atts ),
				'info'     => false,
			),
			$atts
		);

		ob_start();
		?>
			<div class="neuro-prompt-section">                
				<div class="neuro-prompt-field">
					<div class="neuro-prompt-field-user">
						<?php echo wp_kses_post( self::text( $id_atts ) ); ?>

						<?php echo wp_kses_post( self::text( $description_atts ) ); ?>
					</div>
					<div class="neuro-prompt-field-system">
						<?php if ( $data['tag'] === 'p' ) : ?>
							<?php echo wp_kses_post( self::textarea( $placeholder_atts ) ); ?>
						<?php else : ?>
							<?php echo wp_kses_post( self::text( $placeholder_atts ) ); ?>
						<?php endif; ?>

					</div>
				</div>
			</div>
		<?php
		return ob_get_clean();
	}

	// json_image
	public static function json_image( $key, $data, $atts = array() ) {
		$id_atts = shortcode_atts(
			array(
				'name'     => $key,
				'value'    => $key,
				'label'    => __( 'Field ID', 'neurogenesis-styler' ),
				'readonly' => true,
				'class'    => self::get_css_class( $atts ),
				'info'     => false,
			),
			$atts
		);

		$description_atts = shortcode_atts(
			array(
				'name'     => $key . '[description]',
				'value'    => $data['description'],
				'label'    => __( 'Field description', 'neurogenesis-styler' ),
				'required' => false,
				'class'    => self::get_css_class( $atts ),
				'info'     => false,
			),
			$atts
		);

		$placeholder_atts = shortcode_atts(
			array(
				'name'     => $key . '[placeholder]',
				'value'    => $data['placeholder'],
				'label'    => __( 'Field default content', 'neurogenesis-styler' ),
				'required' => false,
				'class'    => self::get_css_class( $atts ),
				'info'     => false,
			),
			$atts
		);

		$img_src_atts = shortcode_atts(
			array(
				'name'     => $key . '[src]',
				'value'    => $data['src'],
				'label'    => __( 'Default Image', 'neurogenesis-styler' ),
				'required' => false,
				'class'    => self::get_css_class( $atts ),
				'info'     => false,
			),
			$atts
		);

		ob_start();
		?>
			<div class="neuro-prompt-section">                
				<div class="neuro-prompt-field">
					<div class="neuro-prompt-field-user">
						<?php echo wp_kses_post( self::text( $id_atts ) ); ?>

						<?php echo wp_kses_post( self::text( $description_atts ) ); ?>
					</div>
					<div class="neuro-prompt-field-system">
						<?php if ( $data['src'] !== '' ) : ?>
							<img src="<?php echo esc_url( $data['src'] ); ?>" alt="<?php echo esc_attr( $data['description'] ); ?>" />
						<?php endif; ?>                        

						<?php echo wp_kses_post( self::text( $placeholder_atts ) ); ?>

						<?php echo wp_kses_post( self::text( $img_src_atts ) ); ?>
					</div>
				</div>
			</div>
		<?php
		return ob_get_clean();
	}



}
