<?php

namespace Neurogenesis\Core\Blocks;

if (! defined('ABSPATH')) {
    exit; // Exit if accessed directly
}
class FrontendSaveCSS
{
    private $page_cache_key;
    private $template_cache_key;

    public function __construct()
    {
        $this->page_cache_key     = FrontendCSS::get_page_cache_key();
        $this->template_cache_key = FrontendCSS::get_template_cache_key();

        add_action('save_post', array($this, 'save_css'));
    }

    public function save_css($post_id)
    {
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Ensure this runs only for the main save operation
        if (wp_is_post_autosave($post_id) || wp_is_post_revision($post_id)) {
            return;
        }

        // Capability guard
        if (! current_user_can('edit_post', $post_id)) {
            return;
        }

        $post = get_post($post_id);
        if (! $post) {
            return;
        }

        // Only proceed if content has blocks (when function exists)
        if (function_exists('has_blocks')) {
            // has_blocks accepts WP_Post or content
            if (! has_blocks($post)) {
                return;
            }
        }

        $block_styles = new BlockStyles();
        $css          = $block_styles->generate($post_id);

        // Skip empty CSS
        if (! is_string($css) || $css === '') {
            return;
        }

        $this->cache_css($post_id, $css);
    }

    public function cache_css($post_id, $css)
    {
        $post = get_post($post_id);

        if (! $post) {
            return;
        }

        if (in_array($post->post_type, array('wp_template', 'wp_template_part'), true)) {
            return $this->save_template_css($post, $css);
        }

        return $this->save_page_css($post, $css);
    }

    public function save_page_css($post, $css)
    {
        // Avoid unnecessary writes if unchanged
        $existing = get_post_meta($post->ID, $this->page_cache_key, true);
        if (is_string($existing) && $existing === $css) {
            return;
        }
        update_post_meta($post->ID, $this->page_cache_key, $css);
    }

    public function save_template_css($post, $css)
    {

        // Build stronger cache key with theme and area (for template parts)
        $theme = function_exists('get_stylesheet') ? get_stylesheet() : 'default';
        if ($post->post_type === 'wp_template_part') {
            $area_term = function_exists('get_the_terms') ? get_the_terms($post, 'wp_theme') : false;
            $area      = 'uncategorized';
            // In block themes, template part areas use 'wp_template_part_area' taxonomy
            if (function_exists('get_the_terms')) {
                $area_terms = get_the_terms($post->ID, 'wp_template_part_area');
                if (is_array($area_terms) && ! empty($area_terms) && isset($area_terms[0]->slug)) {
                    $area = $area_terms[0]->slug;
                }
            }
            $type = $theme . '::template-part::' . $area . '::' . $post->post_name;
        } else {
            $template_name = FrontendCSS::get_template_name($post);
            $type = $theme . '::template::' . $template_name;
        }

        $cache = get_option($this->template_cache_key, null);
        if (! is_array($cache)) {
            $cache = array();
            // Ensure the option exists with autoload disabled on first create
            if (false === get_option($this->template_cache_key, false)) {
                add_option($this->template_cache_key, $cache, '', 'no');
            }
        }

        $cache[$type] = $css;

        update_option($this->template_cache_key, $cache);
    }

    
}
