<?php

namespace Neurogenesis\Core\Blocks;

if (! defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class FrontendDisplayCSS
{
    private $page_cache_key;
    private $template_cache_key;

    public function __construct()
    {
        $this->page_cache_key     = FrontendCSS::get_page_cache_key();
        $this->template_cache_key = FrontendCSS::get_template_cache_key();

        add_action('wp_enqueue_scripts', array($this, 'get_global_styles'));
    }

    public function get_global_styles()
    {
        $css = $this->get_css();

        if (! empty($css)) {
            wp_register_style('neurogenesis-generated', false);
            wp_enqueue_style('neurogenesis-generated');

            wp_add_inline_style('neurogenesis-generated', $css);
        }
    }

    private function get_css()
    {
        $css  = $this->get_page_css();
        $css .= $this->get_template_css();

        return $css;
    }

    /**
     * Regenerates all template and template part CSS and stores it in a global option.
     */
    public function regenerate_all_templates_css()
    {
        $cache = array();

        // Get all templates
        $templates = get_posts(
            array(
                'post_type'      => array('wp_template', 'wp_template_part'),
                'post_status'    => 'publish',
                'posts_per_page' => -1,
            )
        );

        $block_styles = new BlockStyles();
        $theme        = function_exists('get_stylesheet') ? get_stylesheet() : 'default';

        foreach ($templates as $template) {
            $slug = $template->post_name;

            if ($template->post_type === 'wp_template_part') {
                $area = 'uncategorized';
                if (function_exists('get_the_terms')) {
                    $area_terms = get_the_terms($template->ID, 'wp_template_part_area');
                    if (is_array($area_terms) && ! empty($area_terms) && isset($area_terms[0]->slug)) {
                        $area = $area_terms[0]->slug;
                    }
                }
                $key = $theme . '::template-part::' . $area . '::' . $slug;
            } else {
                $key = $theme . '::template::' . $slug;
            }

            $css = $block_styles->generate($template->ID);

            if (is_string($css) && $css !== '') {
                $cache[$key] = $css;
            }
        }

        $existing_option = get_option($this->template_cache_key, null);
        if (false === get_option($this->template_cache_key, false)) {
            add_option($this->template_cache_key, array(), '', 'no');
        }
        // Merge with existing cache if it's an array, otherwise overwrite
        if (is_array($existing_option)) {
            $cache = array_merge($existing_option, $cache);
        }
        update_option($this->template_cache_key, $cache);
    }


    public function get_template_css()
    {
        if (is_admin() || is_feed()) {
            return '';
        }

        $css    = '';
        $cache  = get_option($this->template_cache_key, array());
        $theme  = function_exists('get_stylesheet') ? get_stylesheet() : 'default'; 
        
        $post = get_post(get_the_ID());
        
        $template_name = FrontendCSS::get_template_name($post);
        $template_key  = FrontendCSS::get_template_name_cache_key($template_name);
        $css          .= $cache[$template_key] ?? '';

        // Try multiple methods to get the current template
        $template = $this->get_current_template();

        if (! $template || ! isset($template->content)) {
            return $css;
        }

        // Track already processed template parts to avoid loops
        $visited = array();

        // Start with the main template
        $template_slug = $template->slug;
        
        $template_key  = $theme . '::template::' . $template_slug;
        $css          .= $cache[$template_key] ?? '';
        

        $css .= $this->collect_template_part_css_recursive($template->content, $cache, $visited, $theme);

        return $css;
    }

    private function get_current_template()
    {
        // Method 1: Try to get from global variable
        global $wp_current_template;
        if ($wp_current_template && isset($wp_current_template->content)) {
            return $wp_current_template;
        }

        // Method 2: Try to get default template based on current context
        $template_slug = $this->get_template_slug_by_context();

        if ($template_slug) {
            // Try to get template from the list we already have
            $all_templates = get_block_templates();
            foreach ($all_templates as $template) {
                if ($template->slug === $template_slug && $template->type === 'wp_template' && isset($template->content) && ! empty($template->content)) {
                    return $template;
                }
            }

            // Fallback to get_block_template
            $template = get_block_template($template_slug, 'wp_template');
            if ($template && isset($template->content)) {
                return $template;
            }
        }

        // Method 3: Try to get the current theme's default template
        $template = get_block_template('index', 'wp_template');
        if ($template && isset($template->content)) {
            return $template;
        }

        // Method 4: Try to find template parts that might be in use
        $template_part = $this->find_active_template_part();
        if ($template_part) {
            return $template_part;
        }

        // Method 5: Try to get specific template parts by ID
        $template_part = $this->get_template_part_by_id(378);
        if ($template_part) {
            return $template_part;
        }

        return null;
    }

    private function find_active_template_part()
    {
        // Try to get template parts from the current post/page
        $post_id = get_the_ID();

        if ($post_id) {
            $post = get_post($post_id);

            if ($post && ! empty($post->post_content)) {
                // Look for template-part blocks in the post content
                $blocks = parse_blocks($post->post_content);

                foreach ($blocks as $block) {
                    if ($block['blockName'] === 'core/template-part' && ! empty($block['attrs']['slug'])) {
                        $template_part = get_block_template($block['attrs']['slug'], 'wp_template_part');
                        if ($template_part && isset($template_part->content)) {
                            return $template_part;
                        }
                    }
                }
            }
        }

        // Try to get template parts from the theme
        $template_parts = get_block_templates(array('post_type' => 'wp_template_part'));

        if (! empty($template_parts)) {
            // Return the first available template part
            foreach ($template_parts as $template_part) {
                if (isset($template_part->content) && ! empty($template_part->content)) {
                    return $template_part;
                }
            }
        }

        return null;
    }

    private function get_template_part_by_id($id)
    {
        $template_part = get_post($id);
        if ($template_part && $template_part->post_type === 'wp_template_part' && isset($template_part->post_content) && ! empty($template_part->post_content)) {
            // Convert post to template object
            $template_obj          = new \stdClass();
            $template_obj->slug    = $template_part->post_name;
            $template_obj->content = $template_part->post_content;
            $template_obj->wp_id   = $template_part->ID;
            return $template_obj;
        }
        return null;
    }

    private function get_template_slug_by_context()
    {
        if (is_home() || is_front_page()) {
            return 'home';
        } elseif (is_single()) {
            return 'single';
        } elseif (is_page()) {
            return 'page';
        } elseif (is_archive()) {
            return 'archive';
        } elseif (is_search()) {
            return 'search';
        } elseif (is_404()) {
            return '404';
        }

        return null;
    }


    /**
     * Recursively collect CSS from template parts used inside block content.
     */
    private function collect_template_part_css_recursive(string $block_content, array $cache, array &$visited, string $theme): string
    {
        $css    = '';
        $blocks = parse_blocks($block_content);

        foreach ($blocks as $block) {
            if ($block['blockName'] === 'core/template-part' && ! empty($block['attrs']['slug'])) {
                $slug = $block['attrs']['slug'];
                $area = 'uncategorized';

                // If area is provided in block attrs, prefer it
                if (! empty($block['attrs']['area']) && is_string($block['attrs']['area'])) {
                    $area = $block['attrs']['area'];
                } else {
                    // Otherwise, try to resolve via taxonomy terms
                    $template_part_post = get_page_by_path($slug, OBJECT, 'wp_template_part');
                    if ($template_part_post) {
                        $terms = function_exists('get_the_terms') ? get_the_terms($template_part_post->ID, 'wp_template_part_area') : false;
                        if (is_array($terms) && ! empty($terms) && isset($terms[0]->slug)) {
                            $area = $terms[0]->slug;
                        }
                    }
                }

                $key = $theme . '::template-part::' . $area . '::' . $slug;

                if (in_array($slug, $visited, true)) {
                    continue;
                }
                $visited[] = $slug;

                $css .= $cache[$key] ?? '';

                // Recursively collect nested template part CSS
                $template_part_post = get_page_by_path($slug, OBJECT, 'wp_template_part');
                if ($template_part_post && ! empty($template_part_post->post_content)) {
                    $css .= $this->collect_template_part_css_recursive($template_part_post->post_content, $cache, $visited, $theme);
                }
            }
        }

        return $css;
    }


    public function get_page_css()
    {
        if (is_admin() || is_feed()) {
            return '';
        }
        $post_id = get_the_ID();  

        // if post meta has _wp_page_template get the template with that name and get the css from the cache
        if (get_post_meta($post_id, '_wp_page_template', true)) {
            $template = get_page_by_path(get_post_meta($post_id, '_wp_page_template', true), OBJECT, 'wp_template');
            if ($template) {
                $post_id = $template->ID;                
            }
        }

        $css = get_post_meta($post_id, $this->page_cache_key, true);

        return is_string($css) ? wp_strip_all_tags($css) : '';
    }
}
