<?php
namespace Neurogenesis\Actions;

use Neurogenesis\Core\Form\FormFields;
use Neurogenesis\Traits\HasAction;
use Neurogenesis\Traits\IsCrudAction;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class CrudActions {
	use HasAction;
	use IsCrudAction;

	// __construct
	public function __construct() {
		add_action( 'admin_post_neuro_crud_create', array( $this, 'create' ) );
		add_action( 'admin_post_neuro_crud_update', array( $this, 'update' ) );
		add_action( 'admin_post_neuro_crud_delete', array( $this, 'destroy' ) );
	}

	// create
	public function create() {
		// Nonce verification
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'neuro_crud_create' ) ) {
			$this->nonceInvalidMessage();
			$this->redirectToSettingsPage( null );
			return;
		}

		// Validate and sanitize POST data
		if ( ! isset( $_POST['neuro-crud'] ) || empty( $_POST['neuro-crud'] ) ) {
			neuro_flash_message_add( __( 'Invalid service type provided.', 'neurogenesis-styler' ), 'error' );
			$this->redirectToSettingsPage( null );
			return;
		}

		// Sanitize and unslash POST data
		$service = $this->factory( sanitize_text_field( wp_unslash( $_POST['neuro-crud'] ) ) );

		$fields = new FormFields( $service->model->get_fields() );

		$record = $service->store( $fields->get_values() );

		if ( $record ) {
			neuro_flash_message_add( __( 'The record was saved.', 'neurogenesis-styler' ) );
		} else {
			neuro_flash_message_add( __( 'There was a problem saving the data. The record was not saved', 'neurogenesis-styler' ), 'error' );
		}

		// Safely get the back-to parameter within the nonce verification block
		$back_to = null;
		if ( isset( $_POST['neuro-back-to'] ) && ! empty( $_POST['neuro-back-to'] ) ) {
			$back_to = sanitize_text_field( wp_unslash( $_POST['neuro-back-to'] ) );
		}

		$this->redirectToSettingsPage( $back_to );
	}

	// update
	public function update() {
		// Nonce verification
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'neuro_crud_update' ) ) {
			$this->nonceInvalidMessage();
			$this->redirectToSettingsPage( null );
			return;
		}

		// Validate and sanitize POST data
		if ( ! isset( $_POST['neuro-crud'] ) || empty( $_POST['neuro-crud'] ) ) {
			neuro_flash_message_add( __( 'Invalid service type provided.', 'neurogenesis-styler' ), 'error' );
			$this->redirectToSettingsPage( null );
			return;
		}

		if ( ! isset( $_POST['record-id'] ) || empty( $_POST['record-id'] ) ) {
			neuro_flash_message_add( __( 'Invalid record ID provided.', 'neurogenesis-styler' ), 'error' );
			$this->redirectToSettingsPage( null );
			return;
		}

		// Sanitize and unslash POST data
		$service   = $this->factory( sanitize_text_field( wp_unslash( $_POST['neuro-crud'] ) ) );
		$record_id = absint( wp_unslash( $_POST['record-id'] ) );

		$fields = new FormFields( $service->model->get_fields() );
		$record = $service->update( $fields->get_values(), array( 'id' => $record_id ) );

		if ( $record ) {
			neuro_flash_message_add( __( 'The record was updated.', 'neurogenesis-styler' ) );
		} else {
			neuro_flash_message_add( __( 'There was a problem updating the data. The record was not saved', 'neurogenesis-styler' ), 'error' );
		}

		// Safely get the back-to parameter within the nonce verification block
		$back_to = null;
		if ( isset( $_POST['neuro-back-to'] ) && ! empty( $_POST['neuro-back-to'] ) ) {
			$back_to = sanitize_text_field( wp_unslash( $_POST['neuro-back-to'] ) );
		}

		$this->redirectToSettingsPage( $back_to );
	}

	// destroy
	public function destroy() {
		// Nonce verification
		if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'neuro_crud_delete' ) ) {
			$this->nonceInvalidMessage();
			$this->redirectToSettingsPage( null );
			return;
		}

		// Validate and sanitize GET data
		if ( ! isset( $_GET['neuro-crud'] ) || empty( $_GET['neuro-crud'] ) ) {
			neuro_flash_message_add( __( 'Invalid service type provided.', 'neurogenesis-styler' ), 'error' );
			$this->redirectToSettingsPage( null );
			return;
		}

		if ( ! isset( $_GET['post_id'] ) || empty( $_GET['post_id'] ) ) {
			neuro_flash_message_add( __( 'Invalid record ID provided.', 'neurogenesis-styler' ), 'error' );
			$this->redirectToSettingsPage( null );
			return;
		}

		// Sanitize and unslash GET data
		$service   = $this->factory( sanitize_text_field( wp_unslash( $_GET['neuro-crud'] ) ) );
		$record_id = absint( wp_unslash( $_GET['post_id'] ) );

		$record = $service->destroy( array( 'id' => $record_id ) );

		if ( $record ) {
			neuro_flash_message_add( __( 'The record was deleted.', 'neurogenesis-styler' ) );
		} else {
			neuro_flash_message_add( __( 'There was a problem deleting the data. The record was not deleted', 'neurogenesis-styler' ), 'error' );
		}

		// Safely get the back_to parameter within the nonce verification block
		$back_to = null;
		if ( isset( $_GET['back_to'] ) && ! empty( $_GET['back_to'] ) ) {
			$back_to = sanitize_text_field( wp_unslash( $_GET['back_to'] ) );
		}

		$this->redirectToSettingsPage( neuro_generate_page_url( $back_to ) );
	}
}
