<?php
/**
 * Plugin Name: NP Formshield
 * Description: Advanced spam protection for Contact Form 7, Gravity Forms, Forminator, Ninja Forms, WPForms, Fluent Forms and custom forms.
 * Version: 1.0.5
 * Author: Needs Plugin
 * Author URI: http://needsplugin.com/
 * License: GPL v2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: needsplugin-formshield
 */
// Prevent direct access
if ( !defined( 'ABSPATH' ) ) {
    exit;
}
// Define plugin constants
define( 'NPFO_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'NPFO_PLUGIN_PATH', plugin_dir_path( __FILE__ ) );
define( 'NPFO_VERSION', '1.0.4' );
// Freemius SDK removed for WordPress.org compliance
// Main plugin class
class NPFO_Form_Shield {
    public function __construct() {
        add_action( 'init', array($this, 'init') );
        add_action( 'admin_menu', array($this, 'add_admin_menu') );
        add_action( 'admin_enqueue_scripts', array($this, 'enqueue_admin_scripts') );
        add_action( 'wp_enqueue_scripts', array($this, 'enqueue_frontend_scripts') );
        // Hook into various form plugins
        $this->init_form_hooks();
        // AJAX handlers
        add_action( 'wp_ajax_npfo_get_forms', array($this, 'ajax_get_forms') );
        add_action( 'wp_ajax_npfo_get_form_fields', array($this, 'ajax_get_form_fields') );
        add_action( 'wp_ajax_npfo_save_protection', array($this, 'ajax_save_protection') );
        add_action( 'wp_ajax_npfo_delete_protection', array($this, 'ajax_delete_protection') );
        add_action( 'wp_ajax_npfo_get_protection', array($this, 'ajax_get_protection') );
        add_action( 'wp_ajax_npfo_check_plugin_status', array($this, 'ajax_check_plugin_status') );
        // Frontend AJAX handler for real-time validation
        add_action( 'wp_ajax_npfo_get_form_protection', array($this, 'ajax_get_form_protection') );
        add_action( 'wp_ajax_nopriv_npfo_get_form_protection', array($this, 'ajax_get_form_protection') );
    }

    /**
     * Check if user has premium access
     * Freemius removed - all features now available to all users
     */
    public function is_premium_user() {
        return true; // All features available to all users
    }

    /**
     * Check if specific form plugin is available for free users
     * Freemius removed - all plugins now available to all users
     */
    public function is_form_plugin_free( $plugin ) {
        return true; // All plugins available to all users
    }

    /**
     * Check if user has any form of premium access (license, trial, etc.)
     * Freemius removed - all features now available to all users
     */
    public function has_premium_access() {
        return true; // All features available to all users
    }

    public function init() {
        // WordPress.org automatically loads plugin translations, so load_plugin_textdomain() is not needed
        // Create database table on activation
        $this->create_database_table();
    }

    public function add_admin_menu() {
        add_menu_page(
            'NP Formshield',
            'NP Formshield',
            'manage_options',
            'no-spam-forms',
            array($this, 'admin_dashboard'),
            'dashicons-shield-alt',
            30
        );
        add_submenu_page(
            'no-spam-forms',
            'Protected Forms List',
            'Protected Forms List',
            'manage_options',
            'npfo-protected-forms',
            array($this, 'protected_forms_page')
        );
        add_submenu_page(
            'no-spam-forms',
            'Pro Version',
            'Pro Version',
            'manage_options',
            'npfo-pro-version',
            array($this, 'pro_version_page')
        );
    }

    public function enqueue_admin_scripts( $hook ) {
        if ( strpos( $hook, 'no-spam-forms' ) !== false || strpos( $hook, 'npfo-' ) !== false ) {
            // Register and enqueue admin scripts and styles
            wp_register_script(
                'npfo-admin-script',
                NPFO_PLUGIN_URL . 'assets/admin-script.js',
                array('jquery'),
                NPFO_VERSION,
                true
            );
            wp_register_style(
                'npfo-admin-style',
                NPFO_PLUGIN_URL . 'assets/admin-style.css',
                array(),
                NPFO_VERSION
            );
            
            wp_enqueue_script( 'npfo-admin-script' );
            wp_enqueue_style( 'npfo-admin-style' );
            // All features available to all users (Freemius removed)
            $is_premium = true;
            
            // Get dashboard stats for dashboard page
            $dashboard_stats = array();
            if ( strpos( $hook, 'no-spam-forms' ) !== false && ! strpos( $hook, 'npfo-' ) ) {
                $all_protections = $this->get_all_protections();
                $total_fields = 0;
                if ( is_array( $all_protections ) ) {
                    foreach( $all_protections as $protection ) {
                        if( isset( $protection['field_protections'] ) && is_array( $protection['field_protections'] ) ) {
                            $total_fields += count( $protection['field_protections'] );
                        }
                    }
                    $dashboard_stats = array(
                        'protected_forms_count' => count( $all_protections ),
                        'protected_fields_count' => $total_fields,
                    );
                }
            }
            
            wp_localize_script( 'npfo-admin-script', 'npfo_ajax', array(
                'ajax_url'   => admin_url( 'admin-ajax.php' ),
                'nonce'      => wp_create_nonce( 'npfo_nonce' ),
                'is_premium' => $is_premium,
                'dashboard_stats' => $dashboard_stats,
            ) );
        }
    }

    public function enqueue_frontend_scripts() {
        // Register and enqueue frontend styles
        wp_register_style(
            'npfo-frontend-style',
            NPFO_PLUGIN_URL . 'assets/frontend-style.css',
            array(),
            NPFO_VERSION
        );
        wp_enqueue_style( 'npfo-frontend-style' );
        
        // Register and enqueue frontend scripts
        wp_register_script(
            'npfo-frontend-script',
            NPFO_PLUGIN_URL . 'assets/frontend-script.js',
            array('jquery'),
            NPFO_VERSION,
            true
        );
        wp_register_script(
            'npfo-forminator-errors',
            NPFO_PLUGIN_URL . 'assets/forminator-error-display.js',
            array('jquery'),
            NPFO_VERSION,
            true
        );
        wp_register_script(
            'npfo-ninja-forms-realtime',
            NPFO_PLUGIN_URL . 'assets/ninja-forms-realtime-validation.js',
            array('jquery'),
            NPFO_VERSION,
            true
        );
        
        wp_enqueue_script( 'npfo-frontend-script' );
        wp_enqueue_script( 'npfo-forminator-errors' );
        wp_enqueue_script( 'npfo-ninja-forms-realtime' );
        wp_localize_script( 'npfo-frontend-script', 'npfo_frontend', array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'npfo_frontend_nonce' ),
        ) );
        // Localize script for Ninja Forms real-time validation
        wp_localize_script( 'npfo-ninja-forms-realtime', 'npfo_ajax', array(
            'ajax_url'       => admin_url( 'admin-ajax.php' ),
            'frontend_nonce' => wp_create_nonce( 'npfo_frontend_nonce' ),
        ) );
    }

    public function create_database_table() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'npfo_protections';
        $charset_collate = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE {$table_name} (\r\n      id mediumint(9) NOT NULL AUTO_INCREMENT,\r\n      form_plugin varchar(50) NOT NULL,\r\n      form_id varchar(100) NOT NULL,\r\n      form_title varchar(255) NOT NULL,\r\n      field_protections longtext NOT NULL,\r\n      block_links tinyint(1) DEFAULT 1,\r\n      created_at datetime DEFAULT CURRENT_TIMESTAMP,\r\n      updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,\r\n      PRIMARY KEY (id),\r\n      UNIQUE KEY unique_form (form_plugin, form_id)\r\n    ) {$charset_collate};";
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta( $sql );
    }

    public function admin_dashboard() {
        include NPFO_PLUGIN_PATH . 'templates/dashboard.php';
    }

    public function protected_forms_page() {
        include NPFO_PLUGIN_PATH . 'templates/protected-forms.php';
    }

    public function pro_version_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            // phpcs:ignore WordPress.WP.I18n.TextDomainMismatch -- Text domain uses hyphen (required), folder name has space
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'needsplugin-formshield'));
        }
        
        // Show Pro Version page
        include NPFO_PLUGIN_PATH . 'templates/pro-version.php';
    }

    public function init_form_hooks() {
        // Contact Form 7 - Use higher priority to ensure it runs
        add_filter(
            'wpcf7_validate',
            array($this, 'validate_cf7'),
            5,
            2
        );
        // Gravity Forms - Use higher priority
        add_filter( 'gform_validation', array($this, 'validate_gravity_forms'), 5 );
        // WPForms - Use multiple hooks with different priorities
        add_action(
            'wpforms_process_before',
            array($this, 'validate_wpforms'),
            5,
            3
        );
        add_action(
            'wpforms_process',
            array($this, 'validate_wpforms'),
            5,
            3
        );
        // Ninja Forms - Use multiple hooks with better priorities
        add_action(
            'ninja_forms_process',
            array($this, 'validate_ninja_forms_working'),
            5,
            1
        );
        add_filter(
            'ninja_forms_submit_data',
            array($this, 'validate_ninja_forms_filter'),
            5,
            1
        );
        add_action( 'wp_ajax_ninja_forms_ajax_submit', array($this, 'validate_ninja_forms_ajax'), 5 );
        add_action( 'wp_ajax_nopriv_ninja_forms_ajax_submit', array($this, 'validate_ninja_forms_ajax'), 5 );
        add_filter(
            'ninja_forms_field_validate',
            array($this, 'validate_ninja_forms_field'),
            5,
            2
        );
        // Add early validation hook to catch all submissions
        add_action( 'init', array($this, 'ninja_forms_early_validation') );
        // Forminator - Use higher priority
        add_filter(
            'forminator_custom_form_submit_errors',
            array($this, 'validate_forminator'),
            5,
            3
        );
        // Fluent Forms - Use higher priority
        add_filter(
            'fluentform_validation_errors',
            array($this, 'validate_fluent_forms'),
            5,
            3
        );
        // Debug hook to see which forms are being processed
        add_action( 'init', array($this, 'debug_form_processing') );
    }

    // Early Ninja Forms validation to catch all submissions
    public function ninja_forms_early_validation() {
        // Security: Only process if this is a Ninja Forms submission
        $action = isset( $_POST['action'] ) ? sanitize_text_field( wp_unslash( $_POST['action'] ) ) : '';
        if ( !isset( $_POST['nf_form_submit'] ) && ( empty( $action ) || $action !== 'ninja_forms_ajax_submit' ) ) {
            return;
        }
        
        // Security: Verify Ninja Forms nonce
        if ( isset( $_POST['_wpnonce'] ) ) {
            $nonce = sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) );
            if ( ! wp_verify_nonce( $nonce, 'ninja_forms_ajax_nonce' ) ) {
                wp_send_json_error( array(
                    'message' => 'Invalid request. Please refresh the page and try again.',
                ), 403 );
                return;
            }
        } else {
            // Nonce is required for security
            wp_send_json_error( array(
                'message' => 'Security token missing. Please refresh the page and try again.',
            ), 403 );
            return;
        }
        
        // Get form ID - sanitize input
        $form_id = null;
        if ( isset( $_POST['form_id'] ) ) {
            $form_id = intval( $_POST['form_id'] );
        } elseif ( isset( $_POST['id'] ) ) {
            $form_id = intval( $_POST['id'] );
        }
        // Validate form ID
        if ( !$form_id || $form_id <= 0 ) {
            return;
        }
        // Get protection settings
        $protection = $this->get_protection_settings( 'ninja-forms', $form_id );
        if ( !$protection ) {
            return;
        }
        // Extract form data - only process expected Ninja Forms field keys
        $posted_data = array();
        
        // Filter $_POST keys to only Ninja Forms field keys (nf_field_*)
        $nf_field_keys = array_filter( array_keys( $_POST ), function( $key ) {
            return strpos( $key, 'nf_field_' ) === 0;
        } );
        
        // Process only the filtered keys
        foreach ( $nf_field_keys as $key ) {
            // Sanitize key first
            $sanitized_key = sanitize_key( $key );
            if ( empty( $sanitized_key ) || strpos( $sanitized_key, 'nf_field_' ) !== 0 ) {
                continue;
            }
            
            $field_id = absint( str_replace( 'nf_field_', '', $sanitized_key ) );
            if ( $field_id <= 0 ) {
                continue;
            }
            
            // Get field value - only if key exists
            if ( ! isset( $_POST[ $key ] ) ) {
                continue;
            }
            
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Key is sanitized above, value sanitized below
            $value = wp_unslash( $_POST[ $key ] );
            
            // Get field label
            if ( class_exists( 'Ninja_Forms' ) ) {
                try {
                    $form_obj = Ninja_Forms()->form( $form_id );
                    $field_obj = $form_obj->get_field( $field_id );
                    if ( $field_obj ) {
                        $field_settings = $field_obj->get_settings();
                        if ( !empty( $field_settings['label'] ) ) {
                            $field_label = sanitize_text_field( $field_settings['label'] );
                            // Sanitize field value
                            $sanitized_value = is_array( $value ) ? array_map( 'sanitize_text_field', $value ) : sanitize_text_field( $value );
                            $posted_data[$field_label] = is_array( $sanitized_value ) ? implode( ' ', $sanitized_value ) : $sanitized_value;
                        }
                    }
                } catch ( Exception $e ) {
                    // Silently continue on error
                    continue;
                }
            }
        }
        if ( empty( $posted_data ) ) {
            return;
        }
        // Validate the data
        $validation_result = $this->validate_form_data( $posted_data, $protection['field_protections'], $protection['block_links'] ?? true );
        // If validation fails, stop submission immediately
        if ( !$validation_result['valid'] ) {
            // Send error response and stop
            wp_send_json_error( array(
                'errors'  => $validation_result['errors'],
                'message' => 'Form contains prohibited content. Please remove blocked words and try again.',
            ), 400 );
        }
    }

    // Debug function to track form processing
    // Note: This function is kept for potential future debugging but does not process any data
    public function debug_form_processing() {
        // Removed debug code for production - no $_POST processing
        // This function can be used for future debugging if needed
    }

    // Form validation methods
    public function validate_cf7( $result, $tags ) {
        // Contact Form 7 is always free
        $form_id = $this->get_form_id( 'contact-form-7' );
        if ( !$form_id ) {
            return $result;
        }
        $protection = $this->get_protection_settings( 'contact-form-7', $form_id );
        // Apply fallback protection if no specific protection found
        if ( !$protection && $this->has_fallback_protection( $form_id ) ) {
            $submission = WPCF7_Submission::get_instance();
            if ( $submission ) {
                $posted_data = $submission->get_posted_data();
                $fallback_result = $this->apply_fallback_protection( $posted_data );
                if ( !$fallback_result['valid'] ) {
                    foreach ( $fallback_result['errors'] as $field => $error ) {
                        $result->invalidate( $field, $error );
                    }
                }
            }
            return $result;
        }
        if ( $protection ) {
            $submission = WPCF7_Submission::get_instance();
            if ( $submission ) {
                $posted_data = $submission->get_posted_data();
                $validation_result = $this->validate_form_data( $posted_data, $protection['field_protections'], $protection['block_links'] ?? true );
                if ( !$validation_result['valid'] ) {
                    foreach ( $validation_result['errors'] as $field => $error ) {
                        $result->invalidate( $field, $error );
                    }
                }
            }
        } else {
        }
        return $result;
    }

    public function validate_gravity_forms( $validation_result ) {
        // Check if user has premium access for Gravity Forms
        if ( !$this->has_premium_access() ) {
            return $validation_result;
            // Skip validation for free users
        }
        $form = $validation_result['form'];
        $form_id = $this->get_form_id( 'gravity-forms', $form );
        if ( !$form_id ) {
            return $validation_result;
        }
        $protection = $this->get_protection_settings( 'gravity-forms', $form_id );
        // Apply fallback protection if no specific protection found
        if ( !$protection && $this->has_fallback_protection( $form_id ) ) {
            $posted_data = array();
            foreach ( $form['fields'] as $field ) {
                $field_value = rgpost( 'input_' . $field->id );
                if ( !empty( $field_value ) ) {
                    $posted_data[$field->label] = $field_value;
                }
            }
            $fallback_result = $this->apply_fallback_protection( $posted_data );
            if ( !$fallback_result['valid'] ) {
                $validation_result['is_valid'] = false;
                foreach ( $form['fields'] as &$field ) {
                    if ( isset( $fallback_result['errors'][$field->label] ) ) {
                        $field->failed_validation = true;
                        $field->validation_message = $fallback_result['errors'][$field->label];
                    }
                }
            }
            return $validation_result;
        }
        if ( $protection ) {
            $posted_data = array();
            foreach ( $form['fields'] as $field ) {
                $field_value = '';
                // Handle different field types
                if ( $field->type === 'name' ) {
                    // Name field has multiple inputs - check all possible sub-fields
                    $name_parts = array();
                    // Based on HTML structure: input_1_1 (first), input_1_3 (last), etc.
                    $possible_subfields = array(
                        '1',
                        '2',
                        '3',
                        '4',
                        '5',
                        '6'
                    );
                    foreach ( $possible_subfields as $subfield ) {
                        $subfield_value = rgpost( 'input_' . $field->id . '_' . $subfield );
                        if ( !empty( $subfield_value ) ) {
                            $name_parts[] = $subfield_value;
                        }
                    }
                    // Also check dot notation (backup)
                    $dot_subfields = array('3', '6');
                    foreach ( $dot_subfields as $subfield ) {
                        $subfield_value = rgpost( 'input_' . $field->id . '.' . $subfield );
                        if ( !empty( $subfield_value ) ) {
                            $name_parts[] = $subfield_value;
                        }
                    }
                    // Check for full name (single input)
                    $full_name = rgpost( 'input_' . $field->id );
                    if ( !empty( $full_name ) ) {
                        $name_parts[] = $full_name;
                    }
                    $field_value = implode( ' ', array_unique( $name_parts ) );
                    // Remove duplicates
                } else {
                    // Regular field
                    $field_value = rgpost( 'input_' . $field->id );
                }
                if ( !empty( $field_value ) ) {
                    $posted_data[$field->label] = $field_value;
                }
            }
            $validation_result_custom = $this->validate_form_data( $posted_data, $protection['field_protections'], $protection['block_links'] ?? true );
            if ( !$validation_result_custom['valid'] ) {
                $validation_result['is_valid'] = false;
                foreach ( $form['fields'] as &$field ) {
                    if ( isset( $validation_result_custom['errors'][$field->label] ) ) {
                        $field->failed_validation = true;
                        $field->validation_message = $validation_result_custom['errors'][$field->label];
                    }
                }
            }
        } else {
        }
        return $validation_result;
    }

    public function validate_wpforms( $fields, $entry, $form_data = null ) {
        // Check if user has premium access for WPForms
        if ( !$this->has_premium_access() ) {
            return;
            // Skip validation for free users
        }
        // Handle parameter variations - WPForms passes different parameter counts
        if ( $form_data === null ) {
            // If only 2 parameters passed, $entry might actually be $form_data
            if ( is_array( $entry ) && isset( $entry['id'] ) ) {
                $form_data = $entry;
                $entry = array();
            } else {
                return;
            }
        }
        $protection = $this->get_protection_settings( 'wpforms', $form_data['id'] );
        if ( !$protection ) {
            return;
            // Allow submission if no protection is set
        }
        // Get form structure to understand field labels
        $form_obj = wpforms()->form->get( $form_data['id'] );
        $form_fields = array();
        if ( $form_obj && !empty( $form_obj->post_content ) ) {
            $form_structure = json_decode( $form_obj->post_content, true );
            if ( !empty( $form_structure['fields'] ) ) {
                foreach ( $form_structure['fields'] as $field_data ) {
                    $form_fields[$field_data['id']] = $field_data['label'];
                }
            }
        }
        // Build posted data using actual field labels - INCLUDE EMPTY VALUES for whitelist validation
        $posted_data = array();
        $field_map = array();
        // WPForms passes data in different format - check if fields array contains form data directly
        if ( isset( $fields['fields'] ) ) {
            // Extract field values from the fields array
            foreach ( $fields['fields'] as $field_id => $field_value ) {
                // Get the actual field label from form structure
                $field_label = ( isset( $form_fields[$field_id] ) ? $form_fields[$field_id] : 'Field ' . $field_id );
                // Check if this field has protection settings
                $has_protection = false;
                if ( isset( $protection['field_protections'] ) ) {
                    foreach ( $protection['field_protections'] as $protected_field => $protection_config ) {
                        if ( strtolower( trim( $protected_field ) ) === strtolower( trim( $field_label ) ) ) {
                            $has_protection = true;
                            break;
                        }
                    }
                }
                // Include field if it has protection OR has a value
                if ( $has_protection || $field_value !== '' ) {
                    $posted_data[$field_label] = $field_value;
                    $field_map[$field_label] = $field_id;
                }
            }
        } else {
            // Fallback: process fields in standard format
            foreach ( $fields as $field_id => $field ) {
                // Get the actual field label from form structure
                $field_label = ( isset( $form_fields[$field_id] ) ? $form_fields[$field_id] : (( isset( $field['name'] ) ? $field['name'] : 'Field ' . $field_id )) );
                // Include ALL fields that have protection settings, even if empty
                $field_value = ( isset( $field['value'] ) ? $field['value'] : '' );
                // Check if this field has protection settings
                $has_protection = false;
                if ( isset( $protection['field_protections'] ) ) {
                    foreach ( $protection['field_protections'] as $protected_field => $protection_config ) {
                        if ( strtolower( trim( $protected_field ) ) === strtolower( trim( $field_label ) ) ) {
                            $has_protection = true;
                            break;
                        }
                    }
                }
                // Include field if it has protection OR has a value
                if ( $has_protection || $field_value !== '' ) {
                    $posted_data[$field_label] = $field_value;
                    $field_map[$field_label] = $field_id;
                }
            }
        }
        // Debug logging
        // Validate the data
        $validation_result = $this->validate_form_data( $posted_data, $protection['field_protections'], $protection['block_links'] ?? true );
        // Add errors if validation fails
        if ( !$validation_result['valid'] ) {
            foreach ( $validation_result['errors'] as $field_label => $error_message ) {
                $field_id = null;
                // Find field ID by label - try exact match first
                if ( isset( $field_map[$field_label] ) ) {
                    $field_id = $field_map[$field_label];
                } else {
                    // Try case-insensitive matching
                    foreach ( $field_map as $label => $id ) {
                        if ( strtolower( trim( $label ) ) === strtolower( trim( $field_label ) ) ) {
                            $field_id = $id;
                            break;
                        }
                    }
                }
                // If still not found, try form structure matching
                if ( !$field_id ) {
                    foreach ( $form_fields as $id => $label ) {
                        if ( strtolower( trim( $label ) ) === strtolower( trim( $field_label ) ) ) {
                            $field_id = $id;
                            break;
                        }
                    }
                }
                if ( $field_id ) {
                    wpforms()->process->errors[$form_data['id']][$field_id] = $error_message;
                } else {
                }
            }
        } else {
        }
    }

    // Working Ninja Forms validation method
    public function validate_ninja_forms_working( $form_data ) {
        // Check if user has premium access for Ninja Forms
        if ( !$this->has_premium_access() ) {
            return;
            // Skip validation for free users
        }
        
        // Security: Verify Ninja Forms nonce if $_POST is being used
        if ( ! empty( $_POST ) && ( isset( $_POST['formData'] ) || isset( $_POST['form_id'] ) || isset( $_POST['fields'] ) ) ) {
            if ( isset( $_POST['_wpnonce'] ) ) {
                $nonce = sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) );
                if ( ! wp_verify_nonce( $nonce, 'ninja_forms_ajax_nonce' ) ) {
                    // Invalid nonce - return error
                    return;
                }
            } else {
                // Nonce is required when processing $_POST
                return;
            }
        }
        
        // Check multiple possible data structures for Ninja Forms
        $form_id = null;
        $fields_data = array();
        // Method 1: Check for standard formData structure
        // Security: Sanitize and validate input
        if ( isset( $_POST['formData'] ) && isset( $_POST['form_id'] ) ) {
            $form_id = intval( $_POST['form_id'] );
            if ( $form_id <= 0 ) {
                return;
            }
            // Sanitize formData before parsing
            $form_data_string = sanitize_text_field( wp_unslash( $_POST['formData'] ) );
            parse_str( $form_data_string, $form_fields_data );
            // Security: Only process expected Ninja Forms field keys (nf_field_*)
            // Filter keys before processing to avoid processing entire array
            $form_fields_data = is_array( $form_fields_data ) ? $form_fields_data : array();
            $filtered_keys = array_filter( array_keys( $form_fields_data ), function( $key ) {
                return strpos( $key, 'nf_field_' ) === 0;
            } );
            // Convert to fields_data format - sanitize all values
            foreach ( $filtered_keys as $key ) {
                $field_id = absint( str_replace( 'nf_field_', '', $key ) );
                if ( $field_id > 0 && isset( $form_fields_data[ $key ] ) ) {
                    $value = $form_fields_data[ $key ];
                    $fields_data[] = array(
                        'id'    => $field_id,
                        'value' => is_array( $value ) ? array_map( 'sanitize_text_field', $value ) : sanitize_text_field( $value ),
                    );
                }
            }
        }
        // Method 2: Check for fields array structure
        // Security: Validate and sanitize fields array
        if ( isset( $_POST['fields'] ) && isset( $_POST['form_id'] ) ) {
            $form_id = intval( $_POST['form_id'] );
            if ( $form_id <= 0 ) {
                return;
            }
            // Ensure fields is an array and sanitize
            // Security: Only process expected array structure, not entire $_POST
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized in loop below
            $fields_data = isset( $_POST['fields'] ) && is_array( $_POST['fields'] ) ? wp_unslash( $_POST['fields'] ) : array();
            $sanitized_fields_data = array();
            foreach ( $fields_data as $field_item ) {
                // Validate structure - only process if it's an array with expected keys
                if ( ! is_array( $field_item ) ) {
                    continue;
                }
                // Only process if 'id' key exists (required for field identification)
                if ( ! isset( $field_item['id'] ) ) {
                    continue;
                }
                // Sanitize field ID
                $field_id = absint( $field_item['id'] );
                if ( $field_id <= 0 ) {
                    continue;
                }
                // Sanitize field value if present
                $field_value = isset( $field_item['value'] ) ? $field_item['value'] : '';
                if ( is_array( $field_value ) ) {
                    $field_value = array_map( 'sanitize_text_field', $field_value );
                } else {
                    $field_value = sanitize_text_field( $field_value );
                }
                // Only add valid field items
                $sanitized_fields_data[] = array(
                    'id'    => $field_id,
                    'value' => $field_value,
                );
            }
            $fields_data = $sanitized_fields_data;
        }
        // Method 3: Check form_data parameter (most common for ninja_forms_process hook)
        if ( is_array( $form_data ) && isset( $form_data['form_id'] ) ) {
            $form_id = intval( $form_data['form_id'] );
            if ( isset( $form_data['fields'] ) ) {
                $fields_data = $form_data['fields'];
            }
        }
        // Method 4: Try to get from global $_POST if form_data doesn't have what we need
        // Security: Sanitize form_id from $_POST
        if ( !$form_id && isset( $_POST['form_id'] ) ) {
            $form_id = intval( $_POST['form_id'] );
        }
        // Validate form ID
        if ( !$form_id || $form_id <= 0 ) {
            return;
        }
        $protection = $this->get_protection_settings( 'ninja-forms', $form_id );
        if ( !$protection ) {
            return;
        }
        // Get form fields to map IDs to labels
        $field_map = array();
        if ( class_exists( 'Ninja_Forms' ) ) {
            try {
                $form_obj = Ninja_Forms()->form( $form_id );
                $fields = $form_obj->get_fields();
                foreach ( $fields as $field ) {
                    $settings = $field->get_settings();
                    if ( !empty( $settings['label'] ) ) {
                        $field_map[$field->get_id()] = $settings['label'];
                    }
                }
            } catch ( Exception $e ) {
                return;
            }
        }
        // Extract field values for validation - try multiple methods
        $posted_data = array();
        // If we have fields_data from form_data parameter
        if ( !empty( $fields_data ) ) {
            foreach ( $fields_data as $field_data ) {
                if ( is_array( $field_data ) && isset( $field_data['id'] ) && isset( $field_data['value'] ) ) {
                    $field_id = $field_data['id'];
                    if ( isset( $field_map[$field_id] ) ) {
                        $field_label = $field_map[$field_id];
                        $posted_data[$field_label] = sanitize_text_field( $field_data['value'] );
                    }
                }
            }
        }
        // If no data found, try to extract from $_POST directly
        // Security: Only process expected Ninja Forms field keys
        if ( empty( $posted_data ) && !empty( $_POST ) && is_array( $_POST ) ) {
            // Filter $_POST keys to only Ninja Forms field keys (nf_field_*)
            $nf_field_keys = array_filter( array_keys( $_POST ), function( $key ) {
                return strpos( $key, 'nf_field_' ) === 0;
            } );
            
            // Process only the filtered keys
            foreach ( $nf_field_keys as $key ) {
                // Sanitize key first
                $sanitized_key = sanitize_key( $key );
                if ( empty( $sanitized_key ) || strpos( $sanitized_key, 'nf_field_' ) !== 0 ) {
                    continue;
                }
                
                $field_id = absint( str_replace( 'nf_field_', '', $sanitized_key ) );
                if ( $field_id > 0 && isset( $field_map[$field_id] ) && isset( $_POST[ $key ] ) ) {
                    $field_label = sanitize_text_field( $field_map[$field_id] );
                    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Key is sanitized above, value sanitized below
                    $value = wp_unslash( $_POST[ $key ] );
                    // Sanitize field value
                    $sanitized_value = is_array( $value ) ? array_map( 'sanitize_text_field', $value ) : sanitize_text_field( $value );
                    $posted_data[$field_label] = is_array( $sanitized_value ) ? implode( ' ', $sanitized_value ) : $sanitized_value;
                }
            }
        }
        if ( empty( $posted_data ) ) {
            return;
        }
        // Validate the data
        $validation_result = $this->validate_form_data( $posted_data, $protection['field_protections'], $protection['block_links'] ?? true );
        // If validation fails, prevent form submission
        if ( !$validation_result['valid'] ) {
            // Add errors to Ninja Forms response
            foreach ( $validation_result['errors'] as $field_label => $error_message ) {
                $field_id = null;
                foreach ( $field_map as $id => $label ) {
                    if ( $label === $field_label ) {
                        $field_id = $id;
                        break;
                    }
                }
                if ( $field_id ) {
                    // Use Ninja Forms error system
                    Ninja_Forms()->form()->add_error( 'field_' . $field_id, $error_message );
                }
            }
            // Stop form processing
            wp_send_json_error( array(
                'errors'  => $validation_result['errors'],
                'message' => 'Form contains prohibited content.',
            ), 400 );
        }
    }

    // New Ninja Forms field validation method
    public function validate_ninja_forms_field( $errors, $field ) {
        // Security: Verify form_id exists and is valid
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Form plugin hook, nonce verified by plugin
        if ( !isset( $_POST['form_id'] ) ) {
            return $errors;
        }
        // Sanitize and validate form ID
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Form plugin hook, nonce verified by plugin
        $form_id = intval( $_POST['form_id'] );
        if ( $form_id <= 0 ) {
            return $errors;
        }
        $protection = $this->get_protection_settings( 'ninja-forms', $form_id );
        if ( !$protection ) {
            return $errors;
        }
        // Get field settings
        $field_settings = $field->get_settings();
        $field_label = $field_settings['label'] ?? '';
        $field_value = $field_settings['value'] ?? '';
        if ( !$field_label || !$field_value ) {
            return $errors;
        }
        // Check if this field has protection
        $field_protections = ( is_string( $protection['field_protections'] ) ? json_decode( $protection['field_protections'], true ) : $protection['field_protections'] );
        if ( !isset( $field_protections[$field_label] ) ) {
            return $errors;
        }
        $field_protection = $field_protections[$field_label];
        // Validate field value
        $validation_result = $this->validate_single_field( $field_value, $field_protection );
        if ( !$validation_result['valid'] ) {
            $errors[] = $validation_result['error'];
        }
        return $errors;
    }

    // Helper method to validate a single field
    private function validate_single_field( $field_value, $field_protection ) {
        if ( !$field_value || !$field_protection ) {
            return array(
                'valid' => true,
            );
        }
        $protection_type = $field_protection['type'] ?? 'blacklist';
        $protection_words = $field_protection['words'] ?? '';
        if ( !$protection_words ) {
            return array(
                'valid' => true,
            );
        }
        $words = array_map( 'trim', explode( ',', $protection_words ) );
        $field_value_lower = strtolower( $field_value );
        if ( $protection_type === 'blacklist' ) {
            foreach ( $words as $word ) {
                if ( $word && strpos( $field_value_lower, strtolower( $word ) ) !== false ) {
                    return array(
                        'valid' => false,
                        'error' => 'This field contains prohibited content: ' . $word,
                    );
                }
            }
        } elseif ( $protection_type === 'whitelist' ) {
            $field_words = preg_split( '/\\s+/', $field_value_lower );
            foreach ( $field_words as $field_word ) {
                if ( !$field_word ) {
                    continue;
                }
                $is_whitelisted = false;
                foreach ( $words as $white_word ) {
                    if ( $white_word && strpos( $field_word, strtolower( $white_word ) ) !== false ) {
                        $is_whitelisted = true;
                        break;
                    }
                }
                if ( !$is_whitelisted ) {
                    return array(
                        'valid' => false,
                        'error' => 'This field contains non-whitelisted content: ' . $field_word,
                    );
                }
            }
        }
        return array(
            'valid' => true,
        );
    }

    public function validate_forminator( $errors, $form_id, $field_data_array ) {
        // Check if user has premium access for Forminator
        if ( !$this->has_premium_access() ) {
            return $errors;
            // Skip validation for free users
        }
        $protection = $this->get_protection_settings( 'forminator', $form_id );
        if ( !$protection ) {
            return $errors;
        }
        // Get form object to access field labels
        $form_object = null;
        if ( class_exists( 'Forminator_API' ) ) {
            $form_object = Forminator_API::get_form( $form_id );
        }
        // Build posted data array with proper field labels
        $posted_data = array();
        foreach ( $field_data_array as $field ) {
            if ( !isset( $field['name'] ) || !isset( $field['value'] ) ) {
                continue;
            }
            // Don't skip empty values - they might be required for whitelist validation
            $field_value = '';
            if ( is_array( $field['value'] ) ) {
                $field_value = implode( ' ', array_filter( $field['value'] ) );
                // Remove empty array elements
            } else {
                $field_value = trim( $field['value'] );
            }
            // Get field label from form object
            $field_label = $field['name'];
            // fallback to field name
            if ( $form_object ) {
                foreach ( $form_object->fields as $form_field ) {
                    $field_array = $form_field->to_array();
                    if ( isset( $field_array['element_id'] ) && $field_array['element_id'] === $field['name'] ) {
                        $field_label = ( isset( $field_array['field_label'] ) ? $field_array['field_label'] : $field['name'] );
                        break;
                    }
                }
            }
            // Always add to posted_data, even if empty (for whitelist validation)
            $posted_data[$field_label] = $field_value;
        }
        // Validate the data
        $validation_result = $this->validate_form_data( $posted_data, $protection['field_protections'], $protection['block_links'] ?? true );
        // Add errors if validation fails
        if ( !$validation_result['valid'] ) {
            foreach ( $validation_result['errors'] as $field_label => $error_message ) {
                // Find the field element_id for error assignment
                $element_id = null;
                // First try to find by matching field label with form object
                if ( $form_object ) {
                    foreach ( $form_object->fields as $form_field ) {
                        $field_array = $form_field->to_array();
                        $form_field_label = ( isset( $field_array['field_label'] ) ? $field_array['field_label'] : '' );
                        if ( strtolower( trim( $form_field_label ) ) === strtolower( trim( $field_label ) ) ) {
                            $element_id = ( isset( $field_array['element_id'] ) ? $field_array['element_id'] : null );
                            break;
                        }
                    }
                }
                // Fallback: try to find by matching with field_data_array
                if ( !$element_id ) {
                    foreach ( $field_data_array as $field_data ) {
                        $current_label = ( isset( $field_data['field_label'] ) ? $field_data['field_label'] : (( isset( $field_data['name'] ) ? $field_data['name'] : '' )) );
                        if ( strtolower( trim( $current_label ) ) === strtolower( trim( $field_label ) ) ) {
                            $element_id = $field_data['name'];
                            break;
                        }
                    }
                }
                if ( $element_id ) {
                    // Add error as simple string (Forminator expects string format)
                    $errors[$element_id] = $error_message;
                } else {
                    // Last resort: try direct field name mapping
                    $direct_element_id = null;
                    foreach ( $field_data_array as $field_data ) {
                        if ( isset( $field_data['name'] ) ) {
                            // Check if the field name contains the field label or vice versa
                            $field_name = strtolower( trim( $field_data['name'] ) );
                            $search_label = strtolower( trim( $field_label ) );
                            if ( strpos( $field_name, str_replace( ' ', '-', $search_label ) ) !== false || strpos( $field_name, str_replace( ' ', '_', $search_label ) ) !== false ) {
                                $direct_element_id = $field_data['name'];
                                break;
                            }
                        }
                    }
                    if ( $direct_element_id ) {
                        // Add error as simple string (Forminator expects string format)
                        $errors[$direct_element_id] = $error_message;
                    } else {
                    }
                }
            }
        }
        return $errors;
    }

    // New Ninja Forms AJAX validation method
    public function validate_ninja_forms_ajax() {
        // Security: Verify this is a legitimate request
        if ( ! isset( $_POST['form_id'] ) || ! isset( $_POST['fields'] ) ) {
            return;
        }
        
        // Security: Verify Ninja Forms nonce (required)
        // Ninja Forms uses its own nonce system with 'ninja_forms_ajax_nonce' action
        if ( ! isset( $_POST['_wpnonce'] ) ) {
            wp_send_json_error( array( 'message' => 'Security token missing. Please refresh the page and try again.' ), 403 );
            return;
        }
        
        $nonce = sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) );
        if ( ! wp_verify_nonce( $nonce, 'ninja_forms_ajax_nonce' ) ) {
            wp_send_json_error( array( 'message' => 'Invalid request. Please refresh the page and try again.' ), 403 );
            return;
        }
        
        // Sanitize form ID
        $form_id = intval( $_POST['form_id'] );
        if ( $form_id <= 0 ) {
            return;
        }
        $protection = $this->get_protection_settings( 'ninja-forms', $form_id );
        if ( !$protection ) {
            return;
        }
        // Extract field data
        $posted_data = array();
        $field_map = array();
        if ( class_exists( 'Ninja_Forms' ) ) {
            try {
                $form_obj = Ninja_Forms()->form( $form_id );
                $form_fields = $form_obj->get_fields();
                foreach ( $form_fields as $field ) {
                    $settings = $field->get_settings();
                    if ( !empty( $settings['label'] ) ) {
                        $field_map[$field->get_id()] = $settings['label'];
                    }
                }
            } catch ( Exception $e ) {
                return;
            }
        }
        // Process submitted field data - sanitize all input
        // Security: Only process expected array structure from $_POST['fields'], not entire $_POST
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized in loop below
        $fields_data = isset( $_POST['fields'] ) && is_array( $_POST['fields'] ) ? wp_unslash( $_POST['fields'] ) : array();
        foreach ( $fields_data as $field_data ) {
            // Validate structure - only process arrays with expected keys
            if ( ! is_array( $field_data ) ) {
                continue;
            }
            
            // Only process if 'id' key exists (required for field identification)
            if ( ! isset( $field_data['id'] ) ) {
                continue;
            }
            
            // Sanitize field ID
            $field_id = absint( $field_data['id'] );
            if ( $field_id <= 0 ) {
                continue;
            }
            
            // Only process if field exists in field_map (validates field belongs to form)
            if ( ! isset( $field_map[$field_id] ) ) {
                continue;
            }
            
            // Sanitize field value
            $field_value = isset( $field_data['value'] ) ? $field_data['value'] : '';
            $field_label = sanitize_text_field( $field_map[$field_id] );
            $sanitized_value = is_array( $field_value ) ? array_map( 'sanitize_text_field', $field_value ) : sanitize_text_field( $field_value );
            $posted_data[$field_label] = is_array( $sanitized_value ) ? implode( ' ', $sanitized_value ) : $sanitized_value;
        }
        if ( empty( $posted_data ) ) {
            return;
        }
        // Validate the data
        $validation_result = $this->validate_form_data( $posted_data, $protection['field_protections'], $protection['block_links'] ?? true );
        // If validation fails, send error response
        if ( !$validation_result['valid'] ) {
            $errors = array();
            foreach ( $validation_result['errors'] as $field_label => $error_message ) {
                // Find field ID by label
                $field_id = null;
                foreach ( $field_map as $id => $label ) {
                    if ( $label === $field_label ) {
                        $field_id = $id;
                        break;
                    }
                }
                if ( $field_id ) {
                    $errors[] = array(
                        'field_id' => $field_id,
                        'message'  => $error_message,
                    );
                }
            }
            if ( !empty( $errors ) ) {
                wp_send_json_error( array(
                    'errors'  => $errors,
                    'message' => 'Form contains prohibited content.',
                ) );
                exit;
            }
        }
    }

    // New Ninja Forms filter validation method
    public function validate_ninja_forms_filter( $form_data ) {
        // Ninja Forms uses 'id' not 'form_id' in this context
        if ( !isset( $form_data['id'] ) || !isset( $form_data['fields'] ) ) {
            return $form_data;
        }
        $form_id = intval( $form_data['id'] );
        $protection = $this->get_protection_settings( 'ninja-forms', $form_id );
        if ( !$protection ) {
            return $form_data;
        }
        // Extract field data for validation
        $posted_data = array();
        $field_map = array();
        if ( class_exists( 'Ninja_Forms' ) ) {
            try {
                $form_obj = Ninja_Forms()->form( $form_id );
                $form_fields = $form_obj->get_fields();
                foreach ( $form_fields as $field ) {
                    $settings = $field->get_settings();
                    if ( !empty( $settings['label'] ) ) {
                        $field_map[$field->get_id()] = $settings['label'];
                    }
                }
            } catch ( Exception $e ) {
                return $form_data;
            }
        }
        // Process field data
        foreach ( $form_data['fields'] as $field_data ) {
            if ( isset( $field_data['id'] ) && isset( $field_data['value'] ) && isset( $field_map[$field_data['id']] ) ) {
                $field_label = $field_map[$field_data['id']];
                $posted_data[$field_label] = sanitize_text_field( $field_data['value'] );
            }
        }
        if ( empty( $posted_data ) ) {
            return $form_data;
        }
        // Validate the data
        $validation_result = $this->validate_form_data( $posted_data, $protection['field_protections'], $protection['block_links'] ?? true );
        // If validation fails, add errors to form data
        if ( !$validation_result['valid'] ) {
            // Set form-level errors and halt submission
            $form_data['errors'] = $validation_result['errors'];
            $form_data['halt'] = true;
            // Add errors to individual fields for display
            foreach ( $validation_result['errors'] as $field_label => $error_message ) {
                // Find field ID by label
                $field_id = null;
                foreach ( $field_map as $id => $label ) {
                    if ( $label === $field_label ) {
                        $field_id = $id;
                        break;
                    }
                }
                if ( $field_id ) {
                    // Add error to the specific field using Ninja Forms error format
                    foreach ( $form_data['fields'] as &$field_data ) {
                        if ( $field_data['id'] == $field_id ) {
                            // Use Ninja Forms standard error format
                            $field_data['errors'] = array($error_message);
                            break;
                        }
                    }
                }
            }
            // Force Ninja Forms to display errors by setting response data
            add_filter( 'ninja_forms_submit_data', function ( $form_data_inner ) use($form_data) {
                return $form_data;
            }, 999 );
        }
        return $form_data;
    }

    public function validate_form_data( $posted_data, $field_protections, $block_links = true ) {
        $errors = array();
        $is_valid = true;
        // Handle JSON string field_protections
        if ( is_string( $field_protections ) ) {
            $field_protections = json_decode( $field_protections, true );
            if ( json_last_error() !== JSON_ERROR_NONE ) {
                $field_protections = maybe_unserialize( $field_protections );
            }
        }
        if ( empty( $field_protections ) || !is_array( $field_protections ) ) {
            return array(
                'valid'  => true,
                'errors' => array(),
                'debug'  => 'No protection rules configured',
            );
        }
        // Add fallback validation if no specific rules are set
        if ( empty( $posted_data ) ) {
            return array(
                'valid'  => true,
                'errors' => array(),
                'debug'  => 'No form data submitted',
            );
        }
        foreach ( $field_protections as $field_name => $protection ) {
            // Skip validation if protection is missing required fields
            if ( !isset( $protection['type'] ) || !isset( $protection['words'] ) ) {
                continue;
            }
            $protection_type = $protection['type'];
            // 'blacklist' or 'whitelist'
            $protection_words = array_map( 'trim', explode( ',', $protection['words'] ) );
            // Get field value with improved field name matching
            $field_value = '';
            $matched_field_name = $field_name;
            // Use the new find_matching_field method
            $matched_key = $this->find_matching_field( $field_name, $posted_data );
            if ( $matched_key && isset( $posted_data[$matched_key] ) ) {
                $field_value = ( is_array( $posted_data[$matched_key] ) ? implode( ' ', $posted_data[$matched_key] ) : sanitize_text_field( $posted_data[$matched_key] ) );
                $matched_field_name = $matched_key;
            } else {
                continue;
                // Skip this field if no match found
            }
            // For whitelist validation, we need to check even empty fields
            if ( $protection_type === 'whitelist' ) {
                $found_allowed = false;
                // If field is empty, it fails whitelist validation (unless whitelist is empty)
                if ( empty( $field_value ) ) {
                    if ( !empty( $protection_words[0] ) ) {
                        $errors[$field_name] = 'This field is required and must contain specific content.';
                        $is_valid = false;
                    }
                    continue;
                }
                // Check if field contains required words
                foreach ( $protection_words as $word ) {
                    if ( !empty( $word ) && stripos( $field_value, $word ) !== false ) {
                        $found_allowed = true;
                        break;
                    }
                }
                if ( !$found_allowed && !empty( $protection_words[0] ) ) {
                    $errors[$field_name] = 'This content does not contain required words.';
                    $is_valid = false;
                }
                continue;
                // Skip other validations for whitelist
            }
            // For blacklist validation, skip empty fields
            if ( empty( $field_value ) ) {
                continue;
            }
            // Check for links (except in email fields) if block_links is enabled
            if ( $block_links && strpos( strtolower( $field_name ), 'email' ) === false && preg_match( '/https?:\\/\\/|www\\./i', $field_value ) ) {
                $errors[$field_name] = 'Links are not allowed in this field.';
                $is_valid = false;
                continue;
            }
            // Validate email fields format (but don't skip spam word checking)
            if ( strpos( strtolower( $field_name ), 'email' ) !== false && !empty( $field_value ) && !is_email( $field_value ) ) {
                $errors[$field_name] = 'Please enter a valid email address.';
                $is_valid = false;
                // Don't continue here - still check for spam words even if email format is invalid
            }
            // Check blacklist
            if ( $protection_type === 'blacklist' ) {
                foreach ( $protection_words as $word ) {
                    if ( !empty( $word ) && stripos( $field_value, $word ) !== false ) {
                        $errors[$field_name] = 'This content contains prohibited words.';
                        $is_valid = false;
                        break;
                    }
                }
            }
        }
        return array(
            'valid'  => $is_valid,
            'errors' => $errors,
        );
    }

    // AJAX handlers
    public function ajax_get_forms() {
        // Verify nonce and permissions
        check_ajax_referer( 'npfo_nonce', 'nonce' );
        
        // Check user permissions
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => 'Insufficient permissions' ), 403 );
            return;
        }
        
        if ( ! isset( $_POST['plugin'] ) ) {
            wp_send_json_error( array( 'message' => 'Plugin parameter is required.' ), 400 );
            return;
        }
        
        $plugin = sanitize_text_field( wp_unslash( $_POST['plugin'] ) );
        
            // Check if user has access to this plugin
            if ( ! $this->is_form_plugin_free( $plugin ) && ! $this->has_premium_access() ) {
                wp_send_json_error( array( 'message' => 'This form plugin requires premium access. Please upgrade to use this feature.' ), 403 );
                return;
            }
        
        $forms = array();
        
        try {
            switch ( $plugin ) {
                case 'contact-form-7':
                    if ( class_exists( 'WPCF7_ContactForm' ) ) {
                        $cf7_forms = WPCF7_ContactForm::find();
                        if ( is_array( $cf7_forms ) ) {
                            foreach ( $cf7_forms as $form ) {
                                if ( is_object( $form ) && method_exists( $form, 'id' ) && method_exists( $form, 'title' ) ) {
                                    $forms[] = array(
                                        'id'    => absint( $form->id() ),
                                        'title' => sanitize_text_field( $form->title() ),
                                    );
                                }
                            }
                        }
                    }
                    break;
                case 'gravity-forms':
                    if ( class_exists( 'GFAPI' ) && method_exists( 'GFAPI', 'get_forms' ) ) {
                        $gf_forms = GFAPI::get_forms();
                        if ( is_array( $gf_forms ) ) {
                            foreach ( $gf_forms as $form ) {
                                if ( isset( $form['id'] ) && isset( $form['title'] ) ) {
                                    $forms[] = array(
                                        'id'    => absint( $form['id'] ),
                                        'title' => sanitize_text_field( $form['title'] ),
                                    );
                                }
                            }
                        }
                    }
                    break;
                case 'wpforms':
                    if ( function_exists( 'wpforms' ) ) {
                        $wpforms_instance = wpforms();
                        if ( is_object( $wpforms_instance ) && isset( $wpforms_instance->form ) && method_exists( $wpforms_instance->form, 'get' ) ) {
                            $wpf_forms = $wpforms_instance->form->get();
                            if ( is_array( $wpf_forms ) ) {
                                foreach ( $wpf_forms as $form ) {
                                    if ( is_object( $form ) && isset( $form->ID ) && isset( $form->post_title ) ) {
                                        $forms[] = array(
                                            'id'    => absint( $form->ID ),
                                            'title' => sanitize_text_field( $form->post_title ),
                                        );
                                    }
                                }
                            }
                        }
                    }
                    break;
                case 'ninja-forms':
                    if ( class_exists( 'Ninja_Forms' ) ) {
                        $nf_instance = Ninja_Forms();
                        if ( is_object( $nf_instance ) && method_exists( $nf_instance, 'form' ) ) {
                            $nf_form_handler = $nf_instance->form();
                            if ( is_object( $nf_form_handler ) && method_exists( $nf_form_handler, 'get_forms' ) ) {
                                $nf_forms = $nf_form_handler->get_forms();
                                if ( is_array( $nf_forms ) ) {
                                    foreach ( $nf_forms as $form ) {
                                        if ( is_object( $form ) && method_exists( $form, 'get_id' ) && method_exists( $form, 'get_setting' ) ) {
                                            $forms[] = array(
                                                'id'    => absint( $form->get_id() ),
                                                'title' => sanitize_text_field( $form->get_setting( 'title' ) ),
                                            );
                                        }
                                    }
                                }
                            }
                        }
                    }
                    break;
                case 'forminator':
                    if ( class_exists( 'Forminator_API' ) && method_exists( 'Forminator_API', 'get_forms' ) ) {
                        $forminator_forms = Forminator_API::get_forms( null, 1, 999 );
                        if ( is_array( $forminator_forms ) ) {
                            foreach ( $forminator_forms as $form ) {
                                if ( is_object( $form ) && isset( $form->id ) && isset( $form->name ) ) {
                                    $forms[] = array(
                                        'id'    => absint( $form->id ),
                                        'title' => sanitize_text_field( $form->name ),
                                    );
                                }
                            }
                        }
                    }
                    break;
                case 'fluent-forms':
                    if ( function_exists( 'wpFluentForm' ) ) {
                        global $wpdb;
                        $table_name = $wpdb->prefix . 'fluentform_forms';
                        $table_name_safe = esc_sql( $table_name );
                        // Check if table exists before querying
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Necessary for third-party plugin integration
                        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) ) === $table_name ) {
                            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Necessary for third-party plugin integration, table name is safe
                            $ff_forms = $wpdb->get_results( $wpdb->prepare( "SELECT id, title FROM {$table_name_safe} WHERE status=%s", 'published' ) );
                            if ( is_array( $ff_forms ) ) {
                                foreach ( $ff_forms as $form ) {
                                    if ( is_object( $form ) && isset( $form->id ) && isset( $form->title ) ) {
                                        $forms[] = array(
                                            'id'    => absint( $form->id ),
                                            'title' => sanitize_text_field( $form->title ),
                                        );
                                    }
                                }
                            }
                        }
                    }
                    break;
            }
        } catch ( Exception $e ) {
            // Log error but don't break the site
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Legitimate error logging
            error_log( 'NP Form Shield Error in ajax_get_forms: ' . $e->getMessage() );
            wp_send_json_error( array( 'message' => 'An error occurred while fetching forms. Please try again.' ), 500 );
            return;
        }
        
        wp_send_json_success( $forms );
    }

    public function ajax_get_form_fields() {
        // Verify nonce and permissions
        check_ajax_referer( 'npfo_nonce', 'nonce' );
        
        // Check user permissions
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => 'Insufficient permissions' ), 403 );
            return;
        }
        
        if ( ! isset( $_POST['plugin'] ) || ! isset( $_POST['form_id'] ) ) {
            wp_send_json_error( array( 'message' => 'Plugin and form ID parameters are required.' ), 400 );
            return;
        }
        
        $plugin = sanitize_text_field( wp_unslash( $_POST['plugin'] ) );
        $form_id = sanitize_text_field( wp_unslash( $_POST['form_id'] ) );
        
            // Check if user has access to this plugin
            if ( ! $this->is_form_plugin_free( $plugin ) && ! $this->has_premium_access() ) {
                wp_send_json_error( array( 'message' => 'This form plugin requires premium access. Please upgrade to use this feature.' ), 403 );
                return;
            }
        
        $fields = array();
        
        try {
        switch ( $plugin ) {
            case 'contact-form-7':
                if ( class_exists( 'WPCF7_ContactForm' ) ) {
                    $form = WPCF7_ContactForm::get_instance( $form_id );
                    if ( $form ) {
                        $form_fields = $form->scan_form_tags();
                        foreach ( $form_fields as $field ) {
                            if ( $field->name ) {
                                $fields[] = array(
                                    'name' => sanitize_text_field( $field->name ),
                                    'type' => sanitize_text_field( $field->type ),
                                );
                            }
                        }
                    }
                }
                break;
            case 'gravity-forms':
                if ( class_exists( 'GFAPI' ) ) {
                    $form = GFAPI::get_form( $form_id );
                    if ( $form ) {
                        foreach ( $form['fields'] as $field ) {
                            if ( !empty( $field->label ) ) {
                                $fields[] = array(
                                    'name' => sanitize_text_field( $field->label ),
                                    'type' => sanitize_text_field( $field->type ),
                                );
                            }
                        }
                    }
                }
                break;
            case 'wpforms':
                if ( function_exists( 'wpforms' ) ) {
                    $form = wpforms()->form->get( $form_id );
                    if ( $form && !empty( $form->post_content ) ) {
                        $form_data = json_decode( $form->post_content, true );
                        if ( !empty( $form_data['fields'] ) ) {
                            foreach ( $form_data['fields'] as $field ) {
                                $fields[] = array(
                                    'name' => sanitize_text_field( $field['label'] ?? '' ),
                                    'type' => sanitize_text_field( $field['type'] ?? 'text' ),
                                );
                            }
                        }
                    }
                }
                break;
            case 'ninja-forms':
                if ( class_exists( 'Ninja_Forms' ) ) {
                    try {
                        // Get all fields for the form
                        $form_fields = Ninja_Forms()->form( $form_id )->get_fields();
                        foreach ( $form_fields as $field ) {
                            $settings = $field->get_settings();
                            if ( !empty( $settings['label'] ) && !empty( $settings['type'] ) ) {
                                // Filter out non-input field types
                                $excluded_types = array(
                                    'submit',
                                    'button',
                                    'html',
                                    'hr',
                                    'divider',
                                    'spam',
                                    'recaptcha',
                                    'honeypot'
                                );
                                if ( !in_array( $settings['type'], $excluded_types ) ) {
                                    $fields[] = array(
                                        'name' => sanitize_text_field( $settings['label'] ),
                                        'type' => sanitize_text_field( $settings['type'] ),
                                    );
                                }
                            }
                        }
                    } catch ( Exception $e ) {
                        // Fallback method using direct database query
                        global $wpdb;
                        $table_name = $wpdb->prefix . 'nf3_fields';
                        $table_name_safe = esc_sql( $table_name );
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Necessary for third-party plugin integration, table name is safe
                        $results = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table_name_safe} WHERE parent_id = %d", $form_id ) );
                        if ( $results ) {
                            foreach ( $results as $field ) {
                                $settings = maybe_unserialize( $field->settings );
                                if ( !empty( $settings['label'] ) && !empty( $settings['type'] ) ) {
                                    // Filter out non-input field types
                                    $excluded_types = array(
                                        'submit',
                                        'button',
                                        'html',
                                        'hr',
                                        'divider',
                                        'spam',
                                        'recaptcha',
                                        'honeypot'
                                    );
                                    if ( !in_array( $settings['type'], $excluded_types ) ) {
                                        $fields[] = array(
                                            'name' => $settings['label'],
                                            'type' => $settings['type'] ?? 'text',
                                        );
                                    }
                                }
                            }
                        }
                    }
                }
                break;
            case 'forminator':
                if ( class_exists( 'Forminator_API' ) ) {
                    try {
                        $form = Forminator_API::get_form( $form_id );
                        if ( $form ) {
                            $form_fields = $form->fields;
                            foreach ( $form_fields as $field ) {
                                $field_array = $field->to_array();
                                if ( !empty( $field_array['field_label'] ) ) {
                                    $fields[] = array(
                                        'name' => sanitize_text_field( $field_array['field_label'] ),
                                        'type' => sanitize_text_field( $field_array['type'] ?? 'text' ),
                                    );
                                }
                            }
                        }
                    } catch ( Exception $e ) {
                        // Fallback method using direct POST data
                        // Security: Sanitize $_POST data before processing
                        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized after json_decode
                        if ( !empty( $_POST['fields'] ) && is_string( $_POST['fields'] ) ) {
                            // Get JSON string and decode (sanitize after decode, not before)
                            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized after json_decode
                            $raw_fields = wp_unslash( $_POST['fields'] );
                            $form_fields = json_decode( $raw_fields, true );
                            if ( is_array( $form_fields ) ) {
                                foreach ( $form_fields as $field ) {
                                    if ( !empty( $field['field_label'] ) ) {
                                        $fields[] = array(
                                            'name' => sanitize_text_field( $field['field_label'] ),
                                            'type' => sanitize_text_field( $field['type'] ?? 'text' ),
                                        );
                                    }
                                }
                            }
                        }
                    }
                }
                break;
            case 'fluent-forms':
                if ( function_exists( 'wpFluentForm' ) ) {
                    global $wpdb;
                    // First try to get form data from fluentform_forms table
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Necessary for third-party plugin integration
                    $form = $wpdb->get_row( $wpdb->prepare( "SELECT form_fields FROM {$wpdb->prefix}fluentform_forms WHERE id = %d", $form_id ) );
                    if ( $form && !empty( $form->form_fields ) ) {
                        $form_fields = json_decode( $form->form_fields, true );
                        if ( !empty( $form_fields['fields'] ) ) {
                            foreach ( $form_fields['fields'] as $field ) {
                                // Check different possible field structures
                                $field_name = '';
                                $field_label = '';
                                $field_type = $field['element'] ?? 'text';
                                // Get field name from various possible locations
                                if ( !empty( $field['attributes']['name'] ) ) {
                                    $field_name = $field['attributes']['name'];
                                } elseif ( !empty( $field['settings']['container_class'] ) ) {
                                    $field_name = $field['settings']['container_class'];
                                } elseif ( !empty( $field['settings']['name'] ) ) {
                                    $field_name = $field['settings']['name'];
                                }
                                // Get field label
                                if ( !empty( $field['settings']['label'] ) ) {
                                    $field_label = $field['settings']['label'];
                                } elseif ( !empty( $field['settings']['placeholder'] ) ) {
                                    $field_label = $field['settings']['placeholder'];
                                } else {
                                    $field_label = $field_name;
                                }
                                // Only add fields that have a name and are input fields
                                if ( !empty( $field_name ) && !in_array( $field_type, ['section_break', 'html', 'shortcode'] ) ) {
                                    $fields[] = array(
                                        'name'       => sanitize_text_field( $field_label ),
                                        'type'       => sanitize_text_field( $field_type ),
                                        'field_name' => sanitize_text_field( $field_name ),
                                    );
                                }
                            }
                        }
                    }
                    // If no fields found, try alternative method using form meta
                    if ( empty( $fields ) ) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Necessary for third-party plugin integration
                        $form_meta = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}fluentform_form_meta WHERE form_id = %d AND meta_key = 'formSettings'", $form_id ) );
                        if ( $form_meta && !empty( $form_meta->value ) ) {
                            $form_settings = json_decode( $form_meta->value, true );
                            // Add basic fields if form exists but no specific fields found
                            $fields[] = array(
                                'name'       => 'Email Field',
                                'type'       => 'input_email',
                                'field_name' => 'email',
                            );
                            $fields[] = array(
                                'name'       => 'Name Field',
                                'type'       => 'input_text',
                                'field_name' => 'names',
                            );
                        }
                    }
                }
                    break;
            }
        } catch ( Exception $e ) {
            // Log error but don't break the site
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Legitimate error logging
            error_log( 'NP Form Shield Error in ajax_get_form_fields: ' . $e->getMessage() );
            wp_send_json_error( array( 'message' => 'An error occurred while fetching form fields. Please try again.' ), 500 );
            return;
        }
        
        wp_send_json_success( $fields );
    }

    public function ajax_save_protection() {
        // Verify nonce and permissions
        check_ajax_referer( 'npfo_nonce', 'nonce' );
        
        // Check user permissions
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => 'Insufficient permissions' ), 403 );
            return;
        }
        
        global $wpdb;
        // For update operations, allow all users to update existing protections
        // Only check premium access for new protection creation
        if ( ! isset( $_POST['id'] ) ) {
            if ( ! isset( $_POST['form_plugin'] ) ) {
                wp_send_json_error( array( 'message' => 'Form plugin is required.' ), 400 );
                return;
            }
            $form_plugin = sanitize_text_field( wp_unslash( $_POST['form_plugin'] ) );
            // Check if user has access to this plugin for new protections
            if ( ! $this->is_form_plugin_free( $form_plugin ) && ! $this->has_premium_access() ) {
                wp_send_json_error( array( 'message' => 'This form plugin requires premium access. Please upgrade to use this feature.' ), 403 );
                return;
            }
        }
        // Check if we're updating an existing protection
        if ( isset( $_POST['id'] ) && ! empty( $_POST['id'] ) ) {
            $id = intval( $_POST['id'] );
            // Validate ID
            if ( $id <= 0 ) {
                wp_send_json_error( array( 'message' => 'Invalid protection ID.' ), 400 );
                return;
            }
            
            // Validate required fields
            if ( ! isset( $_POST['field_protections'] ) ) {
                wp_send_json_error( array( 'message' => 'Field protections data is required.' ), 400 );
                return;
            }
            
            // Decode and sanitize field protections
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized after json_decode
            $raw_field_protections = wp_unslash( $_POST['field_protections'] );
            $decoded_field_protections = json_decode( $raw_field_protections, true );
            
            if ( ! is_array( $decoded_field_protections ) || json_last_error() !== JSON_ERROR_NONE ) {
                wp_send_json_error( array( 'message' => 'Invalid field protection data. Please check your input.' ), 400 );
                return;
            }
            
            // Sanitize the decoded array
            $field_protections = $this->sanitize_field_protections( $decoded_field_protections );
            
            if ( empty( $field_protections ) ) {
                wp_send_json_error( array( 'message' => 'No valid field protection data found.' ), 400 );
                return;
            }
            
            // Sanitize block_links boolean value
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized with filter_var below
            $block_links_raw = isset( $_POST['block_links'] ) ? wp_unslash( $_POST['block_links'] ) : '1';
            $block_links = filter_var( $block_links_raw, FILTER_VALIDATE_BOOLEAN );
            if ( false === $block_links ) {
                $block_links = (bool) $block_links_raw;
            }
            // Verify protection exists before updating
            $existing_protection = $this->get_protection_by_id( $id );
            if ( ! $existing_protection ) {
                wp_send_json_error( array( 'message' => 'Protection not found.' ), 404 );
                return;
            }
            
            $table_name = $wpdb->prefix . 'npfo_protections';
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin's own database operations
            $result = $wpdb->update(
                $table_name,
                array(
                    'field_protections' => wp_json_encode( $field_protections ),
                    'block_links'       => ( $block_links ? 1 : 0 ),
                    'updated_at'        => current_time( 'mysql' ),
                ),
                array(
                    'id' => $id,
                ),
                array('%s', '%d', '%s'),
                array('%d')
            );
            if ( $result !== false && $result > 0 ) {
                wp_send_json_success( array( 'message' => 'Protection updated successfully!' ) );
            } else {
                wp_send_json_error( array( 'message' => 'Failed to update protection. No changes made.' ), 400 );
            }
            return;
        } else {
            // Creating a new protection
            // Validate required fields
            if ( ! isset( $_POST['form_plugin'] ) || ! isset( $_POST['form_id'] ) || ! isset( $_POST['form_title'] ) ) {
                wp_send_json_error( array( 'message' => 'Form plugin, form ID, and form title are required.' ), 400 );
                return;
            }
            
            if ( ! isset( $_POST['field_protections'] ) ) {
                wp_send_json_error( array( 'message' => 'Field protections data is required.' ), 400 );
                return;
            }
            
            $form_plugin = sanitize_text_field( wp_unslash( $_POST['form_plugin'] ) );
            $form_id = sanitize_text_field( wp_unslash( $_POST['form_id'] ) );
            $form_title = sanitize_text_field( wp_unslash( $_POST['form_title'] ) );
            
            // Decode and sanitize field protections
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized after json_decode
            $raw_field_protections = wp_unslash( $_POST['field_protections'] );
            $decoded_field_protections = json_decode( $raw_field_protections, true );
            
            if ( ! is_array( $decoded_field_protections ) || json_last_error() !== JSON_ERROR_NONE ) {
                wp_send_json_error( array( 'message' => 'Invalid field protection data. Please check your input.' ), 400 );
                return;
            }
            
            // Sanitize the decoded array
            $field_protections = $this->sanitize_field_protections( $decoded_field_protections );
            
            if ( empty( $field_protections ) ) {
                wp_send_json_error( array( 'message' => 'No valid field protection data found.' ), 400 );
                return;
            }
            
            // Sanitize block_links boolean value
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized with filter_var below
            $block_links_raw = isset( $_POST['block_links'] ) ? wp_unslash( $_POST['block_links'] ) : '1';
            $block_links = filter_var( $block_links_raw, FILTER_VALIDATE_BOOLEAN );
            if ( false === $block_links ) {
                $block_links = (bool) $block_links_raw;
            }
            $table_name = $wpdb->prefix . 'npfo_protections';
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin's own database operations
            $result = $wpdb->replace( $table_name, array(
                'form_plugin'       => $form_plugin,
                'form_id'           => $form_id,
                'form_title'        => $form_title,
                'field_protections' => wp_json_encode( $field_protections ),
                'block_links'       => ( $block_links ? 1 : 0 ),
            ), array(
                '%s',
                '%s',
                '%s',
                '%s',
                '%d'
            ) );
        }
        if ( $result !== false ) {
            wp_send_json_success( array( 'message' => 'Protection saved successfully!' ) );
        } else {
            wp_send_json_error( array( 'message' => 'Failed to save protection. Database error: ' . esc_html( $wpdb->last_error ) ), 500 );
        }
    }

    public function ajax_delete_protection() {
        // Verify nonce and permissions
        check_ajax_referer( 'npfo_nonce', 'nonce' );
        
        // Check user permissions
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => 'Insufficient permissions' ), 403 );
            return;
        }
        
        // Validate required parameter
        if ( ! isset( $_POST['id'] ) ) {
            wp_send_json_error( array( 'message' => 'Protection ID is required.' ), 400 );
            return;
        }
        
        global $wpdb;
        $id = intval( $_POST['id'] );
        
        // Validate ID
        if ( $id <= 0 ) {
            wp_send_json_error( array( 'message' => 'Invalid protection ID.' ), 400 );
            return;
        }
        
        // Verify protection exists before deleting
        $existing_protection = $this->get_protection_by_id( $id );
        if ( ! $existing_protection ) {
            wp_send_json_error( array( 'message' => 'Protection not found.' ), 404 );
            return;
        }
        
        $table_name = $wpdb->prefix . 'npfo_protections';
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin's own database operations
        $result = $wpdb->delete( $table_name, array(
            'id' => $id,
        ), array('%d') );
        
        if ( $result !== false ) {
            wp_send_json_success( array( 'message' => 'Protection deleted successfully!' ) );
        } else {
            wp_send_json_error( array( 'message' => 'Failed to delete protection. Database error: ' . esc_html( $wpdb->last_error ) ), 500 );
        }
    }

    public function ajax_get_protection() {
        // Verify nonce and permissions
        check_ajax_referer( 'npfo_nonce', 'nonce' );
        
        // Check user permissions
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => 'Insufficient permissions' ), 403 );
            return;
        }
        
        if ( ! isset( $_POST['id'] ) ) {
            wp_send_json_error( array( 'message' => 'Protection ID is required.' ), 400 );
            return;
        }
        
        $id = intval( $_POST['id'] );
        
        // Validate ID
        if ( $id <= 0 ) {
            wp_send_json_error( array( 'message' => 'Invalid protection ID.' ), 400 );
            return;
        }
        
        $protection = $this->get_protection_by_id( $id );
        if ( $protection ) {
            wp_send_json_success( $protection );
        } else {
            wp_send_json_error( array( 'message' => 'Protection not found.' ), 404 );
        }
    }

    /**
     * AJAX handler to get form protection settings
     * Handles both admin and frontend requests
     */
    public function ajax_get_form_protection() {
        // Check if this is an admin request (has admin nonce) or frontend request
        $nonce_valid = false;
        
        // Try admin nonce first (for admin dashboard)
        if ( isset( $_POST['nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'npfo_nonce' ) ) {
            // Admin request - also check permissions
            if ( ! current_user_can( 'manage_options' ) ) {
                wp_send_json_error( array( 'message' => 'Insufficient permissions' ), 403 );
                return;
            }
            $nonce_valid = true;
        }
        // Try frontend nonce (for public forms)
        elseif ( isset( $_POST['nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'npfo_frontend_nonce' ) ) {
            $nonce_valid = true;
        }
        
        // If neither nonce is valid, return error
        if ( ! $nonce_valid ) {
            wp_send_json_error( array( 'message' => 'Invalid request. Please refresh the page and try again.' ), 403 );
            return;
        }
        
        // Validate required parameters
        if ( ! isset( $_POST['form_plugin'] ) || ! isset( $_POST['form_id'] ) ) {
            wp_send_json_error( array( 'message' => 'Form plugin and form ID are required.' ), 400 );
            return;
        }
        
        $form_plugin = sanitize_text_field( wp_unslash( $_POST['form_plugin'] ) );
        $form_id = sanitize_text_field( wp_unslash( $_POST['form_id'] ) );
        
        // Validate form_id is numeric
        if ( ! is_numeric( $form_id ) || $form_id <= 0 ) {
            wp_send_json_error( array( 'message' => 'Invalid form ID.' ), 400 );
            return;
        }
        
        $protection = $this->get_protection_settings( $form_plugin, $form_id );
        if ( $protection ) {
            wp_send_json_success( $protection );
        } else {
            wp_send_json_error( array( 'message' => 'No protection settings found for this form.' ) );
        }
    }

    public function ajax_check_plugin_status() {
        // Verify nonce and permissions
        check_ajax_referer( 'npfo_nonce', 'nonce' );
        
        // Check user permissions
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array( 'message' => 'Insufficient permissions' ), 403 );
            return;
        }
        
        $plugin_status = array();
        // Check Contact Form 7
        $plugin_status['contact-form-7'] = array(
            'installed'   => class_exists( 'WPCF7_ContactForm' ),
            'name'        => 'Contact Form 7',
            'plugin_file' => 'contact-form-7/wp-contact-form-7.php',
        );
        // Check Gravity Forms
        $plugin_status['gravity-forms'] = array(
            'installed'   => class_exists( 'GFAPI' ),
            'name'        => 'Gravity Forms',
            'plugin_file' => 'gravityforms/gravityforms.php',
        );
        // Check WPForms
        $plugin_status['wpforms'] = array(
            'installed'   => function_exists( 'wpforms' ),
            'name'        => 'WPForms',
            'plugin_file' => 'wpforms/wpforms.php',
        );
        // Check Ninja Forms
        $plugin_status['ninja-forms'] = array(
            'installed'   => class_exists( 'Ninja_Forms' ),
            'name'        => 'Ninja Forms',
            'plugin_file' => 'ninja-forms/ninja-forms.php',
        );
        // Check Forminator
        $plugin_status['forminator'] = array(
            'installed'   => class_exists( 'Forminator_API' ),
            'name'        => 'Forminator',
            'plugin_file' => 'forminator/forminator.php',
        );
        // Check Fluent Forms
        $plugin_status['fluent-forms'] = array(
            'installed'   => function_exists( 'wpFluentForm' ),
            'name'        => 'Fluent Forms',
            'plugin_file' => 'fluentform/fluentform.php',
        );
        wp_send_json_success( $plugin_status );
    }

    // Fluent Forms validation
    public function validate_fluent_forms( $errors, $form_data, $form ) {
        // Check if user has premium access for Fluent Forms
        if ( !$this->has_premium_access() ) {
            return $errors;
            // Skip validation for free users
        }
        $form_id = $form->id ?? null;
        if ( !$form_id ) {
            return $errors;
        }
        $protection = $this->get_protection_settings( 'fluent-forms', $form_id );
        if ( $protection ) {
            // Convert form data to array format with better field mapping
            $posted_data = array();
            if ( is_array( $form_data ) ) {
                foreach ( $form_data as $field_name => $field_value ) {
                    // Handle different field value formats
                    if ( is_array( $field_value ) ) {
                        $posted_data[$field_name] = implode( ' ', $field_value );
                    } else {
                        $posted_data[$field_name] = $field_value;
                    }
                    // Also try common field name variations for better matching
                    $clean_field_name = str_replace( ['_', '-'], '', strtolower( $field_name ) );
                    if ( !isset( $posted_data[$clean_field_name] ) ) {
                        $posted_data[$clean_field_name] = $posted_data[$field_name];
                    }
                }
            }
            // Only process fields that are in form_data - don't loop through entire $_POST
            // form_data already contains the validated Fluent Forms field data
            // We only need to check $_POST for specific known Fluent Forms field keys if form_data is incomplete
            // But since form_data is provided by Fluent Forms, we should trust it and not process $_POST directly
            // If additional fields are needed, they should come through form_data parameter
            $validation_result = $this->validate_form_data( $posted_data, $protection['field_protections'], $protection['block_links'] ?? true );
            if ( !$validation_result['valid'] ) {
                foreach ( $validation_result['errors'] as $field => $error ) {
                    // Try to map the field name back to Fluent Forms field format
                    $fluent_field_name = $field;
                    foreach ( $form_data as $original_field => $value ) {
                        if ( str_replace( ['_', '-'], '', strtolower( $original_field ) ) === str_replace( ['_', '-'], '', strtolower( $field ) ) ) {
                            $fluent_field_name = $original_field;
                            break;
                        }
                    }
                    $errors[$fluent_field_name] = $error;
                }
            }
        }
        return $errors;
    }

    // Helper methods
    /**
     * Sanitize field protections array from JSON input
     * 
     * @param array $field_protections Raw field protections array
     * @return array Sanitized field protections array
     */
    private function sanitize_field_protections( $field_protections ) {
        if ( ! is_array( $field_protections ) ) {
            return array();
        }
        
        $sanitized = array();
        
        foreach ( $field_protections as $field_name => $protection ) {
            // Sanitize field name
            $sanitized_field_name = sanitize_text_field( $field_name );
            
            if ( ! is_array( $protection ) ) {
                continue;
            }
            
            // Sanitize protection type
            $protection_type = isset( $protection['type'] ) ? sanitize_text_field( $protection['type'] ) : 'blacklist';
            if ( ! in_array( $protection_type, array( 'blacklist', 'whitelist' ), true ) ) {
                $protection_type = 'blacklist';
            }
            
            // Sanitize protection words
            $protection_words = isset( $protection['words'] ) ? sanitize_textarea_field( $protection['words'] ) : '';
            
            $sanitized[ $sanitized_field_name ] = array(
                'type'  => $protection_type,
                'words' => $protection_words,
            );
        }
        
        return $sanitized;
    }
    
    /**
     * Get form ID with multiple fallback methods
     */
    private function get_form_id( $plugin, $form_data = null ) {
        $form_id = null;
        switch ( $plugin ) {
            case 'contact-form-7':
                // Security: Sanitize $_POST data
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                $form_id = isset( $_POST['_wpcf7'] ) ? intval( $_POST['_wpcf7'] ) : ( $form_data['id'] ?? null );
                break;
            case 'gravity-forms':
                if ( isset( $form_data['id'] ) ) {
                    $form_id = $form_data['id'];
                } else {
                    // Try to extract from form HTML
                    preg_match( '/gform_wrapper.*?id="gform_wrapper_(\\d+)"/', ob_get_contents(), $matches );
                    $form_id = isset( $matches[1] ) ? intval( $matches[1] ) : null;
                }
                break;
            case 'wpforms':
                // Security: Sanitize $_POST data
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                if ( isset( $_POST['wpforms']['id'] ) ) {
                    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                    $form_id = intval( $_POST['wpforms']['id'] );
                } else {
                    $form_id = $form_data['id'] ?? null;
                }
                if ( !$form_id ) {
                    // Try to extract from form data attribute
                    preg_match( '/data-formid="(\\d+)"/', ob_get_contents(), $matches );
                    $form_id = isset( $matches[1] ) ? intval( $matches[1] ) : null;
                }
                break;
            case 'ninja-forms':
                // Security: Sanitize $_POST data
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                if ( isset( $_POST['form_id'] ) ) {
                    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                    $form_id = intval( $_POST['form_id'] );
                } else {
                    $form_id = $form_data['id'] ?? $form_data['form_id'] ?? null;
                }
                if ( !$form_id ) {
                    // Try to extract from JavaScript variables
                    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                    if ( isset( $_POST['nf_form_submit'] ) && isset( $_POST['form_id'] ) ) {
                        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                        $form_id = intval( $_POST['form_id'] );
                    }
                }
                break;
            case 'forminator':
                // Security: Sanitize $_POST data
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                if ( isset( $_POST['form_id'] ) ) {
                    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                    $form_id = intval( $_POST['form_id'] );
                } else {
                    $form_id = $form_data['id'] ?? null;
                }
                if ( !$form_id ) {
                    // Try to extract from form data attribute
                    preg_match( '/data-form-id="(\\d+)"/', ob_get_contents(), $matches );
                    $form_id = isset( $matches[1] ) ? intval( $matches[1] ) : null;
                }
                break;
            case 'fluent-forms':
                // Security: Sanitize $_POST data
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                if ( isset( $_POST['form_id'] ) ) {
                    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                    $form_id = intval( $_POST['form_id'] );
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                } elseif ( isset( $_POST['form-id'] ) ) {
                    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Helper function, nonce verified in calling function
                    $form_id = intval( $_POST['form-id'] );
                } else {
                    $form_id = $form_data['id'] ?? null;
                }
                if ( !$form_id ) {
                    // Try to extract from form data attribute
                    preg_match( '/data-form_id="(\\d+)"/', ob_get_contents(), $matches );
                    $form_id = isset( $matches[1] ) ? intval( $matches[1] ) : null;
                }
                break;
        }
        // Convert to integer if possible, otherwise return as string
        if ( is_numeric( $form_id ) ) {
            return (int) $form_id;
        }
        return $form_id;
    }

    /**
     * Normalize field name for consistent matching
     */
    private function normalize_field_name( $field_name ) {
        if ( !$field_name ) {
            return '';
        }
        // Convert to lowercase and remove special characters
        $normalized = strtolower( trim( $field_name ) );
        $normalized = preg_replace( '/[^a-z0-9]/', '', $normalized );
        return $normalized;
    }

    /**
     * Find matching field in posted data using multiple methods
     */
    private function find_matching_field( $field_name, $posted_data ) {
        if ( !$field_name || empty( $posted_data ) ) {
            return null;
        }
        // Try exact match first
        if ( isset( $posted_data[$field_name] ) ) {
            return $field_name;
        }
        // Try case-insensitive match
        foreach ( $posted_data as $key => $value ) {
            if ( strtolower( $key ) === strtolower( $field_name ) ) {
                return $key;
            }
        }
        // Try normalized matching
        $normalized_field = $this->normalize_field_name( $field_name );
        foreach ( $posted_data as $key => $value ) {
            if ( $this->normalize_field_name( $key ) === $normalized_field ) {
                return $key;
            }
        }
        // Try partial matching
        foreach ( $posted_data as $key => $value ) {
            if ( strpos( strtolower( $key ), strtolower( $field_name ) ) !== false || strpos( strtolower( $field_name ), strtolower( $key ) ) !== false ) {
                return $key;
            }
        }
        return null;
    }

    public function get_protection_settings( $plugin, $form_id ) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'npfo_protections';
        $table_name_safe = esc_sql( $table_name );
        // Try multiple query methods for better compatibility
        $result = null;
        // Method 1: Try with string form_id (most flexible)
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Plugin's own database operations, table name is safe
        $result = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table_name_safe} WHERE form_plugin = %s AND form_id = %s", $plugin, $form_id ), ARRAY_A );
        // Method 2: If no result and form_id is numeric, try integer
        if ( !$result && is_numeric( $form_id ) ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Plugin's own database operations, table name is safe
            $result = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table_name_safe} WHERE form_plugin = %s AND form_id = %d", $plugin, (int) $form_id ), ARRAY_A );
        }
        // Method 3: Try case-insensitive plugin matching
        if ( !$result ) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Plugin's own database operations, table name is safe
            $result = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table_name_safe} WHERE LOWER(form_plugin) = %s AND form_id = %s", strtolower( $plugin ), $form_id ), ARRAY_A );
        }
        if ( $result ) {
            // Sanitize string fields from database
            if ( isset( $result['form_plugin'] ) ) {
                $result['form_plugin'] = sanitize_text_field( $result['form_plugin'] );
            }
            if ( isset( $result['form_id'] ) ) {
                $result['form_id'] = sanitize_text_field( $result['form_id'] );
            }
            if ( isset( $result['form_title'] ) ) {
                $result['form_title'] = sanitize_text_field( $result['form_title'] );
            }
            
            // Handle both serialized and JSON data
            if ( is_string( $result['field_protections'] ) ) {
                // Try JSON decode first
                $decoded = json_decode( $result['field_protections'], true );
                if ( json_last_error() === JSON_ERROR_NONE ) {
                    $result['field_protections'] = $decoded;
                } else {
                    // Fallback to unserialize
                    $result['field_protections'] = maybe_unserialize( $result['field_protections'] );
                }
            }
            // Ensure field_protections is an array and sanitize it
            if ( !is_array( $result['field_protections'] ) ) {
                $result['field_protections'] = array();
            } else {
                // Sanitize field protections array
                $result['field_protections'] = $this->sanitize_field_protections( $result['field_protections'] );
            }
            return $result;
        }
        return null;
    }

    /**
     * Check if fallback protection should be applied
     */
    private function has_fallback_protection( $form_id ) {
        // Check if there are any global protection settings
        $global_protection = get_option( 'npfo_global_protection', false );
        return $global_protection && !empty( $global_protection['enabled'] );
    }

    /**
     * Apply fallback protection when specific form protection is not found
     */
    private function apply_fallback_protection( $form_data ) {
        $global_protection = get_option( 'npfo_global_protection', array() );
        if ( empty( $global_protection['enabled'] ) ) {
            return array(
                'valid'  => true,
                'errors' => array(),
            );
        }
        $errors = array();
        $is_valid = true;
        // Apply basic spam protection
        foreach ( $form_data as $field_name => $field_value ) {
            if ( empty( $field_value ) ) {
                continue;
            }
            // Check for common spam patterns
            if ( preg_match( '/\\b(viagra|casino|loan|credit|debt|free|win|prize|click here)\\b/i', $field_value ) ) {
                $errors[$field_name] = 'This content appears to be spam.';
                $is_valid = false;
            }
            // Check for excessive links
            if ( preg_match_all( '/https?:\\/\\/|www\\./i', $field_value ) > 2 ) {
                $errors[$field_name] = 'Too many links detected.';
                $is_valid = false;
            }
        }
        return array(
            'valid'  => $is_valid,
            'errors' => $errors,
            'debug'  => 'Applied fallback protection',
        );
    }

    public function get_protection_by_id( $id ) {
        global $wpdb;
        
        if ( ! is_numeric( $id ) || $id <= 0 ) {
            return null;
        }
        
        $table_name = $wpdb->prefix . 'npfo_protections';
        $table_name_safe = esc_sql( $table_name );
        
        // Check if table exists
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin's own database operations
        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) ) !== $table_name ) {
            return null;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Plugin's own database operations, table name is safe
        $result = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table_name_safe} WHERE id=%d", $id ), ARRAY_A );
        
        if ( $result && is_array( $result ) ) {
            // Sanitize string fields from database
            if ( isset( $result['form_plugin'] ) ) {
                $result['form_plugin'] = sanitize_text_field( $result['form_plugin'] );
            }
            if ( isset( $result['form_id'] ) ) {
                $result['form_id'] = sanitize_text_field( $result['form_id'] );
            }
            if ( isset( $result['form_title'] ) ) {
                $result['form_title'] = sanitize_text_field( $result['form_title'] );
            }
            
            if ( isset( $result['field_protections'] ) && is_string( $result['field_protections'] ) ) {
                $decoded = json_decode( $result['field_protections'], true );
                if ( json_last_error() === JSON_ERROR_NONE ) {
                    $result['field_protections'] = $decoded;
                } else {
                    $result['field_protections'] = maybe_unserialize( $result['field_protections'] );
                }
            }
            // Ensure field_protections is always an array and sanitize it
            if ( ! is_array( $result['field_protections'] ) ) {
                $result['field_protections'] = array();
            } else {
                // Sanitize field protections array
                $result['field_protections'] = $this->sanitize_field_protections( $result['field_protections'] );
            }
            return $result;
        }
        
        return null;
    }

    public function get_all_protections() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'npfo_protections';
        $table_name_safe = esc_sql( $table_name );
        
        // Check if table exists
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin's own database operations
        if ( $wpdb->get_var( $wpdb->prepare( "SHOW TABLES LIKE %s", $table_name ) ) !== $table_name ) {
            return array();
        }
        
        // Safe query - no user input, table name is constructed from prefix
        // Table name is safe as it's constructed from $wpdb->prefix
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Plugin's own database operations, table name is safe
        $results = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table_name_safe} ORDER BY created_at DESC" ), ARRAY_A );
        
        if ( ! is_array( $results ) ) {
            return array();
        }
        
        foreach ( $results as &$result ) {
            // Sanitize string fields from database
            if ( isset( $result['form_plugin'] ) ) {
                $result['form_plugin'] = sanitize_text_field( $result['form_plugin'] );
            }
            if ( isset( $result['form_id'] ) ) {
                $result['form_id'] = sanitize_text_field( $result['form_id'] );
            }
            if ( isset( $result['form_title'] ) ) {
                $result['form_title'] = sanitize_text_field( $result['form_title'] );
            }
            
            if ( isset( $result['field_protections'] ) && is_string( $result['field_protections'] ) ) {
                $decoded = json_decode( $result['field_protections'], true );
                if ( json_last_error() === JSON_ERROR_NONE ) {
                    $result['field_protections'] = $decoded;
                } else {
                    $result['field_protections'] = maybe_unserialize( $result['field_protections'] );
                }
            }
            // Ensure field_protections is always an array and sanitize it
            if ( ! is_array( $result['field_protections'] ) ) {
                $result['field_protections'] = array();
            } else {
                // Sanitize field protections array
                $result['field_protections'] = $this->sanitize_field_protections( $result['field_protections'] );
            }
        }
        unset( $result ); // Break reference
        
        return $results;
    }

    /**
     * Cleanup on uninstall
     * Freemius removed - no special cleanup needed
     */
    public function cleanup_on_uninstall() {
        // Cleanup can be added here if needed
    }

}

// Initialize the plugin
new NPFO_Form_Shield();
// Activation hook
register_activation_hook( __FILE__, function () {
    $plugin = new NPFO_Form_Shield();
    $plugin->create_database_table();
} );