jQuery(document).ready(function($) {
    console.log('ASM Ninja Forms Real-time Validation Loaded');
    
    // Real-time validation for Ninja Forms
    let blockedWords = {};
    let formProtections = {};
    
    // No fallback protection - only use admin settings
    const fallbackProtection = {};
    
    // Get form protection settings
    function getFormProtection(formId) {
        console.log('Getting protection for form ID:', formId);
        
        return new Promise((resolve, reject) => {
            $.ajax({
                url: npfo_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'npfo_get_form_protection',
                    form_plugin: 'ninja-forms',
                    form_id: formId,
                    nonce: npfo_ajax.frontend_nonce
                },
                success: function(response) {
                    console.log('AJAX response for Ninja Forms protection:', response);
                    if (response.success && response.data && response.data.field_protections) {
                        console.log('Using admin-configured protection settings:', response.data.field_protections);
                        resolve(response.data);
                    } else {
                        console.log('No admin protection found - no validation will be applied');
                        resolve(null);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX error getting protection:', error);
                    console.log('No validation will be applied due to AJAX error');
                    resolve(null);
                }
            });
        });
    }
    
    // Check if text contains blocked words
    function checkBlockedWords(text, protectionWords, protectionType) {
        if (!text || !protectionWords) return { hasBlocked: false, blockedWord: null };
        
        const words = protectionWords.split(',').map(w => w.trim().toLowerCase());
        const textLower = text.toLowerCase();
        
        if (protectionType === 'blacklist') {
            // Blacklist: Block if any blacklisted word is found
            for (let word of words) {
                if (word && textLower.includes(word)) {
                    return { hasBlocked: true, blockedWord: word };
                }
            }
        } else if (protectionType === 'whitelist') {
            // Whitelist: Block if text contains words NOT in whitelist
            const textWords = textLower.split(/\s+/).filter(w => w.length > 0);
            for (let textWord of textWords) {
                let isWhitelisted = false;
                for (let whiteWord of words) {
                    if (whiteWord && textWord.includes(whiteWord)) {
                        isWhitelisted = true;
                        break;
                    }
                }
                if (!isWhitelisted) {
                    return { hasBlocked: true, blockedWord: textWord };
                }
            }
        }
        
        return { hasBlocked: false, blockedWord: null };
    }
    
    // Remove blocked words from text
    function removeBlockedWords(text, protectionWords, protectionType) {
        if (!text || !protectionWords) return text;
        
        const words = protectionWords.split(',').map(w => w.trim());
        let cleanText = text;
        
        if (protectionType === 'blacklist') {
            // Remove blacklisted words
            words.forEach(word => {
                if (word) {
                    const regex = new RegExp(word.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'), 'gi');
                    cleanText = cleanText.replace(regex, '');
                }
            });
        } else if (protectionType === 'whitelist') {
            // For whitelist, remove words that are NOT in the whitelist
            const textWords = cleanText.split(/\s+/);
            const filteredWords = textWords.filter(textWord => {
                if (!textWord.trim()) return true; // Keep empty strings/spaces
                
                const textWordLower = textWord.toLowerCase();
                for (let whiteWord of words) {
                    if (whiteWord && textWordLower.includes(whiteWord.toLowerCase())) {
                        return true; // Keep whitelisted words
                    }
                }
                return false; // Remove non-whitelisted words
            });
            cleanText = filteredWords.join(' ');
        }
        
        return cleanText.trim();
    }
    
    // Show warning message
    function showWarning(field, message, protectionType) {
        const $field = $(field);
        const $container = $field.closest('.nf-field-container');
        
        // Remove existing warnings
        $container.find('.asm-warning').remove();
        
        // Add warning styling
        $field.addClass('asm-blocked-field');
        
        // Create appropriate warning message based on protection type
        let warningText = message;
        if (protectionType === 'blacklist') {
            warningText = 'This content contains prohibited words and has been removed.';
        } else if (protectionType === 'whitelist') {
            warningText = 'Only specific words are allowed in this field. Unauthorized content has been removed.';
        }
        
        // Add warning message
        const $warning = $('<div class="asm-warning" style="color: #d63638; font-size: 12px; margin-top: 5px; display: flex; align-items: center;"><span class="dashicons dashicons-warning" style="margin-right: 5px; font-size: 14px;"></span>' + warningText + '</div>');
        $container.append($warning);
    }
    
    // Clear warning
    function clearWarning(field) {
        const $field = $(field);
        const $container = $field.closest('.nf-field-container');
        
        $field.removeClass('asm-blocked-field');
        $container.find('.asm-warning').remove();
    }
    
    // Initialize real-time validation for Ninja Forms
    function initNinjaFormsValidation() {
        console.log('Initializing Ninja Forms validation...');
        
        // Find all Ninja Forms - use exact selectors from HTML structure
        const $forms = $('.nf-form-wrap, .ninja-forms-form-wrap, div[class*="nf-form"]');
        console.log('Found forms:', $forms.length);
        
        $forms.each(function() {
            const $form = $(this);
            console.log('Processing form:', $form);
            
            // Extract form ID from JavaScript variable (from screenshot)
            let formId = null;
            
            // Method 1: Look for FormDisplay variable in page scripts
            if (window.nfForms && window.nfForms.length > 0) {
                formId = window.nfForms[0].id;
                console.log('Form ID from nfForms global:', formId);
            }
            
            // Method 2: Extract from form data attributes or classes
            if (!formId) {
                const formClass = $form.attr('class') || '';
                const formIdMatch = formClass.match(/form-(\d+)/);
                if (formIdMatch) {
                    formId = formIdMatch[1];
                    console.log('Form ID from class:', formId);
                }
            }
            
            // Method 3: Look for form ID in hidden inputs within the form
            if (!formId) {
                $form.find('input[type="hidden"]').each(function() {
                    const name = $(this).attr('name');
                    const value = $(this).val();
                    console.log('Hidden input found:', name, '=', value);
                    
                    if (name === 'form_id' || name === 'id' || (name && name.includes('form') && /^\d+$/.test(value))) {
                        formId = value;
                        console.log('Form ID from hidden input:', formId);
                        return false;
                    }
                });
            }
            
            // Method 4: Extract from script tags containing form data
            if (!formId) {
                $('script').each(function() {
                    const scriptContent = $(this).html();
                    if (scriptContent && scriptContent.includes('form_id')) {
                        const formIdMatch = scriptContent.match(/"form_id"\s*:\s*"?(\d+)"?/);
                        if (formIdMatch) {
                            formId = formIdMatch[1];
                            console.log('Form ID from script:', formId);
                            return false;
                        }
                    }
                });
            }
            
            // Method 5: Default fallback
            if (!formId) {
                console.log('No form ID found - using default form ID 1');
                formId = '1';
            }
            
            console.log('Form ID found:', formId);
            
            // Get protection settings for this form
            getFormProtection(formId).then(function(protection) {
                console.log('Protection settings received:', protection);
                
                if (!protection || !protection.field_protections) {
                    console.log('No protection settings found - skipping validation for this form');
                    return;
                }
                
                const fieldProtections = typeof protection.field_protections === 'string' 
                    ? JSON.parse(protection.field_protections) 
                    : protection.field_protections;
                
                console.log('Parsed field protections:', fieldProtections);
                
                // Check if field protections is empty
                if (!fieldProtections || Object.keys(fieldProtections).length === 0) {
                    console.log('Field protections is empty - no validation will be applied');
                    return;
                }
                
                // Add real-time validation to ALL input fields
                $form.find('input[type="text"], input[type="email"], textarea').each(function() {
                    const $field = $(this);
                    const $container = $field.closest('.nf-field-container, .field-wrap');
                    const $label = $container.find('label, .nf-field-label');
                    let fieldLabel = $label.text().trim();
                    
                    // Try multiple ways to get field label
                    if (!fieldLabel) {
                        fieldLabel = $field.attr('placeholder') || $field.attr('name') || $field.attr('id');
                    }
                    
                    console.log('Found field with label:', fieldLabel);
                    console.log('Available protections:', Object.keys(fieldProtections));
                    
                    // Check if this field has protection - try exact match and partial match
                    let fieldProtection = fieldProtections[fieldLabel];
                    
                    // If no exact match, try partial matching
                    if (!fieldProtection) {
                        for (let protectionKey in fieldProtections) {
                            if (fieldLabel.toLowerCase().includes(protectionKey.toLowerCase()) || 
                                protectionKey.toLowerCase().includes(fieldLabel.toLowerCase())) {
                                fieldProtection = fieldProtections[protectionKey];
                                console.log('Found partial match:', protectionKey, 'for field:', fieldLabel);
                                break;
                            }
                        }
                    }
                    
                    // Also try matching by field position/index
                    if (!fieldProtection) {
                        const fieldIndex = $form.find('input[type="text"], input[type="email"], textarea').index(this);
                        const protectionKeys = Object.keys(fieldProtections);
                        if (protectionKeys[fieldIndex]) {
                            fieldProtection = fieldProtections[protectionKeys[fieldIndex]];
                            console.log('Found by index match:', protectionKeys[fieldIndex], 'for field at index:', fieldIndex);
                        }
                    }
                    
                    if (fieldProtection) {
                        console.log('Adding validation to field:', fieldLabel, fieldProtection);
                        
                        // Add input event listener
                        $field.on('input keyup paste', function() {
                            const currentValue = $(this).val();
                            console.log('Field value changed:', fieldLabel, currentValue);
                            console.log('Protection type:', fieldProtection.type);
                            console.log('Protection words:', fieldProtection.words);
                            
                            const result = checkBlockedWords(currentValue, fieldProtection.words, fieldProtection.type);
                            console.log('Protection check result:', result);
                            
                            if (result.hasBlocked) {
                                console.log('Blocked content found, removing...');
                                
                                // Auto-remove blocked content
                                const cleanValue = removeBlockedWords(currentValue, fieldProtection.words, fieldProtection.type);
                                $(this).val(cleanValue);
                                
                                // Show appropriate warning based on protection type
                                if (fieldProtection.type === 'blacklist') {
                                    showWarning(this, 'Blocked word "' + result.blockedWord + '" removed automatically', 'blacklist');
                                } else if (fieldProtection.type === 'whitelist') {
                                    showWarning(this, 'Non-whitelisted word "' + result.blockedWord + '" removed automatically', 'whitelist');
                                }
                                
                                // Clear warning after 3 seconds
                                setTimeout(() => {
                                    clearWarning(this);
                                }, 3000);
                            } else {
                                clearWarning(this);
                            }
                        });
                    }
                });
            });
        });
    }
    
    // CSS for blocked fields
    $('<style>')
        .prop('type', 'text/css')
        .html(`
            .asm-blocked-field {
                border: 2px solid #d63638 !important;
                background-color: #fef7f7 !important;
            }
            .asm-warning {
                animation: asm-fade-in 0.3s ease-in;
            }
            @keyframes asm-fade-in {
                from { opacity: 0; transform: translateY(-10px); }
                to { opacity: 1; transform: translateY(0); }
            }
        `)
        .appendTo('head');
    
    // Initialize when DOM is ready
    initNinjaFormsValidation();
    
    // Re-initialize when new forms are loaded (AJAX)
    $(document).on('nfFormReady', function() {
        setTimeout(initNinjaFormsValidation, 500);
    });
});
