jQuery(document).ready(function($) {
  // Global variables to store form protection rules
  let formProtections = {};
  let formSettings = {};
  
  // Fetch protection settings for all forms on page load
  fetchAllProtectionSettings();
  
  // Add real-time validation for all supported form types
  setupFormValidation();
  
  /**
   * Fetch all protection settings via AJAX
   */
  function fetchAllProtectionSettings() {
    // For each form on the page, try to identify its plugin and ID
    $('form').each(function() {
      const $form = $(this);
      let formPlugin = '';
      let formId = '';
      
      // Try to identify Contact Form 7 forms
      if ($form.hasClass('wpcf7-form')) {
        formPlugin = 'contact-form-7';
        const $parent = $form.closest('.wpcf7');
        if ($parent.length) {
          formId = $parent.attr('id').replace('wpcf7-f', '').split('-')[0];
        }
        // Fallback: try to get from hidden input
        if (!formId) {
          formId = $form.find('input[name="_wpcf7"]').val();
        }
      }
      
      // Try to identify Gravity Forms
      else if ($form.hasClass('gform_wrapper') || ($form.attr('id') && $form.attr('id').indexOf('gform_') === 0)) {
        formPlugin = 'gravity-forms';
        formId = $form.attr('id').replace('gform_', '');
        // Fallback: try to extract from form HTML
        if (!formId) {
          const formHtml = $form.html();
          const match = formHtml.match(/gform_wrapper_(\d+)/);
          formId = match ? match[1] : null;
        }
      }
      
      // Try to identify WPForms
      else if ($form.hasClass('wpforms-form')) {
        formPlugin = 'wpforms';
        formId = $form.data('formid') || $form.attr('data-formid');
        // Fallback: try to extract from form HTML
        if (!formId) {
          const formHtml = $form.html();
          const match = formHtml.match(/data-formid="(\d+)"/);
          formId = match ? match[1] : null;
        }
      }
      
      // Try to identify Ninja Forms
      else if ($form.hasClass('nf-form-layout') || $form.find('.nf-form-layout').length) {
        formPlugin = 'ninja-forms';
        formId = $form.closest('.nf-form-cont').data('form-id');
        // Fallback: try multiple methods
        if (!formId) {
          formId = $form.data('form-id') || $form.find('input[name="form_id"]').val();
        }
        // Check global JavaScript variables
        if (!formId && window.nfForms && window.nfForms.length > 0) {
          formId = window.nfForms[0].id;
        }
      }
      
      // Try to identify Forminator
      else if ($form.hasClass('forminator-custom-form')) {
        formPlugin = 'forminator';
        formId = $form.data('form-id') || $form.attr('data-form-id') || $form.attr('id').replace('forminator-form-', '');
        // Fallback: try to extract from form HTML
        if (!formId) {
          const formHtml = $form.html();
          const match = formHtml.match(/data-form-id="(\d+)"/);
          formId = match ? match[1] : null;
        }
      }
      
      // Try to identify Fluent Forms
      else if ($form.hasClass('frm-fluent-form') || $form.hasClass('ff-form') || $form.find('.ff-el-group').length) {
        formPlugin = 'fluent-forms';
        formId = $form.data('form_id') || $form.data('form-id') || $form.attr('data-form_id') || $form.find('input[name="form_id"]').val();
        // Fallback: try to extract from form HTML
        if (!formId) {
          const formHtml = $form.html();
          const match = formHtml.match(/data-form_id="(\d+)"/);
          formId = match ? match[1] : null;
        }
      }
      
      console.log('Form detected:', formPlugin, 'ID:', formId);
      
      // If we identified the form, fetch its protection settings
      if (formPlugin && formId) {
        getFormProtectionSettings(formPlugin, formId, $form);
      } else {
        console.log('Could not identify form plugin or ID for form:', $form);
      }
    });
  }
  
  /**
   * Get protection settings for a specific form
   */
  function getFormProtectionSettings(formPlugin, formId, $form) {
    // Check if required variables exist
    if (typeof npfo_frontend === 'undefined' || !npfo_frontend.ajax_url || !npfo_frontend.nonce) {
      console.warn('NP Form Shield: Frontend configuration not available');
      applyFallbackProtection($form, formPlugin, formId);
      return;
    }
    
    $.ajax({
      url: npfo_frontend.ajax_url,
      type: 'POST',
      dataType: 'json',
      data: {
        action: 'npfo_get_form_protection',
        form_plugin: formPlugin,
        form_id: formId,
        nonce: npfo_frontend.nonce
      },
      success: function(response) {
        if (response && response.success && response.data) {
          const formKey = `${formPlugin}_${formId}`;
          formProtections[formKey] = response.data.field_protections || {};
          formSettings[formKey] = {
            blockLinks: response.data.block_links !== undefined ? response.data.block_links : true
          };
          
          console.log('Protection settings loaded for', formKey, ':', formProtections[formKey]);
          
          // Apply validation to this form
          applyFormValidation($form, formKey);
        } else {
          console.log('No protection settings found for', formPlugin, formId);
          // Apply basic fallback protection
          applyFallbackProtection($form, formPlugin, formId);
        }
      },
      error: function(xhr, status, error) {
        console.error('Failed to load protection settings:', error);
        console.log('Response:', xhr.responseText);
        // Apply basic fallback protection on AJAX error
        applyFallbackProtection($form, formPlugin, formId);
      }
    });
  }
  
  /**
   * Apply basic fallback protection when AJAX fails or no settings found
   */
  function applyFallbackProtection($form, formPlugin, formId) {
    console.log('Applying fallback protection for', formPlugin, formId);
    
    // Add basic spam detection
    $form.find('input, textarea').on('keyup change', function() {
      const $field = $(this);
      const value = $field.val();
      
      if (!value) return;
      
      // Check for common spam patterns
      if (/\b(viagra|casino|loan|credit|debt|free|win|prize|click here)\b/i.test(value)) {
        showFieldError($field, 'This content appears to be spam.');
      }
      
      // Check for excessive links
      const linkCount = (value.match(/https?:\/\/|www\./gi) || []).length;
      if (linkCount > 2) {
        showFieldError($field, 'Too many links detected.');
      }
    });
  }
  
  /**
   * Set up form validation for all types of forms
   */
  function setupFormValidation() {
    // Prevent form submission if there are validation errors
    $(document).on('submit', 'form', function(e) {
      const $form = $(this);
      if ($form.find('.asm-field-error').length > 0) {
        e.preventDefault();
        // Scroll to first error
        const $firstError = $form.find('.asm-field-error').first();
        if ($firstError.length) {
          $('html, body').animate({
            scrollTop: $firstError.offset().top - 100
          }, 300);
        }
        return false;
      }
    });
    
    // Real-time validation for Contact Form 7
    $(document).on('keyup change', '.wpcf7-form input, .wpcf7-form textarea', function() {
      const $field = $(this);
      const $form = $field.closest('form');
      const $parent = $form.closest('.wpcf7');
      
      if ($parent.length) {
        const formId = $parent.attr('id').replace('wpcf7-f', '').split('-')[0];
        const formKey = `contact-form-7_${formId}`;
        
        if (formProtections[formKey]) {
          validateField($field, formKey);
        }
      }
    });
    
    // Real-time validation for Gravity Forms - Enhanced for name fields
    $(document).on('keyup change', '.gform_wrapper input, .gform_wrapper textarea', function() {
      const $field = $(this);
      const $form = $field.closest('form');
      const formId = $form.attr('id').replace('gform_', '');
      const formKey = `gravity-forms_${formId}`;
      
      if (formProtections[formKey]) {
        // For Gravity Forms name fields, validate all name sub-fields together
        const fieldName = $field.attr('name');
        if (fieldName && fieldName.includes('input_') && (fieldName.includes('.') || fieldName.includes('_'))) {
          // This is a name sub-field, validate the main name field
          const $gfield = $field.closest('.gfield');
          
          // Collect all name parts from the HTML structure
          let nameValue = '';
          
          // Check for underscore notation: input_1_1, input_1_3, etc.
          $gfield.find('input[name*="input_"]').each(function() {
            const val = $(this).val();
            if (val) {
              nameValue += val + ' ';
            }
          });
          
          // Also check for dot notation as backup
          const baseFieldId = fieldName.split(/[._]/)[0] + '_' + fieldName.split(/[._]/)[1];
          $gfield.find('input[name^="' + baseFieldId + '"]').each(function() {
            const val = $(this).val();
            if (val && nameValue.indexOf(val) === -1) { // Avoid duplicates
              nameValue += val + ' ';
            }
          });
          
          console.log('Gravity Forms name field combined value:', nameValue.trim());
          
          // Create a temporary field object for validation
          const $tempField = $('<input>').val(nameValue.trim());
          $tempField.data('original-field', $field);
          $tempField.data('gfield', $gfield);
          validateField($tempField, formKey);
        } else {
          validateField($field, formKey);
        }
      }
    });
    
    // Real-time validation for WPForms
    $(document).on('keyup change', '.wpforms-form input, .wpforms-form textarea', function() {
      const $field = $(this);
      const $form = $field.closest('form');
      const formId = $form.data('formid');
      const formKey = `wpforms_${formId}`;
      
      if (formProtections[formKey]) {
        validateField($field, formKey);
      }
    });
    
    // Real-time validation for Ninja Forms
    $(document).on('keyup change', '.nf-form-layout input, .nf-form-layout textarea', function() {
      const $field = $(this);
      const $form = $field.closest('.nf-form-layout');
      const formId = $form.closest('.nf-form-cont').data('form-id');
      const formKey = `ninja-forms_${formId}`;
      
      if (formProtections[formKey]) {
        validateField($field, formKey);
      }
    });
    
    // Real-time validation for Forminator
    $(document).on('keyup change', '.forminator-custom-form input, .forminator-custom-form textarea', function() {
      const $field = $(this);
      const $form = $field.closest('form');
      const formId = $form.data('form-id') || $form.attr('id').replace('forminator-form-', '');
      const formKey = `forminator_${formId}`;
      
      if (formProtections[formKey]) {
        validateField($field, formKey);
      }
    });
    
    // Real-time validation for Fluent Forms
    $(document).on('keyup change input', '.frm-fluent-form input, .frm-fluent-form textarea, .ff-form input, .ff-form textarea, .ff-el-group input, .ff-el-group textarea', function() {
      const $field = $(this);
      const $form = $field.closest('form');
      const formId = $form.data('form_id') || $form.data('form-id') || $form.find('input[name="form_id"]').val();
      const formKey = `fluent-forms_${formId}`;
      
      console.log('Fluent Forms field changed:', $field.attr('name'), 'Form ID:', formId, 'Form Key:', formKey);
      
      if (formProtections[formKey]) {
        validateField($field, formKey);
      }
    });
  }
  
  /**
   * Apply validation to a specific form
   */
  function applyFormValidation($form, formKey) {
    // Apply initial validation to all fields
    $form.find('input, textarea').each(function() {
      const $field = $(this);
      validateField($field, formKey);
      
      // Add event listeners for real-time validation
      $field.on('keyup change', function() {
        validateField($field, formKey);
      });
    });
  }
  
  /**
   * Validate a single field against protection rules
   */
  function validateField($field, formKey) {
    // Get field name/label based on form type
    const fieldName = getFieldName($field);
    console.log('Validating field:', fieldName, 'Form key:', formKey);
    
    if (!fieldName) {
      console.log('No field name found, skipping validation');
      return; // Skip if we can't determine field name
    }
    
    // Skip submit buttons, hidden fields, etc.
    if ($field.attr('type') === 'submit' || $field.attr('type') === 'button' || 
        $field.attr('type') === 'hidden' || $field.attr('type') === 'file') {
      return;
    }
    
    // Get field value
    const fieldValue = $field.val();
    if (fieldValue === undefined || fieldValue === null) return;
    
    console.log('Field value:', fieldValue);
    
    // Check if we have protection rules for this field
    const fieldProtections = formProtections[formKey] || {};
    console.log('Available protections:', Object.keys(fieldProtections));
    console.log('Looking for field:', fieldName, 'in protections:', fieldProtections);
    
    // Try to find protection with case-insensitive matching - ONLY for Fluent Forms
    let protection = fieldProtections[fieldName];
    if (!protection) {
      const $form = $field.closest('form');
      const isFluentForm = $form.hasClass('frm-fluent-form') || $form.hasClass('ff-form') || $form.find('.ff-el-group').length;
      
      if (isFluentForm) {
        // For Fluent Forms, try exact key matching from protection settings
        const protectionKeys = Object.keys(fieldProtections);
        console.log('Fluent Forms - Trying to match field:', fieldName, 'against keys:', protectionKeys);
        
        // Try exact match first
        if (fieldProtections[fieldName]) {
          protection = fieldProtections[fieldName];
          console.log('Fluent Forms - Exact match found:', fieldName);
        }
        // Try case-insensitive matching
        else {
          for (const protectionKey in fieldProtections) {
            if (protectionKey.toLowerCase() === fieldName.toLowerCase()) {
              protection = fieldProtections[protectionKey];
              console.log('Fluent Forms - Case-insensitive match found:', protectionKey);
              break;
            }
          }
        }
      } else {
        // For other forms, use original logic
        for (const protectionKey in fieldProtections) {
          if (protectionKey.toLowerCase() === fieldName.toLowerCase() ||
              protectionKey.toLowerCase().replace(/[^a-z]/g, '') === fieldName.toLowerCase().replace(/[^a-z]/g, '')) {
            protection = fieldProtections[protectionKey];
            console.log('Found protection with key:', protectionKey);
            break;
          }
        }
      }
    }
    
    const blockLinks = formSettings[formKey]?.blockLinks !== false; // Default to true
    
    // Remove existing error styling
    removeFieldError($field);
    
    // If no protection rules for this field, only check for links if blockLinks is enabled
    if (!protection) {
      console.log('No protection found for field:', fieldName);
      
      // For Fluent Forms, be more specific about field matching
      const $form = $field.closest('form');
      const isFluentForm = $form.hasClass('frm-fluent-form') || $form.hasClass('ff-form') || $form.find('.ff-el-group').length;
      
      if (isFluentForm) {
        console.log('Fluent Forms - No protection found. Available keys:', Object.keys(fieldProtections));
        console.log('Fluent Forms - Field name we are looking for:', fieldName);
        
        // Try one more time with partial matching for Fluent Forms
        for (const protectionKey in fieldProtections) {
          if (protectionKey.includes('names') && fieldName === 'names') {
            protection = fieldProtections[protectionKey];
            console.log('Fluent Forms - Partial match found for names:', protectionKey);
            break;
          }
          if (protectionKey.includes('Email') && fieldName === 'Email') {
            protection = fieldProtections[protectionKey];
            console.log('Fluent Forms - Partial match found for Email:', protectionKey);
            break;
          }
          if (protectionKey.includes('Subject') && fieldName === 'Subject') {
            protection = fieldProtections[protectionKey];
            console.log('Fluent Forms - Partial match found for Subject:', protectionKey);
            break;
          }
          if (protectionKey.includes('Message') && fieldName === 'Your Message') {
            protection = fieldProtections[protectionKey];
            console.log('Fluent Forms - Partial match found for Message:', protectionKey);
            break;
          }
        }
      }
      
      if (!protection) {
        // Check for links (except email fields) if blockLinks is enabled
        if (blockLinks && isNotEmailField(fieldName) && containsLink(fieldValue)) {
          // Remove the link from the field FIRST
          const newValue = fieldValue.replace(/https?:\/\/\S+|www\.\S+/gi, '');
          $field.val(newValue);
          
          // Show error message
          showFieldError($field, 'Links are not allowed in this field.');
          
          // Focus back to field
          setTimeout(() => {
            $field.focus();
          }, 10);
          
          return true;
        }
        return false;
      }
    }
    
    console.log('Found protection:', protection);
    
    // Check for links (except email fields) if blockLinks is enabled
    if (blockLinks && isNotEmailField(fieldName) && containsLink(fieldValue)) {
      // Remove the link from the field FIRST (same as blocked words)
      const newValue = fieldValue.replace(/https?:\/\/\S+|www\.\S+/gi, '');
      $field.val(newValue);
      
      // Show error message
      showFieldError($field, 'Links are not allowed in this field.');
      
      // Focus back to field
      setTimeout(() => {
        $field.focus();
      }, 10);
      
      return true;
    }
    
    // Validate email fields
    if (!isNotEmailField(fieldName) && fieldValue && !isValidEmail(fieldValue)) {
      showFieldError($field, 'Please enter a valid email address.');
      return true;
    }
    
    // Apply blacklist/whitelist validation
    if (protection.type === 'blacklist') {
      const words = protection.words.split(',').map(word => word.trim());
      console.log('Fluent Forms - Checking blacklist words:', words, 'against value:', fieldValue);
      
      for (let i = 0; i < words.length; i++) {
        const word = words[i];
        if (word && fieldValue.toLowerCase().indexOf(word.toLowerCase()) !== -1) {
          console.log('Fluent Forms - BLOCKED WORD FOUND:', word);
          
          // Remove the blacklisted word from the field FIRST
          const newValue = fieldValue.replace(new RegExp(word, 'gi'), '');
          console.log('Fluent Forms - Removing word, new value:', newValue);
          $field.val(newValue);
          
          // Show error message
          showFieldError($field, 'This content contains prohibited words.');
          
          // Focus back to field to continue typing
          setTimeout(() => {
            $field.focus();
          }, 10);
          
          return true;
        }
      }
    } else if (protection.type === 'whitelist') {
      const words = protection.words.split(',').map(word => word.trim());
      console.log('Checking whitelist words:', words, 'against value:', fieldValue);
      
      if (words.length > 0 && words[0]) {
        let foundAllowed = false;
        for (let i = 0; i < words.length; i++) {
          const word = words[i];
          if (word && fieldValue.toLowerCase().indexOf(word.toLowerCase()) !== -1) {
            foundAllowed = true;
            break;
          }
        }
        
        if (!foundAllowed && fieldValue.trim() !== '') {
          console.log('WHITELIST VIOLATION');
          showFieldError($field, 'This content does not contain required words.');
          return true;
        }
      }
    }
    
    return false;
  }
  
  /**
   * Get field name/label based on form type and field attributes
   */
  function getFieldName($field) {
    // Try to get name from various attributes
    const name = $field.attr('name') || $field.attr('id');
    console.log('Raw field name/id:', name);
    
    if (name) {
      // For Fluent Forms - check if this is a Fluent Forms field first
      const $form = $field.closest('form');
      const isFluentForm = $form.hasClass('frm-fluent-form') || $form.hasClass('ff-form') || $form.find('.ff-el-group').length;
      
      if (isFluentForm) {
        // Fluent Forms specific field name mapping
        if (name.indexOf('names') !== -1 || name.indexOf('first_name') !== -1) {
          console.log('Fluent Forms - Mapped to: names');
          return 'names';
        }
        if (name.indexOf('email') !== -1) {
          console.log('Fluent Forms - Mapped to: Email');
          return 'Email';
        }
        if (name.indexOf('subject') !== -1) {
          console.log('Fluent Forms - Mapped to: Subject');
          return 'Subject';
        }
        if (name.indexOf('message') !== -1) {
          console.log('Fluent Forms - Mapped to: Your Message');
          return 'Your Message';
        }
        
        // If no specific mapping found, return the raw name for Fluent Forms
        console.log('Fluent Forms - Using raw name:', name);
        return name;
      }
      
      // For Contact Form 7 fields
      if (name.indexOf('_wpcf7') === 0) {
        return name.replace('_wpcf7', '');
        if ($label.length) {
          let labelText = $label.text().trim().replace(/\s*\*\s*$/, ''); // Remove asterisk
          
          // For name fields, check if this is a sub-field (first name, last name)
          const fieldName = $field.attr('name');
          if (fieldName && fieldName.includes('input_') && fieldName.includes('.')) {
            // This is a name sub-field, use the main label
            console.log('Gravity Forms name sub-field detected:', fieldName, 'Label:', labelText);
          }
          
          return labelText;
        }
      }
    }
    
    // Contact Form 7
    if ($form.hasClass('wpcf7-form') || $form.find('.wpcf7-form-control').length) {
      const name = $field.attr('name');
      if (name) {
        // Convert name to label format
        return name.replace(/[-_]/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
      }
    }
    
    // Fluent Forms
    if ($form.hasClass('frm-fluent-form') || $form.hasClass('ff-form') || $form.find('.ff-el-group').length) {
      const $group = $field.closest('.ff-el-group');
      if ($group.length) {
        const $label = $group.find('label');
        if ($label.length) {
          return $label.text().trim().replace(/\s*\*\s*$/, ''); // Remove asterisk
        }
      }
    }
    
    // Generic fallback
    const $label = $field.siblings('label').first();
    if ($label.length) {
      return $label.text().trim();
    }
    
    // No name found
    console.log('No field name found');
    return null;
  }
  
  /**
   * Show error message for a field
   */
  function showFieldError($field, message) {
    // Add error class to field
    $field.addClass('asm-field-error');
    
    // Remove any existing error message
    removeErrorMessage($field);
    
    // Create error message
    const $error = $('<div class="asm-error-message">' + message + '</div>');
    
    // Different placement based on form type
    if ($field.closest('.wpcf7-form').length) {
      // Contact Form 7
      $field.closest('.wpcf7-form-control-wrap').after($error);
    } else if ($field.closest('.gfield').length) {
      // Gravity Forms
      $field.closest('.gfield').append($error);
    } else if ($field.closest('.wpforms-field').length) {
      // WPForms
      $field.closest('.wpforms-field').append($error);
    } else if ($field.closest('.nf-field').length) {
      // Ninja Forms
      $field.closest('.nf-field').append($error);
    } else if ($field.closest('.forminator-field').length) {
      // Forminator
      $field.closest('.forminator-field').append($error);
    } else if ($field.closest('.ff-el-group').length) {
      // Fluent Forms
      $field.closest('.ff-el-group').append($error);
    } else {
      // Default placement
      $field.after($error);
    }
  }
  
  /**
   * Remove error styling from a field
   */
  function removeFieldError($field) {
    $field.removeClass('asm-field-error');
    removeErrorMessage($field);
  }
  
  /**
   * Remove error message for a field
   */
  function removeErrorMessage($field) {
    // Different removal based on form type
    if ($field.closest('.wpcf7-form').length) {
      // Contact Form 7
      $field.closest('.wpcf7-form-control-wrap').siblings('.asm-error-message').remove();
    } else if ($field.closest('.gfield').length) {
      // Gravity Forms
      $field.closest('.gfield').find('.asm-error-message').remove();
    } else if ($field.closest('.wpforms-field').length) {
      // WPForms
      $field.closest('.wpforms-field').find('.asm-error-message').remove();
    } else if ($field.closest('.nf-field').length) {
      // Ninja Forms
      $field.closest('.nf-field').find('.asm-error-message').remove();
    } else if ($field.closest('.forminator-field').length) {
      // Forminator
      $field.closest('.forminator-field').find('.asm-error-message').remove();
    } else if ($field.closest('.ff-el-group').length) {
      // Fluent Forms
      $field.closest('.ff-el-group').find('.asm-error-message').remove();
    } else {
      // Default removal
      $field.siblings('.asm-error-message').remove();
    }
  }
  
  /**
   * Check if a string contains a URL
   */
  function containsLink(text) {
    return /https?:\/\/|www\./i.test(text);
  }
  
  /**
   * Check if a field is NOT an email field (based on name)
   */
  function isNotEmailField(fieldName) {
    if (!fieldName) return true;
    return fieldName.toLowerCase().indexOf('email') === -1;
  }
  
  /**
   * Validate email format
   */
  function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
  }
});

// helper names updated to nfs_ where used in AJAX or handles
(function () {
    function fetchProtection(formId, cb) {
        if (typeof npfo_ajax === 'undefined') {
            if (cb) cb(null);
            return;
        }

        var data = {
            action: 'npfo_get_form_protection',
            form_id: formId
        };

        jQuery.post(npfo_ajax.ajax_url, data, function (resp) {
            if (cb) cb(resp);
        });
    }

    // ...existing code...
})();