jQuery(document).ready(function($) {
  let currentStep = 1;
  let selectedPlugin = '';
  let selectedFormId = '';
  let selectedFormTitle = '';
  let editingProtectionId = null;
  let customFormFields = [];
  let blockLinks = true;

  // Ensure modal is hidden immediately on page load (before any other code runs)
  // Only hide modal if it exists (for protected-forms page)
  if ( $('#edit-protection-modal').length ) {
    $('#edit-protection-modal').addClass('asm-hidden').css('display', 'none');
  }

  // Function to update dashboard statistics
  function updateDashboardStats() {
    if (typeof npfo_ajax !== 'undefined' && npfo_ajax.dashboard_stats && Object.keys(npfo_ajax.dashboard_stats).length > 0) {
      const stats = npfo_ajax.dashboard_stats;
      $('#protected-forms-count').text(stats.protected_forms_count || 0);
      $('#protected-fields-count').text(stats.protected_fields_count || 0);
    }
  }

  // Update dashboard stats if on dashboard page
  updateDashboardStats();

  // Ensure step 1 is visible on initial load (only hide steps that should be hidden)
  // Step-1 doesn't have asm-hidden class in template, so it will be visible by default
  // Step-2 and Step-3 have asm-hidden class in template, so they will be hidden by default
  // Just ensure step-1 doesn't have asm-hidden class
  $('#step-1').removeClass('asm-hidden');
  currentStep = 1;

  // Step navigation
  $('.asm-plugin-card').on('click', function(e) {
    // Don't trigger card click if clicking on install button
    if ($(e.target).closest('.asm-install-button').length) {
      return;
    }
    
    // Check if this is a premium card - completely disable click for premium cards
    if ($(this).hasClass('asm-premium-card')) {
      // Show premium upgrade notice but don't allow any action
      showPremiumUpgradeNotice($(this).data('plugin'));
      return false; // Completely prevent any action
    }
    
    // Check if plugin is disabled
    if ($(this).hasClass('asm-plugin-disabled')) {
      return false; // Don't allow selection of disabled plugins
    }
    
    // Check if this is a premium plugin and user doesn't have premium access
    if ($(this).hasClass('asm-premium-card') && !isPremiumUser()) {
      showPremiumUpgradeNotice($(this).data('plugin'));
      return false;
    }
    
    // Custom form functionality removed - coming soon
    // if ($(this).data('plugin') === 'custom') {
    //   // Handle custom form setup
    //   showCustomFormSetup();
    //   return;
    // }

    $('.asm-plugin-card').removeClass('selected');
    $(this).addClass('selected');
    selectedPlugin = $(this).data('plugin');

    setTimeout(() => {
      loadForms(selectedPlugin);
      showStep(2);
    }, 300);
  });

  $('#back-to-step-1').on('click', function() {
    showStep(1);
    $('.asm-plugin-card').removeClass('selected');
    selectedPlugin = '';
  });

  $('#continue-to-step-3').on('click', function() {
    const formSelect = $('#asm-form-select');
    selectedFormId = formSelect.val();
    selectedFormTitle = formSelect.find('option:selected').text();
    
    if (selectedFormId) {
      loadFormFields(selectedPlugin, selectedFormId);
      showStep(3);
    }
  });

  $('#back-to-step-2').on('click', function() {
    // Custom form functionality removed - coming soon
    // if (selectedPlugin === 'custom') {
    //   showStep(1);
    // } else {
      showStep(2);
    // }
  });

  $('#asm-form-select').on('change', function() {
    const hasValue = $(this).val() !== '';
    $('#continue-to-step-3').prop('disabled', !hasValue);
    
    // Check if selected form already has protection
    if (hasValue && selectedPlugin) {
      const formId = $(this).val();
      // Quick check for existing protection
      $.post(
        npfo_ajax.ajax_url,
        {
          action: 'npfo_get_form_protection',
          form_plugin: selectedPlugin,
          form_id: formId,
          nonce: npfo_ajax.nonce
        },
        function(response) {
          // Remove any existing notice
          $('#asm-form-select').siblings('.asm-existing-form-notice').remove();
          
          if (response && response.success && response.data && response.data.field_protections) {
            const protectedFields = Object.keys(response.data.field_protections);
            if (protectedFields.length > 0) {
              const notice = $('<div class="asm-existing-form-notice asm-notice asm-notice-info" style="margin-top: 10px;"><strong>Note:</strong> This form already has protection configured for ' + protectedFields.length + ' field(s). You can modify it in Step 3.</div>');
              $('#asm-form-select').after(notice);
            }
          }
        }
      ).fail(function() {
        // Silently fail - not critical
      });
    } else {
      // Remove notice if no form selected
      $('#asm-form-select').siblings('.asm-existing-form-notice').remove();
    }
  });

  // Block links toggle
  $(document).on('change', '#block-links-toggle', function() {
    blockLinks = $(this).prop('checked');
  });

  // Save protection
  $('#save-protection').on('click', function() {
    saveProtection();
  });

  // Protected forms page functionality - use event delegation for dynamic content
  $(document).on('click', '.edit-protection', function(e) {
    e.preventDefault();
    e.stopPropagation();
    const protectionId = $(this).data('id');
    if (protectionId) {
      editProtection(protectionId);
    } else {
      showNotice('error', 'Invalid protection ID.');
    }
  });

  $(document).on('click', '.delete-protection', function(e) {
    e.preventDefault();
    e.stopPropagation();
    const protectionId = $(this).data('id');
    if (protectionId && confirm('Are you sure you want to delete this protection?')) {
      deleteProtection(protectionId);
    }
  });

  // Modal functionality
  $('.asm-modal-close, #cancel-edit').on('click', function() {
    $('#edit-protection-modal').addClass('asm-hidden');
  });

  $('#update-protection').on('click', function() {
    updateProtection();
  });


  // Custom form functionality removed - coming soon
  // $(document).on('click', '#add-custom-field', function() {
  //   addCustomFormField();
  // });

  // $(document).on('click', '#save-custom-form', function() {
  //   saveCustomForm();
  // });

  // Handle save button for custom forms
  $(document).on('click', '#save-protection', function() {
    // if (selectedPlugin === 'custom') {
    //   saveCustomForm();
    // } else {
      saveProtection();
    // }
  });

  $(document).on('click', '.remove-custom-field', function() {
    $(this).closest('.custom-field-row').remove();
  });

  // Install button functionality
  $(document).on('click', '.asm-install-button', function(e) {
    // Stop event propagation to prevent card click
    e.stopPropagation();
    
    // Prevent action if button is disabled
    if ($(this).prop('disabled') || $(this).hasClass('asm-button-disabled')) {
      e.preventDefault();
      return false;
    }
    
    const pluginName = $(this).data('plugin');
    const pluginFile = $(this).data('plugin-file');
    const $card = $(this).closest('.asm-plugin-card');
    
    // If it's a premium card, show premium notice instead
    if ($card.hasClass('asm-premium-card')) {
      showPremiumUpgradeNotice(pluginName);
      return false;
    }
    
    // Show install message
    showNotice('info', 'Please install ' + pluginName + ' plugin from the WordPress Plugins page to use this form plugin.');
    
    // Open WordPress plugins page in new tab
    const pluginsUrl = npfo_ajax.ajax_url.replace('admin-ajax.php', 'plugin-install.php');
    window.open(pluginsUrl, '_blank');
  });

  // Functions
  function showStep(step) {
    $('.asm-step').addClass('asm-hidden');
    $('#step-' + step).removeClass('asm-hidden');
    currentStep = step;
  }
  
  // Check if user has premium access
  function isPremiumUser() {
    // Check premium status from localized data
    if (typeof npfo_ajax !== 'undefined' && npfo_ajax.is_premium) {
      return true;
    }
    return false;
  }
  
  // Show premium upgrade notice
  function showPremiumUpgradeNotice(pluginName) {
    const pluginNames = {
      'gravity-forms': 'Gravity Forms',
      'wpforms': 'WPForms',
      'ninja-forms': 'Ninja Forms',
      'forminator': 'Forminator',
      'fluent-forms': 'Fluent Forms',
      'custom': 'Custom Forms'
    };
    
    const displayName = pluginNames[pluginName] || pluginName;
    
    const upgradeNotice = `
      <div class="asm-premium-notice">
        <h3><span class="dashicons dashicons-star-filled"></span> Premium Feature</h3>
        <p><strong>${displayName}</strong> protection is available in the premium version.</p>
        <p>Upgrade now to protect all your forms with advanced spam filtering!</p>
        <a href="https://needsplugin.com/np-formshield/" target="_blank" class="asm-upgrade-btn">
          <span class="dashicons dashicons-cart"></span> Upgrade to Premium
        </a>
      </div>
    `;
    
    // Remove existing notices
    $('.asm-premium-notice').remove();
    
    // Add upgrade notice
    $('.asm-content').prepend(upgradeNotice);
    
    // Scroll to notice
    $('html, body').animate({
      scrollTop: $('.asm-premium-notice').offset().top - 100
    }, 500);
  }

  function loadForms(plugin) {
    // Check if required variables exist
    if (typeof npfo_ajax === 'undefined' || !npfo_ajax.ajax_url || !npfo_ajax.nonce) {
      showNotice('error', 'Plugin configuration error. Please refresh the page.');
      return;
    }
    
    $('#forms-loading').removeClass('asm-hidden');
    $('#asm-form-select').html('<option value="">Loading...</option>').prop('disabled', true);

    $.post(
      npfo_ajax.ajax_url,
      {
        action: 'npfo_get_forms',
        plugin: plugin,
        nonce: npfo_ajax.nonce
      },
      function(response) {
        $('#forms-loading').addClass('asm-hidden');
        
        if (response && response.success && response.data && Array.isArray(response.data) && response.data.length > 0) {
          let options = '<option value="">Select a form...</option>';
          response.data.forEach(function(form) {
            options += `<option value="${form.id}">${form.title}</option>`;
          });
          
          $('#asm-form-select').html(options).prop('disabled', false);
        } else if (response.success === false && response.data && response.data.includes('premium access')) {
          // Show premium upgrade notice
          showPremiumUpgradeNotice(plugin);
          $('#asm-form-select').html('<option value="">Premium Feature</option>');
        } else {
          $('#asm-form-select').html('<option value="">No forms found</option>');
          showNotice('error', 'Failed to load forms. Please check if the plugin is installed and has forms.');
        }
      }
    ).fail(function(xhr, status, error) {
      $('#forms-loading').addClass('asm-hidden');
      $('#asm-form-select').html('<option value="">Error loading forms</option>');
      
      // Log error for debugging but show user-friendly message
      if (typeof console !== 'undefined' && console.error) {
        console.error('Failed to load forms:', status, error);
      }
      
      showNotice('error', 'Failed to load forms. Please check if the form plugin is installed and try again.');
    });
  }

  function loadFormFields(plugin, formId) {
    $('#form-fields-container').html('<div class="asm-loading"><span class="spinner"></span> Loading form fields...</div>');
    
    // First, check if this form already has protection configured
    $.post(
      npfo_ajax.ajax_url,
      {
        action: 'npfo_get_form_protection',
        form_plugin: plugin,
        form_id: formId,
        nonce: npfo_ajax.nonce
      },
      function(protectionResponse) {
        console.log('Existing protection response:', protectionResponse);
        
        // Store existing protection data
        let existingProtection = null;
        if (protectionResponse && protectionResponse.success && protectionResponse.data) {
          existingProtection = protectionResponse.data;
        }
        
        // Then load form fields
        $.post(
          npfo_ajax.ajax_url,
          {
            action: 'npfo_get_form_fields',
            plugin: plugin,
            form_id: formId,
            nonce: npfo_ajax.nonce
          },
          function(response) {
            console.log('Form fields response:', response); // Debug log
            
            if (response.success && response.data && response.data.length > 0) {
              // Pass existing protection to render function
              renderFormFields(response.data, existingProtection);
            } else {
              // More detailed error message
              let errorMsg = 'No fields found for this form.';
              if (plugin === 'fluent-forms') {
                errorMsg += ' Please ensure the Fluent Form has input fields configured and is published.';
              } else {
                errorMsg += ' Please ensure the form has fields configured.';
              }
              
              $('#form-fields-container').html('<p class="asm-notice asm-notice-warning">' + errorMsg + '</p>');
              showNotice('warning', errorMsg);
            }
          }
        ).fail(function(xhr, status, error) {
          console.error('AJAX Error:', xhr.responseText, status, error); // Debug log
          $('#form-fields-container').html('<p class="asm-notice asm-notice-error">Error loading fields. Please try again.</p>');
          showNotice('error', 'Failed to load form fields. Please try again.');
        });
      }
    ).fail(function(xhr, status, error) {
      console.log('Protection check failed, loading fields without existing protection:', xhr.responseText);
      // If protection check fails, just load form fields normally
      $.post(
        npfo_ajax.ajax_url,
        {
          action: 'npfo_get_form_fields',
          plugin: plugin,
          form_id: formId,
          nonce: npfo_ajax.nonce
        },
        function(response) {
          if (response.success && response.data && response.data.length > 0) {
            renderFormFields(response.data, null);
          } else {
            $('#form-fields-container').html('<p class="asm-notice asm-notice-warning">No fields found for this form.</p>');
          }
        }
      );
    });
  }

  function renderFormFields(fields, existingProtection = null) {
    let html = '';
    
    // Add the block links toggle at the top
    const blockLinksChecked = existingProtection ? (existingProtection.block_links == 1) : true;
    html += `
      <div class="asm-global-settings">
        <div class="asm-toggle-container">
          <label class="asm-toggle-switch">
            <input type="checkbox" id="block-links-toggle" ${blockLinksChecked ? 'checked' : ''}>
            <span class="asm-toggle-slider"></span>
          </label>
          <span class="asm-toggle-label">Block links in all fields (except email)</span>
        </div>
        <div class="asm-help-text">
          When enabled, submissions containing URLs will be blocked in all fields except email fields.
        </div>
      </div>
    `;
    
    // Show existing protection notice if found
    if (existingProtection && existingProtection.field_protections) {
      const protectedFields = Object.keys(existingProtection.field_protections);
      html += `
        <div class="asm-existing-protection-notice">
          <div class="asm-notice asm-notice-info">
            <strong>Existing Protection Found:</strong> This form already has protection configured for ${protectedFields.length} field(s): ${protectedFields.join(', ')}. 
            You can modify existing protections or add new ones below.
          </div>
        </div>
      `;
    }
    
    if (!fields || fields.length === 0) {
      html += '<p class="asm-notice asm-notice-warning">No fields found for this form. Please ensure the form has fields configured.</p>';
    } else {
      html += '<div class="asm-fields-container">';
      fields.forEach(function(field) {
        if (field && field.name) {
          // Check if this field already has protection
          // Try exact match first
          let existingFieldProtection = null;
          let matchedKey = null;
          
          if (existingProtection && existingProtection.field_protections) {
            // Try exact match
            if (existingProtection.field_protections[field.name]) {
              existingFieldProtection = existingProtection.field_protections[field.name];
              matchedKey = field.name;
            } else {
              // Try case-insensitive match
              const fieldNameLower = field.name.toLowerCase().trim();
              for (const key in existingProtection.field_protections) {
                if (key.toLowerCase().trim() === fieldNameLower) {
                  existingFieldProtection = existingProtection.field_protections[key];
                  matchedKey = key;
                  break;
                }
              }
              
              // Try partial match if still not found
              if (!existingFieldProtection) {
                for (const key in existingProtection.field_protections) {
                  const keyLower = key.toLowerCase().trim();
                  const fieldNameLower = field.name.toLowerCase().trim();
                  // Check if field name contains key or key contains field name
                  if (fieldNameLower.includes(keyLower) || keyLower.includes(fieldNameLower)) {
                    existingFieldProtection = existingProtection.field_protections[key];
                    matchedKey = key;
                    break;
                  }
                }
              }
            }
          }
          
          let protectionType = 'blacklist';
          let protectionWords = '';
          let isExistingField = false;
          
          if (existingFieldProtection) {
            protectionType = existingFieldProtection.type || 'blacklist';
            protectionWords = existingFieldProtection.words || '';
            isExistingField = true;
          }
          
          html += `
            <div class="asm-field-protection ${isExistingField ? 'asm-existing-field' : ''}" data-field="${field.name}">
              <h4>${field.name} (${field.type || 'text'}) ${isExistingField ? '<span class="asm-protected-badge">Protected</span>' : ''}</h4>
              ${isExistingField ? '<div class="asm-existing-protection-msg">This field already has protection configured. You can edit it again below.</div>' : ''}
              <div class="asm-protection-type">
                <div class="asm-radio-group">
                  <input type="radio" id="blacklist_${field.name}" name="protection_type_${field.name}" value="blacklist" ${protectionType === 'blacklist' ? 'checked' : ''}>
                  <label for="blacklist_${field.name}">Blacklist</label>
                </div>
                <div class="asm-radio-group">
                  <input type="radio" id="whitelist_${field.name}" name="protection_type_${field.name}" value="whitelist" ${protectionType === 'whitelist' ? 'checked' : ''}>
                  <label for="whitelist_${field.name}">Whitelist</label>
                </div>
              </div>
              <textarea class="asm-textarea" placeholder="Enter words/phrases separated by commas..." data-field="${field.name}">${protectionWords}</textarea>
              <div class="asm-help-text">
                <strong>Blacklist:</strong> Block submissions containing these words<br>
                <strong>Whitelist:</strong> Only allow submissions containing these words
                ${isExistingField ? '<br><em>This field already has protection configured.</em>' : ''}
              </div>
            </div>
          `;
        }
      });
      html += '</div>';
    }
    
    $('#form-fields-container').html(html);
  }

  function saveProtection() {
    const fieldProtections = {};
    
    $('.asm-field-protection').each(function() {
      const fieldName = $(this).data('field');
      const protectionType = $(this).find('input[type="radio"]:checked').val();
      const words = $(this).find('.asm-textarea').val().trim();
      
      if (words) {
        fieldProtections[fieldName] = {
          type: protectionType,
          words: words
        };
      }
    });
    
    if (Object.keys(fieldProtections).length === 0) {
      showNotice('warning', 'Please configure at least one field protection.');
      return;
    }
    
    showLoadingOverlay();
    
    // Check if this is an update to existing protection
    const isUpdate = $('.asm-existing-field').length > 0;
    const actionText = isUpdate ? 'updated' : 'saved';
    
    $.post(
      npfo_ajax.ajax_url,
      {
        action: 'npfo_save_protection',
        form_plugin: selectedPlugin,
        form_id: selectedFormId,
        form_title: selectedFormTitle,
        field_protections: JSON.stringify(fieldProtections),
        block_links: blockLinks,
        nonce: npfo_ajax.nonce
      },
      function(response) {
        hideLoadingOverlay();
        
        if (response.success) {
          showNotice('success', `Protection ${actionText} successfully!`);
          setTimeout(() => {
            window.location.href = 'admin.php?page=npfo-protected-forms';
          }, 1500);
        } else {
          showNotice('error', response.data || `Failed to ${actionText} protection.`);
        }
      }
    ).fail(function(xhr) {
      hideLoadingOverlay();
      showNotice('error', `Failed to ${actionText} protection. ` + (xhr.responseJSON ? xhr.responseJSON.data : 'Please try again.'));
    });
  }

  function editProtection(protectionId) {
    // Ensure protectionId is a valid number
    const id = parseInt(protectionId);
    if (isNaN(id) || id <= 0) {
      showNotice('error', 'Invalid protection ID.');
      return;
    }
    
    editingProtectionId = id;
    console.log('Setting editingProtectionId to:', editingProtectionId);
    console.log('Type of editingProtectionId:', typeof editingProtectionId);
    showLoadingOverlay();
    
    $.post(
      npfo_ajax.ajax_url,
      {
        action: 'npfo_get_protection',
        id: protectionId,
        nonce: npfo_ajax.nonce
      },
      function(response) {
        hideLoadingOverlay();
        if (response.success) {
          // Store current protection data for "Add More Protection" functionality
          window.currentEditingProtection = response.data;
          renderEditForm(response.data);
          
          // Show modal - ensure it exists and remove hidden class
          const $modal = $('#edit-protection-modal');
          if ($modal.length) {
            $modal.removeClass('asm-hidden').css('display', 'flex');
          } else {
            console.error('Edit protection modal not found in DOM');
            showNotice('error', 'Modal element not found. Please refresh the page.');
          }
        } else {
          showNotice('error', response.data || 'Failed to load protection data.');
        }
      }
    ).fail(function() {
      hideLoadingOverlay();
      showNotice('error', 'Failed to load protection data. Please try again.');
    });
  }

  function renderEditForm(protection) {
    let html = '';
    
    // Add the block links toggle at the top
    html += `
      <div class="asm-global-settings">
        <div class="asm-toggle-container">
          <label class="asm-toggle-switch">
            <input type="checkbox" id="edit-block-links-toggle" ${protection.block_links == 1 ? 'checked' : ''}>
            <span class="asm-toggle-slider"></span>
          </label>
          <span class="asm-toggle-label">Block links in all fields (except email)</span>
        </div>
        <div class="asm-help-text">
          When enabled, submissions containing URLs will be blocked in all fields except email fields.
        </div>
      </div>
    `;
    
    if (protection && protection.field_protections) {
      Object.keys(protection.field_protections).forEach(function(fieldName) {
        const fieldProtection = protection.field_protections[fieldName];
        html += `
          <div class="asm-field-protection" data-field="${fieldName}">
            <h4>${fieldName}</h4>
            <div class="asm-protection-type">
              <div class="asm-radio-group">
                <input type="radio" id="edit_blacklist_${fieldName}" name="edit_protection_type_${fieldName}" value="blacklist" ${fieldProtection.type === 'blacklist' ? 'checked' : ''}>
                <label for="edit_blacklist_${fieldName}">Blacklist</label>
              </div>
              <div class="asm-radio-group">
                <input type="radio" id="edit_whitelist_${fieldName}" name="edit_protection_type_${fieldName}" value="whitelist" ${fieldProtection.type === 'whitelist' ? 'checked' : ''}>
                <label for="edit_whitelist_${fieldName}">Whitelist</label>
              </div>
            </div>
            <textarea class="asm-textarea" placeholder="Enter words/phrases separated by commas..." data-field="${fieldName}" >${fieldProtection.words}</textarea>
            <div class="asm-help-text">
              <strong>Blacklist:</strong> Block submissions containing these words<br>
              <strong>Whitelist:</strong> Only allow submissions containing these words
            </div>
          </div>
        `;
      });
    }
    
    $('#edit-form-fields').html(html || '<p>No field protections found.</p>');
    
    // Save the current block links value
    blockLinks = $('#edit-block-links-toggle').prop('checked');
    
    // Add event handler for the toggle
    $('#edit-block-links-toggle').on('change', function() {
      blockLinks = $(this).prop('checked');
    });
  }

  function updateProtection() {
    const fieldProtections = {};
    const blockLinks = $('#edit-block-links-toggle').prop('checked');
    
    $('#edit-form-fields .asm-field-protection').each(function() {
      const fieldName = $(this).data('field');
      const protectionType = $(this).find('input[type="radio"]:checked').val();
      const words = $(this).find('.asm-textarea').val().trim();
      
      if (words) {
        fieldProtections[fieldName] = {
          type: protectionType,
          words: words
        };
      }
    });
    
    if (Object.keys(fieldProtections).length === 0) {
      showNotice('warning', 'Please configure at least one field protection.');
      return;
    }
    
    // Debug: Check if editingProtectionId is set
    console.log('Editing Protection ID:', editingProtectionId);
    console.log('Field Protections:', fieldProtections);
    console.log('Block Links:', blockLinks);
    
    if (!editingProtectionId || editingProtectionId === 'undefined' || editingProtectionId === null) {
      showNotice('error', 'No protection ID found. Please try again.');
      return;
    }
    
    // Ensure we're updating, not creating
    if (typeof editingProtectionId !== 'number' && isNaN(parseInt(editingProtectionId))) {
      showNotice('error', 'Invalid protection ID. Please try again.');
      return;
    }
    
    // Show loading overlay on modal
    showModalLoadingOverlay();
    
    $.post(
      npfo_ajax.ajax_url,
      {
        action: 'npfo_save_protection',
        id: editingProtectionId,
        field_protections: JSON.stringify(fieldProtections),
        block_links: blockLinks,
        nonce: npfo_ajax.nonce
      },
      function(response) {
        hideModalLoadingOverlay();
        
        if (response.success) {
          showNotice('success', 'Protection updated successfully!');
          $('#edit-protection-modal').addClass('asm-hidden');
          setTimeout(() => {
            location.reload();
          }, 1000);
        } else {
          showNotice('error', response.data || 'Failed to update protection.');
        }
      }
    ).fail(function() {
      hideModalLoadingOverlay();
      showNotice('error', 'Failed to update protection. Please try again.');
    });
  }


  function deleteProtection(protectionId) {
    showLoadingOverlay();
    
    $.post(
      npfo_ajax.ajax_url,
      {
        action: 'npfo_delete_protection',
        id: protectionId,
        nonce: npfo_ajax.nonce
      },
      function(response) {
        hideLoadingOverlay();
        
        if (response.success) {
          showNotice('success', 'Protection deleted successfully!');
          setTimeout(() => {
            location.reload();
          }, 1000);
        } else {
          showNotice('error', response.data || 'Failed to delete protection.');
        }
      }
    ).fail(function() {
      hideLoadingOverlay();
      showNotice('error', 'Failed to delete protection. Please try again.');
    });
  }

  // Custom form functionality removed - coming soon
  // function showCustomFormSetup() {
  //   selectedPlugin = 'custom';
  //   selectedFormId = 'custom-form-' + Date.now();
  //   selectedFormTitle = 'Custom Form';
  //   
  //   // Show custom form setup UI
  //   $('#form-fields-container').html(`
  //     <div class="asm-custom-form-setup">
  //       <h4>Custom Form Setup</h4>
  //       <p>Add fields for your custom form:</p>
  //       
  //       <div class="asm-global-settings">
  //         <div class="asm-toggle-container">
  //           <label class="asm-toggle-switch">
  //             <input type="checkbox" id="block-links-toggle" checked>
  //             <span class="asm-toggle-slider"></span>
  //           </label>
  //           <span class="asm-toggle-label">Block links in all fields (except email)</span>
  //         </div>
  //         <div class="asm-help-text">
  //           When enabled, submissions containing URLs will be blocked in all fields except email fields.
  //         </div>
  //       </div>
  //       
  //       <div class="custom-fields-container">
  //         <!-- Fields will be added here -->
  //       </div>
  //       
  //       <div class="asm-custom-form-actions">
  //         <button type="button" class="asm-btn asm-btn-secondary" id="add-custom-field">
  //           <span class="dashicons dashicons-plus"></span> Add Field
  //         </button>
  //       </div>
  //       
  //     </div>
  //   `);
  //   
  //   // Add initial field
  //   addCustomFormField();
  //   
  //   // Show step 3
  //   showStep(3);
  // }

  // Custom form functionality removed - coming soon
  // function showCustomFormSetupWithExistingProtection(formId, formTitle) {
  //   // Load existing protection data for this custom form
  //   $.post(
  //     npfo_ajax.ajax_url,
  //     {
  //       action: 'npfo_get_form_protection',
  //       form_plugin: 'custom',
  //       form_id: formId,
  //       nonce: npfo_ajax.nonce
  //     },
  //     function(response) {
  //       if (response.success && response.data) {
  //         const existingProtection = response.data;
  //         const fieldProtections = existingProtection.field_protections || {};
  //         
  //         // Show custom form setup UI with existing protection
  //         $('#form-fields-container').html(`
  //           <div class="asm-custom-form-setup">
  //             <h4>Edit Custom Form Protection</h4>
  //             <p>Modify existing protection or add new fields for your custom form:</p>
  //             
  //             <div class="asm-global-settings">
  //               <div class="asm-toggle-container">
  //                 <label class="asm-toggle-switch">
  //                   <input type="checkbox" id="block-links-toggle" ${existingProtection.block_links ? 'checked' : ''}>
  //                   <span class="asm-toggle-slider"></span>
  //                 </label>
  //                 <span class="asm-toggle-label">Block links in all fields (except email)</span>
  //               </div>
  //               <div class="asm-help-text">
  //                 When enabled, submissions containing URLs will be blocked in all fields except email fields.
  //               </div>
  //             </div>
  //             
  //             <div class="custom-fields-container">
  //               <!-- Existing fields will be loaded here -->
  //             </div>
  //             
  //             <div class="asm-custom-form-actions">
  //               <button type="button" class="asm-btn asm-btn-secondary" id="add-custom-field">
  //                 <span class="dashicons dashicons-plus"></span> Add New Field
  //               </button>
  //             </div>
  //             
  //           </div>
  //         `);
  //         
  //         // Load existing fields
  //         loadExistingCustomFields(fieldProtections);
  //         
  //         // Show step 3
  //         showStep(3);
  //       } else {
  //         // Fallback to regular custom form setup
  //         showCustomFormSetup();
  //       }
  //     }
  //   ).fail(function() {
  //     // Fallback to regular custom form setup
  //     showCustomFormSetup();
  //   });
  // }

  // Custom form functionality removed - coming soon
  // function loadExistingCustomFields(fieldProtections) {
  //   const fieldNames = Object.keys(fieldProtections);
  //   
  //   if (fieldNames.length === 0) {
  //     // No existing fields, add one new field
  //     addCustomFormField();
  //     return;
  //   }
  //   
  //   // Load existing fields with their protection settings
  //   fieldNames.forEach((fieldName, index) => {
  //     const protection = fieldProtections[fieldName];
  //     const fieldId = 'field_' + Date.now() + '_' + index;
  //     
  //     const fieldHtml = `
  //       <div class="custom-field-row" data-field-id="${fieldId}">
  //         <div class="custom-field-inputs">
  //           <input type="text" class="custom-field-name" value="${fieldName}" placeholder="Field Name (e.g. Name, Email, Message)" required>
  //           <select class="custom-field-type">
  //             <option value="text" ${protection.field_type === 'text' ? 'selected' : ''}>Text</option>
  //             <option value="email" ${protection.field_type === 'email' ? 'selected' : ''}>Email</option>
  //             <option value="textarea" ${protection.field_type === 'textarea' ? 'selected' : ''}>Textarea</option>
  //             <option value="number" ${protection.field_type === 'number' ? 'selected' : ''}>Number</option>
  //             <option value="tel" ${protection.field_type === 'tel' ? 'selected' : ''}>Telephone</option>
  //           </select>
  //           <button type="button" class="asm-btn asm-btn-small asm-btn-danger remove-custom-field">
  //             <span class="dashicons dashicons-trash"></span>
  //           </button>
  //         </div>
  //         <div class="asm-protection-type">
  //           <div class="asm-radio-group">
  //             <input type="radio" id="blacklist_${fieldId}" name="protection_type_${fieldId}" value="blacklist" ${protection.type === 'blacklist' ? 'checked' : ''}>
  //             <label for="blacklist_${fieldId}">Blacklist</label>
  //           </div>
  //           <div class="asm-radio-group">
  //             <input type="radio" id="whitelist_${fieldId}" name="protection_type_${fieldId}" value="whitelist" ${protection.type === 'whitelist' ? 'checked' : ''}>
  //             <label for="whitelist_${fieldId}">Whitelist</label>
  //           </div>
  //         </div>
  //         <textarea class="asm-textarea" placeholder="Enter words/phrases separated by commas..." data-field="${fieldName}">${protection.words || ''}</textarea>
  //         <div class="asm-help-text">
  //           <strong>Blacklist:</strong> Block submissions containing these words<br>
  //           <strong>Whitelist:</strong> Only allow submissions containing these words
  //           <br><em>This field already has protection configured.</em>
  //         </div>
  //       </div>
  //     `;
  //     
  //     $('.custom-fields-container').append(fieldHtml);
  //   });
  // }

  // Custom form functionality removed - coming soon
  // function addCustomFormField() {
  //   const fieldId = 'field_' + Date.now();
  //   const fieldHtml = `
  //     <div class="custom-field-row" data-field-id="${fieldId}">
  //       <div class="custom-field-inputs">
  //         <input type="text" class="custom-field-name" placeholder="Field Name (e.g. Name, Email, Message)" required>
  //         <select class="custom-field-type">
  //           <option value="text">Text</option>
  //           <option value="email">Email</option>
  //           <option value="textarea">Textarea</option>
  //           <option value="number">Number</option>
  //           <option value="tel">Telephone</option>
  //         </select>
  //         <button type="button" class="asm-btn asm-btn-small asm-btn-danger remove-custom-field">
  //           <span class="dashicons dashicons-trash"></span>
  //         </button>
  //       </div>
  //       <div class="asm-protection-type">
  //         <div class="asm-radio-group">
  //           <input type="radio" id="blacklist_${fieldId}" name="protection_type_${fieldId}" value="blacklist" checked>
  //           <label for="blacklist_${fieldId}">Blacklist</label>
  //         </div>
  //         <div class="asm-radio-group">
  //           <input type="radio" id="whitelist_${fieldId}" name="protection_type_${fieldId}" value="whitelist">
  //           <label for="whitelist_${fieldId}">Whitelist</label>
  //         </div>
  //       </div>
  //       <textarea class="asm-textarea" placeholder="Enter words/phrases separated by commas..."></textarea>
  //     </div>
  //   `;
  //   
  //   $('.custom-fields-container').append(fieldHtml);
  // }

  // Custom form functionality removed - coming soon
  // function saveCustomForm() {
  //   const fieldProtections = {};
  //   let hasErrors = false;
  //   
  //   $('.custom-field-row').each(function() {
  //     const fieldName = $(this).find('.custom-field-name').val().trim();
  //     const fieldType = $(this).find('.custom-field-type').val();
  //     const protectionType = $(this).find('input[type="radio"]:checked').val();
  //     const words = $(this).find('.asm-textarea').val().trim();
  //     
  //     if (!fieldName) {
  //       showNotice('error', 'Please enter a name for all fields.');
  //       hasErrors = true;
  //       return false;
  //     }
  //     
  //     if (words) {
  //       fieldProtections[fieldName] = {
  //         type: protectionType,
  //         words: words
  //       };
  //     }
  //   });
  //   
  //   if (hasErrors) {
  //     return;
  //   }
  //   
  //   if (Object.keys(fieldProtections).length === 0) {
  //     showNotice('warning', 'Please configure at least one field protection.');
  //     return;
  //   }
  //   
  //   // Get block links value
  //   const blockLinks = $('#block-links-toggle').prop('checked');
  //   
  //   // Save custom form setup
  //   showLoadingOverlay();
  //   
  //   $.post(
  //     npfo_ajax.ajax_url,
  //     {
  //       action: 'npfo_save_protection',
  //       form_plugin: 'custom',
  //       form_id: selectedFormId,
  //       form_title: 'Custom Form',
  //       field_protections: JSON.stringify(fieldProtections),
  //       block_links: blockLinks,
  //       nonce: npfo_ajax.nonce
  //     },
  //     function(response) {
  //       hideLoadingOverlay();
  //       
  //       if (response.success) {
  //         showNotice('success', 'Custom form protection saved successfully!');
  //         setTimeout(() => {
  //           window.location.href = 'admin.php?page=nsf-protected-forms';
  //         }, 1500);
  //       } else {
  //         showNotice('error', response.data || 'Failed to save protection.');
  //       }
  //     }
  //   ).fail(function() {
  //     hideLoadingOverlay();
  //     showNotice('error', 'Failed to save protection. Please try again.');
  //   });
  // }

  function showNotice(type, message) {
    const noticeClass = `asm-notice asm-notice-${type}`;
    const notice = `<div class="${noticeClass}">${message}</div>`;
    
    // Remove existing notices
    $('.asm-notice').remove();
    
    // Add new notice at the top of the dashboard
    $('.asm-content').prepend(notice);
    
    // Auto-hide after 5 seconds
    setTimeout(() => {
      $('.asm-notice').fadeOut();
    }, 5000);
  }

  function showLoadingOverlay() {
    $('body').append('<div class="asm-loading-overlay"><div class="asm-loading-spinner"></div></div>');
  }

  function hideLoadingOverlay() {
    $('.asm-loading-overlay').remove();
  }

  // Show loading overlay specifically on modal footer
  function showModalLoadingOverlay() {
    const $modal = $('#edit-protection-modal');
    const $modalFooter = $modal.find('.asm-modal-footer');
    if ($modalFooter.length) {
      // Remove any existing modal loading overlay
      $modalFooter.find('.asm-modal-loading-overlay').remove();
      // Add loading overlay inside modal footer
      $modalFooter.append('<div class="asm-modal-loading-overlay"><div class="asm-loading-spinner"></div></div>');
    }
  }

  // Hide loading overlay from modal
  function hideModalLoadingOverlay() {
    $('#edit-protection-modal .asm-modal-loading-overlay').remove();
  }

  // Check plugin statuses on page load
  checkPluginStatuses();
  
  // Handle URL parameters for pre-selecting forms
  handleUrlParameters();
  
  function handleUrlParameters() {
    const urlParams = new URLSearchParams(window.location.search);
    const formPlugin = urlParams.get('form_plugin');
    const formId = urlParams.get('form_id');
    const formTitle = urlParams.get('form_title');
    const step = urlParams.get('step');
    const editMode = urlParams.get('edit_mode');
    
    if (formPlugin && formId && formTitle) {
      // Custom form functionality removed - coming soon
      // Handle custom forms with edit mode
      // if (formPlugin === 'custom' && editMode === 'true') {
      //   // Set the selected plugin and form data
      //   selectedPlugin = formPlugin;
      //   selectedFormId = formId;
      //   selectedFormTitle = formTitle;
      //   
      //   // Show custom form setup with existing protection
      //   showCustomFormSetupWithExistingProtection(formId, formTitle);
      //   showNotice('info', `Custom Form "${decodeURIComponent(formTitle)}" was pre-selected for adding more protection.`);
      //   return;
      // }
      
      // Pre-select the plugin and form for regular forms
      setTimeout(() => {
        // Select the plugin
        $(`.asm-plugin-card[data-plugin="${formPlugin}"]`).click();
        
        // Wait for forms to load, then select the form
        setTimeout(() => {
          $(`#asm-form-select option[value="${formId}"]`).prop('selected', true);
          $('#asm-form-select').trigger('change');
          
          // If step=3 is specified, go directly to Step 3
          if (step === '3') {
            setTimeout(() => {
              showStep(3);
              loadFormFields(formPlugin, formId);
              showNotice('info', `Form "${decodeURIComponent(formTitle)}" was pre-selected for adding more protection.`);
            }, 500);
          } else {
            // Show a notice that this form was pre-selected for adding more protection
            showNotice('info', `Form "${decodeURIComponent(formTitle)}" was pre-selected for adding more protection.`);
          }
        }, 1000);
      }, 500);
    }
  }
  
  function checkPluginStatuses() {
    $.post(
      npfo_ajax.ajax_url,
      {
        action: 'npfo_check_plugin_status',
        nonce: npfo_ajax.nonce
      },
      function(response) {
        if (response.success) {
          const pluginStatuses = response.data;
          
          Object.keys(pluginStatuses).forEach(function(pluginKey) {
            const plugin = pluginStatuses[pluginKey];
            const $card = $(`.asm-plugin-card[data-plugin="${pluginKey}"]`);
            const $status = $card.find('.asm-plugin-status');
            
            if (plugin.installed) {
              $status.text('Available').removeClass('asm-status-not-installed').addClass('asm-status-available');
              $card.removeClass('asm-plugin-disabled').addClass('asm-plugin-enabled');
            } else {
              $status.text('Install Required').removeClass('asm-status-available').addClass('asm-status-not-installed');
              $card.removeClass('asm-plugin-enabled').addClass('asm-plugin-disabled');
              
              // Add install button
              if (!$card.find('.asm-install-button').length) {
                // Check if this is a premium card - if yes, disable the button
                const isPremiumCard = $card.hasClass('asm-premium-card');
                const disabledAttr = isPremiumCard ? ' disabled' : '';
                const disabledClass = isPremiumCard ? ' asm-button-disabled' : '';
                $status.after('<button class="asm-install-button' + disabledClass + '" data-plugin="' + pluginKey + '" data-plugin-file="' + plugin.plugin_file + '"' + disabledAttr + '>Install ' + plugin.name + '</button>');
              }
            }
          });
        } else {
          console.error('Failed to check plugin statuses:', response.data);
        }
      }
    ).fail(function() {
      console.error('AJAX error checking plugin statuses');
    });
  }
});
(function($){
    // ...existing code...

    function getFormProtection(formId, callback) {
        $.post(npfo_ajax.ajax_url, {
            action: 'npfo_get_form_protection',
            form_id: formId
        }, function(response) {
            if (typeof callback === 'function') {
                callback(response);
            }
        });
    }

    function saveFormProtection(data, callback) {
        $.post(npfo_ajax.ajax_url, $.extend({}, data, {
            action: 'npfo_save_form_protection'
        }), function(response) {
            if (typeof callback === 'function') {
                callback(response);
            }
        });
    }

    // ...existing code...
})(jQuery);