<?php
/*
 * Plugin Name: Chat AgentX Widget
 * Description: Embeds the Chat AgentX widget. Set a default Agent ID in Settings → Chat AgentX, or use the shortcode [chatagentx agent_id="..."].
 * Version:     1.0.1
 * Author:      Chat AgentX
 * Author URI:  https://chatagentx.com
 * License:     GPL-2.0-or-later
 * Text Domain: chatagentx-widget
 */

if (! defined('ABSPATH')) exit;

class ChatAgentX_Widget_Plugin
{

    const OPTION_KEY     = 'chatagentx_options';
    const SCRIPT_HANDLE  = 'chatagentx-widget';
    const SCRIPT_SRC     = 'https://chatagentx.com/widget.js';

    public function __construct()
    {
        // Settings
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_menu', [$this, 'add_settings_page']);

        // Enqueue (sitewide, optional)
        add_action('wp_enqueue_scripts', [$this, 'conditionally_enqueue_widget']);

        // Inject data-agent-id into the enqueued script tag
        add_filter('script_loader_tag', [$this, 'add_data_attribute_to_script'], 10, 3);

        // Shortcode
        add_shortcode('chatagentx', [$this, 'shortcode']);
    }

    /** -------- Settings -------- */

    public function register_settings()
    {
        register_setting(
            self::OPTION_KEY,
            self::OPTION_KEY,
            [
                'type'              => 'array',
                'sanitize_callback' => [$this, 'sanitize_options'],
                'default'           => [
                    'agent_id'      => '',
                    'load_sitewide' => 0,
                ],
            ]
        );

        add_settings_section(
            'chatagentx_main',
            'Chat AgentX Settings',
            function () {
                echo '<p>Set a default Agent ID and optionally load the widget on every page. You can also use the shortcode <code>[chatagentx agent_id="YOUR_ID"]</code>.</p>';
            },
            self::OPTION_KEY
        );

        add_settings_field(
            'chatagentx_agent_id',
            'Default Agent ID',
            [$this, 'field_agent_id'],
            self::OPTION_KEY,
            'chatagentx_main'
        );

        add_settings_field(
            'chatagentx_load_sitewide',
            'Load on all pages',
            [$this, 'field_load_sitewide'],
            self::OPTION_KEY,
            'chatagentx_main'
        );
    }

    public function sanitize_options($input)
    {
        $out                  = [];
        $out['agent_id']      = isset($input['agent_id']) ? sanitize_text_field($input['agent_id']) : '';
        $out['load_sitewide'] = isset($input['load_sitewide']) ? (int) (bool) $input['load_sitewide'] : 0;
        return $out;
    }

    public function field_agent_id()
    {
        $opts = get_option(self::OPTION_KEY);
        $val  = isset($opts['agent_id']) ? $opts['agent_id'] : '';
        echo '<input type="text" name="' . esc_attr(self::OPTION_KEY) . '[agent_id]" value="' . esc_attr($val) . '" class="regular-text" placeholder="e.g. ag_12345" />';
    }

    public function field_load_sitewide()
    {
        $opts    = get_option(self::OPTION_KEY);
        $checked = ! empty($opts['load_sitewide']) ? 'checked' : '';
        printf(
            '<label><input type="checkbox" name="%s[load_sitewide]" value="1" %s /> Load the widget on every page using the default Agent ID.</label>',
            esc_attr( self::OPTION_KEY ),
            checked( $opts['load_sitewide'], 1, false )
);
    }

    public function add_settings_page()
    {
        add_options_page(
            'Chat AgentX',
            'Chat AgentX',
            'manage_options',
            self::OPTION_KEY,
            [$this, 'render_settings_page']
        );
    }

    public function render_settings_page()
    {
?>
        <div class="wrap">
            <h1>Chat AgentX</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields(self::OPTION_KEY);
                do_settings_sections(self::OPTION_KEY);
                submit_button();
                ?>
            </form>
        </div>
<?php
    }

    /** -------- Front-end embed (sitewide) -------- */

    public function conditionally_enqueue_widget()
    {
        $opts = get_option(self::OPTION_KEY);
        if (empty($opts['load_sitewide'])) {
            return;
        }

        $agent_id = isset($opts['agent_id']) ? trim($opts['agent_id']) : '';
        if ($agent_id === '') {
            // Require Agent ID for sitewide loading.
            return;
        }

        // Store the agent id for this request so the filter can access it.
        $GLOBALS['chatagentx_current_agent_id'] = $agent_id;

        // Enqueue the external script; we'll add data-agent-id via the filter below.
        wp_enqueue_script(
            self::SCRIPT_HANDLE,
            self::SCRIPT_SRC,
            [],
            null,
            true // in footer
        );
    }

    /**
     * Inject data-agent-id into the script tag generated by wp_enqueue_script.
     *
     * @param string $tag    The <script> tag for the enqueued script.
     * @param string $handle The script's registered handle.
     * @param string $src    The script's source URL.
     *
     * @return string
     */
    public function add_data_attribute_to_script($tag, $handle, $src)
    {
        if ($handle !== self::SCRIPT_HANDLE) {
            return $tag;
        }

        $agent_id = isset($GLOBALS['chatagentx_current_agent_id']) ? $GLOBALS['chatagentx_current_agent_id'] : '';
        if ($agent_id === '') {
            return $tag;
        }

        // Safely inject data-agent-id before the src attribute in the existing script tag.
        $attribute = ' data-agent-id="' . esc_attr($agent_id) . '"';

        // If the tag already has a src attribute, inject before it.
        if (strpos($tag, ' src=') !== false) {
            $tag = str_replace(' src=', $attribute . ' src=', $tag);
        } else {
            // Fallback: just append the attribute before closing '>'.
            $tag = str_replace('></script>', $attribute . '></script>', $tag);
        }

        return $tag;
    }

    /** -------- Shortcode: [chatagentx agent_id="..."] -------- */

    public function shortcode($atts)
    {
        static $loaded = false;

        $opts = get_option(self::OPTION_KEY);

        $atts = shortcode_atts(
            [
                'agent_id' => isset($opts['agent_id']) ? $opts['agent_id'] : '',
            ],
            $atts,
            'chatagentx'
        );

        $agent_id = trim($atts['agent_id']);
        if ($agent_id === '') {
            return '<!-- Chat AgentX: missing agent_id -->';
        }

        // Prevent multiple inserts if the shortcode is used more than once.
        if ($loaded) {
            return '<!-- Chat AgentX: already loaded -->';
        }
        $loaded = true;

        // Store the agent id for this request so the script tag gets the correct data-agent-id.
        $GLOBALS['chatagentx_current_agent_id'] = $agent_id;

        // Enqueue the script (WordPress will output the <script> tag in the footer).
        wp_enqueue_script(
            self::SCRIPT_HANDLE,
            self::SCRIPT_SRC,
            [],
            null,
            true // in footer
        );

        // Nothing to output here; the script will be printed by WordPress.
        return '<!-- Chat AgentX: script enqueued -->';
    }
}

new ChatAgentX_Widget_Plugin();
