<?php

namespace EazyAddonsForElementor\Widgets\Renders;

use Elementor\Icons_Manager;

if (!defined('ABSPATH')) {
  exit; // Exit if accessed directly
}

/**
 * Render function for EAFE_Social_Icons_Widget
 *
 * @param array $settings The widget settings passed from the main widget class.
 */
function render_eafe_social_icons_widget($settings)
{
  $social_icon_list = $settings['social_icon_list'] ?? [];
  $view             = $settings['view'] ?? 'default';
  $shape            = $settings['shape'] ?? 'circle';
  $show_label       = $settings['show_label'] === 'yes';

  if (empty($social_icon_list)) {
    return;
  }

  $container_classes   = ['eafe-social-icons'];
  $container_classes[] = 'eafe-social-view-' . $view;
  $container_classes[] = 'eafe-social-shape-' . $shape;

  echo '<div class="' . esc_attr(implode(' ', $container_classes)) . '">';

  foreach ($social_icon_list as $index => $item) {
    $social_icon  = $item['social_icon'] ?? '';
    $social_label = $item['social_label'] ?? '';
    $link         = $item['link'] ?? [];

    if (empty($social_icon)) {
      continue;
    }

    $icon_classes   = ['eafe-social-icon'];
    $icon_classes[] = 'elementor-repeater-item-' . $item['_id'];

    // Add social platform class for automatic styling (only if custom colors are not enabled)
    $custom_colors = $item['custom_colors'] ?? 'no';
    if ($custom_colors !== 'yes') {
      $social_class = get_social_platform_class($social_icon);
      if ($social_class) {
        $icon_classes[] = 'eafe-social-' . $social_class;
      }
    }

    $link_attrs = [
      'class'      => 'eafe-social-link',
      'aria-label' => esc_attr($social_label ?: __('Social Link', 'nb-eazy-addons-for-elementor')),
    ];

    if (!empty($link['url'])) {
      $link_attrs['href'] = esc_url($link['url']);

      if (!empty($link['is_external'])) {
        $link_attrs['target'] = '_blank';
      }

      if (!empty($link['nofollow'])) {
        $link_attrs['rel'] = 'nofollow';
      }
    } else {
      $link_attrs['href']  = '#';
      $link_attrs['class'] .= ' eafe-social-no-link';
    }

    echo '<a ' . generate_attributes($link_attrs) . '>';
    echo '<span class="' . esc_attr(implode(' ', $icon_classes)) . '">';

    // Render icon
    Icons_Manager::render_icon($social_icon, ['aria-hidden' => 'true']);

    // Render label if enabled
    if ($show_label && $social_label) {
      echo '<span class="eafe-social-label">' . esc_html($social_label) . '</span>';
    }

    echo '</span>';
    echo '</a>';
  }

  echo '</div>';
}

/**
 * Generate HTML attributes string
 */
function generate_attributes($attributes)
{
  $atts = [];
  foreach ($attributes as $key => $value) {
    $atts[] = $key . '="' . $value . '"';
  }
  return implode(' ', $atts);
}

/**
 * Get social platform class for automatic styling
 */
function get_social_platform_class($icon_data)
{
  if (empty($icon_data['value'])) {
    return '';
  }

  $icon_value       = $icon_data['value'];
  $social_platforms = [
    'facebook'   => ['facebook', 'facebook-f', 'facebook-square'],
    'twitter'    => ['twitter', 'twitter-square'],
    'instagram'  => ['instagram'],
    'linkedin'   => ['linkedin', 'linkedin-in'],
    'youtube'    => ['youtube'],
    'pinterest'  => ['pinterest', 'pinterest-p'],
    'whatsapp'   => ['whatsapp'],
    'tiktok'     => ['tiktok'],
    'telegram'   => ['telegram'],
    'discord'    => ['discord'],
    'slack'      => ['slack'],
    'github'     => ['github'],
    'dribbble'   => ['dribbble'],
    'behance'    => ['behance'],
    'spotify'    => ['spotify'],
    'soundcloud' => ['soundcloud'],
    'twitch'     => ['twitch'],
    'vimeo'      => ['vimeo'],
    'reddit'     => ['reddit'],
    'skype'      => ['skype'],
    'snapchat'   => ['snapchat'],
    'email'      => ['envelope', 'envelope-o'],
    'phone'      => ['phone'],
    'website'    => ['globe'],
  ];

  foreach ($social_platforms as $platform => $keywords) {
    foreach ($keywords as $keyword) {
      if (strpos($icon_value, $keyword) !== false) {
        return $platform;
      }
    }
  }

  return 'custom';
}
