<?php

namespace EazyAddonsForElementor\Widgets\Renders;

use Elementor\Icons_Manager;

if (!defined('ABSPATH')) {
  exit; // Exit if accessed directly
}

/**
 * Render function for EAFE_News_Ticker_Widget
 *
 * @param array $settings The widget settings passed from the main widget class.
 */
function render_eafe_news_ticker_widget($settings)
{
  // Get news items based on source
  $news_items = get_news_items($settings);

  if (empty($news_items)) {
    echo '<div class="eafe-news-ticker-empty">' . esc_html__('No news items found.', 'nb-eazy-addons-for-elementor') . '</div>';
    return;
  }

  $ticker_style      = $settings['ticker_style'] ?? 'horizontal';
  $show_controls     = $settings['show_controls'] === 'yes';
  $show_progress_bar = $settings['show_progress_bar'] === 'yes' && in_array($ticker_style, ['vertical', 'fade']);

  // Prepare data attributes for JavaScript
  $data_attributes = [
    'data-ticker-style'    => $ticker_style,
    'data-animation-speed' => $settings['animation_speed'] ?? 300,
    'data-autoplay-speed'  => $settings['autoplay_speed'] ?? 3000,
    'data-pause-on-hover'  => $settings['pause_on_hover'] === 'yes' ? 'true' : 'false',
  ];

  $data_string = '';
  foreach ($data_attributes as $key => $value) {
    $data_string .= ' ' . $key . '="' . esc_attr($value) . '"';
  }

  echo '<div class="eafe-news-ticker eafe-ticker-style-' . esc_attr($ticker_style) . '"' . $data_string . '>';

  // Ticker Label
  if (!empty($settings['label_text'])) {
    echo '<div class="eafe-ticker-label">';
    if (!empty($settings['label_icon']['value'])) {
      Icons_Manager::render_icon($settings['label_icon'], ['aria-hidden' => 'true']);
    }
    echo '<span>' . esc_html($settings['label_text']) . '</span>';
    echo '</div>';
  }

  // Ticker Content
  echo '<div class="eafe-ticker-content">';
  echo '<div class="eafe-ticker-items">';

  foreach ($news_items as $index => $item) {
    $item_class = 'eafe-ticker-item';
    if ($index === 0) {
      $item_class .= ' eafe-ticker-active';
    }

    echo '<div class="' . esc_attr($item_class) . '">';

    // Badge
    if (!empty($item['badge'])) {
      echo '<span class="eafe-news-badge">' . esc_html($item['badge']) . '</span>';
    }

    // Title with link
    if (!empty($item['link'])) {
      $link_attrs = '';
      if (!empty($item['link']['is_external'])) {
        $link_attrs .= ' target="_blank"';
      }
      if (!empty($item['link']['nofollow'])) {
        $link_attrs .= ' rel="nofollow"';
      }
      echo '<a href="' . esc_url($item['link']['url']) . '"' . $link_attrs . '>';
      echo esc_html($item['title']);
      echo '</a>';
    } else {
      echo '<span>' . esc_html($item['title']) . '</span>';
    }

    // Date
    if (!empty($item['date'])) {
      echo '<span class="eafe-news-date">' . esc_html($item['date']) . '</span>';
    }

    echo '</div>';

    // Separator for horizontal style
    if ($ticker_style === 'horizontal' && $index < count($news_items) - 1) {
      echo '<span class="eafe-ticker-separator">' . esc_html($settings['item_separator'] ?? '•') . '</span>';
    }
  }

  echo '</div>'; // .eafe-ticker-items

  // Progress Bar
  if ($show_progress_bar) {
    echo '<div class="eafe-ticker-progress"></div>';
  }

  echo '</div>'; // .eafe-ticker-content

  // Controls
  if ($show_controls) {
    echo '<div class="eafe-ticker-controls">';
    echo '<button class="eafe-ticker-prev" aria-label="' . esc_attr__('Previous news', 'nb-eazy-addons-for-elementor') . '">';
    echo '<svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="15 18 9 12 15 6"></polyline></svg>';
    echo '</button>';
    echo '<button class="eafe-ticker-pause-play" aria-label="' . esc_attr__('Pause ticker', 'nb-eazy-addons-for-elementor') . '">';
    echo '<svg class="eafe-pause-icon" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="6" y="4" width="4" height="16"></rect><rect x="14" y="4" width="4" height="16"></rect></svg>';
    echo '<svg class="eafe-play-icon" width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="display:none;"><polygon points="5 3 19 12 5 21 5 3"></polygon></svg>';
    echo '</button>';
    echo '<button class="eafe-ticker-next" aria-label="' . esc_attr__('Next news', 'nb-eazy-addons-for-elementor') . '">';
    echo '<svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="9 18 15 12 9 6"></polyline></svg>';
    echo '</button>';
    echo '</div>';
  }

  echo '</div>'; // .eafe-news-ticker
}

/**
 * Get news items based on settings
 */
function get_news_items($settings)
{
  $news_items  = [];
  $news_source = $settings['news_source'] ?? 'custom';

  switch ($news_source) {
    case 'custom':
      $news_items = get_custom_news_items($settings);
      break;

    case 'posts':
      $news_items = get_recent_posts_items($settings);
      break;

    case 'category':
      $news_items = get_category_posts_items($settings);
      break;
  }

  return $news_items;
}

/**
 * Get custom news items from repeater
 */
function get_custom_news_items($settings)
{
  $items        = [];
  $custom_items = $settings['custom_news_items'] ?? [];

  foreach ($custom_items as $item) {
    $items[] = [
      'title' => $item['news_title'] ?? '',
      'link'  => $item['news_link'] ?? '',
      'badge' => $item['news_badge'] ?? '',
    ];
  }

  return $items;
}

/**
 * Get recent posts items
 */
function get_recent_posts_items($settings)
{
  $posts_count = $settings['posts_count'] ?? 5;
  $show_date   = $settings['show_post_date'] === 'yes';

  $args = [
    'post_type'      => 'post',
    'posts_per_page' => $posts_count,
    'post_status'    => 'publish',
  ];

  $posts = get_posts($args);
  $items = [];

  foreach ($posts as $post) {
    $item = [
      'title' => get_the_title($post),
      'link'  => [
        'url'         => get_permalink($post),
        'is_external' => false,
        'nofollow'    => false,
      ],
    ];

    if ($show_date) {
      $item['date'] = format_post_date($post->post_date, $settings);
    }

    $items[] = $item;
  }

  return $items;
}

/**
 * Get category posts items
 */
function get_category_posts_items($settings)
{
  $posts_count = $settings['posts_count'] ?? 5;
  $category_id = $settings['posts_category'] ?? '';
  $show_date   = $settings['show_post_date'] === 'yes';

  $args = [
    'post_type'      => 'post',
    'posts_per_page' => $posts_count,
    'post_status'    => 'publish',
  ];

  if (!empty($category_id)) {
    $args['cat'] = $category_id;
  }

  $posts = get_posts($args);
  $items = [];

  foreach ($posts as $post) {
    $item = [
      'title' => get_the_title($post),
      'link'  => [
        'url'         => get_permalink($post),
        'is_external' => false,
        'nofollow'    => false,
      ],
    ];

    if ($show_date) {
      $item['date'] = format_post_date($post->post_date, $settings);
    }

    $items[] = $item;
  }

  return $items;
}

/**
 * Format post date based on settings
 */
function format_post_date($post_date, $settings)
{
  $date_format = $settings['date_format'] ?? 'relative';

  switch ($date_format) {
    case 'relative':
      return human_time_diff(strtotime($post_date), current_time('timestamp')) . ' ' . __('ago', 'nb-eazy-addons-for-elementor');

    case 'custom':
      $custom_format = $settings['custom_date_format'] ?? 'F j, Y';
      return date_i18n($custom_format, strtotime($post_date));

    default:
      return date_i18n(get_option('date_format'), strtotime($post_date));
  }
}
