<?php

namespace EazyAddonsForElementor\Widgets\Renders;

if ( !defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Render function for EAFE_Justified_Grid_Widget
 *
 * @param array $settings The widget settings passed from the main widget class.
 */
function render_eafe_justified_grid_widget( $settings ) {
    // Ensure we have gallery items
    if ( empty( $settings['gallery_items'] ) ) {
        return;
    }

    $layout_style = !empty($settings['layout_style']) ? $settings['layout_style'] : 'style-1';
    $show_filter = !empty($settings['show_filter']) && $settings['show_filter'] === 'yes' && in_array($layout_style, ['style-1', 'style-6']);
    $show_all_filter = !empty($settings['show_all_filter']) && $settings['show_all_filter'] === 'yes';
    $all_filter_text = !empty($settings['all_filter_text']) ? $settings['all_filter_text'] : __('All', 'nb-eazy-addons-for-elementor');
    $show_caption = !empty($settings['show_caption']) && $settings['show_caption'] === 'yes';
    $caption_style = !empty($settings['caption_style']) ? $settings['caption_style'] : 'hover';
    $gutter_size = !empty($settings['gutter_size']['size']) ? $settings['gutter_size']['size'] : 15;
    $image_height = !empty($settings['image_height']['size']) ? $settings['image_height']['size'] : 300;
    $masonry_columns = !empty($settings['masonry_columns']) ? $settings['masonry_columns'] : '3';
    $image_hover_effect = !empty($settings['image_hover_effect']) ? $settings['image_hover_effect'] : 'zoom';

    // Get all unique categories
    $all_categories = [];
    foreach ($settings['gallery_items'] as $item) {
        if (!empty($item['category'])) {
            $categories = array_map('trim', explode(',', $item['category']));
            foreach ($categories as $category) {
                if (!empty($category) && !in_array($category, $all_categories)) {
                    $all_categories[] = $category;
                }
            }
        }
    }

    // Create wrapper classes
    $wrapper_classes = [
        'eafe-justified-grid-container',
        'eafe-justified-grid-' . $layout_style,
        'eafe-justified-grid-caption-' . $caption_style,
    ];

    // Add hover effect class
    if ($image_hover_effect !== 'none') {
        $wrapper_classes[] = 'eafe-hover-effect-' . $image_hover_effect;
    }

    // Add masonry class
    if ($layout_style === 'style-6') {
        $wrapper_classes[] = 'eafe-masonry-grid';
        $wrapper_classes[] = 'eafe-masonry-columns-' . $masonry_columns;
    }

    // Add gutter class for styles that need it
    if ($layout_style !== 'style-3') {
        $wrapper_classes[] = 'eafe-justified-grid-has-gutter';
    }

    // Output the justified grid
    echo '<div class="' . esc_attr( implode( ' ', $wrapper_classes ) ) . '" data-gutter="' . esc_attr($gutter_size) . '" data-height="' . esc_attr($image_height) . '" data-columns="' . esc_attr($masonry_columns) . '">';

    // Filter buttons
    if ($show_filter && !empty($all_categories)) {
        echo '<div class="eafe-justified-grid-filter">';

        if ($show_all_filter) {
            echo '<button class="eafe-filter-button active" data-filter="*">' . esc_html($all_filter_text) . '</button>';
        }

        foreach ($all_categories as $category) {
            $filter_class = sanitize_title($category);
            echo '<button class="eafe-filter-button" data-filter=".' . esc_attr($filter_class) . '">' . esc_html($category) . '</button>';
        }

        echo '</div>';
    }

    // Grid items
    echo '<div class="eafe-justified-grid">';

    foreach ($settings['gallery_items'] as $index => $item) {
        $image_url = !empty($item['image']['url']) ? $item['image']['url'] : '';
        $title = !empty($item['title']) ? $item['title'] : '';
        $description = !empty($item['description']) ? $item['description'] : '';
        $categories = !empty($item['category']) ? array_map('trim', explode(',', $item['category'])) : [];
        $link = !empty($item['link']['url']) ? $item['link']['url'] : '';
        $is_external = !empty($item['link']['is_external']) ? ' target="_blank"' : '';
        $nofollow = !empty($item['link']['nofollow']) ? ' rel="nofollow"' : '';

        // Create item classes
        $item_classes = ['eafe-justified-grid-item'];

        // Add category classes for filtering
        foreach ($categories as $category) {
            if (!empty($category)) {
                $item_classes[] = sanitize_title($category);
            }
        }

        // Output grid item
        echo '<div class="' . esc_attr( implode( ' ', $item_classes ) ) . '">';

        // Image with optional link
        if ($image_url) {
            if ($link) {
                echo '<a href="' . esc_url($link) . '"' . $is_external . $nofollow . '>';
            }

            echo '<img src="' . esc_url($image_url) . '" alt="' . esc_attr($title) . '">';

            // Caption
            if ($show_caption && ($title || $description)) {
                echo '<div class="eafe-justified-grid-caption">';
                if ($title) {
                    echo '<h3 class="eafe-justified-grid-title">' . esc_html($title) . '</h3>';
                }
                if ($description) {
                    echo '<div class="eafe-justified-grid-description">' . esc_html($description) . '</div>';
                }
                echo '</div>';
            }

            if ($link) {
                echo '</a>';
            } else {
                // Still show caption even without link
                if ($show_caption && ($title || $description)) {
                    echo '<div class="eafe-justified-grid-caption">';
                    if ($title) {
                        echo '<h3 class="eafe-justified-grid-title">' . esc_html($title) . '</h3>';
                    }
                    if ($description) {
                        echo '<div class="eafe-justified-grid-description">' . esc_html($description) . '</div>';
                    }
                    echo '</div>';
                }
            }
        }

        echo '</div>'; // End grid item
    }

    echo '</div>'; // End grid
    echo '</div>'; // End container
}