<?php

namespace EazyAddonsForElementor\Widgets\Renders;

use Elementor\Icons_Manager;

if ( !defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Render function for EAFE_Icon_Box_Widget
 *
 * @param array $settings The widget settings passed from the main widget class.
 */
function render_eafe_icon_box_widget( $settings ) {
    $template_class = !empty( $settings['template_style'] ) ? $settings['template_style'] : 'template-1';
    $position_class = !empty( $settings['icon_position'] ) ? 'icon-position-' . $settings['icon_position'] : 'icon-position-top';
    $box_tag = 'div';
    $box_attributes = '';

    // Handle entire box linking
    if ( $settings['link_entire_box'] === 'yes' && $settings['enable_button'] !== 'yes' && !empty( $settings['box_url']['url'] ) ) {
        $box_tag = 'a';
        $target = $settings['box_url']['is_external'] ? ' target="_blank"' : '';
        $nofollow = $settings['box_url']['nofollow'] ? ' rel="nofollow"' : '';
        $box_attributes = 'href="' . esc_url( $settings['box_url']['url'] ) . '"' . $target . $nofollow . ' class="eafe-icon-box-link"';
    }

    echo '<' . $box_tag . ' class="eafe-icon-box ' . esc_attr( $template_class ) . ' ' . esc_attr( $position_class ) . '"' . $box_attributes . '>';

    // Add background image for template 7
    if ( $template_class === 'template-7' && !empty( $settings['background_image']['url'] ) ) {
        echo '<div class="eafe-icon-box-bg-image" style="background-image: url(' . esc_url( $settings['background_image']['url'] ) . ');"></div>';

        // Add overlay if enabled
        if ( $settings['background_overlay'] === 'yes' ) {
            echo '<div class="eafe-icon-box-overlay"></div>';
        }
    }

    // Add content wrapper for better positioning
    echo '<div class="eafe-icon-box-content">';

    // Render Icon or Image
    $icon_type = !empty( $settings['icon_type'] ) ? $settings['icon_type'] : 'icon';

    if ( $icon_type === 'icon' && !empty( $settings['icon'] ) && is_array( $settings['icon'] ) ) {
        echo '<div class="eafe-icon-box-icon">';
        \Elementor\Icons_Manager::render_icon( $settings['icon'], ['aria-hidden' => 'true'] );
        echo '</div>';
    } elseif ( $icon_type === 'image' && !empty( $settings['icon_image']['url'] ) ) {
        $alt_text = !empty( $settings['title'] ) ? esc_attr( $settings['title'] ) : '';
        echo '<div class="eafe-icon-box-icon eafe-icon-box-image">';
        echo '<img src="' . esc_url( $settings['icon_image']['url'] ) . '" alt="' . $alt_text . '">';
        echo '</div>';
    }

    // Add text content wrapper
    echo '<div class="eafe-icon-box-text-content">';

    // Render Title
    if ( !empty( $settings['title'] ) ) {
        $title_tag = !empty( $settings['title_tag'] ) ? $settings['title_tag'] : 'h3';
        echo '<' . $title_tag . ' class="eafe-icon-box-title">' . esc_html( $settings['title'] ) . '</' . $title_tag . '>';
    }

    // Render Description
    if ( !empty( $settings['description'] ) ) {
        echo '<p class="eafe-icon-box-description">' . wp_kses_post( $settings['description'] ) . '</p>';
    }

    // Render Button
    if ( $settings['enable_button'] === 'yes' && !empty( $settings['button_text'] ) && !empty( $settings['button_url']['url'] ) ) {
        $target = $settings['button_url']['is_external'] ? ' target="_blank"' : '';
        $nofollow = $settings['button_url']['nofollow'] ? ' rel="nofollow"' : '';
        echo '<a href="' . esc_url( $settings['button_url']['url'] ) . '" class="eafe-icon-box-button"' . $target . $nofollow . '>';
        echo esc_html( $settings['button_text'] );
        echo '</a>';
    }

    echo '</div>'; // Close text content wrapper
    echo '</div>'; // Close content wrapper
    echo '</' . $box_tag . '>';
}

/**
 * Get template-specific CSS classes
 *
 * @param string $template The template name
 * @return string CSS classes for the template
 */
function get_template_classes( $template ) {
    $classes = [];

    switch( $template ) {
        case 'template-1':
            $classes[] = 'outline-box';
            break;
        case 'template-2':
            $classes[] = 'borderless-box';
            break;
        case 'template-3':
            $classes[] = 'multicolor-badge';
            break;
        case 'template-4':
            $classes[] = 'offset-badge';
            break;
        case 'template-5':
            $classes[] = 'gradient-bg';
            break;
        case 'template-6':
            $classes[] = 'circular-badge';
            break;
        case 'template-7':
            $classes[] = 'image-bg';
            break;
    }

    return implode( ' ', $classes );
}

