<?php

namespace EazyAddonsForElementor\Widgets\Renders;

use Elementor\Icons_Manager;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * Render function for EAFE_Dual_Button_Widget
 *
 * @param array $settings The widget settings passed from the main widget class.
 */
function render_eafe_dual_button_widget( $settings ) {

    // ====== Helpers ==========================================================
    $has_icon = static function( $icon_conf, $fa4_fallback ) {
        return ( is_array( $icon_conf ) && ! empty( $icon_conf['value'] ) ) || ! empty( $fa4_fallback );
    };

    $render_icon = static function( $icon_conf, $fa4_fallback ) {
        if ( is_array( $icon_conf ) && ! empty( $icon_conf['value'] ) ) {
            Icons_Manager::render_icon( $icon_conf, [
                'aria-hidden' => 'true',
                'class'       => 'eafe-icon',
            ] );
        } elseif ( ! empty( $fa4_fallback ) ) {
            echo '<i class="' . esc_attr( $fa4_fallback ) . ' eafe-icon" aria-hidden="true"></i>';
        }
    };

    // Enhanced icon asset enqueuing
    $enqueue_icon_assets = static function( $icon_conf, $fa4_fallback ) {
        // Handle modern icon configuration
        if ( is_array( $icon_conf ) && ! empty( $icon_conf['library'] ) ) {
            switch ( $icon_conf['library'] ) {
                case 'eicons':
                    wp_enqueue_style( 'elementor-icons' );
                    break;
                case 'fa-solid':
                case 'fa-regular':
                case 'fa-light':
                case 'fa-duotone':
                case 'fa-brands':
                    wp_enqueue_style( 'elementor-icons-fa-solid' );
                    wp_enqueue_style( 'elementor-icons-fa-regular' );
                    wp_enqueue_style( 'elementor-icons-fa-brands' );
                    break;
                case 'svg':
                    // SVG icons don't need stylesheets
                    break;
                default:
                    // Try to enqueue based on library name
                    $style_handle = 'elementor-icons-' . $icon_conf['library'];
                    if ( wp_style_is( $style_handle, 'registered' ) ) {
                        wp_enqueue_style( $style_handle );
                    }
                    break;
            }
        }

        // Handle FA4 fallback
        if ( ! empty( $fa4_fallback ) ) {
            if ( method_exists( Icons_Manager::class, 'enqueue_shim' ) ) {
                Icons_Manager::enqueue_shim();
            }
            // Also try to enqueue FontAwesome 4 if available
            if ( wp_style_is( 'font-awesome', 'registered' ) ) {
                wp_enqueue_style( 'font-awesome' );
            }
        }
    };

    $build_link_attrs = static function( $url, $is_external, $nofollow, $label ) {
        $attrs = '';
        if ( $url ) {
            $attrs .= ' href="' . esc_url( $url ) . '"';
            if ( $is_external ) {
                $attrs .= ' target="_blank"';
            }
            $rel = [];
            if ( $nofollow )    { $rel[] = 'nofollow'; }
            if ( $is_external ) { $rel[] = 'noopener'; $rel[] = 'noreferrer'; }
            if ( $rel ) {
                $attrs .= ' rel="' . esc_attr( implode( ' ', array_unique( $rel ) ) ) . '"';
            }
        } else {
            // Keep focusable when no link provided
            $attrs .= ' role="button" tabindex="0"';
        }
        if ( $label ) {
            $attrs .= ' aria-label="' . esc_attr( $label ) . '"';
        }
        return $attrs;
    };

    // ====== Button 1 settings ===============================================
    $button1_text            = ! empty( $settings['button1_text'] ) ? $settings['button1_text'] : '';
    $button1_link            = ! empty( $settings['button1_link']['url'] ) ? $settings['button1_link']['url'] : '';
    $button1_is_external     = ! empty( $settings['button1_link']['is_external'] );
    $button1_nofollow        = ! empty( $settings['button1_link']['nofollow'] );
    $button1_icon_conf       = isset( $settings['button1_icon'] ) ? $settings['button1_icon'] : null;
    $button1_icon_position   = ! empty( $settings['button1_icon_position'] ) ? $settings['button1_icon_position'] : 'before';
    $button1_hover_animation = ! empty( $settings['button1_hover_animation'] ) ? ' elementor-animation-' . $settings['button1_hover_animation'] : '';

    // FA4 fallback value (if old content / migration)
    $button1_icon_fa4 = '';
    if ( is_string( $button1_icon_conf ) ) {
        $button1_icon_fa4 = $button1_icon_conf;
        $button1_icon_conf = null; // Reset to prevent double rendering
    } elseif ( ! empty( $settings['icon'] ) && is_string( $settings['icon'] ) ) {
        $button1_icon_fa4 = $settings['icon'];
    }
    $button1_has_icon = $has_icon( $button1_icon_conf, $button1_icon_fa4 );

    // Enqueue assets for button 1 icon
    if ( $button1_has_icon ) {
        $enqueue_icon_assets( $button1_icon_conf, $button1_icon_fa4 );
    }

    // ====== Button 2 settings ===============================================
    $button2_text            = ! empty( $settings['button2_text'] ) ? $settings['button2_text'] : '';
    $button2_link            = ! empty( $settings['button2_link']['url'] ) ? $settings['button2_link']['url'] : '';
    $button2_is_external     = ! empty( $settings['button2_link']['is_external'] );
    $button2_nofollow        = ! empty( $settings['button2_link']['nofollow'] );
    $button2_icon_conf       = isset( $settings['button2_icon'] ) ? $settings['button2_icon'] : null;
    $button2_icon_position   = ! empty( $settings['button2_icon_position'] ) ? $settings['button2_icon_position'] : 'after';
    $button2_hover_animation = ! empty( $settings['button2_hover_animation'] ) ? ' elementor-animation-' . $settings['button2_hover_animation'] : '';

    $button2_icon_fa4 = '';
    if ( is_string( $button2_icon_conf ) ) {
        $button2_icon_fa4 = $button2_icon_conf;
        $button2_icon_conf = null; // Reset to prevent double rendering
    } elseif ( ! empty( $settings['icon2'] ) && is_string( $settings['icon2'] ) ) {
        $button2_icon_fa4 = $settings['icon2'];
    }
    $button2_has_icon = $has_icon( $button2_icon_conf, $button2_icon_fa4 );

    // Enqueue assets for button 2 icon
    if ( $button2_has_icon ) {
        $enqueue_icon_assets( $button2_icon_conf, $button2_icon_fa4 );
    }

    // ====== Middle & Layout ==================================================
    $middle_text              = ! empty( $settings['middle_text'] ) ? $settings['middle_text'] : 'OR';
    $middle_text_aria_hidden  = ! empty( $settings['middle_text_aria_hidden'] ) ? $settings['middle_text_aria_hidden'] : 'yes';

    $layout_style = ! empty( $settings['layout_style'] ) ? $settings['layout_style'] : 'horizontal';
    $button_style = ! empty( $settings['button_style'] ) ? $settings['button_style'] : 'style-1';
    $alignment    = ! empty( $settings['alignment'] ) ? $settings['alignment'] : 'center';
    $full_width   = ! empty( $settings['full_width'] ) ? $settings['full_width'] : 'no';

    $wrapper_classes = [
        'eafe-dual-button-container',
        'eafe-dual-button-layout-' . $layout_style,
        'eafe-dual-button-align-' . $alignment,
        'eafe-dual-button-style-' . $button_style,
    ];
    if ( $full_width === 'yes' ) {
        $wrapper_classes[] = 'eafe-dual-button-full-width';
    }

    $button1_classes = [ 'eafe-dual-button', 'eafe-dual-button-1', $button1_hover_animation ];
    $button2_classes = [ 'eafe-dual-button', 'eafe-dual-button-2', $button2_hover_animation ];

    // ====== Output ===========================================================
    echo '<div class="' . esc_attr( implode( ' ', array_filter( $wrapper_classes ) ) ) . '">';

    // Button 1
    if ( $button1_text || $button1_has_icon ) {
        $button1_attrs = $build_link_attrs( $button1_link, $button1_is_external, $button1_nofollow, $button1_text );
        echo '<a ' . $button1_attrs . ' class="' . esc_attr( implode( ' ', array_filter( $button1_classes ) ) ) . '">';

        if ( $button1_has_icon && $button1_icon_position === 'before' ) {
            echo '<span class="eafe-button-icon eafe-button-icon-before">';
            $render_icon( $button1_icon_conf, $button1_icon_fa4 );
            echo '</span>';
        }

        if ( $button1_text ) {
            echo '<span class="eafe-button-text">' . esc_html( $button1_text ) . '</span>';
        }

        if ( $button1_has_icon && $button1_icon_position === 'after' ) {
            echo '<span class="eafe-button-icon eafe-button-icon-after">';
            $render_icon( $button1_icon_conf, $button1_icon_fa4 );
            echo '</span>';
        }

        echo '</a>';
    }

    // Middle text (Style 6)
    if ( $button_style === 'style-6' && $middle_text ) {
        $aria_hidden = ( $middle_text_aria_hidden === 'yes' ) ? ' aria-hidden="true"' : '';
        echo '<span class="eafe-dual-button-middle-text"' . $aria_hidden . '>' . esc_html( $middle_text ) . '</span>';
    }

    // Button 2
    if ( $button2_text || $button2_has_icon ) {
        $button2_attrs = $build_link_attrs( $button2_link, $button2_is_external, $button2_nofollow, $button2_text );
        echo '<a ' . $button2_attrs . ' class="' . esc_attr( implode( ' ', array_filter( $button2_classes ) ) ) . '">';

        if ( $button2_has_icon && $button2_icon_position === 'before' ) {
            echo '<span class="eafe-button-icon eafe-button-icon-before">';
            $render_icon( $button2_icon_conf, $button2_icon_fa4 );
            echo '</span>';
        }

        if ( $button2_text ) {
            echo '<span class="eafe-button-text">' . esc_html( $button2_text ) . '</span>';
        }

        if ( $button2_has_icon && $button2_icon_position === 'after' ) {
            echo '<span class="eafe-button-icon eafe-button-icon-after">';
            $render_icon( $button2_icon_conf, $button2_icon_fa4 );
            echo '</span>';
        }

        echo '</a>';
    }

    echo '</div>';
}