<?php

namespace EazyAddonsForElementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Widget_Base;
use Elementor\Repeater;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class EAFE_Smart_Slider_Widget extends Widget_Base
{
    public function get_name()
    {
        return EAFE_WIDGET_PREFIX . 'smart_slider';
    }

    public function get_title()
    {
        return __('Smart Slider', 'nb-eazy-addons-for-elementor');
    }

    public function get_style_depends()
    {
        return ['swiper', 'ele-addons-slider'];
    }

    public function get_script_depends()
    {
        return ['swiper', 'eafe-smart-slider-js'];
    }

    public function get_icon()
    {
        return 'eicon-slider-push';
    }

    public function get_categories()
    {
        return ['eazy-addons-for-elementor-category'];
    }

    protected function _register_controls()
    {
        // Slides Section
        $this->start_controls_section(
            'slides_section',
            [
                'label' => __('Slides', 'nb-eazy-addons-for-elementor'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        // Slide Type
        $repeater->add_control(
            'slide_type',
            [
                'label'   => __('Slide Type', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'image'  => __('Image', 'nb-eazy-addons-for-elementor'),
                    'content' => __('Content', 'nb-eazy-addons-for-elementor'),
                    'video'  => __('Video', 'nb-eazy-addons-for-elementor'),
                ],
                'default' => 'image',
            ]
        );

        // Image Slide
        $repeater->add_control(
            'slide_image',
            [
                'label'     => __('Image', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::MEDIA,
                'default'   => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'slide_type' => ['image', 'content'],
                ],
            ]
        );

        // Image Size Control
        $repeater->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name'      => 'slide_image_size',
                'default'   => 'full',
                'separator' => 'none',
                'condition' => [
                    'slide_type' => ['image', 'content'],
                ],
            ]
        );

        // Video Slide
        $repeater->add_control(
            'video_url',
            [
                'label'       => __('Video URL', 'nb-eazy-addons-for-elementor'),
                'type'        => Controls_Manager::URL,
                'placeholder' => __('https://your-link.com/video.mp4', 'nb-eazy-addons-for-elementor'),
                'condition'   => [
                    'slide_type' => 'video',
                ],
            ]
        );

        $repeater->add_control(
            'video_poster',
            [
                'label'     => __('Video Poster', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::MEDIA,
                'condition' => [
                    'slide_type' => 'video',
                ],
            ]
        );

        // Content Slide
        $repeater->add_control(
            'slide_title',
            [
                'label'       => __('Title', 'nb-eazy-addons-for-elementor'),
                'type'        => Controls_Manager::TEXT,
                'default'     => __('Slide Title', 'nb-eazy-addons-for-elementor'),
                'placeholder' => __('Enter slide title', 'nb-eazy-addons-for-elementor'),
                'condition'   => [
                    'slide_type' => 'content',
                ],
            ]
        );

        $repeater->add_control(
            'slide_description',
            [
                'label'       => __('Description', 'nb-eazy-addons-for-elementor'),
                'type'        => Controls_Manager::TEXTAREA,
                'default'     => __('Lorem ipsum dolor sit amet, consectetur adipiscing elit.', 'nb-eazy-addons-for-elementor'),
                'placeholder' => __('Enter slide description', 'nb-eazy-addons-for-elementor'),
                'condition'   => [
                    'slide_type' => 'content',
                ],
            ]
        );

        $repeater->add_control(
            'slide_button_text',
            [
                'label'       => __('Button Text', 'nb-eazy-addons-for-elementor'),
                'type'        => Controls_Manager::TEXT,
                'placeholder' => __('Learn More', 'nb-eazy-addons-for-elementor'),
                'condition'   => [
                    'slide_type' => 'content',
                ],
            ]
        );

        $repeater->add_control(
            'slide_button_link',
            [
                'label'       => __('Button Link', 'nb-eazy-addons-for-elementor'),
                'type'        => Controls_Manager::URL,
                'placeholder' => __('https://your-link.com', 'nb-eazy-addons-for-elementor'),
                'condition'   => [
                    'slide_type' => 'content',
                ],
            ]
        );

        // Slide Background
        $repeater->add_control(
            'slide_background',
            [
                'label'     => __('Background Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}}' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'slides',
            [
                'label'       => __('Slides', 'nb-eazy-addons-for-elementor'),
                'type'        => Controls_Manager::REPEATER,
                'fields'      => $repeater->get_controls(),
                'default'     => [
                    [
                        'slide_type'        => 'content',
                        'slide_title'       => __('First Slide', 'nb-eazy-addons-for-elementor'),
                        'slide_description' => __('This is the first slide description.', 'nb-eazy-addons-for-elementor'),
                    ],
                    [
                        'slide_type'        => 'content',
                        'slide_title'       => __('Second Slide', 'nb-eazy-addons-for-elementor'),
                        'slide_description' => __('This is the second slide description.', 'nb-eazy-addons-for-elementor'),
                    ],
                    [
                        'slide_type'        => 'content',
                        'slide_title'       => __('Third Slide', 'nb-eazy-addons-for-elementor'),
                        'slide_description' => __('This is the third slide description.', 'nb-eazy-addons-for-elementor'),
                    ],
                ],
                'title_field' => '{{{ slide_title }}}',
            ]
        );

        $this->end_controls_section();

        // Slider Settings Section
        $this->start_controls_section(
            'slider_settings',
            [
                'label' => __('Slider Settings', 'nb-eazy-addons-for-elementor'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        // Transition Effect
        $this->add_control(
            'transition_effect',
            [
                'label'   => __('Transition Effect', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'slide'     => __('Slide', 'nb-eazy-addons-for-elementor'),
                    'fade'      => __('Fade', 'nb-eazy-addons-for-elementor'),
                    'cube'      => __('Cube', 'nb-eazy-addons-for-elementor'),
                    'coverflow' => __('Coverflow', 'nb-eazy-addons-for-elementor'),
                    'flip'      => __('Flip', 'nb-eazy-addons-for-elementor'),
                ],
                'default' => 'slide',
            ]
        );

        // Direction
        $this->add_control(
            'direction',
            [
                'label'   => __('Direction', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'horizontal' => __('Horizontal', 'nb-eazy-addons-for-elementor'),
                    'vertical'   => __('Vertical', 'nb-eazy-addons-for-elementor'),
                ],
                'default' => 'horizontal',
            ]
        );

        // Slides Per View
        $this->add_control(
            'slides_per_view',
            [
                'label'   => __('Slides Per View', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::NUMBER,
                'min'     => 1,
                'max'     => 10,
                'step'    => 1,
                'default' => 1,
            ]
        );

        // Space Between
        $this->add_responsive_control(
            'space_between',
            [
                'label'   => __('Space Between', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::NUMBER,
                'min'     => 0,
                'max'     => 100,
                'step'    => 1,
                'default' => 0,
            ]
        );

        // Transition Speed
        $this->add_control(
            'speed',
            [
                'label'   => __('Transition Speed (ms)', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::NUMBER,
                'default' => 500,
                'min'     => 100,
                'max'     => 5000,
                'step'    => 100,
            ]
        );

        // Autoplay
        $this->add_control(
            'autoplay',
            [
                'label'   => __('Autoplay', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'autoplay_speed',
            [
                'label'     => __('Autoplay Speed (ms)', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::NUMBER,
                'default'   => 3000,
                'condition' => [
                    'autoplay' => 'yes',
                ],
            ]
        );

        // Loop
        $this->add_control(
            'loop',
            [
                'label'   => __('Infinite Loop', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        // Navigation Arrows
        $this->add_control(
            'show_arrows',
            [
                'label'   => __('Show Arrows', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        // Pagination Dots
        $this->add_control(
            'show_dots',
            [
                'label'   => __('Show Dots', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        // Mousewheel Control
        $this->add_control(
            'mousewheel',
            [
                'label'   => __('Mousewheel Control', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SWITCHER,
                'default' => 'no',
            ]
        );

        // Keyboard Control
        $this->add_control(
            'keyboard',
            [
                'label'   => __('Keyboard Control', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        // Pause on Hover
        $this->add_control(
            'pause_on_hover',
            [
                'label'   => __('Pause on Hover', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'condition' => [
                    'autoplay' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Content Settings Section
        $this->start_controls_section(
            'content_settings',
            [
                'label' => __('Content Settings', 'nb-eazy-addons-for-elementor'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'content_position',
            [
                'label'   => __('Content Position', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'top-left'      => __('Top Left', 'nb-eazy-addons-for-elementor'),
                    'top-center'    => __('Top Center', 'nb-eazy-addons-for-elementor'),
                    'top-right'     => __('Top Right', 'nb-eazy-addons-for-elementor'),
                    'center-left'   => __('Center Left', 'nb-eazy-addons-for-elementor'),
                    'center'        => __('Center', 'nb-eazy-addons-for-elementor'),
                    'center-right'  => __('Center Right', 'nb-eazy-addons-for-elementor'),
                    'bottom-left'   => __('Bottom Left', 'nb-eazy-addons-for-elementor'),
                    'bottom-center' => __('Bottom Center', 'nb-eazy-addons-for-elementor'),
                    'bottom-right'  => __('Bottom Right', 'nb-eazy-addons-for-elementor'),
                ],
                'default' => 'center',
            ]
        );

        $this->add_control(
            'content_alignment',
            [
                'label'   => __('Content Alignment', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'left'   => __('Left', 'nb-eazy-addons-for-elementor'),
                    'center' => __('Center', 'nb-eazy-addons-for-elementor'),
                    'right'  => __('Right', 'nb-eazy-addons-for-elementor'),
                ],
                'default' => 'center',
            ]
        );

        $this->add_control(
            'content_width',
            [
                'label' => __('Content Width', 'nb-eazy-addons-for-elementor'),
                'type'  => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 100,
                        'max' => 1200,
                    ],
                ],
                'default' => [
                    'size' => 600,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .eafe-slide-content' => 'max-width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Section - Slider
        $this->start_controls_section(
            'style_slider',
            [
                'label' => __('Slider', 'nb-eazy-addons-for-elementor'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'slide_height',
            [
                'label' => __('Slide Height', 'nb-eazy-addons-for-elementor'),
                'type'  => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 100,
                        'max' => 1000,
                    ],
                    'vh' => [
                        'min' => 10,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'size' => 500,
                    'unit' => 'px',
                ],
                'size_units' => ['px', 'vh'],
                'selectors' => [
                    '{{WRAPPER}} .eafe-slide' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'     => 'slide_border',
                'selector' => '{{WRAPPER}} .eafe-slide',
            ]
        );

        $this->add_control(
            'slide_border_radius',
            [
                'label'      => __('Border Radius', 'nb-eazy-addons-for-elementor'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .eafe-slide' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'slide_box_shadow',
                'selector' => '{{WRAPPER}} .eafe-slide',
            ]
        );

        $this->end_controls_section();

        // Style Section - Content
        $this->start_controls_section(
            'style_content',
            [
                'label' => __('Content', 'nb-eazy-addons-for-elementor'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'content_background',
            [
                'label'     => __('Background Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .eafe-slide-content' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'content_padding',
            [
                'label'      => __('Padding', 'nb-eazy-addons-for-elementor'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .eafe-slide-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'content_border_radius',
            [
                'label'      => __('Border Radius', 'nb-eazy-addons-for-elementor'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .eafe-slide-content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        // Title Style
        $this->add_control(
            'heading_title',
            [
                'label'     => __('Title', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label'     => __('Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .eafe-slide-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'title_typography',
                'selector' => '{{WRAPPER}} .eafe-slide-title',
            ]
        );

        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name'     => 'title_text_shadow',
                'selector' => '{{WRAPPER}} .eafe-slide-title',
            ]
        );

        // Description Style
        $this->add_control(
            'heading_description',
            [
                'label'     => __('Description', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label'     => __('Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .eafe-slide-description' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'description_typography',
                'selector' => '{{WRAPPER}} .eafe-slide-description',
            ]
        );

        $this->end_controls_section();

        // Style Section - Button
        $this->start_controls_section(
            'style_button',
            [
                'label' => __('Button', 'nb-eazy-addons-for-elementor'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'button_typography',
                'selector' => '{{WRAPPER}} .eafe-slide-button',
            ]
        );

        $this->start_controls_tabs('button_tabs');

        // Normal State
        $this->start_controls_tab(
            'button_normal',
            [
                'label' => __('Normal', 'nb-eazy-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'button_text_color',
            [
                'label'     => __('Text Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .eafe-slide-button' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name'     => 'button_background',
                'selector' => '{{WRAPPER}} .eafe-slide-button',
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'     => 'button_border',
                'selector' => '{{WRAPPER}} .eafe-slide-button',
            ]
        );

        $this->add_control(
            'button_border_radius',
            [
                'label'      => __('Border Radius', 'nb-eazy-addons-for-elementor'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .eafe-slide-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_tab();

        // Hover State
        $this->start_controls_tab(
            'button_hover',
            [
                'label' => __('Hover', 'nb-eazy-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'button_text_color_hover',
            [
                'label'     => __('Text Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .eafe-slide-button:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name'     => 'button_background_hover',
                'selector' => '{{WRAPPER}} .eafe-slide-button:hover',
            ]
        );

        $this->add_control(
            'button_border_color_hover',
            [
                'label'     => __('Border Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .eafe-slide-button:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->add_control(
            'button_padding',
            [
                'label'      => __('Padding', 'nb-eazy-addons-for-elementor'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .eafe-slide-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'separator' => 'before',
            ]
        );

        $this->end_controls_section();

        // Style Section - Navigation
        $this->start_controls_section(
            'style_navigation',
            [
                'label' => __('Navigation', 'nb-eazy-addons-for-elementor'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        // Arrows
        $this->add_control(
            'heading_arrows',
            [
                'label'     => __('Arrows', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::HEADING,
                'condition' => [
                    'show_arrows' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'arrows_size',
            [
                'label' => __('Size', 'nb-eazy-addons-for-elementor'),
                'type'  => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 20,
                        'max' => 60,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .eafe-slider-arrow' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .eafe-slider-arrow' => 'width: calc({{SIZE}}{{UNIT}} * 2); height: calc({{SIZE}}{{UNIT}} * 2);',
                ],
                'condition' => [
                    'show_arrows' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'arrows_color',
            [
                'label'     => __('Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .eafe-slider-arrow' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_arrows' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'arrows_background',
            [
                'label'     => __('Background Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .eafe-slider-arrow' => 'background-color: {{VALUE}};',
                ],
                'condition' => [
                    'show_arrows' => 'yes',
                ],
            ]
        );

        // Dots
        $this->add_control(
            'heading_dots',
            [
                'label'     => __('Dots', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'show_dots' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'dots_size',
            [
                'label' => __('Size', 'nb-eazy-addons-for-elementor'),
                'type'  => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 5,
                        'max' => 15,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .swiper-pagination-bullet' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'show_dots' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'dots_color',
            [
                'label'     => __('Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .swiper-pagination-bullet' => 'background-color: {{VALUE}};',
                ],
                'condition' => [
                    'show_dots' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'dots_active_color',
            [
                'label'     => __('Active Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .swiper-pagination-bullet-active' => 'background-color: {{VALUE}};',
                ],
                'condition' => [
                    'show_dots' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        // Prepare slide attributes
        $slides_data = [];
        foreach ($settings['slides'] as $index => $slide) {
            $slide_key = $this->get_repeater_setting_key('slide', 'slides', $index);
            $this->add_render_attribute($slide_key, [
                'class' => ['swiper-slide', 'eafe-slide', 'elementor-repeater-item-' . $slide['_id']],
            ]);

            // FIX: Check if slide_background exists and is not empty
            if (!empty($slide['slide_background'])) {
                $this->add_render_attribute($slide_key, 'style', 'background-color: ' . esc_attr($slide['slide_background']) . ';');
            }

            $slides_data[] = [
                'slide' => $slide,
                'attributes' => $this->get_render_attribute_string($slide_key),
            ];
        }

        // Include the render file and call the render function
        require_once __DIR__ . '/renders/render-eafe-smart-slider-widget.php';
        \EazyAddonsForElementor\Widgets\Renders\render_eafe_smart_slider_widget($settings, $slides_data, $this);
    }

    protected function content_template()
    {
        ?>
        <#
        var sliderSettings = {
        effect: settings.transition_effect,
        direction: settings.direction,
        slidesPerView: settings.slides_per_view,
        spaceBetween: settings.space_between,
        speed: settings.speed,
        autoplay: settings.autoplay === 'yes',
        autoplaySpeed: settings.autoplay_speed,
        loop: settings.loop === 'yes',
        mousewheel: settings.mousewheel === 'yes',
        keyboard: settings.keyboard === 'yes',
        arrows: settings.show_arrows === 'yes',
        dots: settings.show_dots === 'yes',
        pauseOnHover: settings.pause_on_hover === 'yes',
        contentPosition: settings.content_position,
        contentAlignment: settings.content_alignment
        };
        #>
        <div class="eafe-smart-slider"
             data-settings='{{{ JSON.stringify(sliderSettings) }}}'>
            <div class="swiper-wrapper">
                <# _.each(settings.slides, function(slide, index) { #>
                <div class="swiper-slide eafe-slide elementor-repeater-item-{{{ slide._id }}}">
                    <div class="eafe-slide-inner">
                        <# if (slide.slide_type === 'image' || slide.slide_type === 'content') {
                        if (slide.slide_image && slide.slide_image.url) {
                        var image = {
                        id: slide.slide_image.id,
                        url: slide.slide_image.url,
                        size: slide.slide_image_size_size,
                        dimension: slide.slide_image_size_custom_dimension,
                        model: view.getEditModel()
                        };
                        var image_url = elementor.imagesManager.getImageUrl( image );
                        #>
                        <img src="{{{ image_url }}}" alt="{{ slide.slide_title }}" class="eafe-slide-image">
                        <# } else { #>
                        <img src="<?php echo esc_url(\Elementor\Utils::get_placeholder_image_src()); ?>" alt="" class="eafe-slide-image">
                        <# } } #>
                        <# if (slide.slide_type === 'video') { #>
                        <video class="eafe-slide-video"
                               src="{{ slide.video_url.url }}"
                        <# if (slide.video_poster && slide.video_poster.url) { #>poster="{{ slide.video_poster.url }}"<# } #>
                        controls>
                        </video>
                        <# } #>
                        <# if (slide.slide_type === 'content') { #>
                        <div class="eafe-slide-content position-{{ settings.content_position }} text-{{ settings.content_alignment }}">
                            <# if (slide.slide_title) { #>
                            <h2 class="eafe-slide-title">{{{ slide.slide_title }}}</h2>
                            <# } #>
                            <# if (slide.slide_description) { #>
                            <p class="eafe-slide-description">{{{ slide.slide_description }}}</p>
                            <# } #>
                            <# if (slide.slide_button_text) { #>
                            <a class="eafe-slide-button" href="{{ slide.slide_button_link.url }}">{{{ slide.slide_button_text }}}</a>
                            <# } #>
                        </div>
                        <# } #>
                    </div>
                </div>
                <# }); #>
            </div>

            <# if (settings.show_arrows === 'yes') { #>
            <div class="eafe-slider-arrows">
                <div class="eafe-slider-arrow eafe-slider-prev swiper-button-prev"></div>
                <div class="eafe-slider-arrow eafe-slider-next swiper-button-next"></div>
            </div>
            <# } #>

            <# if (settings.show_dots === 'yes') { #>
            <div class="eafe-slider-pagination swiper-pagination"></div>
            <# } #>
        </div>
        <?php
    }
}