<?php

namespace EazyAddonsForElementor\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Widget_Base;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class EAFE_Post_Carousel_Widget extends Widget_Base
{

    public function get_name()
    {
        return EAFE_WIDGET_PREFIX . 'post_carousel';
    }

    public function get_title()
    {
        return __('Post Carousel', 'nb-eazy-addons-for-elementor');
    }

    public function get_style_depends() {
        return ['swiper', 'eafe-post-carousel'];
    }

    public function get_script_depends() {
        return ['swiper', 'eafe-post-carousel'];
    }

    public function get_icon()
    {
        return 'eicon-post-slider';
    }

    public function get_categories()
    {
        return ['eazy-addons-for-elementor-category'];
    }

    protected function _register_controls()
    {
        // Query Section (same as post grid)
        $this->start_controls_section(
            'query_section',
            [
                'label' => __('Query', 'nb-eazy-addons-for-elementor'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'post_type',
            [
                'label'   => __('Post Type', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'post' => __('Post', 'nb-eazy-addons-for-elementor'),
                    'page' => __('Page', 'nb-eazy-addons-for-elementor'),
                ],
                'default' => 'post',
            ]
        );

        $this->add_control(
            'posts_per_page',
            [
                'label'       => __('Posts Per Page', 'nb-eazy-addons-for-elementor'),
                'type'        => Controls_Manager::NUMBER,
                'default'     => 6,
                'placeholder' => 6,
            ]
        );

        $this->add_control(
            'order',
            [
                'label'   => __('Order', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'ASC'  => __('Ascending', 'nb-eazy-addons-for-elementor'),
                    'DESC' => __('Descending', 'nb-eazy-addons-for-elementor'),
                ],
                'default' => 'DESC',
            ]
        );

        $this->add_control(
            'orderby',
            [
                'label'   => __('Order By', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'date'       => __('Date', 'nb-eazy-addons-for-elementor'),
                    'title'      => __('Title', 'nb-eazy-addons-for-elementor'),
                    'menu_order' => __('Menu Order', 'nb-eazy-addons-for-elementor'),
                    'rand'       => __('Random', 'nb-eazy-addons-for-elementor'),
                ],
                'default' => 'date',
            ]
        );

        $this->add_control(
            'category',
            [
                'label'    => __('Category', 'nb-eazy-addons-for-elementor'),
                'type'     => Controls_Manager::SELECT2,
                'options'  => $this->get_post_categories(),
                'multiple' => true,
            ]
        );

        $this->end_controls_section();

        // Image Settings Section
        $this->start_controls_section(
            'image_settings',
            [
                'label' => __('Image Settings', 'nb-eazy-addons-for-elementor'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'default_thumbnail',
            [
                'label' => __('Default Thumbnail', 'nb-eazy-addons-for-elementor'),
                'type' => Controls_Manager::MEDIA,
                'description' => __('Upload a default image to show when posts don\'t have featured images.', 'nb-eazy-addons-for-elementor'),
                'default' => [
                    'url' => '',
                ],
            ]
        );

        $this->add_control(
            'show_default_thumbnail',
            [
                'label' => __('Show Default Thumbnail', 'nb-eazy-addons-for-elementor'),
                'type' => Controls_Manager::SWITCHER,
                'description' => __('Show default thumbnail when post has no featured image.', 'nb-eazy-addons-for-elementor'),
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'image_size',
            [
                'label' => __('Image Size', 'nb-eazy-addons-for-elementor'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'thumbnail' => __('Thumbnail (150x150)', 'nb-eazy-addons-for-elementor'),
                    'medium' => __('Medium (300x300)', 'nb-eazy-addons-for-elementor'),
                    'medium_large' => __('Medium Large (768x0)', 'nb-eazy-addons-for-elementor'),
                    'large' => __('Large (1024x1024)', 'nb-eazy-addons-for-elementor'),
                    'full' => __('Full Size', 'nb-eazy-addons-for-elementor'),
                ],
                'default' => 'medium_large',
            ]
        );

        $this->end_controls_section();

        // Carousel Settings Section
        $this->start_controls_section(
            'carousel_settings',
            [
                'label' => __('Carousel Settings', 'nb-eazy-addons-for-elementor'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'slides_to_show',
            [
                'label'   => __('Slides to Show', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::NUMBER,
                'min'     => 1,
                'max'     => 10,
                'step'    => 1,
                'default' => 3,
            ]
        );

        $this->add_control(
            'slides_to_scroll',
            [
                'label'   => __('Slides to Scroll', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::NUMBER,
                'min'     => 1,
                'max'     => 10,
                'step'    => 1,
                'default' => 1,
            ]
        );

        $this->add_control(
            'autoplay',
            [
                'label'   => __('Autoplay', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'autoplay_speed',
            [
                'label'     => __('Autoplay Speed (ms)', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::NUMBER,
                'default'   => 3000,
                'condition' => [
                    'autoplay' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'infinite',
            [
                'label'   => __('Infinite Loop', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_arrows',
            [
                'label'   => __('Show Arrows', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_dots',
            [
                'label'   => __('Show Dots', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SWITCHER,
                'default' => 'no',
            ]
        );

        $this->add_control(
            'pause_on_hover',
            [
                'label'   => __('Pause on Hover', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_responsive_control(
            'space_between',
            [
                'label'   => __('Space Between', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::NUMBER,
                'min'     => 0,
                'max'     => 100,
                'step'    => 1,
                'default' => 30,
            ]
        );

        $this->end_controls_section();

        // Style Section
        $this->start_controls_section(
            'style_section',
            [
                'label' => __('Style', 'nb-eazy-addons-for-elementor'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'title_typography',
                'label'    => __('Title Typography', 'nb-eazy-addons-for-elementor'),
                'selector' => '{{WRAPPER}} .eafe-post-title',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label'     => __('Title Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .eafe-post-title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'hover_effect',
            [
                'label'   => __('Hover Effect', 'nb-eazy-addons-for-elementor'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'none'   => __('None', 'nb-eazy-addons-for-elementor'),
                    'zoom'   => __('Zoom', 'nb-eazy-addons-for-elementor'),
                    'fade'   => __('Fade', 'nb-eazy-addons-for-elementor'),
                    'shadow' => __('Shadow', 'nb-eazy-addons-for-elementor'),
                ],
                'default' => 'none',
            ]
        );

        $this->add_control(
            'arrow_color',
            [
                'label'     => __('Arrow Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .eafe-carousel-arrow' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_arrows' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'dot_color',
            [
                'label'     => __('Dot Color', 'nb-eazy-addons-for-elementor'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .eafe-carousel-dots .swiper-pagination-bullet' => 'background-color: {{VALUE}};',
                ],
                'condition' => [
                    'show_dots' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        // Include the render file and call the render function
        require_once __DIR__ . '/renders/render-eafe-post-carousel-widget.php';
        \EazyAddonsForElementor\Widgets\Renders\render_eafe_post_carousel_widget($settings);
    }

    // Get posts data for editor
    private function get_posts_data($settings) {
        $args = [
            'post_type'      => $settings['post_type'],
            'posts_per_page' => $settings['posts_per_page'],
            'order'          => $settings['order'],
            'orderby'        => $settings['orderby'],
        ];

        if (!empty($settings['category'])) {
            $args['cat'] = implode(',', $settings['category']);
        }

        $query = new \WP_Query($args);
        $posts_data = [];

        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();

                // Get thumbnail with fallback to default
                $thumbnail = '';
                if (has_post_thumbnail()) {
                    $thumbnail = get_the_post_thumbnail_url(get_the_ID(), $settings['image_size'] ?? 'medium_large');
                } elseif ($settings['show_default_thumbnail'] === 'yes' && !empty($settings['default_thumbnail']['url'])) {
                    $thumbnail = $settings['default_thumbnail']['url'];
                }

                $posts_data[] = [
                    'id' => get_the_ID(),
                    'title' => get_the_title(),
                    'permalink' => get_permalink(),
                    'thumbnail' => $thumbnail,
                    'excerpt' => get_the_excerpt(),
                ];
            }
        }

        wp_reset_postdata();
        return $posts_data;
    }

    // Add this method for editor content template
    /**
     * Clean content_template method without any inline scripts
     * Place this in your EAFE_Post_Carousel_Widget class
     */
    protected function content_template() {
        ?>
        <#
        var slidesToShow = settings.slides_to_show || 3;
        var spaceBetween = settings.space_between || 30;
        var showArrows = settings.show_arrows === 'yes';
        var showDots = settings.show_dots === 'yes';

        var carouselSettings = {
        slidesToShow: slidesToShow,
        spaceBetween: spaceBetween,
        slidesToScroll: settings.slides_to_scroll || 1,
        infinite: settings.infinite === 'yes',
        autoplay: settings.autoplay === 'yes',
        autoplaySpeed: settings.autoplay_speed || 3000,
        arrows: showArrows,
        dots: showDots,
        pauseOnHover: settings.pause_on_hover === 'yes'
        };

        // Generate a unique ID for this instance
        var uniqueId = 'eafe-carousel-' + Math.random().toString(36).substr(2, 9);
        #>

        <div class="eafe-post-carousel elementor-editor-carousel"
             id="{{{ uniqueId }}}"
             data-settings='{{{ JSON.stringify(carouselSettings) }}}'
             data-widget-settings='{{{ JSON.stringify(settings) }}}'
             data-editor-mode="true"
             data-unique-id="{{{ uniqueId }}}">

            <div class="swiper-wrapper" id="carousel-content-{{{ uniqueId }}}">
                <div class="eafe-loading-posts">
                    <p><?php esc_html_e('Loading posts...', 'nb-eazy-addons-for-elementor'); ?></p>
                </div>
            </div>

            <# if (showArrows) { #>
            <div class="eafe-carousel-arrows">
                <div class="eafe-carousel-arrow eafe-carousel-prev swiper-button-prev"></div>
                <div class="eafe-carousel-arrow eafe-carousel-next swiper-button-next"></div>
            </div>
            <# } #>

            <# if (showDots) { #>
            <div class="eafe-carousel-dots swiper-pagination"></div>
            <# } #>
        </div>
        <?php
    }

    private function get_post_categories()
    {
        $categories = get_categories();
        $options    = [];
        foreach ($categories as $category) {
            $options[$category->term_id] = $category->name;
        }
        return $options;
    }
}