<?php

namespace EazyAddonsForElementor\Ajax;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * AJAX Handler for Post Carousel Widget
 */
class PostCarouselAjax
{
    /**
     * Initialize AJAX handlers
     */
    public function init()
    {
        // Hook for logged-in users
        add_action('wp_ajax_eafe_get_carousel_posts', [$this, 'get_carousel_posts']);

        // Hook for non-logged-in users (if you want to support frontend editing)
        add_action('wp_ajax_nopriv_eafe_get_carousel_posts', [$this, 'get_carousel_posts']);
    }

    /**
     * Handle AJAX request for carousel posts
     */
    public function get_carousel_posts()
    {
        // Verify nonce for security
        if (!wp_verify_nonce($_POST['nonce'] ?? '', 'eafe_carousel_nonce')) {
            wp_send_json_error([
                'message' => __('Security check failed.', 'nb-eazy-addons-for-elementor')
            ]);
        }

        // Get and sanitize settings
        $settings = $this->sanitize_settings($_POST['settings'] ?? []);

        if (empty($settings)) {
            wp_send_json_error([
                'message' => __('Invalid settings provided.', 'nb-eazy-addons-for-elementor')
            ]);
        }

        // Get posts data
        $posts_data = $this->get_posts_data($settings);

        // Generate HTML
        $html = $this->generate_carousel_html($posts_data, $settings);

        wp_send_json_success([
            'html' => $html,
            'posts_count' => count($posts_data)
        ]);
    }

    /**
     * Sanitize widget settings
     *
     * @param array $settings Raw settings from AJAX request
     * @return array Sanitized settings
     */
    private function sanitize_settings($settings)
    {
        if (!is_array($settings)) {
            return [];
        }

        $sanitized = [
            'post_type'      => sanitize_text_field($settings['post_type'] ?? 'post'),
            'posts_per_page' => absint($settings['posts_per_page'] ?? 6),
            'order'          => in_array($settings['order'] ?? '', ['ASC', 'DESC']) ? $settings['order'] : 'DESC',
            'orderby'        => sanitize_text_field($settings['orderby'] ?? 'date'),
            'category'       => array_map('absint', (array) ($settings['category'] ?? [])),
            'hover_effect'   => sanitize_text_field($settings['hover_effect'] ?? 'none'),
            'image_size'     => sanitize_text_field($settings['image_size'] ?? 'medium_large'),
            'show_default_thumbnail' => sanitize_text_field($settings['show_default_thumbnail'] ?? 'yes'),
        ];

        // Handle default thumbnail setting
        if (!empty($settings['default_thumbnail']['url'])) {
            $sanitized['default_thumbnail'] = [
                'url' => esc_url_raw($settings['default_thumbnail']['url']),
                'id'  => absint($settings['default_thumbnail']['id'] ?? 0)
            ];
        }

        return $sanitized;
    }

    /**
     * Get posts data based on settings
     *
     * @param array $settings Sanitized settings
     * @return array Posts data
     */
    private function get_posts_data($settings)
    {
        $args = [
            'post_type'      => $settings['post_type'],
            'posts_per_page' => $settings['posts_per_page'],
            'order'          => $settings['order'],
            'orderby'        => $settings['orderby'],
            'post_status'    => 'publish',
        ];

        // Add category filter if specified
        if (!empty($settings['category']) && $settings['post_type'] === 'post') {
            $args['cat'] = implode(',', $settings['category']);
        }

        $query = new \WP_Query($args);
        $posts_data = [];

        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $posts_data[] = [
                    'id'        => get_the_ID(),
                    'title'     => get_the_title(),
                    'permalink' => get_permalink(),
                    'thumbnail' => $this->get_post_thumbnail($settings),
                    'excerpt'   => $this->get_post_excerpt(),
                    'date'      => get_the_date(),
                    'author'    => get_the_author(),
                ];
            }
        }

        wp_reset_postdata();
        return $posts_data;
    }

    /**
     * Get post thumbnail URL with default fallback
     *
     * @param array $settings Widget settings
     * @return string Thumbnail URL or empty string
     */
    private function get_post_thumbnail($settings)
    {
        $image_size = $settings['image_size'] ?? 'medium_large';

        // Try to get featured image first
        if (has_post_thumbnail()) {
            return get_the_post_thumbnail_url(get_the_ID(), $image_size);
        }

        // Check if default thumbnail should be used
        if ($settings['show_default_thumbnail'] === 'yes' && !empty($settings['default_thumbnail']['url'])) {
            return $settings['default_thumbnail']['url'];
        }

        // Return empty string if no image available
        return '';
    }

    /**
     * Get post excerpt with fallback
     *
     * @return string Post excerpt
     */
    private function get_post_excerpt()
    {
        $excerpt = get_the_excerpt();

        if (empty($excerpt)) {
            $excerpt = wp_trim_words(get_the_content(), 20, '...');
        }

        return $excerpt;
    }

    /**
     * Generate carousel HTML
     *
     * @param array $posts_data Posts data
     * @param array $settings Widget settings
     * @return string Generated HTML
     */
    private function generate_carousel_html($posts_data, $settings)
    {
        if (empty($posts_data)) {
            return '<div class="eafe-no-posts"><p>' .
                esc_html__('No posts found.', 'nb-eazy-addons-for-elementor') .
                '</p></div>';
        }

        ob_start();

        foreach ($posts_data as $post) {
            $this->render_carousel_slide($post, $settings);
        }

        return ob_get_clean();
    }

    /**
     * Render individual carousel slide
     *
     * @param array $post Post data
     * @param array $settings Widget settings
     */
    private function render_carousel_slide($post, $settings)
    {
        $hover_class = !empty($settings['hover_effect']) ? 'eafe-hover-' . $settings['hover_effect'] : '';
        ?>
        <div class="swiper-slide">
            <div class="eafe-post-item <?php echo esc_attr($hover_class); ?>">
                <?php if (!empty($post['thumbnail'])): ?>
                    <div class="eafe-post-thumbnail">
                        <a href="<?php echo esc_url($post['permalink']); ?>">
                            <img src="<?php echo esc_url($post['thumbnail']); ?>"
                                 alt="<?php echo esc_attr($post['title']); ?>"
                                 loading="lazy">
                        </a>
                    </div>
                <?php endif; ?>

                <div class="eafe-post-content">
                    <h3 class="eafe-post-title">
                        <a href="<?php echo esc_url($post['permalink']); ?>">
                            <?php echo esc_html($post['title']); ?>
                        </a>
                    </h3>

                    <?php if (!empty($post['excerpt'])): ?>
                        <div class="eafe-post-excerpt">
                            <p><?php echo esc_html($post['excerpt']); ?></p>
                        </div>
                    <?php endif; ?>

                    <div class="eafe-post-meta">
                        <span class="eafe-post-date"><?php echo esc_html($post['date']); ?></span>
                        <span class="eafe-post-author"><?php echo esc_html($post['author']); ?></span>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
}