<?php

namespace EazyAddonsForElementor;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Loader
{
    private $widgets_list = [];

    public function init()
    {
        // Register the custom widget category
        $this->register_widget_category();

        // Hook to register widgets once Elementor is initialized
        add_action('elementor/widgets/register', [$this, 'register_widgets']);

        // Initialize AJAX handlers
        $this->init_ajax_handlers();

        // Hook for admin-specific functionality
        if (is_admin()) {
            $this->init_admin();
        }

        // Clear cache when settings change
        $this->clear_elementor_cache();
    }

    private function register_widget_category()
    {
        add_action('elementor/elements/categories_registered', function ($elements_manager) {
            $elements_manager->add_category(
                'eazy-addons-for-elementor-category',
                [
                    'title' => __('Eazy Addons For Elementor', 'nb-eazy-addons-for-elementor'),
                    'icon'  => 'fa fa-plug',
                ]
            );
        });
    }

    public function register_widgets($widgets_manager)
    {
        $widget_files = glob(__DIR__ . '/widgets/*.php');

        foreach ($widget_files as $file) {
            require_once $file;

            $class_name = $this->get_class_name_from_file($file);

            if (class_exists($class_name)) {
                // Store widget info for admin dashboard
                $this->store_widget_info($class_name, $file);

                // Check if widget is enabled before registering
                if ($this->is_widget_enabled($class_name)) {
                    $widgets_manager->register(new $class_name());
                }

                // Enqueue assets for this widget
                $this->enqueue_widget_assets($class_name);
            }
        }
    }

    /**
     * Store widget information for admin dashboard
     */
    private function store_widget_info($class_name, $file)
    {
        $widget_name = $this->get_widget_name($class_name);
        $base_class_name = substr(strrchr($class_name, '\\'), 1);
        $widget_title = $this->get_widget_title($base_class_name);

        $this->widgets_list[$widget_name] = [
            'title' => $widget_title,
            'class' => $class_name,
            'file' => $file,
            'enabled' => $this->is_widget_enabled($class_name)
        ];
    }

    /**
     * Get widget title from class name or constant
     */
    private function get_widget_title($class_name)
    {
        $full_class_name = 'EazyAddonsForElementor\\Widgets\\' . $class_name;
        if (class_exists($full_class_name) && defined("$full_class_name::TITLE")) {
            return $full_class_name::TITLE;
        }

        $title = str_replace(['EAFE_', '_Widget'], '', $class_name);
        $title = str_replace('_', ' ', $title);
        return ucwords(strtolower($title));
    }

    /**
     * Check if widget is enabled in settings
     */
    private function is_widget_enabled($class_name)
    {
        $widget_name = $this->get_widget_name($class_name);
        $settings = get_option('eafe_settings', []);

        if (!isset($settings['widgets'][$widget_name])) {
            return true;
        }

        return (bool) $settings['widgets'][$widget_name];
    }

    /**
     * Get the list of widgets for admin dashboard
     */
    public function get_widgets_list()
    {
        return $this->widgets_list;
    }

    private function enqueue_widget_assets($class_name)
    {
        $widget_name = $this->get_widget_name($class_name);
        $css_url = EAFE_URL . "assets/css/{$widget_name}.css";
        $js_url = EAFE_URL . "assets/js/{$widget_name}.js";
        $css_file_path = EAFE_PATH . "assets/css/{$widget_name}.css";
        $js_file_path = EAFE_PATH . "assets/js/{$widget_name}.js";
        $script_handle = 'eazy-addons-for-elementor-' . $widget_name;

        // FRONTEND enqueue - only if widget is enabled
        add_action('elementor/frontend/after_enqueue_scripts', function () use ($widget_name, $css_file_path, $js_file_path, $css_url, $js_url, $script_handle) {
            if (!$this->is_widget_enabled_by_name($widget_name)) {
                return;
            }

            if (file_exists($css_file_path)) {
                wp_enqueue_style($script_handle . '-style', $css_url, [], EAFE_VERSION);
            }

            if (file_exists($js_file_path)) {
                $dependencies = ['jquery'];

                // Set widget-specific dependencies
                if ($widget_name === 'eafe-post-carousel-widget' || $widget_name === 'eafe-slider-widget') {
                    $dependencies[] = 'swiper';
                } elseif ($widget_name === 'eafe-fun-counter-widget') {
                    $dependencies[] = 'elementor-frontend';
                }

                wp_enqueue_script($script_handle, $js_url, $dependencies, EAFE_VERSION, true);
                $this->localize_widget_script($script_handle, $widget_name);
            }
        });

        // ELEMENTOR EDITOR enqueue - always enqueue in editor
        add_action('elementor/editor/after_enqueue_scripts', function () use ($widget_name, $css_file_path, $js_file_path, $css_url, $js_url, $script_handle) {
            if (file_exists($css_file_path)) {
                wp_enqueue_style($script_handle . '-editor-style', $css_url, [], EAFE_VERSION);
            }

            if (file_exists($js_file_path)) {
                $dependencies = ['jquery'];

                if ($widget_name === 'eafe-post-carousel-widget' || $widget_name === 'eafe-slider-widget') {
                    $dependencies[] = 'swiper';
                } elseif ($widget_name === 'eafe-fun-counter-widget') {
                    $dependencies[] = 'elementor-frontend';
                }

                wp_enqueue_script($script_handle . '-editor', $js_url, $dependencies, EAFE_VERSION, true);
                $this->localize_widget_script($script_handle . '-editor', $widget_name);
            }
        });
    }

    /**
     * Check if widget is enabled by widget name
     */
    private function is_widget_enabled_by_name($widget_name)
    {
        $settings = get_option('eafe_settings', []);
        return !isset($settings['widgets'][$widget_name]) ? true : (bool) $settings['widgets'][$widget_name];
    }

    /**
     * Localize script with widget-specific data
     */
    private function localize_widget_script($script_handle, $widget_name)
    {
        $localization_data = [
            'ajaxUrl'   => admin_url('admin-ajax.php'),
            'nonce'     => wp_create_nonce('eafe_nonce'),
            'isEditor'  => (defined('ELEMENTOR_VERSION') && \Elementor\Plugin::$instance->editor->is_edit_mode()),
            'isPreview' => (defined('ELEMENTOR_VERSION') && \Elementor\Plugin::$instance->preview->is_preview_mode()),
        ];

        // Widget-specific localization
        switch ($widget_name) {
            case 'eafe-post-carousel-widget':
                $localization_data = array_merge($localization_data, [
                    'carouselNonce' => wp_create_nonce('eafe_carousel_nonce'),
                    'strings' => [
                        'loadingPosts' => __('Loading posts...', 'nb-eazy-addons-for-elementor'),
                        'noPostsFound' => __('No posts found.', 'nb-eazy-addons-for-elementor'),
                        'errorLoading' => __('Error loading posts.', 'nb-eazy-addons-for-elementor'),
                    ],
                ]);
                break;

            case 'eafe-progress-bar-widget':
                $localization_data['animationSpeed'] = apply_filters('eafe_progress_bar_animation_speed', 1000);
                break;

            case 'eafe-fun-counter-widget':
                $localization_data['animationEnabled'] = true;
                break;
        }

        $object_name = 'eafe' . str_replace('-', '', ucwords($widget_name, '-')) . 'Data';
        wp_localize_script($script_handle, $object_name, $localization_data);
    }

    /**
     * Extract the widget name from the class name
     */
    private function get_widget_name($class_name)
    {
        $base_class_name = substr(strrchr($class_name, '\\'), 1);
        $widget_name = strtolower(preg_replace('/([a-z])([A-Z])/', '$1-$2', $base_class_name));
        return str_replace('_', '-', $widget_name);
    }

    private function init_admin()
    {
        require_once __DIR__ . '/../admin/Dashboard.php';
        $dashboard = new Admin\Dashboard();
        $dashboard->init();
    }

    private function get_class_name_from_file($file)
    {
        $base_name = basename($file, '.php');
        return 'EazyAddonsForElementor\\Widgets\\' . $base_name;
    }

    /**
     * Initialize AJAX handlers
     */
    private function init_ajax_handlers()
    {
        require_once __DIR__ . '/ajax/PostCarouselAjax.php';
        $carousel_ajax = new Ajax\PostCarouselAjax();
        $carousel_ajax->init();
    }

    /**
     * Get available widgets list for admin dashboard
     */
    public static function get_widgets_for_dashboard()
    {
        $widget_files = glob(__DIR__ . '/widgets/*.php');
        $widgets_list = [];

        foreach ($widget_files as $file) {
            $base_name = basename($file, '.php');

            // Skip if not a widget file
            if (strpos($base_name, 'EAFE_') !== 0) {
                continue;
            }

            $widget_name = strtolower(preg_replace('/([a-z])([A-Z])/', '$1-$2', $base_name));
            $widget_name = str_replace('_', '-', $widget_name);

            $title = str_replace(['EAFE_', '_Widget'], '', $base_name);
            $title = str_replace('_', ' ', $title);
            $title = preg_replace('/([a-z])([A-Z])/', '$1 $2', $title);
            $title = ucwords($title);

            $widgets_list[$widget_name] = [
                'title' => $title,
                'class' => 'EazyAddonsForElementor\\Widgets\\' . $base_name,
                'file' => $file,
                'enabled' => true
            ];
        }

        return $widgets_list;
    }

    /**
     * Clear Elementor cache when settings change
     */
    private function clear_elementor_cache()
    {
        add_action('update_option_eafe_settings', function() {
            if (class_exists('\Elementor\Plugin')) {
                \Elementor\Plugin::$instance->files_manager->clear_cache();
            }
        });
    }
}