/**
 * Smart Slider Widget JavaScript
 * Follows WordPress Coding Standards for JavaScript
 */

/* global jQuery, elementorFrontend, elementor, Swiper */

(function($, window, document) {
  'use strict';

  /**
   * Smart Slider Handler Class
   */
  var SmartSliderHandler = {

    /**
     * Initialize the handler
     */
    init: function() {
      this.bindEvents();
    },

    /**
     * Bind events
     */
    bindEvents: function() {
      var self = this;

      // Elementor frontend initialization
      $(window).on('elementor/frontend/init', function() {
        self.onElementorFrontendInit();
      });

      // Document ready fallback
      $(document).ready(function() {
        self.initializeAllSliders();
      });

      // Editor-specific bindings
      if (typeof elementor !== 'undefined') {
        self.bindEditorEvents();
      }
    },

    /**
     * Handle Elementor frontend initialization
     */
    onElementorFrontendInit: function() {
      var self = this;

      if (typeof elementorFrontend !== 'undefined') {
        // Register handler for specific widget
        elementorFrontend.hooks.addAction(
            'frontend/element_ready/eafe-smart-slider.default',
            function($scope) {
              self.handleWidget($scope, $);
            }
        );

        // Fallback for generic widget handler
        elementorFrontend.hooks.addAction(
            'frontend/element_ready/widget',
            function($scope) {
              if ($scope.find('.eafe-smart-slider').length) {
                self.handleWidget($scope, $);
              }
            }
        );
      }
    },

    /**
     * Bind editor-specific events
     */
    bindEditorEvents: function() {
      var self = this;

      // Listen for widget changes in the editor
      elementor.channels.editor.on('change', function() {
        setTimeout(function() {
          self.initializeAllSliders();
        }, 100);
      });

      // Listen for preview refresh
      elementor.channels.data.on('element:after:add', function() {
        setTimeout(function() {
          self.initializeAllSliders();
        }, 200);
      });
    },

    /**
     * Widget-specific handler
     *
     * @param {jQuery} $scope - The widget scope
     * @param {jQuery} $ - jQuery instance
     */
    handleWidget: function($scope, $) {
      var $slider = $scope.find('.eafe-smart-slider');

      if ($slider.length) {
        this.initializeSingleSlider($slider);
      }
    },

    /**
     * Initialize all sliders on the page
     */
    initializeAllSliders: function() {
      var self = this;

      $('.eafe-smart-slider').each(function() {
        self.initializeSingleSlider($(this));
      });
    },

    /**
     * Initialize a single slider
     *
     * @param {jQuery} $slider - The slider element
     */
    initializeSingleSlider: function($slider) {
      var self = this;

      // Destroy existing instance
      if ($slider[0].swiper) {
        $slider[0].swiper.destroy(true, true);
      }

      var settings = $slider.data('settings') || this.getDefaultSettings();

      // Small delay to ensure DOM is ready
      setTimeout(function() {
        self.createSwiperInstance($slider, settings);
      }, 50);
    },

    /**
     * Get default slider settings
     *
     * @return {Object} Default settings
     */
    getDefaultSettings: function() {
      return {
        effect: 'slide',
        direction: 'horizontal',
        slidesPerView: 1,
        spaceBetween: 0,
        speed: 500,
        autoplay: true,
        autoplaySpeed: 3000,
        loop: true,
        mousewheel: false,
        keyboard: true,
        arrows: true,
        dots: true,
        pauseOnHover: true
      };
    },

    /**
     * Create Swiper instance
     *
     * @param {jQuery} $slider - The slider element
     * @param {Object} settings - Slider settings
     */
    createSwiperInstance: function($slider, settings) {
      var swiperConfig = this.buildSwiperConfig($slider, settings);

      try {
        var swiper = new Swiper($slider[0], swiperConfig);
        this.handleVideoPlayback(swiper);
      } catch (error) {
        console.warn('EAFE: Swiper initialization failed:', error);
      }
    },

    /**
     * Build Swiper configuration
     *
     * @param {jQuery} $slider - The slider element
     * @param {Object} settings - Slider settings
     * @return {Object} Swiper configuration
     */
    buildSwiperConfig: function($slider, settings) {
      var slideCount = $slider.find('.swiper-slide').length;
      var canLoop = settings.loop && slideCount > settings.slidesPerView;

      var config = {
        effect: settings.effect,
        direction: settings.direction,
        slidesPerView: settings.slidesPerView,
        spaceBetween: settings.spaceBetween,
        speed: settings.speed,
        loop: canLoop,
        autoplay: this.getAutoplayConfig(settings),
        mousewheel: settings.mousewheel,
        keyboard: settings.keyboard,
        navigation: this.getNavigationConfig($slider, settings),
        pagination: this.getPaginationConfig($slider, settings),
        on: {
          init: function() {
            $slider.addClass('swiper-initialized');
          }
        }
      };

      // Additional configs for effects
      if (settings.effect === 'coverflow') {
        config.coverflowEffect = {
          rotate: 30,
          slideShadows: false
        };
      }

      if (settings.effect === 'cube') {
        config.cubeEffect = {
          slideShadows: false
        };
      }

      return config;
    },

    /**
     * Get autoplay configuration
     *
     * @param {Object} settings - Slider settings
     * @return {Object|boolean} Autoplay config or false
     */
    getAutoplayConfig: function(settings) {
      if (!settings.autoplay) {
        return false;
      }

      return {
        delay: settings.autoplaySpeed,
        disableOnInteraction: false,
        pauseOnMouseEnter: settings.pauseOnHover
      };
    },

    /**
     * Get navigation configuration
     *
     * @param {jQuery} $slider - The slider element
     * @param {Object} settings - Slider settings
     * @return {Object|boolean} Navigation config or false
     */
    getNavigationConfig: function($slider, settings) {
      if (!settings.arrows) {
        return false;
      }

      return {
        nextEl: $slider.find('.eafe-slider-next')[0],
        prevEl: $slider.find('.eafe-slider-prev')[0]
      };
    },

    /**
     * Get pagination configuration
     *
     * @param {jQuery} $slider - The slider element
     * @param {Object} settings - Slider settings
     * @return {Object|boolean} Pagination config or false
     */
    getPaginationConfig: function($slider, settings) {
      if (!settings.dots) {
        return false;
      }

      return {
        el: $slider.find('.eafe-slider-pagination')[0],
        clickable: true
      };
    },

    /**
     * Handle video playback in slides
     *
     * @param {Swiper} swiper - The swiper instance
     */
    handleVideoPlayback: function(swiper) {
      swiper.on('slideChangeTransitionStart', function() {
        var previousSlide = swiper.slides[swiper.previousIndex];
        var activeSlide = swiper.slides[swiper.activeIndex];

        // Pause video in previous slide
        var prevVideo = $(previousSlide).find('video')[0];
        if (prevVideo) {
          prevVideo.pause();
        }

        // Play video in active slide
        var activeVideo = $(activeSlide).find('video')[0];
        if (activeVideo) {
          activeVideo.play();
        }
      });
    }
  };

  // Make functions globally available for compatibility
  window.initializeSliders = function() {
    SmartSliderHandler.initializeAllSliders();
  };

  // Initialize the handler
  SmartSliderHandler.init();

})(jQuery, window, document);