/**
 * Post Carousel Widget JavaScript
 * Follows WordPress Coding Standards for JavaScript
 */

/* global jQuery, elementorFrontend, elementor, Swiper, eafePostCarouselWidgetData */

(function($, window, document) {
  'use strict';

  /**
   * Post Carousel Handler Class
   */
  var PostCarouselHandler = {

    /**
     * Initialize the handler
     */
    init: function() {
      this.bindEvents();
    },

    /**
     * Bind events
     */
    bindEvents: function() {
      var self = this;

      // Elementor frontend initialization
      $(window).on('elementor/frontend/init', function() {
        self.onElementorFrontendInit();
      });

      // Document ready fallback
      $(document).ready(function() {
        self.initializeAllCarousels();
      });

      // Editor-specific bindings
      if (typeof elementor !== 'undefined') {
        self.bindEditorEvents();
      }
    },

    /**
     * Handle Elementor frontend initialization
     */
    onElementorFrontendInit: function() {
      var self = this;

      if (typeof elementorFrontend !== 'undefined') {
        // Register handler for specific widget
        elementorFrontend.hooks.addAction(
            'frontend/element_ready/eafe-post-carousel.default',
            function($scope) {
              self.handleWidget($scope, $);
            }
        );

        // Fallback for generic widget handler
        elementorFrontend.hooks.addAction(
            'frontend/element_ready/widget',
            function($scope) {
              if ($scope.find('.eafe-post-carousel').length) {
                self.handleWidget($scope, $);
              }
            }
        );
      }
    },

    /**
     * Bind editor-specific events
     */
    bindEditorEvents: function() {
      var self = this;

      // Listen for widget changes in the editor
      elementor.channels.editor.on('change', function() {
        setTimeout(function() {
          self.initializeAllCarousels();
        }, 100);
      });

      // Listen for preview refresh
      elementor.channels.data.on('element:after:add', function() {
        setTimeout(function() {
          self.initializeAllCarousels();
        }, 200);
      });

      // Listen for editor carousel events
      EditorCarouselHandler.init();
    },

    /**
     * Widget-specific handler
     *
     * @param {jQuery} $scope - The widget scope
     * @param {jQuery} $ - jQuery instance
     */
    handleWidget: function($scope, $) {
      var $carousel = $scope.find('.eafe-post-carousel');

      if ($carousel.length) {
        this.initializeSingleCarousel($carousel);
      }
    },

    /**
     * Initialize all carousels on the page
     */
    initializeAllCarousels: function() {
      var self = this;

      $('.eafe-post-carousel').each(function() {
        self.initializeSingleCarousel($(this));
      });
    },

    /**
     * Initialize a single carousel
     *
     * @param {jQuery} $carousel - The carousel element
     */
    initializeSingleCarousel: function($carousel) {
      var self = this;

      // Destroy existing instance
      if ($carousel[0].swiper) {
        $carousel[0].swiper.destroy(true, true);
      }

      var settings = $carousel.data('settings') || this.getDefaultSettings();

      // Small delay to ensure DOM is ready
      setTimeout(function() {
        self.createSwiperInstance($carousel, settings);
      }, 50);
    },

    /**
     * Get default carousel settings
     *
     * @return {Object} Default settings
     */
    getDefaultSettings: function() {
      return {
        slidesToShow: 3,
        spaceBetween: 30,
        slidesToScroll: 1,
        infinite: true,
        autoplay: true,
        autoplaySpeed: 3000,
        arrows: true,
        dots: false,
        pauseOnHover: true
      };
    },

    /**
     * Create Swiper instance
     *
     * @param {jQuery} $carousel - The carousel element
     * @param {Object} settings - Carousel settings
     */
    createSwiperInstance: function($carousel, settings) {
      var swiperConfig = this.buildSwiperConfig($carousel, settings);

      try {
        new Swiper($carousel[0], swiperConfig);
      } catch (error) {
        console.warn('EAFE: Swiper initialization failed:', error);
      }
    },

    /**
     * Build Swiper configuration
     *
     * @param {jQuery} $carousel - The carousel element
     * @param {Object} settings - Carousel settings
     * @return {Object} Swiper configuration
     */
    buildSwiperConfig: function($carousel, settings) {
      var slideCount = $carousel.find('.swiper-slide').length;
      var canLoop = settings.infinite && slideCount > settings.slidesToShow;

      return {
        slidesPerView: settings.slidesToShow,
        spaceBetween: settings.spaceBetween,
        slidesPerGroup: settings.slidesToScroll,
        loop: canLoop,
        autoplay: this.getAutoplayConfig(settings),
        navigation: this.getNavigationConfig($carousel, settings),
        pagination: this.getPaginationConfig($carousel, settings),
        breakpoints: this.getBreakpointsConfig(settings),
        on: {
          init: function() {
            $carousel.addClass('swiper-initialized');
          }
        }
      };
    },

    /**
     * Get autoplay configuration
     *
     * @param {Object} settings - Carousel settings
     * @return {Object|boolean} Autoplay config or false
     */
    getAutoplayConfig: function(settings) {
      if (!settings.autoplay) {
        return false;
      }

      return {
        delay: settings.autoplaySpeed,
        disableOnInteraction: false,
        pauseOnMouseEnter: settings.pauseOnHover
      };
    },

    /**
     * Get navigation configuration
     *
     * @param {jQuery} $carousel - The carousel element
     * @param {Object} settings - Carousel settings
     * @return {Object|boolean} Navigation config or false
     */
    getNavigationConfig: function($carousel, settings) {
      if (!settings.arrows) {
        return false;
      }

      return {
        nextEl: $carousel.find('.eafe-carousel-next')[0],
        prevEl: $carousel.find('.eafe-carousel-prev')[0]
      };
    },

    /**
     * Get pagination configuration
     *
     * @param {jQuery} $carousel - The carousel element
     * @param {Object} settings - Carousel settings
     * @return {Object|boolean} Pagination config or false
     */
    getPaginationConfig: function($carousel, settings) {
      if (!settings.dots) {
        return false;
      }

      return {
        el: $carousel.find('.eafe-carousel-dots')[0],
        clickable: true
      };
    },

    /**
     * Get breakpoints configuration
     *
     * @param {Object} settings - Carousel settings
     * @return {Object} Breakpoints config
     */
    getBreakpointsConfig: function(settings) {
      return {
        1024: {
          slidesPerView: Math.min(settings.slidesToShow, 4),
          spaceBetween: settings.spaceBetween
        },
        768: {
          slidesPerView: Math.min(settings.slidesToShow, 2),
          spaceBetween: settings.spaceBetween
        },
        480: {
          slidesPerView: 1,
          spaceBetween: settings.spaceBetween
        }
      };
    }
  };

  /**
   * Editor Carousel Handler for live preview
   */
  var EditorCarouselHandler = {

    /**
     * Initialize editor-specific functionality
     */
    init: function() {
      this.bindEditorEvents();
      this.initializeEditorCarousels();
    },

    /**
     * Bind editor-specific events
     */
    bindEditorEvents: function() {
      var self = this;

      // Watch for new carousels being added to the editor
      $(document).on('DOMNodeInserted', function(e) {
        var $target = $(e.target);
        if ($target.hasClass('eafe-post-carousel') && $target.data('editor-mode')) {
          self.handleEditorCarousel($target);
        }
      });

      // Use MutationObserver for modern browsers (fallback to DOMNodeInserted)
      if (typeof MutationObserver !== 'undefined') {
        var observer = new MutationObserver(function(mutations) {
          mutations.forEach(function(mutation) {
            if (mutation.type === 'childList') {
              $(mutation.addedNodes).each(function() {
                var $node = $(this);
                if ($node.hasClass && $node.hasClass('eafe-post-carousel') && $node.data('editor-mode')) {
                  self.handleEditorCarousel($node);
                }

                // Check children too
                $node.find('.eafe-post-carousel[data-editor-mode="true"]').each(function() {
                  self.handleEditorCarousel($(this));
                });
              });
            }
          });
        });

        observer.observe(document.body, {
          childList: true,
          subtree: true
        });
      }

      // Listen for Elementor editor changes
      if (typeof elementor !== 'undefined') {
        elementor.channels.editor.on('change', function() {
          setTimeout(function() {
            self.initializeEditorCarousels();
          }, 300);
        });

        // Listen for panel changes (when settings are modified)
        elementor.channels.data.on('element:after:add element:after:duplicate', function() {
          setTimeout(function() {
            self.initializeEditorCarousels();
          }, 500);
        });
      }
    },

    /**
     * Initialize all editor carousels
     */
    initializeEditorCarousels: function() {
      var self = this;

      $('.eafe-post-carousel[data-editor-mode="true"]').each(function() {
        self.handleEditorCarousel($(this));
      });
    },

    /**
     * Handle individual editor carousel
     *
     * @param {jQuery} $carousel - The carousel element
     */
    handleEditorCarousel: function($carousel) {
      var uniqueId = $carousel.data('unique-id') || $carousel.attr('id');
      var settings = $carousel.data('widget-settings');

      if (uniqueId && settings) {
        this.loadCarouselContent(uniqueId, settings);
      }
    },

    /**
     * Load carousel content via AJAX
     *
     * @param {string} uniqueId - Unique carousel ID
     * @param {Object} settings - Widget settings
     */
    loadCarouselContent: function(uniqueId, settings) {
      var $carousel = $('#' + uniqueId);
      var $contentContainer = $('#carousel-content-' + uniqueId);

      if (!$carousel.length || !$contentContainer.length) {
        return;
      }

      // Prevent multiple simultaneous requests
      if ($carousel.data('loading')) {
        return;
      }
      $carousel.data('loading', true);

      // Get localized data
      var ajaxData = this.getAjaxData(settings);

      $.ajax({
        url: ajaxData.ajaxUrl,
        type: 'POST',
        data: {
          action: 'eafe_get_carousel_posts',
          settings: settings,
          nonce: ajaxData.nonce
        },
        success: function(response) {
          EditorCarouselHandler.handleAjaxSuccess(response, $contentContainer, uniqueId);
        },
        error: function() {
          EditorCarouselHandler.handleAjaxError($contentContainer);
        },
        complete: function() {
          $carousel.removeData('loading');
        }
      });
    },

    /**
     * Get AJAX data from localized script
     *
     * @param {Object} settings - Widget settings
     * @return {Object} AJAX data
     */
    getAjaxData: function(settings) {
      // Try to get data from localized script
      if (typeof eafePostCarouselWidgetData !== 'undefined') {
        return {
          ajaxUrl: eafePostCarouselWidgetData.ajaxUrl,
          nonce: eafePostCarouselWidgetData.carouselNonce
        };
      }

      // Fallback
      return {
        ajaxUrl: (typeof ajaxurl !== 'undefined') ? ajaxurl : '/wp-admin/admin-ajax.php',
        nonce: ''
      };
    },

    /**
     * Handle successful AJAX response
     *
     * @param {Object} response - AJAX response
     * @param {jQuery} $contentContainer - Content container
     * @param {string} uniqueId - Unique carousel ID
     */
    handleAjaxSuccess: function(response, $contentContainer, uniqueId) {
      if (response.success && response.data.html) {
        $contentContainer.html(response.data.html);

        // Initialize carousel after content is loaded
        setTimeout(function() {
          PostCarouselHandler.initializeAllCarousels();
        }, 100);
      } else {
        var noPostsMessage = this.getLocalizedString('noPostsFound', 'No posts found.');
        $contentContainer.html('<div class="eafe-no-posts"><p>' + noPostsMessage + '</p></div>');
      }
    },

    /**
     * Handle AJAX error
     *
     * @param {jQuery} $contentContainer - Content container
     */
    handleAjaxError: function($contentContainer) {
      var errorMessage = this.getLocalizedString('errorLoading', 'Error loading posts.');
      $contentContainer.html('<div class="eafe-error"><p>' + errorMessage + '</p></div>');
    },

    /**
     * Get localized string
     *
     * @param {string} key - String key
     * @param {string} fallback - Fallback string
     * @return {string} Localized string
     */
    getLocalizedString: function(key, fallback) {
      if (typeof eafePostCarouselWidgetData !== 'undefined' &&
          eafePostCarouselWidgetData.strings &&
          eafePostCarouselWidgetData.strings[key]) {
        return eafePostCarouselWidgetData.strings[key];
      }
      return fallback;
    }
  };

  // Make functions globally available for compatibility
  window.initializePostCarousels = function() {
    PostCarouselHandler.initializeAllCarousels();
  };

  // Initialize the handlers
  PostCarouselHandler.init();

  // Initialize editor handler if in editor context
  if (typeof elementor !== 'undefined' ||
      (typeof eafePostCarouselWidgetData !== 'undefined' && eafePostCarouselWidgetData.isEditor)) {
    EditorCarouselHandler.init();
  }

})(jQuery, window, document);