document.addEventListener('DOMContentLoaded', function () {
  const newsTickers = document.querySelectorAll('.eafe-news-ticker')

  newsTickers.forEach((ticker) => {
    initializeNewsTicker(ticker)
  })
})

function initializeNewsTicker(ticker) {
  const style = ticker.getAttribute('data-ticker-style') || 'horizontal'
  const animationSpeed =
    parseInt(ticker.getAttribute('data-animation-speed')) || 300
  const autoplaySpeed =
    parseInt(ticker.getAttribute('data-autoplay-speed')) || 3000
  const pauseOnHover = ticker.getAttribute('data-pause-on-hover') === 'true'

  const itemsContainer = ticker.querySelector('.eafe-ticker-items')
  const tickerContent = ticker.querySelector('.eafe-ticker-content')
  const items = ticker.querySelectorAll('.eafe-ticker-item')
  const progressBar = ticker.querySelector('.eafe-ticker-progress')
  const prevBtn = ticker.querySelector('.eafe-ticker-prev')
  const nextBtn = ticker.querySelector('.eafe-ticker-next')
  const pausePlayBtn = ticker.querySelector('.eafe-ticker-pause-play')
  const pauseIcon = ticker.querySelector('.eafe-pause-icon')
  const playIcon = ticker.querySelector('.eafe-play-icon')

  let currentIndex = 0
  let isPlaying = true
  let animationInterval
  let progressInterval

  // Check if we have enough items
  if (items.length <= 1) {
    if (pausePlayBtn) pausePlayBtn.style.display = 'none'
    if (prevBtn) prevBtn.style.display = 'none'
    if (nextBtn) nextBtn.style.display = 'none'
    return
  }

  // Initialize based on style
  switch (style) {
    case 'horizontal':
      initHorizontalTicker()
      break
    case 'vertical':
      initVerticalTicker()
      break
    case 'fade':
      initFadeTicker()
      break
  }

  function initHorizontalTicker() {
    // For horizontal, all items are visible and scroll continuously
    items.forEach((item, index) => {
      item.style.display = 'inline-block'
    })

    // Start the horizontal animation
    startHorizontalAnimation()
  }

  function initVerticalTicker() {
    // Hide all items except the first one
    items.forEach((item, index) => {
      if (index !== 0) {
        item.style.display = 'none'
      } else {
        item.style.display = 'block'
      }
    })
  }

  function initFadeTicker() {
    // Position all items absolutely and show only the first one
    items.forEach((item, index) => {
      item.style.position = 'absolute'
      item.style.top = '0'
      item.style.left = '0'
      item.style.right = '0'
      item.style.opacity = index === 0 ? '1' : '0'
      item.style.visibility = index === 0 ? 'visible' : 'hidden'
      item.style.transition = `opacity ${animationSpeed}ms ease-in-out`
    })
  }

  function startHorizontalAnimation() {
    if (style !== 'horizontal') return

    if (animationInterval) {
      clearInterval(animationInterval)
    }

    // For horizontal, we use CSS animation, so we don't need interval for item switching
    // But we still need interval for progress bar if enabled
    if (progressBar) {
      startProgress()
    }
  }

  function nextItem() {
    if (style === 'horizontal') {
      // Horizontal style doesn't use discrete items, so we don't change currentIndex
      return
    }

    currentIndex = (currentIndex + 1) % items.length
    updateDisplay()
    resetProgress()
  }

  function prevItem() {
    if (style === 'horizontal') {
      // Horizontal style doesn't use discrete items
      return
    }

    currentIndex = (currentIndex - 1 + items.length) % items.length
    updateDisplay()
    resetProgress()
  }

  function updateDisplay() {
    if (style === 'horizontal') {
      return // Horizontal uses continuous scroll, not discrete items
    }

    switch (style) {
      case 'vertical':
        items.forEach((item, index) => {
          item.style.display = index === currentIndex ? 'block' : 'none'
        })
        break
      case 'fade':
        items.forEach((item, index) => {
          if (index === currentIndex) {
            item.style.opacity = '1'
            item.style.visibility = 'visible'
          } else {
            item.style.opacity = '0'
            setTimeout(() => {
              if (index !== currentIndex) {
                item.style.visibility = 'hidden'
              }
            }, animationSpeed)
          }
        })
        break
    }

    // Update active class
    items.forEach((item, index) => {
      item.classList.toggle('eafe-ticker-active', index === currentIndex)
    })
  }

  function startAutoplay() {
    if (style === 'horizontal') {
      startHorizontalAnimation()
      return
    }

    if (animationInterval) {
      clearInterval(animationInterval)
    }

    animationInterval = setInterval(nextItem, autoplaySpeed)
    startProgress()
  }

  function pauseAutoplay() {
    if (animationInterval) {
      clearInterval(animationInterval)
      animationInterval = null
    }
    stopProgress()

    // Also pause CSS animation for horizontal style
    if (style === 'horizontal' && itemsContainer) {
      itemsContainer.style.animationPlayState = 'paused'
    }
  }

  function resumeAutoplay() {
    if (style === 'horizontal' && itemsContainer) {
      itemsContainer.style.animationPlayState = 'running'
      startProgress()
    } else if (isPlaying) {
      startAutoplay()
    }
  }

  function togglePlayPause() {
    isPlaying = !isPlaying

    if (isPlaying) {
      resumeAutoplay()
      if (pauseIcon) pauseIcon.style.display = 'block'
      if (playIcon) playIcon.style.display = 'none'
      if (pausePlayBtn) pausePlayBtn.setAttribute('aria-label', 'Pause ticker')
    } else {
      pauseAutoplay()
      if (pauseIcon) pauseIcon.style.display = 'none'
      if (playIcon) playIcon.style.display = 'block'
      if (pausePlayBtn) pausePlayBtn.setAttribute('aria-label', 'Play ticker')
    }
  }

  function startProgress() {
    if (!progressBar) return

    stopProgress()
    progressBar.style.width = '0%'
    progressBar.style.transition = `width ${autoplaySpeed}ms linear`

    setTimeout(() => {
      progressBar.style.width = '100%'
    }, 50)
  }

  function stopProgress() {
    if (!progressBar) return
    progressBar.style.transition = 'none'
    progressBar.style.width = '0%'
  }

  function resetProgress() {
    stopProgress()
    if (isPlaying) {
      setTimeout(startProgress, 50)
    }
  }

  // Event listeners for controls - FIXED
  if (prevBtn) {
    prevBtn.addEventListener('click', (e) => {
      e.preventDefault()
      e.stopPropagation()
      if (style !== 'horizontal') {
        prevItem()
        if (isPlaying) resetProgress()
      }
    })
  }

  if (nextBtn) {
    nextBtn.addEventListener('click', (e) => {
      e.preventDefault()
      e.stopPropagation()
      if (style !== 'horizontal') {
        nextItem()
        if (isPlaying) resetProgress()
      }
    })
  }

  if (pausePlayBtn) {
    pausePlayBtn.addEventListener('click', (e) => {
      e.preventDefault()
      e.stopPropagation()
      togglePlayPause()
    })
  }

  // Pause on hover - ONLY for ticker content area
  if (pauseOnHover && tickerContent) {
    tickerContent.addEventListener('mouseenter', () => {
      if (isPlaying) {
        pauseAutoplay()
      }
    })

    tickerContent.addEventListener('mouseleave', () => {
      if (isPlaying) {
        resumeAutoplay()
      }
    })
  }

  // Initialize autoplay based on reduced motion preference
  const reducedMotion = window.matchMedia(
    '(prefers-reduced-motion: reduce)'
  ).matches
  if (reducedMotion) {
    isPlaying = false
    pauseAutoplay()
    if (pauseIcon) pauseIcon.style.display = 'none'
    if (playIcon) playIcon.style.display = 'block'
    if (pausePlayBtn) pausePlayBtn.setAttribute('aria-label', 'Play ticker')
  } else {
    startAutoplay()
  }

  // Handle visibility change
  document.addEventListener('visibilitychange', () => {
    if (document.hidden) {
      pauseAutoplay()
    } else if (isPlaying) {
      resumeAutoplay()
    }
  })

  // Make sure controls are visible and functional
  if (prevBtn) prevBtn.style.display = 'flex'
  if (nextBtn) nextBtn.style.display = 'flex'
  if (pausePlayBtn) pausePlayBtn.style.display = 'flex'
}
