(function($) {
    'use strict';

    // Check if Swiper is available
    if (typeof Swiper === 'undefined' && typeof window.Swiper === 'undefined') {
        console.warn('Swiper.js is not loaded. Logo Grid carousel will not work.');
        return;
    }

    const SwiperClass = window.Swiper || Swiper;

    // Store swiper instances
    let swiperInstances = {};

    // Initialize logo grid
    function initLogoGrid() {
        $('.eafe-logo-grid-container').each(function() {
            const container = $(this);
            const containerId = container.attr('id') || 'logo-grid-' + Math.random().toString(36).substr(2, 9);

            if (!container.attr('id')) {
                container.attr('id', containerId);
            }

            const isCarousel = container.hasClass('eafe-logo-grid-style-4');

            // Destroy existing swiper instance if it exists
            if (swiperInstances[containerId]) {
                swiperInstances[containerId].destroy(true, true);
                delete swiperInstances[containerId];
            }

            // Handle carousel functionality
            if (isCarousel) {
                initLogoCarousel(container, containerId);
            }

            // Handle tooltips
            initLogoTooltips(container);
        });
    }

    // Initialize logo carousel with Swiper
    function initLogoCarousel(container, containerId) {
        // Get settings from data attributes
        const autoplay = container.data('autoplay') === 'yes' || container.data('autoplay') === true;
        const autoplaySpeed = parseInt(container.data('autoplay-speed')) || 3000;
        const loop = container.data('loop') === 'yes' || container.data('loop') === true;
        const showArrows = container.find('.eafe-logo-swiper').length > 0;
        const showDots = container.data('show-dots') === 'yes' || container.data('show-dots') === true;

        // Find the swiper container
        const swiperContainer = container.find('.eafe-logo-swiper');

        if (swiperContainer.length === 0) {
            console.warn('Swiper container not found');
            return;
        }

        // Add class for loop handling
        if (!loop) {
            container.addClass('no-loop');
        } else {
            container.removeClass('no-loop');
        }

        // Calculate breakpoints based on columns setting
        const columns = parseInt(container.attr('class').match(/eafe-logo-grid-columns-(\d+)/)?.[1]) || 4;

        const breakpoints = {
            320: {
                slidesPerView: 1,
                spaceBetween: 15,
            },
            576: {
                slidesPerView: Math.min(2, columns),
                spaceBetween: 20,
            },
            768: {
                slidesPerView: Math.min(3, columns),
                spaceBetween: 25,
            },
            1024: {
                slidesPerView: Math.min(4, columns),
                spaceBetween: 30,
            },
            1200: {
                slidesPerView: columns,
                spaceBetween: 30,
            }
        };

        // Swiper configuration
        const swiperConfig = {
            slidesPerView: columns,
            spaceBetween: 30,
            loop: loop,
            speed: 600,
            autoplay: autoplay ? {
                delay: autoplaySpeed,
                disableOnInteraction: false,
                pauseOnMouseEnter: true,
            } : false,
            navigation: showArrows ? {
                nextEl: container.find('.swiper-button-next')[0],
                prevEl: container.find('.swiper-button-prev')[0],
            } : false,
            pagination: showDots ? {
                el: container.find('.swiper-pagination')[0],
                clickable: true,
                dynamicBullets: false,
            } : false,
            breakpoints: breakpoints,
            watchSlidesProgress: true,
            watchOverflow: true,
            centerInsufficientSlides: true,
            // Prevent clicks during transition
            preventClicks: true,
            preventClicksPropagation: true,
            // Accessibility
            a11y: {
                enabled: true,
                prevSlideMessage: 'Previous logo',
                nextSlideMessage: 'Next logo',
                firstSlideMessage: 'This is the first logo',
                lastSlideMessage: 'This is the last logo',
            },
            // Keyboard control
            keyboard: {
                enabled: true,
                onlyInViewport: true,
            },
            // Mouse wheel
            mousewheel: {
                forceToAxis: true,
            },
            // Touch events
            touchEventsTarget: 'wrapper',
            simulateTouch: true,
            // Resize observer
            observer: true,
            observeParents: true,
            observeSlideChildren: true,
        };

        try {
            // Initialize Swiper
            const swiper = new SwiperClass(swiperContainer[0], swiperConfig);

            // Store instance
            swiperInstances[containerId] = swiper;

            // Handle autoplay pause on hover
            if (autoplay) {
                container.on('mouseenter', function() {
                    swiper.autoplay.stop();
                });

                container.on('mouseleave', function() {
                    swiper.autoplay.start();
                });
            }

            // Custom events
            swiper.on('slideChange', function() {
                // Custom slide change logic if needed
                container.trigger('logoGridSlideChange', {
                    activeIndex: swiper.activeIndex,
                    realIndex: swiper.realIndex
                });
            });

            swiper.on('reachEnd', function() {
                container.trigger('logoGridReachEnd');
            });

            swiper.on('reachBeginning', function() {
                container.trigger('logoGridReachBeginning');
            });

            // Handle resize
            let resizeTimeout;
            $(window).on('resize.logoGrid', function() {
                clearTimeout(resizeTimeout);
                resizeTimeout = setTimeout(function() {
                    if (swiper && !swiper.destroyed) {
                        swiper.update();
                    }
                }, 250);
            });

            // Update swiper after images load
            swiperContainer.find('img').on('load', function() {
                setTimeout(function() {
                    if (swiper && !swiper.destroyed) {
                        swiper.update();
                    }
                }, 100);
            });

        } catch (error) {
            console.error('Error initializing Swiper:', error);
        }
    }

    // Initialize logo tooltips
    function initLogoTooltips(container) {
        const items = container.find('.eafe-logo-grid-item');

        items.each(function() {
            const item = $(this);
            const tooltip = item.find('.eafe-logo-grid-tooltip');

            if (tooltip.length) {
                item.on('mouseenter', function() {
                    const tooltipWidth = tooltip.outerWidth();
                    const itemWidth = item.outerWidth();
                    const leftPos = (itemWidth - tooltipWidth) / 2;

                    tooltip.css({
                        'left': '50%',
                        'margin-left': -tooltipWidth / 2
                    });
                });
            }
        });
    }

    // Destroy all swiper instances
    function destroyAllSwipers() {
        Object.keys(swiperInstances).forEach(function(key) {
            if (swiperInstances[key] && !swiperInstances[key].destroyed) {
                swiperInstances[key].destroy(true, true);
            }
            delete swiperInstances[key];
        });
    }

    // Initialize on document ready
    $(document).ready(function() {
        // Wait a bit for Swiper to be fully loaded
        setTimeout(function() {
            initLogoGrid();
        }, 100);
    });

    // Reinitialize when Elementor editor changes
    if (typeof elementor !== 'undefined') {
        elementor.hooks.addAction('panel/open_editor/widget', function(panel, model, view) {
            if (model.attributes.widgetType && model.attributes.widgetType.includes('logo-grid')) {
                setTimeout(function() {
                    destroyAllSwipers();
                    initLogoGrid();
                }, 500);
            }
        });
    }

    // Handle Elementor frontend refresh
    $(document).on('elementor/popup/show', function() {
        setTimeout(initLogoGrid, 300);
    });

    // Global functions for external use
    window.reinitLogoGrid = function() {
        destroyAllSwipers();
        initLogoGrid();
    };

    window.destroyLogoGridSwipers = function() {
        destroyAllSwipers();
    };

    // Clean up on page unload
    $(window).on('beforeunload', function() {
        destroyAllSwipers();
        $(window).off('resize.logoGrid');
    });

})(jQuery);