;(function ($) {
  'use strict'

  // Initialize image comparison sliders
  function initImageComparison() {
    $('.eafe-image-comparison-container').each(function () {
      const container = $(this)
      const wrapper = container.closest('.eafe-image-comparison-wrapper')
      const before = container.find('.eafe-image-comparison-before')
      const handle = container.find('.eafe-image-comparison-handle')
      const beforeBadge = container.find('.eafe-badge-before')
      const afterBadge = container.find('.eafe-badge-after')
      const percentageIndicator = container.find('.eafe-percentage')
      const orientation = wrapper.data('orientation') || 'horizontal'
      const interactionMode = container.data('interaction') || 'drag'
      const hoverSensitivity = parseInt(container.data('sensitivity')) || 10

      // Get initial position from data attribute or default to 50%
      let position = container.data('position') || 50
      let isDragging = false
      let isHovering = false

      // Update slider position
      function updateSlider(pos, animate = false) {
        position = pos
        if (position < 0) position = 0
        if (position > 100) position = 100

        if (animate) {
          before
            .stop()
            .animate(
              orientation === 'horizontal'
                ? { width: position + '%' }
                : { height: position + '%' },
              300
            )
          handle
            .stop()
            .animate(
              orientation === 'horizontal'
                ? { left: position + '%' }
                : { top: position + '%' },
              300
            )
        } else {
          if (orientation === 'horizontal') {
            before.css('width', position + '%')
            handle.css('left', position + '%')
          } else {
            before.css('height', position + '%')
            handle.css('top', position + '%')
          }
        }

        if (percentageIndicator.length) {
          percentageIndicator.text(Math.round(position) + '%')
        }

        // Update badge visibility based on position
        if (position < 10) {
          beforeBadge.addClass('hidden')
          afterBadge.removeClass('hidden')
        } else if (position > 90) {
          beforeBadge.removeClass('hidden')
          afterBadge.addClass('hidden')
        } else {
          beforeBadge.removeClass('hidden')
          afterBadge.removeClass('hidden')
        }
      }

      // Set to before view
      function setToBefore() {
        updateSlider(0, true)
      }

      // Set to after view
      function setToAfter() {
        updateSlider(100, true)
      }

      // Reset to default position
      function resetPosition() {
        const defaultPos = container.data('position') || 50
        updateSlider(defaultPos, true)
      }

      // Handle mouse events
      if (interactionMode === 'drag' || interactionMode === 'both') {
        container.on('mousedown', function (e) {
          isDragging = true
          handle.css(orientation === 'horizontal' ? 'width' : 'height', '6px')

          let offset =
            orientation === 'horizontal'
              ? ((e.clientX - container.offset().left) / container.width()) *
                100
              : ((e.clientY - container.offset().top) / container.height()) *
                100

          updateSlider(offset)
        })

        $(document).on('mouseup', function () {
          isDragging = false
          handle.css(orientation === 'horizontal' ? 'width' : 'height', '4px')
        })

        $(document).on('mouseleave', function () {
          isDragging = false
          handle.css(orientation === 'horizontal' ? 'width' : 'height', '4px')
        })

        $(document).on('mousemove', function (e) {
          if (!isDragging) return

          let offset =
            orientation === 'horizontal'
              ? ((e.clientX - container.offset().left) / container.width()) *
                100
              : ((e.clientY - container.offset().top) / container.height()) *
                100

          updateSlider(offset)
        })

        // Touch support
        container.on('touchstart', function (e) {
          isDragging = true
          let touch = e.originalEvent.touches[0]
          let offset =
            orientation === 'horizontal'
              ? ((touch.clientX - container.offset().left) /
                  container.width()) *
                100
              : ((touch.clientY - container.offset().top) /
                  container.height()) *
                100

          updateSlider(offset)
          e.preventDefault()
        })

        $(document).on('touchend', function () {
          isDragging = false
        })

        $(document).on('touchcancel', function () {
          isDragging = false
        })

        $(document).on('touchmove', function (e) {
          if (!isDragging) return

          let touch = e.originalEvent.touches[0]
          let offset =
            orientation === 'horizontal'
              ? ((touch.clientX - container.offset().left) /
                  container.width()) *
                100
              : ((touch.clientY - container.offset().top) /
                  container.height()) *
                100

          updateSlider(offset)
          e.preventDefault()
        })
      }

      // Handle hover interaction
      if (interactionMode === 'hover' || interactionMode === 'both') {
        container.on('mousemove', function (e) {
          if (isDragging) return // Don't interfere with drag interaction

          const rect = container[0].getBoundingClientRect()
          let offset, sensitivity

          if (orientation === 'horizontal') {
            offset = ((e.clientX - rect.left) / rect.width) * 100
            sensitivity = hoverSensitivity
          } else {
            offset = ((e.clientY - rect.top) / rect.height) * 100
            sensitivity = hoverSensitivity
          }

          // Only update if movement is beyond sensitivity threshold
          if (Math.abs(offset - position) > sensitivity || !isHovering) {
            updateSlider(offset)
            isHovering = true
          }
        })

        container.on('mouseleave', function () {
          isHovering = false
        })
      }

      // Keyboard support
      $(document).on('keydown', function (e) {
        if (
          container.is(':visible') &&
          (e.key === 'ArrowLeft' ||
            e.key === 'ArrowRight' ||
            e.key === 'ArrowUp' ||
            e.key === 'ArrowDown')
        ) {
          const isHorizontal = orientation === 'horizontal'
          const isLeftOrUp = e.key === 'ArrowLeft' || e.key === 'ArrowUp'

          if (
            (isHorizontal &&
              (e.key === 'ArrowLeft' || e.key === 'ArrowRight')) ||
            (!isHorizontal && (e.key === 'ArrowUp' || e.key === 'ArrowDown'))
          ) {
            updateSlider(position + (isLeftOrUp ? -5 : 5))
            e.preventDefault()
          }
        }
      })

      // Control buttons
      container
        .closest('.eafe-image-comparison-wrapper')
        .find('.eafe-comparison-controls button')
        .on('click', function () {
          const action = $(this).data('action')

          switch (action) {
            case 'before':
              setToBefore()
              break
            case 'after':
              setToAfter()
              break
            case 'reset':
              resetPosition()
              break
          }
        })

      // Initialize
      updateSlider(position)
    })
  }

  // Initialize on document ready
  $(document).ready(function () {
    initImageComparison()
  })

  // Reinitialize when Elementor editor changes
  if (window.elementor) {
    elementor.hooks.addAction(
      'panel/open_editor/widget',
      function (panel, model, view) {
        if (model.attributes.widgetType === 'eafe-image-comparison') {
          setTimeout(initImageComparison, 300)
        }
      }
    )

    // Handle responsive preview changes
    elementor.hooks.addAction(
      'frontend/element_ready/widget',
      function ($scope) {
        if ($scope.find('.eafe-image-comparison-container').length) {
          initImageComparison()
        }
      }
    )
  }

  // Make functions available globally for potential external control
  window.EAFEImageComparison = {
    init: initImageComparison,
    setPosition: function (containerId, position) {
      const container = $('#' + containerId)
      const before = container.find('.eafe-image-comparison-before')
      const handle = container.find('.eafe-image-comparison-handle')
      const orientation =
        container
          .closest('.eafe-image-comparison-wrapper')
          .data('orientation') || 'horizontal'

      if (orientation === 'horizontal') {
        before.css('width', position + '%')
        handle.css('left', position + '%')
      } else {
        before.css('height', position + '%')
        handle.css('top', position + '%')
      }

      container.find('.eafe-percentage').text(Math.round(position) + '%')
    },
  }
})(jQuery)
