/**
 * EAFE Icon Box Widget JavaScript
 * Handles dynamic interactions and animations
 */

(function($) {
    'use strict';

    // Initialize when DOM is ready
    $(document).ready(function() {
        initIconBoxes();
    });

    // Initialize when Elementor frontend loads
    $(window).on('elementor/frontend/init', function() {
        elementorFrontend.hooks.addAction('frontend/element_ready/eafe-icon-box.default', function($scope) {
            initIconBoxes($scope);
        });
    });

    /**
     * Initialize icon box functionality
     * @param {jQuery} $scope - The scope to search within (optional)
     */
    function initIconBoxes($scope) {
        var $iconBoxes = $scope ? $scope.find('.eafe-icon-box') : $('.eafe-icon-box');

        $iconBoxes.each(function() {
            var $iconBox = $(this);

            // Add enhanced hover effects
            addHoverEffects($iconBox);

            // Add click animations
            addClickAnimations($iconBox);

            // Add scroll animations
            addScrollAnimations($iconBox);

            // Add accessibility features
            addAccessibilityFeatures($iconBox);

            // Template-specific enhancements
            addTemplateSpecificFeatures($iconBox);
        });
    }

    /**
     * Add enhanced hover effects
     * @param {jQuery} $iconBox - The icon box element
     */
    function addHoverEffects($iconBox) {
        var $icon = $iconBox.find('.eafe-icon-box-icon');
        var $title = $iconBox.find('.eafe-icon-box-title');
        var $description = $iconBox.find('.eafe-icon-box-description');
        var $button = $iconBox.find('.eafe-icon-box-button');
        var $image = $iconBox.find('.eafe-icon-box-image img');

        $iconBox.on('mouseenter', function() {
            // Add subtle parallax effect to icon
            if ($icon.length) {
                if ($image.length) {
                    // Special handling for image icons
                    $image.css('transform', 'scale(1.1)');
                } else {
                    $icon.css('transform', 'translateZ(10px)');
                }
            }

            // Add glow effect for certain templates
            if ($iconBox.hasClass('template-3') || $iconBox.hasClass('template-5')) {
                $iconBox.addClass('glow-effect');
            }

            // Position-specific animations
            if ($iconBox.hasClass('icon-position-left') || $iconBox.hasClass('icon-position-right')) {
                $title.css('transform', 'translateX(' + ($iconBox.hasClass('icon-position-left') ? '5px' : '-5px') + ')');
            } else {
                $title.css('transform', 'translateY(-2px)');
            }

            $description.css('opacity', '1');

        }).on('mouseleave', function() {
            // Reset transformations
            $icon.css('transform', '');
            $image.css('transform', '');
            $title.css('transform', '');
            $iconBox.removeClass('glow-effect');
        });

        // Button hover effects
        $button.on('mouseenter', function() {
            $(this).addClass('button-hover-active');
        }).on('mouseleave', function() {
            $(this).removeClass('button-hover-active');
        });
    }

    /**
     * Add click animations
     * @param {jQuery} $iconBox - The icon box element
     */
    function addClickAnimations($iconBox) {
        var $button = $iconBox.find('.eafe-icon-box-button');
        var $link = $iconBox.find('.eafe-icon-box-link');

        // Button click animation
        $button.on('click', function(e) {
            var $btn = $(this);

            // Add ripple effect
            addRippleEffect($btn, e);

            // Add click animation
            $btn.addClass('clicked');
            setTimeout(function() {
                $btn.removeClass('clicked');
            }, 200);
        });

        // Entire box click animation
        $link.on('click', function() {
            $iconBox.addClass('box-clicked');
            setTimeout(function() {
                $iconBox.removeClass('box-clicked');
            }, 300);
        });
    }

    /**
     * Add ripple effect to buttons
     * @param {jQuery} $element - The button element
     * @param {Event} event - The click event
     */
    function addRippleEffect($element, event) {
        var $ripple = $('<span class="ripple"></span>');
        var btnOffset = $element.offset();
        var xPos = event.pageX - btnOffset.left;
        var yPos = event.pageY - btnOffset.top;

        $ripple.css({
            'left': xPos + 'px',
            'top': yPos + 'px'
        });

        $element.append($ripple);

        setTimeout(function() {
            $ripple.remove();
        }, 600);
    }

    /**
     * Add scroll animations using Intersection Observer
     * @param {jQuery} $iconBox - The icon box element
     */
    function addScrollAnimations($iconBox) {
        // Check if Intersection Observer is supported
        if (!window.IntersectionObserver) {
            return;
        }

        var observer = new IntersectionObserver(function(entries) {
            entries.forEach(function(entry) {
                if (entry.isIntersecting) {
                    var $box = $(entry.target);

                    // Add entrance animation
                    $box.addClass('animate-in');

                    // Stagger animation for child elements
                    setTimeout(function() {
                        $box.find('.eafe-icon-box-icon').addClass('animate-icon');
                    }, 200);

                    setTimeout(function() {
                        $box.find('.eafe-icon-box-title').addClass('animate-title');
                    }, 400);

                    setTimeout(function() {
                        $box.find('.eafe-icon-box-description').addClass('animate-description');
                    }, 600);

                    setTimeout(function() {
                        $box.find('.eafe-icon-box-button').addClass('animate-button');
                    }, 800);

                    // Unobserve after animation
                    observer.unobserve(entry.target);
                }
            });
        }, {
            threshold: 0.2,
            rootMargin: '0px 0px -50px 0px'
        });

        observer.observe($iconBox[0]);
    }

    /**
     * Add accessibility features
     * @param {jQuery} $iconBox - The icon box element
     */
    function addAccessibilityFeatures($iconBox) {
        var $button = $iconBox.find('.eafe-icon-box-button');
        var $link = $iconBox.find('.eafe-icon-box-link');

        // Add ARIA labels if missing
        if ($button.length && !$button.attr('aria-label')) {
            var buttonText = $button.text().trim();
            var titleText = $iconBox.find('.eafe-icon-box-title').text().trim();
            $button.attr('aria-label', buttonText + ' - ' + titleText);
        }

        if ($link.length && !$link.attr('aria-label')) {
            var titleText = $iconBox.find('.eafe-icon-box-title').text().trim();
            $link.attr('aria-label', 'Read more about ' + titleText);
        }

        // Add keyboard navigation
        $iconBox.attr('tabindex', '0');

        $iconBox.on('keydown', function(e) {
            // Enter or Space key
            if (e.which === 13 || e.which === 32) {
                e.preventDefault();

                if ($link.length) {
                    $link[0].click();
                } else if ($button.length) {
                    $button[0].click();
                }
            }
        });

        // Focus indicators
        $iconBox.on('focus', function() {
            $(this).addClass('keyboard-focused');
        }).on('blur', function() {
            $(this).removeClass('keyboard-focused');
        });
    }

    /**
     * Add template-specific features
     * @param {jQuery} $iconBox - The icon box element
     */
    function addTemplateSpecificFeatures($iconBox) {
        // Template 3: Multicolor badge - Add color cycling
        if ($iconBox.hasClass('template-3')) {
            addColorCycling($iconBox);
        }

        // Template 4: Offset badge - Add floating animation
        if ($iconBox.hasClass('template-4')) {
            addFloatingAnimation($iconBox);
        }

        // Template 5: Gradient background - Add gradient shifting
        if ($iconBox.hasClass('template-5')) {
            addGradientShifting($iconBox);
        }

        // Template 6: Circular - Add rotation on hover
        if ($iconBox.hasClass('template-6')) {
            addCircularEffects($iconBox);
        }

        // Template 7: Image background - Add parallax effect
        if ($iconBox.hasClass('template-7')) {
            addParallaxEffect($iconBox);
        }
    }

    /**
     * Add color cycling for template 3
     * @param {jQuery} $iconBox - The icon box element
     */
    function addColorCycling($iconBox) {
        var colors = [
            'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
            'linear-gradient(135deg, #f093fb 0%, #f5576c 100%)',
            'linear-gradient(135deg, #4facfe 0%, #00f2fe 100%)',
            'linear-gradient(135deg, #43e97b 0%, #38f9d7 100%)',
            'linear-gradient(135deg, #fa709a 0%, #fee140 100%)'
        ];

        var currentIndex = 0;

        $iconBox.on('mouseenter', function() {
            currentIndex = (currentIndex + 1) % colors.length;
            var $icon = $iconBox.find('.eafe-icon-box-icon');
            $icon.css('background', colors[currentIndex]);
        });
    }

    /**
     * Add floating animation for template 4
     * @param {jQuery} $iconBox - The icon box element
     */
    function addFloatingAnimation($iconBox) {
        var $icon = $iconBox.find('.eafe-icon-box-icon');

        setInterval(function() {
            if ($iconBox.is(':hover')) {
                $icon.addClass('floating');
                setTimeout(function() {
                    $icon.removeClass('floating');
                }, 2000);
            }
        }, 3000);
    }

    /**
     * Add gradient shifting for template 5
     * @param {jQuery} $iconBox - The icon box element
     */
    function addGradientShifting($iconBox) {
        var gradients = [
            'linear-gradient(135deg, #667eea 0%, #764ba2 100%)',
            'linear-gradient(135deg, #764ba2 0%, #667eea 100%)',
            'linear-gradient(45deg, #667eea 0%, #764ba2 100%)',
            'linear-gradient(225deg, #667eea 0%, #764ba2 100%)'
        ];

        var currentGradient = 0;

        $iconBox.on('mouseenter', function() {
            currentGradient = (currentGradient + 1) % gradients.length;
            $iconBox.css('background', gradients[currentGradient]);
        });
    }

    /**
     * Add circular effects for template 6
     * @param {jQuery} $iconBox - The icon box element
     */
    function addCircularEffects($iconBox) {
        var $icon = $iconBox.find('.eafe-icon-box-icon');
        var rotationAngle = 0;

        $iconBox.on('mouseenter', function() {
            rotationAngle += 360;
            $icon.css('transform', 'rotate(' + rotationAngle + 'deg) scale(1.15)');
        });

        $iconBox.on('mouseleave', function() {
            $icon.css('transform', 'rotate(' + rotationAngle + 'deg) scale(1)');
        });
    }

    /**
     * Add parallax effect for template 7
     * @param {jQuery} $iconBox - The icon box element
     */
    function addParallaxEffect($iconBox) {
        var $bgImage = $iconBox.find('.eafe-icon-box-bg-image');

        if (!$bgImage.length) return;

        $iconBox.on('mousemove', function(e) {
            var rect = this.getBoundingClientRect();
            var x = e.clientX - rect.left;
            var y = e.clientY - rect.top;

            var centerX = rect.width / 2;
            var centerY = rect.height / 2;

            var deltaX = (x - centerX) / centerX;
            var deltaY = (y - centerY) / centerY;

            var moveX = deltaX * 10;
            var moveY = deltaY * 10;

            $bgImage.css('transform', 'translate(' + moveX + 'px, ' + moveY + 'px) scale(1.1)');
        });

        $iconBox.on('mouseleave', function() {
            $bgImage.css('transform', 'translate(0, 0) scale(1)');
        });
    }

    /**
     * Utility function to debounce scroll events
     * @param {Function} func - Function to debounce
     * @param {number} wait - Wait time in milliseconds
     * @param {boolean} immediate - Execute immediately
     * @returns {Function} Debounced function
     */
    function debounce(func, wait, immediate) {
        var timeout;
        return function() {
            var context = this, args = arguments;
            var later = function() {
                timeout = null;
                if (!immediate) func.apply(context, args);
            };
            var callNow = immediate && !timeout;
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
            if (callNow) func.apply(context, args);
        };
    }

    /**
     * Add performance optimizations
     */
    function optimizePerformance() {
        // Use CSS transforms instead of changing layout properties
        var style = document.createElement('style');
        style.textContent = `
            .eafe-icon-box * {
                will-change: transform, opacity;
            }
            
            .eafe-icon-box.animate-in {
                animation: slideInUp 0.6s ease-out forwards;
            }
            
            .animate-icon {
                animation: bounceIn 0.6s ease-out forwards;
            }
            
            .animate-title {
                animation: fadeInUp 0.4s ease-out forwards;
            }
            
            .animate-description {
                animation: fadeInUp 0.4s ease-out 0.1s forwards;
            }
            
            .animate-button {
                animation: fadeInUp 0.4s ease-out 0.2s forwards;
            }
            
            .clicked {
                transform: scale(0.95);
                transition: transform 0.1s ease;
            }
            
            .box-clicked {
                transform: scale(0.98);
                transition: transform 0.2s ease;
            }
            
            .button-hover-active {
                transform: translateY(-2px);
            }
            
            .glow-effect {
                box-shadow: 0 0 30px rgba(102, 126, 234, 0.3);
            }
            
            .floating {
                animation: float 2s ease-in-out infinite;
            }
            
            .keyboard-focused {
                outline: 2px solid #007cba;
                outline-offset: 2px;
            }
            
            .ripple {
                position: absolute;
                border-radius: 50%;
                background: rgba(255, 255, 255, 0.6);
                transform: scale(0);
                animation: ripple-animation 0.6s linear;
                pointer-events: none;
            }
            
            @keyframes ripple-animation {
                to {
                    transform: scale(4);
                    opacity: 0;
                }
            }
            
            @keyframes slideInUp {
                from {
                    opacity: 0;
                    transform: translateY(30px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
            
            @keyframes bounceIn {
                0% {
                    opacity: 0;
                    transform: scale(0.3);
                }
                50% {
                    opacity: 1;
                    transform: scale(1.05);
                }
                70% {
                    transform: scale(0.9);
                }
                100% {
                    opacity: 1;
                    transform: scale(1);
                }
            }
            
            @keyframes fadeInUp {
                from {
                    opacity: 0;
                    transform: translateY(20px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
            
            @keyframes float {
                0%, 100% {
                    transform: translateY(0px);
                }
                50% {
                    transform: translateY(-10px);
                }
            }
            
            /* Reduce motion for users who prefer it */
            @media (prefers-reduced-motion: reduce) {
                .eafe-icon-box,
                .eafe-icon-box *,
                .eafe-icon-box::before,
                .eafe-icon-box::after {
                    animation-duration: 0.01ms !important;
                    animation-iteration-count: 1 !important;
                    transition-duration: 0.01ms !important;
                }
            }
        `;
        document.head.appendChild(style);
    }

    // Initialize performance optimizations
    optimizePerformance();

})(jQuery);

