(function($) {
    'use strict';

    class EAFEFunCounter {
        constructor(element) {
            this.$element = $(element);
            this.counters = this.$element.find('.eafe-factor-number[data-end]');
            this.animated = false;
            this.animationFrames = [];

            this.init();
        }

        init() {
            if (this.counters.length === 0) return;

            // Create Intersection Observer for viewport detection
            this.createObserver();

            // Handle Elementor editor mode
            if (this.isEditMode()) {
                this.showFinalValues();
            }
        }

        isEditMode() {
            return typeof elementorFrontend !== 'undefined' &&
                elementorFrontend.isEditMode();
        }

        createObserver() {
            const options = {
                root: null,
                rootMargin: '0px',
                threshold: 0.3 // Trigger when 30% of element is visible
            };

            this.observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting && !this.animated) {
                        this.animated = true;
                        this.startAnimation();
                        this.observer.disconnect(); // Run animation only once
                    }
                });
            }, options);

            this.observer.observe(this.$element[0]);
        }

        startAnimation() {
            const duration = parseInt(this.$element.data('animation-duration')) || 2000;

            this.counters.each((index, element) => {
                this.animateCounter($(element), duration);
            });
        }

        animateCounter($counter, duration) {
            const start = parseInt($counter.data('start')) || 0;
            const end = parseInt($counter.data('end')) || 0;
            const suffix = $counter.data('suffix') || '';
            const useDecimals = (end % 1 !== 0); // Check if end number has decimals

            // Update aria-label for accessibility
            $counter.attr('aria-live', 'polite');
            $counter.attr('aria-label', `Counter animating from ${start} to ${end}${suffix}`);

            let startTime = null;
            const animationId = {cancelled: false};
            this.animationFrames.push(animationId);

            const step = (timestamp) => {
                if (animationId.cancelled) return;

                if (!startTime) startTime = timestamp;
                const progress = Math.min((timestamp - startTime) / duration, 1);

                // Easing function (easeOutExpo)
                const easeProgress = progress === 1 ? 1 : 1 - Math.pow(2, -10 * progress);

                const current = start + (end - start) * easeProgress;
                const displayValue = useDecimals ? current.toFixed(1) : Math.round(current);

                $counter.text(displayValue + suffix);

                if (progress < 1) {
                    requestAnimationFrame(step);
                } else {
                    // Animation complete, update aria-label
                    $counter.attr('aria-label', `${end}${suffix} ${$counter.siblings('.eafe-factor-title').text()}`);
                    $counter.removeAttr('aria-live');
                }
            };

            requestAnimationFrame(step);
        }

        showFinalValues() {
            // In edit mode, show final values immediately
            this.counters.each((index, element) => {
                const $counter = $(element);
                const end = parseInt($counter.data('end')) || 0;
                const suffix = $counter.data('suffix') || '';
                $counter.text(end + suffix);
            });
        }

        destroy() {
            // Clean up observer
            if (this.observer) {
                this.observer.disconnect();
            }

            // Cancel all running animations
            this.animationFrames.forEach(frame => {
                frame.cancelled = true;
            });

            this.animationFrames = [];
        }
    }

    // Initialize widget
    const initFunCounter = function() {
        $('.eafe-fun-counter[data-animation-duration]').each(function() {
            if (!$(this).data('eafe-counter-instance')) {
                const instance = new EAFEFunCounter(this);
                $(this).data('eafe-counter-instance', instance);
            }
        });
    };

    // Initialize on page load
    $(window).on('load', function() {
        initFunCounter();
    });

    // Re-initialize in Elementor editor
    if (typeof elementorFrontend !== 'undefined') {
        elementorFrontend.hooks.addAction('frontend/element_ready/widget', function($scope) {
            if ($scope.find('.eafe-fun-counter').length) {
                const $counter = $scope.find('.eafe-fun-counter');

                // Destroy old instance if exists
                const oldInstance = $counter.data('eafe-counter-instance');
                if (oldInstance && typeof oldInstance.destroy === 'function') {
                    oldInstance.destroy();
                }

                // Create new instance
                if ($counter.data('animation-duration')) {
                    const instance = new EAFEFunCounter($counter[0]);
                    $counter.data('eafe-counter-instance', instance);
                }
            }
        });
    }

})(jQuery);