(function($) {
    'use strict';

    class EazyAddonsDashboard {
        constructor() {
            this.init();
        }

        init() {
            this.bindEvents();
        }

        bindEvents() {
            // Tabs
            $(document).on('click', '.eafe-tab-button', this.switchTab.bind(this));

            // Quick Action shortcuts
            $(document).on('click', '.eafe-quick-action-button[data-tab]', this.quickSwitchTab.bind(this));

            // Bulk actions
            $(document).on('click', '.eafe-bulk-enable', this.bulkEnableWidgets.bind(this));
            $(document).on('click', '.eafe-bulk-disable', this.bulkDisableWidgets.bind(this));

            // Save button
            $(document).on('click', '.eafe-save-settings', this.saveSettings.bind(this));

            // Auto-save (with debounce)
            $(document).on('change', '.eafe-widget-toggle', this.debounce(this.autoSaveSettings.bind(this), 1000));
        }

        /* -----------------------------
           TAB SWITCHING LOGIC
        ----------------------------- */
        switchTab(e) {
            e.preventDefault();
            const $button = $(e.currentTarget);
            const tabName = $button.data('tab');

            // Update buttons
            $('.eafe-tab-button').removeClass('active');
            $button.addClass('active');

            // Update content
            $('.eafe-tab-pane').removeClass('active');
            $('#' + tabName + '-tab').addClass('active');

            // Smooth scroll top of container
            $('.eafe-tabs-content').animate({ scrollTop: 0 }, 300);
        }

        quickSwitchTab(e) {
            e.preventDefault();
            const tabName = $(e.currentTarget).data('tab');
            $(`.eafe-tab-button[data-tab="${tabName}"]`).trigger('click');
        }

        /* -----------------------------
           WIDGETS BULK ACTIONS
        ----------------------------- */
        bulkEnableWidgets() {
            $('.eafe-widget-toggle').prop('checked', true).trigger('change');
            this.showToast('✅ All widgets enabled!');
        }

        bulkDisableWidgets() {
            $('.eafe-widget-toggle').prop('checked', false).trigger('change');
            this.showToast('⚠️ All widgets disabled!');
        }

        /* -----------------------------
           SAVE SETTINGS (MANUAL)
        ----------------------------- */
        saveSettings() {
            const $saveButton = $('.eafe-save-settings');
            const $message = $('.eafe-save-message');

            // Visual feedback
            $saveButton.prop('disabled', true)
                .html('<span class="dashicons dashicons-update spin"></span> Saving...');

            $message.removeClass('success error').text('');

            const formData = this.getFormData();

            $.ajax({
                url: eafeAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'eafe_save_settings',
                    nonce: eafeAdmin.nonce,
                    ...formData
                },
                success: (response) => {
                    if (response.success) {
                        this.showToast('✅ Settings saved successfully!');
                        $message.addClass('success').text('Settings saved successfully!');
                    } else {
                        $message.addClass('error').text(response.data || 'Failed to save settings.');
                        this.showToast('❌ Failed to save settings.', 'error');
                    }
                },
                error: (xhr, status, error) => {
                    console.error('Save settings error:', error);
                    $message.addClass('error').text('Error: ' + error);
                    this.showToast('❌ AJAX request failed.', 'error');
                },
                complete: () => {
                    // Restore button
                    $saveButton.prop('disabled', false)
                        .html('<span class="eafe-save-text">💾 Save Settings</span>');
                }
            });
        }

        /* -----------------------------
           AUTO SAVE SETTINGS
        ----------------------------- */
        autoSaveSettings() {
            const $message = $('.eafe-save-message');
            $message.removeClass('success error').text('Auto-saving...');

            const formData = this.getFormData();

            $.ajax({
                url: eafeAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'eafe_save_settings',
                    nonce: eafeAdmin.nonce,
                    ...formData
                },
                success: (response) => {
                    if (response.success) {
                        $message.addClass('success').text('Auto-saved!');
                        this.showToast('✅ Auto-saved!');
                        setTimeout(() => $message.text(''), 2000);
                    }
                },
                error: (xhr, status, error) => {
                    console.error('Auto-save error:', error);
                    this.showToast('❌ Auto-save failed.', 'error');
                }
            });
        }

        /* -----------------------------
           GET FORM DATA
        ----------------------------- */
        getFormData() {
            const formData = { widgets: {} };
            $('.eafe-widget-toggle').each(function() {
                const widgetName = $(this).attr('name').match(/widgets\[(.*?)\]/)[1];
                formData.widgets[widgetName] = $(this).is(':checked') ? 1 : 0;
            });
            return formData;
        }

        /* -----------------------------
           TOAST MESSAGE SYSTEM
        ----------------------------- */
        showToast(message, type = 'success') {
            // Remove existing toast
            $('.eafe-toast').remove();

            const bgColor = type === 'error' ? '#ef4444' : '#22c55e';
            const $toast = $(`
                <div class="eafe-toast" style="
                    position: fixed;
                    bottom: 25px;
                    right: 25px;
                    background: ${bgColor};
                    color: #fff;
                    padding: 12px 20px;
                    border-radius: 8px;
                    font-weight: 500;
                    box-shadow: 0 6px 20px rgba(0,0,0,0.2);
                    z-index: 9999;
                    opacity: 0;
                    transform: translateY(20px);
                    transition: all 0.3s ease;
                ">
                    ${message}
                </div>
            `);

            $('body').append($toast);

            // Animate in
            setTimeout(() => {
                $toast.css({ opacity: 1, transform: 'translateY(0)' });
            }, 50);

            // Remove after delay
            setTimeout(() => {
                $toast.css({ opacity: 0, transform: 'translateY(20px)' });
                setTimeout(() => $toast.remove(), 300);
            }, 3000);
        }

        /* -----------------------------
           DEBOUNCE (UTILITY)
        ----------------------------- */
        debounce(func, wait) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    clearTimeout(timeout);
                    func(...args);
                };
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
            };
        }
    }

    // Initialize dashboard when DOM is ready
    $(document).ready(() => {
        new EazyAddonsDashboard();
    });

})(jQuery);
