<?php

namespace EazyAddonsForElementor\Admin;

use EazyAddonsForElementor\Core\WidgetManager;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Dashboard
{
    private $widgets;
    private $settings;

    public function init()
    {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_action('wp_ajax_eafe_save_settings', [$this, 'save_settings']);

        // Remove unwanted elements from our dashboard
        add_action('admin_init', [$this, 'clean_dashboard']);

        // Initialize data
        $this->widgets = $this->get_available_widgets();
        $this->settings = $this->get_plugin_settings();
    }

    /**
     * Clean up dashboard - remove notices and Elementor admin bar
     */
    public function clean_dashboard()
    {
        // Check if we're on our dashboard page
        if (isset($_GET['page']) && $_GET['page'] === 'nb-eazy-addons-for-elementor') {
            // Remove admin notices
            remove_all_actions('admin_notices');
            remove_all_actions('all_admin_notices');

            // Remove Elementor admin top bar
            $this->remove_elementor_admin_bar();

            // Remove specific plugin notices
            $this->remove_plugin_notices();
        }
    }

    /**
     * Remove Elementor admin top bar
     */
    private function remove_elementor_admin_bar()
    {
        // Remove Elementor admin bar hooks
        remove_action('in_admin_header', ['\Elementor\Plugin', 'add_admin_top_bar']);
        remove_action('in_admin_header', ['\Elementor\Core\Admin\Admin', 'add_admin_top_bar']);

        // Remove via filters
        add_filter('elementor/admin-top-bar/init', '__return_false');

        // Additional Elementor admin bar removal
        if (class_exists('\Elementor\Core\Admin\Admin')) {
            $admin = \Elementor\Plugin::$instance->admin;
            if ($admin) {
                remove_action('in_admin_header', [$admin, 'add_admin_top_bar']);
            }
        }
    }

    /**
     * Remove specific plugin notices
     */
    private function remove_plugin_notices()
    {
        // Rank Math
        remove_action('admin_notices', 'rank_math_admin_notices');
        remove_action('admin_notices', ['RankMath\Admin\Admin', 'notices']);

        // Yoast SEO
        remove_action('admin_notices', 'wpseo_admin_notices');
        remove_action('admin_notices', ['WPSEO_Admin', 'admin_notices']);

        // WooCommerce
        remove_action('admin_notices', 'woocommerce_admin_notices');

        // LiteSpeed Cache
        remove_action('admin_notices', 'litespeed_admin_notices');

        // Add more as needed for other plugins
    }

    public function add_admin_menu()
    {
        add_menu_page(
            __('Eazy Addons For Elementor', 'nb-eazy-addons-for-elementor'),
            __('Eazy Addons', 'nb-eazy-addons-for-elementor'),
            'manage_options',
            'nb-eazy-addons-for-elementor',
            [$this, 'render_dashboard'],
            'dashicons-admin-generic',
            58
        );
    }

    public function enqueue_admin_assets($hook)
    {
        if ('toplevel_page_nb-eazy-addons-for-elementor' !== $hook) {
            return;
        }

        // Load Elementor eicons font
        wp_enqueue_style(
            'elementor-icons',
            ELEMENTOR_ASSETS_URL . 'lib/eicons/css/elementor-icons.min.css'
        );

        wp_enqueue_style(
            'eafe-admin-dashboard',
            EAFE_URL . 'admin/assets/css/dashboard.css',
            [],
            EAFE_VERSION
        );

        wp_enqueue_script(
            'eafe-admin-dashboard',
            EAFE_URL . 'admin/assets/js/dashboard.js',
            ['jquery'],
            EAFE_VERSION,
            true
        );

        wp_localize_script('eafe-admin-dashboard', 'eafeAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('eafe_admin_nonce'),
            'savingText' => __('Saving...', 'nb-eazy-addons-for-elementor'),
            'savedText' => __('Settings Saved!', 'nb-eazy-addons-for-elementor')
        ]);

        // Add inline CSS as final backup to hide any remaining elements
        wp_add_inline_style('eafe-admin-dashboard', '
            /* Hide Elementor admin bar and notices */
            #e-admin-top-bar-root,
            .wrap.eafe-dashboard .notice,
            .wrap.eafe-dashboard .updated,
            .wrap.eafe-dashboard .error,
            .wrap.eafe-dashboard .is-dismissible,
            .wrap.eafe-dashboard .rank-math-notice,
            .wrap.eafe-dashboard .yoast-notice {
                display: none !important;
            }
            
            /* Ensure our header stays clean */
            .eafe-header {
                margin-top: 0 !important;
            }
        ');
    }

    public function render_dashboard()
    {
        // Start output buffering to capture and filter content
        ob_start();

        // Pass variables to the view
        $widgets = $this->widgets;
        $settings = $this->settings;

        require_once __DIR__ . '/views/dashboard.php';

        $content = ob_get_clean();

        // Remove any remaining notices or unwanted elements from the content
        $content = $this->clean_dashboard_content($content);

        echo $content;
    }

    /**
     * Clean dashboard content - remove any remaining HTML elements
     */
    private function clean_dashboard_content($content)
    {
        // Remove common notice patterns
        $patterns = [
            // WordPress notices
            '/<div class="notice[^>]*>.*?<\/div>/is',
            '/<div class="updated[^>]*>.*?<\/div>/is',
            '/<div class="error[^>]*>.*?<\/div>/is',
            '/<div class="is-dismissible[^>]*>.*?<\/div>/is',

            // Elementor admin bar
            '/<div id="e-admin-top-bar-root"[^>]*>.*?<\/div>/is',
            '/<div[^>]*class="[^"]*e-admin-top-bar[^"]*"[^>]*>.*?<\/div>/is',

            // Specific plugin notices
            '/<div class="rank-math-notice[^>]*>.*?<\/div>/is',
            '/<div class="yoast-notice[^>]*>.*?<\/div>/is',
            '/<div class="woocommerce-message[^>]*>.*?<\/div>/is',

            // Any script tags related to Elementor admin bar
            '/<script[^>]*e-admin-top-bar[^>]*>.*?<\/script>/is',
        ];

        foreach ($patterns as $pattern) {
            $content = preg_replace($pattern, '', $content);
        }

        return $content;
    }

    private function get_available_widgets()
    {
        return WidgetManager::get_widgets_for_dashboard();
    }

    private function get_plugin_settings()
    {
        $default_widgets = [];

        foreach (array_keys($this->widgets) as $widget_slug) {
            $default_widgets[$widget_slug] = true;
        }

        $defaults = [
            'widgets' => $default_widgets
        ];

        $saved_settings = get_option('eafe_settings', []);

        // Merge with defaults to ensure all widgets are included
        if (isset($saved_settings['widgets'])) {
            $saved_settings['widgets'] = array_merge($default_widgets, $saved_settings['widgets']);
        }

        return wp_parse_args($saved_settings, $defaults);
    }

    public function save_settings()
    {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'eafe_admin_nonce')) {
            wp_send_json_error('Security check failed');
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        $settings = [
            'widgets' => isset($_POST['widgets']) ? array_map('boolval', $_POST['widgets']) : []
        ];

        update_option('eafe_settings', $settings);

        // Clear Elementor cache after saving settings
        if (class_exists('\Elementor\Plugin')) {
            \Elementor\Plugin::$instance->files_manager->clear_cache();
        }

        wp_send_json_success(['message' => 'Settings saved successfully']);
    }
}