<?php
/**
 * Plugin Name: NanoTOC — Fast Lightweight Table of Contents
 * Description: Fast, lightweight table of contents (TOC) for WordPress. Auto insert or shortcode, nested/flat lists, smooth scrolling, and optional offset.
 * Version: 1.0.0
 * Author: Saqib Hanif
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: nanotoc
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
	exit;
}

final class NanoTOC_Plugin {
	private const OPTION_KEY = 'nanotoc_options';
	private const PLACEHOLDER = '<!--nano-toc-->';
	private const SCRIPT_HANDLE = 'nanotoc';
	private const STYLE_HANDLE = 'nanotoc-style';

	public static function init(): void {
		add_action('admin_init', [self::class, 'register_settings']);
		add_action('admin_menu', [self::class, 'register_menu']);

		add_action('wp_enqueue_scripts', [self::class, 'enqueue_assets']);
		add_filter('the_content', [self::class, 'filter_the_content'], 20);

		add_shortcode('nanotoc', [self::class, 'shortcode']);
	}

	public static function defaults(): array {
		return [
			'enabled' => 1,
			'post_types' => ['post', 'page'],
			'heading_levels' => [2, 3],
			'list_style' => 'nested', // nested | flat
			'auto_insert' => 1,
			'label' => 'Table of Contents',
			'smooth_scroll' => 1,
			'scroll_offset' => 0,
		];
	}

	public static function get_options(): array {
		$defaults = self::defaults();
		$raw = get_option(self::OPTION_KEY);
		if (!is_array($raw)) {
			return $defaults;
		}

		$opts = array_merge($defaults, $raw);
		$opts['enabled'] = !empty($opts['enabled']) ? 1 : 0;
		$opts['auto_insert'] = !empty($opts['auto_insert']) ? 1 : 0;
		$opts['smooth_scroll'] = !empty($opts['smooth_scroll']) ? 1 : 0;

		$opts['post_types'] = is_array($opts['post_types']) ? array_values(array_filter(array_map('sanitize_key', $opts['post_types']))) : $defaults['post_types'];

		$levels = is_array($opts['heading_levels']) ? $opts['heading_levels'] : $defaults['heading_levels'];
		$levels = array_values(array_unique(array_map('intval', $levels)));
		$levels = array_values(array_filter($levels, static fn($n) => $n >= 1 && $n <= 6));
		if (empty($levels)) {
			$levels = $defaults['heading_levels'];
		}
		sort($levels);
		$opts['heading_levels'] = $levels;

		$opts['list_style'] = ($opts['list_style'] === 'flat') ? 'flat' : 'nested';
		$opts['label'] = is_string($opts['label']) && $opts['label'] !== '' ? wp_strip_all_tags($opts['label']) : $defaults['label'];
		$opts['scroll_offset'] = intval($opts['scroll_offset'] ?? 0);
		if ($opts['scroll_offset'] < 0) {
			$opts['scroll_offset'] = 0;
		}

		return $opts;
	}

	public static function register_settings(): void {
		register_setting(
			'nanotoc_settings',
			self::OPTION_KEY,
			[
				'type' => 'array',
				'sanitize_callback' => [self::class, 'sanitize_options'],
				'default' => self::defaults(),
			]
		);

		add_settings_section('nanotoc_main', 'NanoTOC', '__return_false', 'nanotoc');

		add_settings_field('enabled', 'Enable', [self::class, 'render_field_enabled'], 'nanotoc', 'nanotoc_main');
		add_settings_field('post_types', 'Post types', [self::class, 'render_field_post_types'], 'nanotoc', 'nanotoc_main');
		add_settings_field('heading_levels', 'Headings to include', [self::class, 'render_field_heading_levels'], 'nanotoc', 'nanotoc_main');
		add_settings_field('list_style', 'List style', [self::class, 'render_field_list_style'], 'nanotoc', 'nanotoc_main');
		add_settings_field('auto_insert', 'Auto insert', [self::class, 'render_field_auto_insert'], 'nanotoc', 'nanotoc_main');
		add_settings_field('label', 'Label', [self::class, 'render_field_label'], 'nanotoc', 'nanotoc_main');
		add_settings_field('smooth_scroll', 'Smooth scroll', [self::class, 'render_field_smooth_scroll'], 'nanotoc', 'nanotoc_main');
		add_settings_field('scroll_offset', 'Scroll offset (px)', [self::class, 'render_field_scroll_offset'], 'nanotoc', 'nanotoc_main');
	}

	public static function sanitize_options($input): array {
		$defaults = self::defaults();
		if (!is_array($input)) {
			return $defaults;
		}

		$out = [];
		$out['enabled'] = !empty($input['enabled']) ? 1 : 0;
		$out['auto_insert'] = !empty($input['auto_insert']) ? 1 : 0;
		$out['smooth_scroll'] = !empty($input['smooth_scroll']) ? 1 : 0;

		$out['post_types'] = isset($input['post_types']) && is_array($input['post_types'])
			? array_values(array_filter(array_map('sanitize_key', $input['post_types'])))
			: $defaults['post_types'];

		$out['heading_levels'] = isset($input['heading_levels']) && is_array($input['heading_levels'])
			? array_values(array_unique(array_map('intval', $input['heading_levels'])))
			: $defaults['heading_levels'];

		$out['heading_levels'] = array_values(array_filter($out['heading_levels'], static fn($n) => $n >= 1 && $n <= 6));
		if (empty($out['heading_levels'])) {
			$out['heading_levels'] = $defaults['heading_levels'];
		}
		sort($out['heading_levels']);

		$out['list_style'] = (isset($input['list_style']) && $input['list_style'] === 'flat') ? 'flat' : 'nested';
		$out['label'] = isset($input['label']) && is_string($input['label']) ? wp_strip_all_tags($input['label']) : $defaults['label'];

		$out['scroll_offset'] = isset($input['scroll_offset']) ? max(0, intval($input['scroll_offset'])) : 0;

		return $out;
	}

	public static function register_menu(): void {
		add_options_page('NanoTOC', 'NanoTOC', 'manage_options', 'nanotoc', [self::class, 'render_settings_page']);
	}

	public static function render_settings_page(): void {
		if (!current_user_can('manage_options')) {
			return;
		}

		echo '<div class="wrap">';
		echo '<h1>NanoTOC</h1>';
		echo '<form method="post" action="options.php">';
		settings_fields('nanotoc_settings');
		do_settings_sections('nanotoc');
		submit_button();
		echo '</form>';
		echo '<p>Use <code>[nanotoc]</code> to place the table of contents manually. If Auto insert is enabled, it will be added at the top when the shortcode is not used.</p>';
		echo '<p>If you find an issue or want to request a feature, please contact the author.</p>';
		echo '<p>Created by Saqib Hanif</p>';
		echo '</div>';
	}

	private static function field_name(string $key): string {
		return self::OPTION_KEY . '[' . $key . ']';
	}

	public static function render_field_enabled(): void {
		$opts = self::get_options();
		echo '<label><input type="checkbox" name="' . esc_attr(self::field_name('enabled')) . '" value="1" ' . checked(1, $opts['enabled'], false) . '> Enabled</label>';
	}

	public static function render_field_post_types(): void {
		$opts = self::get_options();
		$post_types = get_post_types(['public' => true], 'objects');
		foreach ($post_types as $pt) {
			$checked = in_array($pt->name, $opts['post_types'], true);
			echo '<label style="display:block;margin:2px 0;">';
			echo '<input type="checkbox" name="' . esc_attr(self::field_name('post_types')) . '[]" value="' . esc_attr($pt->name) . '" ' . checked(true, $checked, false) . '> ' . esc_html($pt->labels->singular_name);
			echo '</label>';
		}
	}

	public static function render_field_heading_levels(): void {
		$opts = self::get_options();
		for ($i = 1; $i <= 6; $i++) {
			$checked = in_array($i, $opts['heading_levels'], true);
			echo '<label style="margin-right:12px;">';
			echo '<input type="checkbox" name="' . esc_attr(self::field_name('heading_levels')) . '[]" value="' . esc_attr((string)$i) . '" ' . checked(true, $checked, false) . '> H' . esc_html((string)$i);
			echo '</label>';
		}
		echo '<p class="description">Select which headings will be included in the TOC.</p>';
	}

	public static function render_field_list_style(): void {
		$opts = self::get_options();
		echo '<label><input type="radio" name="' . esc_attr(self::field_name('list_style')) . '" value="nested" ' . checked('nested', $opts['list_style'], false) . '> Nested</label>&nbsp;&nbsp;';
		echo '<label><input type="radio" name="' . esc_attr(self::field_name('list_style')) . '" value="flat" ' . checked('flat', $opts['list_style'], false) . '> Flat</label>';
	}

	public static function render_field_auto_insert(): void {
		$opts = self::get_options();
		echo '<label><input type="checkbox" name="' . esc_attr(self::field_name('auto_insert')) . '" value="1" ' . checked(1, $opts['auto_insert'], false) . '> Insert at the top (when shortcode is not used)</label>';
	}

	public static function render_field_label(): void {
		$opts = self::get_options();
		echo '<input type="text" class="regular-text" name="' . esc_attr(self::field_name('label')) . '" value="' . esc_attr($opts['label']) . '">';
	}

	public static function render_field_smooth_scroll(): void {
		$opts = self::get_options();
		echo '<label><input type="checkbox" name="' . esc_attr(self::field_name('smooth_scroll')) . '" value="1" ' . checked(1, $opts['smooth_scroll'], false) . '> Enable smooth scrolling on click</label>';
	}

	public static function render_field_scroll_offset(): void {
		$opts = self::get_options();
		echo '<input type="number" min="0" class="small-text" name="' . esc_attr(self::field_name('scroll_offset')) . '" value="' . esc_attr((string)$opts['scroll_offset']) . '">';
		echo '<p class="description">Useful if your theme has a sticky header.</p>';
	}

	public static function enqueue_assets(): void {
		$opts = self::get_options();
		if (!$opts['enabled']) {
			return;
		}
		if (!is_singular()) {
			return;
		}

		wp_register_style(
			self::STYLE_HANDLE,
			false,
			[],
			'1.0.0'
		);

		if ($opts['smooth_scroll'] || intval($opts['scroll_offset']) > 0) {
			wp_register_script(
				self::SCRIPT_HANDLE,
				plugins_url('assets/light-toc.js', __FILE__),
				[],
				'1.0.0',
				true
			);
		}
	}

	private static function maybe_enqueue_assets(array $opts): void {
		wp_enqueue_style(self::STYLE_HANDLE);
		$css = '.nanotoc__title{margin:0 0 .5em}.nanotoc__list{margin:0;padding-left:1.25em}.nanotoc__item{margin:6px 0}.nanotoc__link{text-decoration:inherit}.nanotoc__link:hover{text-decoration:underline}';
		wp_add_inline_style(self::STYLE_HANDLE, $css);

		if (!$opts['smooth_scroll'] && intval($opts['scroll_offset']) <= 0) {
			return;
		}
		wp_enqueue_script(self::SCRIPT_HANDLE);
		wp_add_inline_script(self::SCRIPT_HANDLE, 'window.NanoTOC = window.NanoTOC || {}; window.NanoTOC.scrollOffset = ' . intval($opts['scroll_offset']) . '; window.NanoTOC.smoothScroll = ' . intval($opts['smooth_scroll']) . ';', 'before');
	}

	public static function shortcode(): string {
		return self::PLACEHOLDER;
	}

	public static function filter_the_content(string $content): string {
		if (is_admin()) {
			return $content;
		}

		$opts = self::get_options();
		if (!$opts['enabled']) {
			return $content;
		}

		if (!is_singular()) {
			return $content;
		}

		if (!in_the_loop() || !is_main_query()) {
			return $content;
		}

		$post_type = get_post_type();
		if (!is_string($post_type) || !in_array($post_type, $opts['post_types'], true)) {
			return $content;
		}

		$processed = self::build_toc_and_update_headings($content, $opts);
		if (empty($processed['headings']) || empty($processed['toc_html'])) {
			return $processed['content'];
		}

		$has_placeholder = strpos($processed['content'], self::PLACEHOLDER) !== false;
		if ($has_placeholder) {
			self::maybe_enqueue_assets($opts);
			return str_replace(self::PLACEHOLDER, $processed['toc_html'], $processed['content']);
		}

		if ($opts['auto_insert']) {
			self::maybe_enqueue_assets($opts);
			return $processed['toc_html'] . $processed['content'];
		}

		return $processed['content'];
	}

	private static function build_toc_and_update_headings(string $content, array $opts): array {
		$levels = $opts['heading_levels'];
		$used_ids = [];
		$headings = [];

		$pattern = '~<h([1-6])([^>]*)>(.*?)</h\1>~is';
		$new_content = preg_replace_callback($pattern, static function (array $m) use ($levels, &$used_ids, &$headings) {
			$level = intval($m[1]);
			$attrs = $m[2] ?? '';
			$inner = $m[3] ?? '';

			if (!in_array($level, $levels, true)) {
				return $m[0];
			}

			if (stripos($attrs, 'data-toc="false"') !== false || stripos($attrs, "data-toc='false'") !== false) {
				return $m[0];
			}

			$id = '';
			if (preg_match('~\sid=(\")(.*?)(\1)~i', $attrs, $idm)) {
				$id = $idm[2];
			} elseif (preg_match("~\\sid=(')(.*?)(\\1)~i", $attrs, $idm)) {
				$id = $idm[2];
			} elseif (preg_match('~\sid=([^\s>]+)~i', $attrs, $idm)) {
				$id = trim($idm[1], "\"'");
			}

			$label = wp_strip_all_tags($inner);
			$label = html_entity_decode($label, ENT_QUOTES, get_bloginfo('charset'));
			$label = trim(preg_replace('~\s+~', ' ', $label));

			if ($id === '') {
				$base = sanitize_title($label);
				if ($base === '') {
					$base = 'section';
				}
				$candidate = $base;
				$i = 2;
				while (isset($used_ids[$candidate])) {
					$candidate = $base . '-' . $i;
					$i++;
				}
				$id = $candidate;
				$used_ids[$id] = true;

				$attrs .= ' id="' . esc_attr($id) . '"';
			} else {
				$id = trim($id);
				if ($id === '') {
					$id = 'section';
				}
				if (isset($used_ids[$id])) {
					$base = $id;
					$candidate = $base;
					$i = 2;
					while (isset($used_ids[$candidate])) {
						$candidate = $base . '-' . $i;
						$i++;
					}
					$id = $candidate;
					$attrs = preg_replace('~\sid=(?:"[^"]*"|\'[^\']*\'|[^\s>]+)~i', '', $attrs);
					$attrs .= ' id="' . esc_attr($id) . '"';
				}
				$used_ids[$id] = true;
			}

			$headings[] = [
				'level' => $level,
				'id' => $id,
				'label' => $label,
			];

			return '<h' . $level . $attrs . '>' . $inner . '</h' . $level . '>';
		}, $content);

		if (!is_string($new_content)) {
			$new_content = $content;
		}

		if (count($headings) < 2) {
			return [
				'content' => $new_content,
				'headings' => [],
				'toc_html' => '',
			];
		}

		$toc_html = self::render_toc($headings, $opts);

		return [
			'content' => $new_content,
			'headings' => $headings,
			'toc_html' => $toc_html,
		];
	}

	private static function render_toc(array $headings, array $opts): string {
		$label = $opts['label'];
		$list_style = $opts['list_style'];

		$nav = '<nav class="nanotoc" aria-label="' . esc_attr($label) . '">';
		$nav .= '<p class="nanotoc__title"><strong>' . esc_html($label) . '</strong></p>';

		if ($list_style === 'flat') {
			$nav .= '<ul class="nanotoc__list">';
			foreach ($headings as $h) {
				$nav .= '<li class="nanotoc__item"><a class="nanotoc__link" href="#' . esc_attr($h['id']) . '">' . esc_html($h['label']) . '</a></li>';
			}
			$nav .= '</ul>';
			$nav .= '</nav>';
			return $nav;
		}

		$tree = self::build_heading_tree($headings);
		$nav .= self::render_heading_tree($tree, true);
		$nav .= '</nav>';
		return $nav;
	}

	private static function build_heading_tree(array $headings): array {
		$min = 7;
		foreach ($headings as $h) {
			$lvl = intval($h['level']);
			if ($lvl >= 1 && $lvl <= 6 && $lvl < $min) {
				$min = $lvl;
			}
		}
		if ($min === 7) {
			$min = 2;
		}

		$root = [
			'level' => $min - 1,
			'children' => [],
		];
		$stack = [&$root];

		foreach ($headings as $h) {
			$lvl = intval($h['level']);
			if ($lvl < $min) {
				$lvl = $min;
			}

			$parent_level = intval($stack[count($stack) - 1]['level']);
			if ($lvl > $parent_level + 1) {
				$lvl = $parent_level + 1;
			}

			while (count($stack) > 1 && $lvl <= intval($stack[count($stack) - 1]['level'])) {
				array_pop($stack);
			}

			$node = [
				'level' => $lvl,
				'id' => $h['id'],
				'label' => $h['label'],
				'children' => [],
			];

			$stack[count($stack) - 1]['children'][] = $node;
			$index = count($stack[count($stack) - 1]['children']) - 1;
			$stack[] = &$stack[count($stack) - 1]['children'][$index];
		}

		return $root['children'];
	}

	private static function render_heading_tree(array $nodes, bool $is_root = false): string {
		if (empty($nodes)) {
			return '';
		}

		$html = '<ul class="nanotoc__list">';
		foreach ($nodes as $n) {
			$lvl = intval($n['level'] ?? 2);
			$id = (string)($n['id'] ?? '');
			$label = (string)($n['label'] ?? '');

			$html .= '<li class="nanotoc__item nanotoc__item--h' . $lvl . '">';
			$html .= '<a class="nanotoc__link" href="#' . esc_attr($id) . '">' . esc_html($label) . '</a>';
			if (!empty($n['children'])) {
				$html .= self::render_heading_tree($n['children'], false);
			}
			$html .= '</li>';
		}
		$html .= '</ul>';

		return $html;
	}
}

NanoTOC_Plugin::init();
