<?php
namespace ExpandOps\Nanomailer\Services;
defined( 'ABSPATH' ) || exit;
use ExpandOps\Nanomailer\Utilities\Log;
use ExpandOps\Nanomailer\Config;
use ExpandOps\Nanomailer\Utilities\RecordActivity;
use ExpandOps\Nanomailer\Utilities\TestOutput;
class WpMailToSes
{
    public static function process( $to_email, $subject, $message, $from_email, $from_name, $access_key, $secret_key, $region, $reply_to, $cc = [], $bcc = [] )
    {
        $ajax_test = expandops_shared_verify_ajax_action('expandops_nanomailer_ajax_test');
        $to_email = is_array($to_email) ? array_map('sanitize_email', $to_email) : [ sanitize_email($to_email) ];
        $to_email_log = implode(', ', $to_email); 
        $to_email_log = ( mb_strlen($to_email_log) > 60 ) ? mb_substr($to_email_log, 0, 57) . '...' : $to_email_log; 
        $subject = sanitize_text_field($subject);
        $subject_log = ( mb_strlen($subject) > 30 ) ? mb_substr($subject, 0, 27) . '...' : $subject; 
        $message = wp_kses_post($message);
        $from_email = sanitize_email($from_email);
        $from_name = sanitize_text_field($from_name); 
        $formatted_from = !empty($from_name) ? "$from_name <$from_email>" : $from_email; 
        $cc = array_map('sanitize_email', $cc);
        $bcc = array_map('sanitize_email', $bcc);
        $reply_to = sanitize_text_field($reply_to);
        $service = 'ses';
        $method = 'POST';
        $uri = '/v2/email/outbound-emails';
        $body = json_encode([
            'FromEmailAddress' => $formatted_from,
            'ReplyToAddresses' => [ $reply_to ],
            'Destination' => [
                'ToAddresses' => $to_email,
                'CcAddresses' => $cc,
                'BccAddresses' => $bcc,
            ],
            'Content' => [
                'Simple' => [
                    'Subject' => [
                        'Data' => $subject,
                        'Charset' => 'UTF-8',
                    ],
                    'Body' => [
                        'Text' => [
                            'Data' => $message,
                            'Charset' => 'UTF-8',
                        ],
                        'Html' => [
                            'Data' => "<p>$message</p>",
                            'Charset' => 'UTF-8',
                        ],
                    ],
                ],
            ],
        ]);
        $headers = self::AwsSignRequest($method, $uri, '', $body, $region, $service, $access_key, $secret_key);
        if ( $ajax_test ){ TestOutput::info('Payload body and signature ready for SES API. Sending...'); }
        $response = wp_remote_post("https://email.$region.amazonaws.com$uri", [
            'method' => $method,
            'headers' => $headers,
            'body' => $body,
            'timeout' => 15,
        ]);
        if ( is_wp_error($response) )
        {
            $error = 'Problem encountered with POST to AWS, see below:';
            Log::error( $error );
            Log::error( $response->get_error_message() );
            RecordActivity::error( "To: $to_email_log. Subject: $subject_log. Check wp-expandops.log" );
            if ( $ajax_test )
            { 
                TestOutput::error( $error ); 
                TestOutput::error( $response->get_error_message() ); 
            }
            return [ 'success' => false ]; 
        }
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        if ( $response_code === 200 )
        {
            $success = "To: $to_email_log. Subject: $subject_log.";
            RecordActivity::success( $success );
            if ( $ajax_test ) { TestOutput::success( $success ); } 
            return [ 'success' => true ]; 
        } 
        else
        {
            $short_body = mb_substr( $response_body, 0, 500 ); 
            $error = "Response error encountered when sending to AWS. HTTP Code: $response_code. Response: $short_body";
            Log::error( $error );
            if ( $ajax_test ){ TestOutput::error( $error ); } 
            RecordActivity::error( "To: $to_email_log. Subject: $subject_log. Check wp-expandops.log." );
            return [ 'success' => false ]; 
        }
    } 
    private static function awsSignRequest( $method, $uri, $query_params, $body, $region, $service, $access_key, $secret_key )
    {
        $host = "email.$region.amazonaws.com";
        $endpoint = "https://$host$uri";
        $timestamp = gmdate('Ymd\THis\Z');
        $date = gmdate('Ymd');
        $hashed_body = hash('sha256', $body);
        $canonical_request = "$method\n$uri\n$query_params\nhost:$host\n\nhost\n$hashed_body";
        $algorithm = 'AWS4-HMAC-SHA256';
        $credential_scope = "$date/$region/$service/aws4_request";
        $string_to_sign = "$algorithm\n$timestamp\n$credential_scope\n" . hash('sha256', $canonical_request);
        $k_date = hash_hmac('sha256', $date, 'AWS4' . $secret_key, true);
        $k_region = hash_hmac('sha256', $region, $k_date, true);
        $k_service = hash_hmac('sha256', $service, $k_region, true);
        $k_signing = hash_hmac('sha256', 'aws4_request', $k_service, true);
        $signature = hash_hmac('sha256', $string_to_sign, $k_signing);
        $authorization = "$algorithm Credential=$access_key/$credential_scope, SignedHeaders=host, Signature=$signature";
        return [
            'Host' => $host,
            'X-Amz-Date' => $timestamp,
            'Authorization' => $authorization,
            'Content-Type' => 'application/json',
        ];
    } 
} 