<?php
namespace ExpandOps\Nanomailer\Rules;
defined('ABSPATH') || exit;
use ExpandOps\AAAFoundation\Classes\V1\BaseValidationRule;
use ExpandOps\Nanomailer\Utilities\Log;
class Validation extends BaseValidationRule
{
    public function __construct($field_id, $value, $saved_options)
    {
        parent::__construct($field_id, $value, $saved_options);
    }
    protected function route_wp_mail_to_ses( $value )
    {
        $value = (string) $value;
        if( $value === 'disabled' )
        {
            return $value; 
        }
        $failure_message = '';
        if( !defined('NANOMAILER_AWS_ACCESS_KEY') || empty(NANOMAILER_AWS_ACCESS_KEY) )
        {
            $failure_message .= '"AWS Access Key" is not set. ';
        }
        if( !defined('NANOMAILER_AWS_SECRET_KEY') || empty(NANOMAILER_AWS_SECRET_KEY) )
        {
            $failure_message .= '"AWS Secret Key" is not set. ';
        }
        if( !$this->getSavedOptions('region') )
        {
            $failure_message .= '"Region" is not set. '; 
        }
        if( !$this->getSavedOptions('verified_email_identities') )
        {
            $failure_message .= '"Verified Email Identities" is not set. '; 
        }
        if( $failure_message !== '' )
        {
            return $this->failureMessage( '"Route All wp_mail() Emails to AWS SES" can not be enabled if ' . $failure_message );
        }
        if( $value === 'enabled' )
        {
            return $value;
        }
        return $this->failureMessage('The value of "Route All wp_mail() Emails to AWS SES" must be either enabled or disabled.');
    } 
    protected function region($value)
    {
        $value = (string) $value;
        $label = '"Region"';
        $allow_result = $this->allowEmptyOrFail( $value, $label );
        if ( $allow_result !== null ) { return $allow_result; } 
        $allowed_regions = [
            'af-south-1',
            'ap-east-1',
            'ap-south-2',
            'ap-southeast-3',
            'ap-southeast-5',
            'ap-southeast-4',
            'ap-south-1',
            'ap-northeast-3	',
            'ap-northeast-2',
            'ap-southeast-1',
            'ap-southeast-2	',
            'ap-southeast-7',
            'ap-northeast-1',
            'ca-central-1',
            'ca-west-1',
            'eu-central-1',
            'eu-west-1',
            'eu-west-2',
            'eu-south-1',
            'eu-west-3',
            'eu-south-2',
            'eu-north-1',
            'eu-central-2',
            'il-central-1',
            'mx-central-1',
            'me-south-1',
            'me-central-1',
            'sa-east-1',
            'us-east-1',
            'us-east-2',
            'us-west-1',
            'us-west-2',
        ];
        if(  in_array($value, $allowed_regions) )
        {
            return $value;
        }
        return $this->failureMessage($label . 'Select a Region from the list.');
    } 
    protected function verified_email_identities($value)
    {
        $value = (string) $value;
        $label = '"Verified Email Identities"';
        $allow_result = $this->allowEmptyOrFail( $value, $label );
        if ( $allow_result !== null ) { return $allow_result; } 
        $emails = array_map( 'trim', explode(',', $value) );
        $emails = array_slice( $emails, 0, 3);
        foreach ( $emails as $email )
        {
            if ( !is_email($email) )
            {
                return $this->failureMessage( "$label must be a single email identity, or a comma-separated list of email identities (maximum of 3). ");
            }
        }
        if ( mb_strlen($value) > 200 )
        {
            return $this->failureMessage( "$label must not exceed 200 characters in length.");
        }
        return implode(',', $emails); 
    } 
    protected function send_from_name($value)
    {
        $value = (string) $value;
        if ( $value === '' ) {
            return $value; 
        }
        if ( preg_match('/^[a-zA-Z0-9\- ]{2,35}$/', $value) ) 
        {
            return $value;
        }
        return $this->failureMessage('"Send From Name" can only contain letters, numbers, and dashes, and be between 2 and 35 characters long.');
    } 
    protected function reply_to($value)
    {
        $value = (string) $value;
        if ( $value === '' ) {
            return $value; 
        }
        if ( !is_email($value) )
        {
            return $this->failureMessage("Value must be a valid email.");
        }
        if ( mb_strlen($value) > 100 )
        {
            return $this->failureMessage("Value must not exceed 100 characters.");
        }
        return $value; 
    } 
    protected function allowEmptyOrFail( $value, $label = '"LABEL" ')
    {
        if( $value !== '' )
        {
            return null;  
        }
        if( $this->getSavedOptions('route_wp_mail_to_ses') === 'enabled' )
        {
            return $this->failureMessage($label .' is required for "Route All wp_mail() Emails to AWS SES" to be enabled.');
        }
        return ''; 
    }    
}