<?php
/**
 * Plugin Name: Nandann AI Smart 404 Redirects - Auto Fix Broken Links & Typos
 * Plugin URI: https://nandann.com/contact
 * Description: Automatically fix 404 errors using AI-powered smart matching and typo detection. Redirect lost visitors to the most relevant page or your custom destination.
 * Version: 1.2.5
 * Author: Prakhar Bhatia
 * Author URI: https://nandann.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: nandann-ai-smart-404-redirect
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.2
 *
 * @package NandannAISmart404Redirect
 * @author Prakhar Bhatia <prakhar@nandann.com>
 * @copyright 2025 Nandann Creative Agency
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('NANDANN_SMART_AI_404_VERSION', '1.2.5');
define('NANDANN_SMART_AI_404_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('NANDANN_SMART_AI_404_PLUGIN_URL', plugin_dir_url(__FILE__));
define('NANDANN_SMART_AI_404_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('NANDANN_SMART_AI_404_OPTIONS_KEY', 'nandann_smart_ai_404_options');
define('NANDANN_SMART_AI_404_TABLE_NAME', 'nandann_smart_ai_404_logs');

// Include required files
require_once NANDANN_SMART_AI_404_PLUGIN_DIR . 'includes/functions.php';
require_once NANDANN_SMART_AI_404_PLUGIN_DIR . 'includes/smart-matcher.php';
require_once NANDANN_SMART_AI_404_PLUGIN_DIR . 'includes/email-notifications.php';

// Activation and deactivation hooks
register_activation_hook(__FILE__, 'nandann_smart_ai_404_activate_plugin');
register_deactivation_hook(__FILE__, 'nandann_smart_ai_404_deactivate_plugin');

/**
 * Plugin activation
 */
function nandann_smart_ai_404_activate_plugin() {
    nandann_smart_ai_404_create_database_table();
    nandann_smart_ai_404_initialize_options();
    flush_rewrite_rules();
}

/**
 * Plugin deactivation
 */
function nandann_smart_ai_404_deactivate_plugin() {
    nandann_smart_ai_404_clear_scheduled_events();
    flush_rewrite_rules();
}

/**
 * Create database table for logging
 */
function nandann_smart_ai_404_create_database_table() {
    global $wpdb;
    
    $table_name = $wpdb->prefix . NANDANN_SMART_AI_404_TABLE_NAME;
    $charset_collate = $wpdb->get_charset_collate();
    
    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        requested_url text NOT NULL,
        redirected_to text,
        redirect_type varchar(20) NOT NULL DEFAULT 'fallback',
        visitor_ip varchar(45),
        referrer text,
        user_agent text,
        hit_count int(11) NOT NULL DEFAULT 1,
        match_score int(11) DEFAULT NULL,
        first_occurrence datetime NOT NULL,
        last_occurrence datetime NOT NULL,
        PRIMARY KEY (id),
        KEY idx_requested_url (requested_url(191)),
        KEY idx_last_occurrence (last_occurrence),
        KEY idx_redirect_type (redirect_type)
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

/**
 * Initialize default options
 */
function nandann_smart_ai_404_initialize_options() {
    $defaults = array(
        'enabled' => '1',
        'redirect_method' => 'smart', // smart, homepage, custom
        'fallback_url' => home_url(),
        'redirect_status' => '301',
        'smart_matching_enabled' => '1',
        'keyword_threshold' => '60',
        'typo_detection_enabled' => '1',
        'typo_threshold' => '3',
        'exclude_media' => '1',
        'excluded_paths' => '', // Comma-separated list of path prefixes to exclude (e.g., /go/,/out/,/link/)
        'log_enabled' => '1',
        'log_retention_days' => '30',
        'email_enabled' => '1',
        'email_frequency' => 'weekly',
        'email_recipient' => get_option('admin_email'),
        'install_date' => current_time('mysql')
    );
    
    if (!get_option(NANDANN_SMART_AI_404_OPTIONS_KEY)) {
        add_option(NANDANN_SMART_AI_404_OPTIONS_KEY, $defaults);
    }
}

/**
 * Main redirect handler
 */
function nandann_smart_ai_404_handle_404_redirect() {
    if (!is_404()) {
        return;
    }
    
    $options = get_option(NANDANN_SMART_AI_404_OPTIONS_KEY);
    
    if (!$options || $options['enabled'] !== '1') {
        return;
    }
    
    $requested_url = nandann_smart_ai_404_get_current_url();
    
    // Check if this is a media file and should be excluded
    if ($options['exclude_media'] === '1' && nandann_smart_ai_404_is_media_url($requested_url)) {
        return;
    }
    
    // Check if this path should be excluded (e.g., affiliate links, pretty links)
    if (!empty($options['excluded_paths']) && nandann_smart_ai_404_is_excluded_path($requested_url, $options['excluded_paths'])) {
        return;
    }
    
    // Determine redirect target
    $redirect_target = '';
    $redirect_type = 'fallback';
    $match_score = 0;
    
    if ($options['redirect_method'] === 'smart' && $options['smart_matching_enabled'] === '1') {
        // Use AI Smart Matching
        $smart_match = nandann_smart_ai_404_find_smart_match($requested_url, $options);
        
        if ($smart_match && !empty($smart_match['url'])) {
            $redirect_target = $smart_match['url'];
            $redirect_type = $smart_match['type'];
            $match_score = $smart_match['score'];
        }
    }
    
    // Fallback to default if no smart match found
    if (empty($redirect_target)) {
        if ($options['redirect_method'] === 'custom' && !empty($options['fallback_url'])) {
            $redirect_target = $options['fallback_url'];
        } else {
            $redirect_target = home_url();
        }
        $redirect_type = 'fallback';
    }
    
    // Prevent infinite loops
    if ($redirect_target === $requested_url) {
        return;
    }
    
    // Log the redirect
    if ($options['log_enabled'] === '1') {
        nandann_smart_ai_404_log_redirect($requested_url, $redirect_target, $redirect_type, $match_score);
    }
    
    // Perform the redirect
    $status_code = intval($options['redirect_status']);
    wp_safe_redirect($redirect_target, $status_code);
    exit;
}
add_action('template_redirect', 'nandann_smart_ai_404_handle_404_redirect', 1);

/**
 * Handle CSV export early before any output
 */
function nandann_smart_ai_404_handle_export() {
    // Check if we're on the plugin page and export is requested
    if (!is_admin()) {
        return;
    }
    
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce is verified in the next line
    if (isset($_GET['page']) && $_GET['page'] === 'nandann-ai-smart-404-redirect' && 
        isset($_GET['action']) && $_GET['action'] === 'export') {
        
        // Verify nonce
        if (!isset($_GET['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['nonce'])), 'nandann_smart_ai_404_export_nonce')) {
            wp_die(esc_html__('Security check failed', 'nandann-ai-smart-404-redirect'));
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions', 'nandann-ai-smart-404-redirect'));
        }
        
        // Export and exit
        nandann_smart_ai_404_export_logs_csv();
        exit;
    }
}
add_action('admin_init', 'nandann_smart_ai_404_handle_export', 1);

/**
 * Add admin menu
 */
function nandann_smart_ai_404_add_admin_menu() {
    // Use icon file from assets folder
    $icon_url = NANDANN_SMART_AI_404_PLUGIN_URL . 'assets/icon.svg';
    
    add_menu_page(
        __('AI Smart 404 Redirect', 'nandann-ai-smart-404-redirect'),
        __('AI 404 Redirect', 'nandann-ai-smart-404-redirect'),
        'manage_options',
        'nandann-ai-smart-404-redirect',
        'nandann_smart_ai_404_render_admin_page',
        $icon_url,
        25
    );
}
add_action('admin_menu', 'nandann_smart_ai_404_add_admin_menu');

/**
 * Add custom CSS for menu icon
 */
function nandann_smart_ai_404_admin_menu_icon_css() {
    // Ensure CSS loads on ALL admin pages by attaching to the core 'common' admin stylesheet
    wp_enqueue_style('common');

    $custom_css = "
        /* Constrain AI 404 Redirect menu icon size at all times */
        #adminmenu .toplevel_page_nandann-ai-smart-404-redirect .wp-menu-image img,
        #adminmenu .menu-top.toplevel_page_nandann-ai-smart-404-redirect > a .wp-menu-image img {
            max-width: 20px !important;
            width: 20px !important;
            height: auto !important;
            padding: 7px 0 !important;
            margin-top: 2px !important;
            opacity: 1 !important;
        }
        #adminmenu .toplevel_page_nandann-ai-smart-404-redirect .wp-menu-image svg,
        #adminmenu .menu-top.toplevel_page_nandann-ai-smart-404-redirect > a .wp-menu-image svg {
            width: 20px !important;
            height: auto !important;
        }
    ";
    wp_add_inline_style('common', $custom_css);
}
add_action('admin_enqueue_scripts', 'nandann_smart_ai_404_admin_menu_icon_css');

/**
 * Render admin page
 */
function nandann_smart_ai_404_render_admin_page() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'nandann-ai-smart-404-redirect'));
    }
    
    require_once NANDANN_SMART_AI_404_PLUGIN_DIR . 'includes/admin-page.php';
}

/**
 * Enqueue admin styles and scripts
 */
function nandann_smart_ai_404_enqueue_admin_assets($hook) {
    // Only load on our plugin main page based on $hook suffix to avoid direct $_GET access
    // Example $hook: 'toplevel_page_nandann-ai-smart-404-redirect'
    if (strpos($hook, 'toplevel_page_nandann-ai-smart-404-redirect') === false) {
        return;
    }
    
    wp_enqueue_style(
        'nandann-smart-ai-404-admin-styles',
        NANDANN_SMART_AI_404_PLUGIN_URL . 'assets/css/admin-styles.css',
        array(),
        NANDANN_SMART_AI_404_VERSION . '-' . time()
    );
    
    wp_enqueue_script(
        'nandann-smart-ai-404-admin-scripts',
        NANDANN_SMART_AI_404_PLUGIN_URL . 'assets/js/admin-scripts.js',
        array('jquery'),
        NANDANN_SMART_AI_404_VERSION . '-' . time(),
        true
    );
    
    wp_localize_script('nandann-smart-ai-404-admin-scripts', 'aismart404Ajax', array(
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('nandann-smart-ai-404-ajax-nonce')
    ));
}
add_action('admin_enqueue_scripts', 'nandann_smart_ai_404_enqueue_admin_assets');

/**
 * Add admin bar stats widget
 */
function nandann_smart_ai_404_add_admin_bar_stats($wp_admin_bar) {
    if (!current_user_can('manage_options')) {
        return;
    }
    
    $stats = nandann_smart_ai_404_get_quick_stats();
    
    /* translators: %s: Number of 404 errors today */
    $title = sprintf(__('404s: %s', 'nandann-ai-smart-404-redirect'), number_format($stats['today']));
    
    $wp_admin_bar->add_node(array(
        'id'    => 'nandann-smart-ai-404-stats',
        'title' => '<span class="ab-icon dashicons dashicons-warning"></span> ' . $title,
        'href'  => admin_url('admin.php?page=nandann-ai-smart-404-redirect&tab=logs'),
        'meta'  => array(
            'class' => 'nandann-smart-ai-404-admin-bar-item'
        )
    ));
}
add_action('admin_bar_menu', 'nandann_smart_ai_404_add_admin_bar_stats', 100);

/**
 * Add plugin action links
 */
function nandann_smart_ai_404_plugin_action_links($links) {
    $settings_link = sprintf(
        '<a href="%s">%s</a>',
        admin_url('admin.php?page=nandann-ai-smart-404-redirect'),
        __('Settings', 'nandann-ai-smart-404-redirect')
    );
    array_unshift($links, $settings_link);
    return $links;
}
add_filter('plugin_action_links_' . NANDANN_SMART_AI_404_PLUGIN_BASENAME, 'nandann_smart_ai_404_plugin_action_links');

/**
 * AJAX handler for clearing logs
 */
function nandann_smart_ai_404_ajax_clear_logs() {
    check_ajax_referer('nandann-smart-ai-404-ajax-nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Insufficient permissions', 'nandann-ai-smart-404-redirect'));
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . NANDANN_SMART_AI_404_TABLE_NAME;
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe
    $wpdb->query("TRUNCATE TABLE $table_name");
    
    // Clear all caches
    nandann_smart_ai_404_clear_cache();
    
    wp_send_json_success(__('Logs cleared successfully', 'nandann-ai-smart-404-redirect'));
}
add_action('wp_ajax_nandann_smart_ai_404_clear_logs', 'nandann_smart_ai_404_ajax_clear_logs');

/**
 * AJAX handler for testing email
 */
function nandann_smart_ai_404_ajax_test_email() {
    check_ajax_referer('nandann-smart-ai-404-ajax-nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Insufficient permissions', 'nandann-ai-smart-404-redirect'));
    }
    
    $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
    $result = nandann_smart_ai_404_send_test_email($email);
    
    if ($result) {
        wp_send_json_success(__('Test email sent successfully', 'nandann-ai-smart-404-redirect'));
    } else {
        $last_error = get_transient('nandann_smart_ai_404_last_mail_error');
        // translators: %s is replaced with the specific error message returned by wp_mail
        $message = $last_error ? sprintf(__('Failed to send test email: %s', 'nandann-ai-smart-404-redirect'), $last_error) : __('Failed to send test email', 'nandann-ai-smart-404-redirect');
        wp_send_json_error($message);
    }
}
add_action('wp_ajax_nandann_smart_ai_404_test_email', 'nandann_smart_ai_404_ajax_test_email');

/**
 * AJAX handler to submit contact form
 */
function nandann_smart_ai_404_ajax_submit_contact() {
    check_ajax_referer('nandann_smart_ai_404_contact_nonce', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized', 'nandann-ai-smart-404-redirect'));
    }
    
    $name = isset($_POST['name']) ? sanitize_text_field(wp_unslash($_POST['name'])) : '';
    $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
    $message = isset($_POST['message']) ? sanitize_textarea_field(wp_unslash($_POST['message'])) : '';
    
    // Validate inputs
    if (empty($name) || empty($email) || empty($message)) {
        wp_send_json_error(__('All fields are required', 'nandann-ai-smart-404-redirect'));
    }
    
    if (!is_email($email)) {
        wp_send_json_error(__('Invalid email address', 'nandann-ai-smart-404-redirect'));
    }
    
    // Get site info
    $site_name = get_bloginfo('name');
    $site_url = get_site_url();
    
    // Prepare email
    $to = 'prakhar@nandann.com';
    $subject = sprintf('[AI Smart 404 Redirect] Contact Form - %s', $site_name);
    
    $email_body = sprintf(
        "New contact form submission from AI Smart 404 Redirect plugin:\n\n" .
        "Site: %s (%s)\n" .
        "Name: %s\n" .
        "Email: %s\n\n" .
        "Message:\n%s\n\n" .
        "---\n" .
        "Plugin Version: %s\n" .
        "WordPress Version: %s\n" .
        "Submitted: %s",
        $site_name,
        $site_url,
        $name,
        $email,
        $message,
        NANDANN_SMART_AI_404_VERSION,
        get_bloginfo('version'),
        current_time('mysql')
    );
    
    $headers = array(
        'Content-Type: text/plain; charset=UTF-8',
        'From: ' . $site_name . ' <' . get_option('admin_email') . '>',
        'Reply-To: ' . $name . ' <' . $email . '>'
    );
    
    $result = wp_mail($to, $subject, $email_body, $headers);
    
    if ($result) {
        wp_send_json_success(__('Thank you! Your message has been sent successfully.', 'nandann-ai-smart-404-redirect'));
    } else {
        wp_send_json_error(__('Failed to send message. Please try again later.', 'nandann-ai-smart-404-redirect'));
    }
}
add_action('wp_ajax_nandann_smart_ai_404_submit_contact', 'nandann_smart_ai_404_ajax_submit_contact');

/**
 * Schedule cleanup of old logs
 */
function nandann_smart_ai_404_schedule_log_cleanup() {
    if (!wp_next_scheduled('nandann_smart_ai_404_cleanup_old_logs')) {
        wp_schedule_event(time(), 'daily', 'nandann_smart_ai_404_cleanup_old_logs');
    }
}
add_action('wp', 'nandann_smart_ai_404_schedule_log_cleanup');

/**
 * Cleanup old logs based on retention setting
 */
function nandann_smart_ai_404_cleanup_old_logs() {
    $options = get_option(NANDANN_SMART_AI_404_OPTIONS_KEY);
    $retention_days = isset($options['log_retention_days']) ? intval($options['log_retention_days']) : 30;
    
    if ($retention_days <= 0) {
        return;
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . NANDANN_SMART_AI_404_TABLE_NAME;
    $cutoff_date = gmdate('Y-m-d H:i:s', strtotime("-{$retention_days} days", current_time('timestamp')));
    
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Intentional direct query for data cleanup, no caching needed for DELETE operations
    $wpdb->query($wpdb->prepare(
        "DELETE FROM $table_name WHERE last_occurrence < %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe
        $cutoff_date
    ));
    
    // Clear all caches after cleanup
    nandann_smart_ai_404_clear_cache();
}
add_action('nandann_smart_ai_404_cleanup_old_logs', 'nandann_smart_ai_404_cleanup_old_logs');

/**
 * Clear scheduled events
 */
function nandann_smart_ai_404_clear_scheduled_events() {
    wp_clear_scheduled_hook('nandann_smart_ai_404_cleanup_old_logs');
    wp_clear_scheduled_hook('nandann_smart_ai_404_send_email_report_daily');
    wp_clear_scheduled_hook('nandann_smart_ai_404_send_email_report_weekly');
    wp_clear_scheduled_hook('nandann_smart_ai_404_send_email_report_monthly');
}

// Note: load_plugin_textdomain() is not needed when plugin is hosted on WordPress.org
// WordPress will automatically load translations for the plugin slug

