<?php
/**
 * Helper Functions
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Get current URL
 */
function nandann_smart_ai_404_get_current_url() {
    $protocol = is_ssl() ? 'https://' : 'http://';
    $host = isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : '';
    $uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
    
    return $protocol . $host . $uri;
}

/**
 * Check if URL is a media file
 */
function nandann_smart_ai_404_is_media_url($url) {
    $media_extensions = array('jpg', 'jpeg', 'png', 'gif', 'bmp', 'svg', 'webp', 'ico', 'pdf', 'zip', 'mp3', 'mp4', 'avi', 'mov', 'css', 'js');
    $extension = strtolower(pathinfo(wp_parse_url($url, PHP_URL_PATH), PATHINFO_EXTENSION));
    
    return in_array($extension, $media_extensions);
}

/**
 * Check if URL path should be excluded from redirection
 * 
 * @param string $url The requested URL
 * @param string $excluded_paths Comma-separated list of path prefixes to exclude
 * @return bool True if path should be excluded, false otherwise
 */
function nandann_smart_ai_404_is_excluded_path($url, $excluded_paths) {
    if (empty($excluded_paths)) {
        return false;
    }
    
    $path = wp_parse_url($url, PHP_URL_PATH);
    if (empty($path)) {
        return false;
    }
    
    // Parse excluded paths (comma-separated)
    $excluded_list = array_map('trim', explode(',', $excluded_paths));
    
    foreach ($excluded_list as $excluded) {
        if (empty($excluded)) {
            continue;
        }
        
        // Ensure excluded path starts with /
        if (substr($excluded, 0, 1) !== '/') {
            $excluded = '/' . $excluded;
        }
        
        // Check if the requested path starts with the excluded prefix
        if (strpos($path, $excluded) === 0) {
            return true;
        }
    }
    
    return false;
}

/**
 * Clear all plugin caches
 */
function nandann_smart_ai_404_clear_cache() {
    delete_transient('nandann_smart_ai_404_quick_stats');
    delete_transient('nandann_smart_ai_404_statistics_today');
    delete_transient('nandann_smart_ai_404_statistics_week');
    delete_transient('nandann_smart_ai_404_statistics_month');
    delete_transient('nandann_smart_ai_404_statistics_all');
    delete_transient('nandann_smart_ai_404_chart_data');
    delete_transient('nandann_smart_ai_404_total_log_count');
    delete_transient('nandann_smart_ai_404_email_stats_daily');
    delete_transient('nandann_smart_ai_404_email_stats_weekly');
    delete_transient('nandann_smart_ai_404_email_stats_monthly');
}

/**
 * Log a redirect
 * 
 * phpcs:disable WordPress.DB.DirectDatabaseQuery -- Direct queries intentional for real-time logging
 */
function nandann_smart_ai_404_log_redirect($requested_url, $redirected_to, $redirect_type, $match_score = 0) {
    global $wpdb;
    
    $table_name = $wpdb->prefix . NANDANN_SMART_AI_404_TABLE_NAME;
    
    // Get visitor information
    $visitor_ip = nandann_smart_ai_404_get_visitor_ip();
    $referrer = isset($_SERVER['HTTP_REFERER']) ? esc_url_raw(wp_unslash($_SERVER['HTTP_REFERER'])) : '';
    $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
    
    // Check if this URL already exists
    $existing = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $table_name WHERE requested_url = %s ORDER BY last_occurrence DESC LIMIT 1", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $requested_url
    ));
    
    if ($existing) {
        // Update existing record
        $wpdb->update(
            $table_name,
            array(
                'redirected_to' => $redirected_to,
                'redirect_type' => $redirect_type,
                'visitor_ip' => $visitor_ip,
                'referrer' => $referrer,
                'user_agent' => $user_agent,
                'hit_count' => $existing->hit_count + 1,
                'match_score' => $match_score,
                'last_occurrence' => current_time('mysql')
            ),
            array('id' => $existing->id),
            array('%s', '%s', '%s', '%s', '%s', '%d', '%d', '%s'),
            array('%d')
        );
    } else {
        // Insert new record
        $wpdb->insert(
            $table_name,
            array(
                'requested_url' => $requested_url,
                'redirected_to' => $redirected_to,
                'redirect_type' => $redirect_type,
                'visitor_ip' => $visitor_ip,
                'referrer' => $referrer,
                'user_agent' => $user_agent,
                'hit_count' => 1,
                'match_score' => $match_score,
                'first_occurrence' => current_time('mysql'),
                'last_occurrence' => current_time('mysql')
            ),
            array('%s', '%s', '%s', '%s', '%s', '%s', '%d', '%d', '%s', '%s')
        );
    }
    
    // Clear cache after logging
    nandann_smart_ai_404_clear_cache();
}
// phpcs:enable WordPress.DB.DirectDatabaseQuery

/**
 * Get visitor IP address
 */
function nandann_smart_ai_404_get_visitor_ip() {
    $ip_keys = array(
        'HTTP_CF_CONNECTING_IP',
        'HTTP_CLIENT_IP',
        'HTTP_X_FORWARDED_FOR',
        'HTTP_X_FORWARDED',
        'HTTP_X_CLUSTER_CLIENT_IP',
        'HTTP_FORWARDED_FOR',
        'HTTP_FORWARDED',
        'REMOTE_ADDR'
    );
    
    foreach ($ip_keys as $key) {
        if (isset($_SERVER[$key])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER[$key]));
            if (filter_var($ip, FILTER_VALIDATE_IP)) {
                return $ip;
            }
        }
    }
    
    return 'unknown';
}

/**
 * Get quick stats for admin bar
 * 
 * phpcs:disable WordPress.DB.DirectDatabaseQuery -- Direct queries for real-time stats, cached for performance
 */
function nandann_smart_ai_404_get_quick_stats() {
    // Try to get cached stats
    $cached_stats = get_transient('nandann_smart_ai_404_quick_stats');
    if (false !== $cached_stats) {
        return $cached_stats;
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . NANDANN_SMART_AI_404_TABLE_NAME;
    
    $today = $wpdb->get_var($wpdb->prepare(
        "SELECT SUM(hit_count) FROM $table_name WHERE DATE(last_occurrence) = %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        current_time('Y-m-d')
    ));
    
    $week = $wpdb->get_var($wpdb->prepare(
        "SELECT SUM(hit_count) FROM $table_name WHERE last_occurrence >= %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        gmdate('Y-m-d H:i:s', strtotime('-7 days', current_time('timestamp')))
    ));
    
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe
    $total = $wpdb->get_var("SELECT SUM(hit_count) FROM $table_name");
    
    $stats = array(
        'today' => intval($today),
        'week' => intval($week),
        'total' => intval($total)
    );
    
    // Cache for 1 minute
    set_transient('nandann_smart_ai_404_quick_stats', $stats, MINUTE_IN_SECONDS);
    
    return $stats;
}
// phpcs:enable WordPress.DB.DirectDatabaseQuery

/**
 * Get detailed statistics
 */
// phpcs:disable WordPress.DB.DirectDatabaseQuery -- Direct queries for real-time statistics, cached for performance
function nandann_smart_ai_404_get_statistics($period = 'week') {
    // Try to get cached stats
    $cache_key = 'nandann_smart_ai_404_statistics_' . $period;
    $cached_stats = get_transient($cache_key);
    if (false !== $cached_stats) {
        return $cached_stats;
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . NANDANN_SMART_AI_404_TABLE_NAME;
    $escaped_table = esc_sql( $table_name );
    
    $date_filter = '';
    switch ($period) {
        case 'today':
            $date_filter = $wpdb->prepare("WHERE DATE(last_occurrence) = %s", current_time('Y-m-d'));
            break;
        case 'week':
            $date_filter = $wpdb->prepare("WHERE last_occurrence >= %s", gmdate('Y-m-d H:i:s', strtotime('-7 days', current_time('timestamp'))));
            break;
        case 'month':
            $date_filter = $wpdb->prepare("WHERE last_occurrence >= %s", gmdate('Y-m-d H:i:s', strtotime('-30 days', current_time('timestamp'))));
            break;
        case 'all':
        default:
            $date_filter = '';
            break;
    }
    
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name is escaped, date filter is prepared
    $total_errors = $wpdb->get_var("SELECT SUM(hit_count) FROM {$escaped_table} {$date_filter}");
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name is escaped, date filter is prepared
    $unique_urls = $wpdb->get_var("SELECT COUNT(DISTINCT requested_url) FROM {$escaped_table} {$date_filter}");
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name is escaped, date filter is prepared
    $smart_matches = $wpdb->get_var("SELECT COUNT(*) FROM {$escaped_table} {$date_filter} AND redirect_type != 'fallback'");
    
    $stats = array(
        'total_errors' => intval($total_errors),
        'unique_urls' => intval($unique_urls),
        'smart_matches' => intval($smart_matches),
        'fallback_redirects' => intval($total_errors) - intval($smart_matches)
    );
    
    // Cache for 1 minute
    set_transient($cache_key, $stats, MINUTE_IN_SECONDS);
    
    return $stats;
}
// phpcs:enable WordPress.DB.DirectDatabaseQuery

/**
 * Get top 404 URLs
 */
// phpcs:disable WordPress.DB.DirectDatabaseQuery -- Direct queries for log retrieval
function nandann_smart_ai_404_get_top_404_urls($limit = 10, $offset = 0) {
    global $wpdb;
    $table_name = $wpdb->prefix . NANDANN_SMART_AI_404_TABLE_NAME;
    
    return $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM $table_name ORDER BY last_occurrence DESC LIMIT %d OFFSET %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $limit,
        $offset
    ));
}
// phpcs:enable WordPress.DB.DirectDatabaseQuery

/**
 * Get total log count
 */
// phpcs:disable WordPress.DB.DirectDatabaseQuery -- Direct queries for log count, cached for performance
function nandann_smart_ai_404_get_total_log_count() {
    // Try to get cached count
    $cached_count = get_transient('nandann_smart_ai_404_total_log_count');
    if (false !== $cached_count) {
        return $cached_count;
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . NANDANN_SMART_AI_404_TABLE_NAME;
    
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe
    $count = intval($wpdb->get_var("SELECT COUNT(*) FROM $table_name"));
    
    // Cache for 1 minute
    set_transient('nandann_smart_ai_404_total_log_count', $count, MINUTE_IN_SECONDS);
    
    return $count;
}
// phpcs:enable WordPress.DB.DirectDatabaseQuery

/**
 * Format redirect type for display
 */
function nandann_smart_ai_404_format_redirect_type($type) {
    $types = array(
        'typo_match' => __('Typo Match', 'nandann-ai-smart-404-redirect'),
        'keyword_match' => __('Keyword Match', 'nandann-ai-smart-404-redirect'),
        'category_match' => __('Category Match', 'nandann-ai-smart-404-redirect'),
        'tag_match' => __('Tag Match', 'nandann-ai-smart-404-redirect'),
        'taxonomy_similar' => __('Similar Taxonomy', 'nandann-ai-smart-404-redirect'),
        'fallback' => __('Fallback', 'nandann-ai-smart-404-redirect')
    );
    
    return isset($types[$type]) ? $types[$type] : ucfirst(str_replace('_', ' ', $type));
}

/**
 * Get chart data for dashboard
 */
// phpcs:disable WordPress.DB.DirectDatabaseQuery -- Direct queries for chart data, cached for performance
function nandann_smart_ai_404_get_chart_data($days = 7) {
    // Try to get cached chart data
    $cached_data = get_transient('nandann_smart_ai_404_chart_data');
    if (false !== $cached_data) {
        return $cached_data;
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . NANDANN_SMART_AI_404_TABLE_NAME;
    
    $data = array();
    
    for ($i = $days - 1; $i >= 0; $i--) {
        $date = gmdate('Y-m-d', strtotime("-$i days", current_time('timestamp')));
        
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT SUM(hit_count) FROM $table_name WHERE DATE(last_occurrence) = %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $date
        ));
        
        $data[] = array(
            'date' => gmdate('M j', strtotime($date)),
            'count' => intval($count)
        );
    }
    
    // Cache for 1 minute
    set_transient('nandann_smart_ai_404_chart_data', $data, MINUTE_IN_SECONDS);
    
    return $data;
}
// phpcs:enable WordPress.DB.DirectDatabaseQuery

/**
 * Delete old logs
 */
// phpcs:disable WordPress.DB.DirectDatabaseQuery -- Direct queries for log deletion
function nandann_smart_ai_404_delete_log($id) {
    global $wpdb;
    $table_name = $wpdb->prefix . NANDANN_SMART_AI_404_TABLE_NAME;
    
    $result = $wpdb->delete($table_name, array('id' => $id), array('%d'));
    
    // Clear cache after deletion
    if ($result) {
        nandann_smart_ai_404_clear_cache();
    }
    
    return $result;
}
// phpcs:enable WordPress.DB.DirectDatabaseQuery

/**
 * Export logs to CSV
 */
// phpcs:disable WordPress.DB.DirectDatabaseQuery -- Direct queries for CSV export
function nandann_smart_ai_404_export_logs_csv() {
    global $wpdb;
    $table_name = $wpdb->prefix . NANDANN_SMART_AI_404_TABLE_NAME;
    
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe
    $logs = $wpdb->get_results("SELECT * FROM $table_name ORDER BY last_occurrence DESC", ARRAY_A);
    
    if (empty($logs)) {
        return false;
    }
    
    $filename = 'nandann-smart-ai-404-404-logs-' . gmdate('Y-m-d', current_time('timestamp')) . '.csv';
    
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = fopen('php://output', 'w');
    
    // Write headers
    fputcsv($output, array('ID', 'Requested URL', 'Redirected To', 'Type', 'Score', 'Hits', 'IP', 'Referrer', 'First Seen', 'Last Seen'));
    
    // Write data
    foreach ($logs as $log) {
        fputcsv($output, array(
            $log['id'],
            $log['requested_url'],
            $log['redirected_to'],
            $log['redirect_type'],
            $log['match_score'],
            $log['hit_count'],
            $log['visitor_ip'],
            $log['referrer'],
            $log['first_occurrence'],
            $log['last_occurrence']
        ));
    }
    
    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing output stream for CSV export
    fclose($output);
    exit;
}
// phpcs:enable WordPress.DB.DirectDatabaseQuery

/**
 * Format number for display
 */
function nandann_smart_ai_404_format_number($number) {
    if ($number >= 1000000) {
        return number_format($number / 1000000, 1) . 'M';
    } elseif ($number >= 1000) {
        return number_format($number / 1000, 1) . 'K';
    }
    return number_format($number);
}

/**
 * Get match score badge HTML
 */
function nandann_smart_ai_404_get_score_badge($score) {
    if ($score >= 80) {
        $class = 'success';
    } elseif ($score >= 60) {
        $class = 'warning';
    } else {
        $class = 'default';
    }
    
    return sprintf('<span class="aismart404-badge aismart404-badge-%s">%d%%</span>', $class, $score);
}

