<?php
/**
 * Email Notification System
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Setup email schedules
 */
function nandann_smart_ai_404_setup_email_schedule() {
    $options = get_option(NANDANN_SMART_AI_404_OPTIONS_KEY);
    
    if (!$options || $options['email_enabled'] !== '1') {
        nandann_smart_ai_404_clear_email_schedules();
        return;
    }
    
    $frequency = $options['email_frequency'];
    
    // Clear existing schedules
    nandann_smart_ai_404_clear_email_schedules();
    
    // Schedule new event
    $hook = 'nandann_smart_ai_404_send_email_report_' . $frequency;
    
    if (!wp_next_scheduled($hook)) {
        $next_run = nandann_smart_ai_404_calculate_next_email_time($frequency);
        wp_schedule_event($next_run, $frequency, $hook);
    }
}

/**
 * Calculate next email time
 */
function nandann_smart_ai_404_calculate_next_email_time($frequency) {
    switch ($frequency) {
        case 'daily':
            return strtotime('tomorrow 9:00 AM');
        case 'weekly':
            return strtotime('next Monday 9:00 AM');
        case 'monthly':
            return strtotime('first day of next month 9:00 AM');
        default:
            return strtotime('tomorrow 9:00 AM');
    }
}

/**
 * Clear email schedules
 */
function nandann_smart_ai_404_clear_email_schedules() {
    wp_clear_scheduled_hook('nandann_smart_ai_404_send_email_report_daily');
    wp_clear_scheduled_hook('nandann_smart_ai_404_send_email_report_weekly');
    wp_clear_scheduled_hook('nandann_smart_ai_404_send_email_report_monthly');
}

/**
 * Send scheduled email report
 */
function nandann_smart_ai_404_send_scheduled_email($period) {
    $options = get_option(NANDANN_SMART_AI_404_OPTIONS_KEY);
    
    if (!$options || $options['email_enabled'] !== '1') {
        return;
    }
    
    $recipient = $options['email_recipient'];
    $stats = nandann_smart_ai_404_get_email_stats($period);
    
    if ($stats['total_errors'] == 0) {
        return; // No errors to report
    }
    
    $subject = nandann_smart_ai_404_get_email_subject($period, $stats);
    $message = nandann_smart_ai_404_generate_email_html($period, $stats);
    
    $headers = array(
        'Content-Type: text/html; charset=UTF-8',
        'From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>'
    );
    
    wp_mail($recipient, $subject, $message, $headers);
}

// Hook email actions
add_action('nandann_smart_ai_404_send_email_report_daily', function() { nandann_smart_ai_404_send_scheduled_email('daily'); });
add_action('nandann_smart_ai_404_send_email_report_weekly', function() { nandann_smart_ai_404_send_scheduled_email('weekly'); });
add_action('nandann_smart_ai_404_send_email_report_monthly', function() { nandann_smart_ai_404_send_scheduled_email('monthly'); });

/**
 * Get email statistics
 * 
 * phpcs:disable WordPress.DB.DirectDatabaseQuery -- Direct queries for email reports, cached for performance
 */
function nandann_smart_ai_404_get_email_stats($period) {
    // Try to get cached stats
    $cache_key = 'nandann_smart_ai_404_email_stats_' . $period;
    $cached_stats = get_transient($cache_key);
    if (false !== $cached_stats) {
        return $cached_stats;
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . NANDANN_SMART_AI_404_TABLE_NAME;
    
    // Determine date range
    $current_timestamp = current_time('timestamp');
    switch ($period) {
        case 'daily':
            $start_date = gmdate('Y-m-d 00:00:00', $current_timestamp);
            $end_date = gmdate('Y-m-d 23:59:59', $current_timestamp);
            break;
        case 'weekly':
            $start_date = gmdate('Y-m-d 00:00:00', strtotime('-7 days', $current_timestamp));
            $end_date = gmdate('Y-m-d 23:59:59', $current_timestamp);
            break;
        case 'monthly':
            $start_date = gmdate('Y-m-01 00:00:00', $current_timestamp);
            $end_date = gmdate('Y-m-t 23:59:59', $current_timestamp);
            break;
        default:
            $start_date = gmdate('Y-m-d 00:00:00', strtotime('-7 days', $current_timestamp));
            $end_date = gmdate('Y-m-d 23:59:59', $current_timestamp);
    }
    
    // Get total errors
    $total_errors = $wpdb->get_var($wpdb->prepare(
        "SELECT SUM(hit_count) FROM $table_name WHERE last_occurrence BETWEEN %s AND %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $start_date, $end_date
    ));
    
    // Get unique URLs
    $unique_urls = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(DISTINCT requested_url) FROM $table_name WHERE last_occurrence BETWEEN %s AND %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $start_date, $end_date
    ));
    
    // Get smart matches
    $smart_matches = $wpdb->get_var($wpdb->prepare(
        "SELECT SUM(hit_count) FROM $table_name WHERE last_occurrence BETWEEN %s AND %s AND redirect_type != 'fallback'", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $start_date, $end_date
    ));
    
    // Get top URLs
    // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
    $top_urls = $wpdb->get_results($wpdb->prepare(
        "SELECT requested_url, redirected_to, redirect_type, match_score, SUM(hit_count) as total_hits 
        FROM $table_name 
        WHERE last_occurrence BETWEEN %s AND %s 
        GROUP BY requested_url 
        ORDER BY total_hits DESC 
        LIMIT 5",
        $start_date, $end_date
    ), ARRAY_A);
    // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
    
    // Get redirect type breakdown
    // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
    $type_breakdown = $wpdb->get_results($wpdb->prepare(
        "SELECT redirect_type, SUM(hit_count) as count 
        FROM $table_name 
        WHERE last_occurrence BETWEEN %s AND %s 
        GROUP BY redirect_type 
        ORDER BY count DESC",
        $start_date, $end_date
    ), ARRAY_A);
    // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
    
    $stats = array(
        'total_errors' => intval($total_errors),
        'unique_urls' => intval($unique_urls),
        'smart_matches' => intval($smart_matches),
        'fallback_redirects' => intval($total_errors) - intval($smart_matches),
        'top_urls' => $top_urls,
        'type_breakdown' => $type_breakdown,
        'start_date' => $start_date,
        'end_date' => $end_date
    );
    
    // Cache for 1 minute
    set_transient($cache_key, $stats, MINUTE_IN_SECONDS);
    
    return $stats;
}
// phpcs:enable WordPress.DB.DirectDatabaseQuery

/**
 * Get email subject
 */
function nandann_smart_ai_404_get_email_subject($period, $stats) {
    $site_name = get_bloginfo('name');
    $period_text = ucfirst($period);
    
    return sprintf(
        '[%s] 404 Error Report (%s) - %s errors detected',
        $site_name,
        $period_text,
        number_format($stats['total_errors'])
    );
}

/**
 * Generate email HTML
 */
function nandann_smart_ai_404_generate_email_html($period, $stats) {
    $site_name = get_bloginfo('name');
    $period_text = ucfirst($period);
    $date_range = nandann_smart_ai_404_format_email_date_range($period, $stats['start_date'], $stats['end_date']);
    
    ob_start();
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>404 Error Report</title>
    </head>
    <body style="margin: 0; padding: 0; font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif; background-color: #f4f7fa;">
        <table width="100%" cellpadding="0" cellspacing="0" border="0" style="background-color: #f4f7fa; padding: 40px 20px;">
            <tr>
                <td align="center">
                    <table width="600" cellpadding="0" cellspacing="0" border="0" style="background-color: #ffffff; border-radius: 12px; box-shadow: 0 4px 12px rgba(0,0,0,0.1);">
                        <!-- Header -->
                        <tr>
                            <td style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px; text-align: center; border-radius: 12px 12px 0 0;">
                                <h1 style="margin: 0; color: #ffffff; font-size: 28px; font-weight: 700;">🔍 404 Error Report</h1>
                                <p style="margin: 10px 0 0; color: #e2e8f0; font-size: 16px;"><?php echo esc_html($site_name); ?></p>
                                <p style="margin: 5px 0 0; color: #cbd5e0; font-size: 14px;"><?php echo esc_html($date_range); ?></p>
                            </td>
                        </tr>
                        
                        <!-- Summary Stats -->
                        <tr>
                            <td style="padding: 30px;">
                                <h2 style="margin: 0 0 20px; color: #2d3748; font-size: 22px;">📊 Summary Statistics</h2>
                                
                                <table width="100%" cellpadding="10" cellspacing="0" border="0">
                                    <tr>
                                        <td width="50%" style="background-color: #fef5f5; padding: 20px; border-radius: 8px; border-left: 4px solid #f56565;">
                                            <div style="font-size: 36px; font-weight: 700; color: #c53030; margin-bottom: 5px;"><?php echo esc_html(number_format($stats['total_errors'])); ?></div>
                                            <div style="color: #718096; font-size: 14px;">Total 404 Errors</div>
                                        </td>
                                        <td width="10"></td>
                                        <td width="50%" style="background-color: #f0fff4; padding: 20px; border-radius: 8px; border-left: 4px solid #48bb78;">
                                            <div style="font-size: 36px; font-weight: 700; color: #2f855a; margin-bottom: 5px;"><?php echo esc_html(number_format($stats['smart_matches'])); ?></div>
                                            <div style="color: #718096; font-size: 14px;">Smart Matches</div>
                                        </td>
                                    </tr>
                                    <tr><td colspan="3" height="10"></td></tr>
                                    <tr>
                                        <td style="background-color: #ebf8ff; padding: 20px; border-radius: 8px; border-left: 4px solid #4299e1;">
                                            <div style="font-size: 28px; font-weight: 700; color: #2c5282; margin-bottom: 5px;"><?php echo esc_html(number_format($stats['unique_urls'])); ?></div>
                                            <div style="color: #718096; font-size: 14px;">Unique URLs</div>
                                        </td>
                                        <td width="10"></td>
                                        <td style="background-color: #faf5ff; padding: 20px; border-radius: 8px; border-left: 4px solid #9f7aea;">
                                            <div style="font-size: 28px; font-weight: 700; color: #553c9a; margin-bottom: 5px;"><?php echo esc_html(number_format($stats['fallback_redirects'])); ?></div>
                                            <div style="color: #718096; font-size: 14px;">Fallback Redirects</div>
                                        </td>
                                    </tr>
                                </table>
                            </td>
                        </tr>
                        
                        <?php if (!empty($stats['top_urls'])): ?>
                        <!-- Top 404 URLs -->
                        <tr>
                            <td style="padding: 0 30px 30px;">
                                <h2 style="margin: 0 0 15px; color: #2d3748; font-size: 20px;">🔝 Most Frequent 404 Errors</h2>
                                
                                <table width="100%" cellpadding="12" cellspacing="0" border="0" style="background-color: #f7fafc; border-radius: 8px;">
                                    <thead>
                                        <tr style="background-color: #edf2f7;">
                                            <th style="padding: 12px; text-align: left; color: #4a5568; font-size: 13px; font-weight: 600;">Requested URL</th>
                                            <th style="padding: 12px; text-align: center; color: #4a5568; font-size: 13px; font-weight: 600; width: 80px;">Hits</th>
                                            <th style="padding: 12px; text-align: center; color: #4a5568; font-size: 13px; font-weight: 600; width: 100px;">Match Type</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($stats['top_urls'] as $url): ?>
                                        <tr>
                                            <td style="padding: 12px; border-top: 1px solid #e2e8f0; font-size: 12px; color: #2d3748; word-break: break-all;">
                                                <?php echo esc_html($url['requested_url']); ?>
                                            </td>
                                            <td style="padding: 12px; border-top: 1px solid #e2e8f0; text-align: center; font-weight: 600; color: #e53e3e;">
                                                <?php echo esc_html(number_format($url['total_hits'])); ?>
                                            </td>
                                            <td style="padding: 12px; border-top: 1px solid #e2e8f0; text-align: center; font-size: 11px;">
                                                <span style="background-color: <?php echo esc_attr($url['redirect_type'] !== 'fallback' ? '#c6f6d5' : '#fed7d7'); ?>; 
                                                             color: <?php echo esc_attr($url['redirect_type'] !== 'fallback' ? '#22543d' : '#742a2a'); ?>; 
                                                             padding: 4px 8px; border-radius: 4px; font-weight: 600;">
                                                    <?php echo esc_html(nandann_smart_ai_404_format_redirect_type($url['redirect_type'])); ?>
                                                </span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </td>
                        </tr>
                        <?php endif; ?>
                        
                        <!-- Recommendations -->
                        <tr>
                            <td style="padding: 0 30px 30px;">
                                <h2 style="margin: 0 0 15px; color: #2d3748; font-size: 20px;">💡 Recommendations</h2>
                                
                                <div style="background-color: #fffbeb; border-left: 4px solid #f6e05e; padding: 20px; border-radius: 8px;">
                                    <ul style="margin: 0; padding-left: 20px; color: #744210; line-height: 1.8;">
                                        <?php if ($stats['fallback_redirects'] > $stats['smart_matches']): ?>
                                        <li>Consider improving your content to better match common 404 patterns</li>
                                        <?php endif; ?>
                                        <li>Review the top 404 URLs and create redirects or content for them</li>
                                        <li>Check for broken links in your content</li>
                                        <li>Update sitemap and internal linking structure</li>
                                        <?php if ($stats['total_errors'] > 100): ?>
                                        <li><strong>High error count detected!</strong> Immediate action recommended</li>
                                        <?php endif; ?>
                                    </ul>
                                </div>
                            </td>
                        </tr>
                        
                        <!-- Action Button -->
                        <tr>
                            <td style="padding: 0 30px 40px; text-align: center;">
                                <a href="<?php echo esc_url(admin_url('admin.php?page=nandann-ai-smart-404-redirect&tab=logs')); ?>" 
                                   style="display: inline-block; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); 
                                          color: #ffffff; padding: 14px 32px; text-decoration: none; border-radius: 8px; 
                                          font-weight: 600; font-size: 16px; box-shadow: 0 4px 12px rgba(102, 126, 234, 0.3);">
                                    View Full Report →
                                </a>
                            </td>
                        </tr>
                        
                        <!-- Footer -->
                        <tr>
                            <td style="background-color: #f7fafc; padding: 20px; text-align: center; border-radius: 0 0 12px 12px; border-top: 1px solid #e2e8f0;">
                                <p style="margin: 0; color: #718096; font-size: 12px;">
                                    This email was generated by <strong>AI Smart 404 Redirect</strong> plugin<br>
                                    <?php echo esc_html(get_site_url()); ?>
                                </p>
                                <p style="margin: 10px 0 0; color: #a0aec0; font-size: 11px;">
                                    Report generated on <?php echo esc_html(gmdate('F j, Y \a\t g:i A', current_time('timestamp'))); ?>
                                </p>
                            </td>
                        </tr>
                    </table>
                </td>
            </tr>
        </table>
    </body>
    </html>
    <?php
    return ob_get_clean();
}

/**
 * Format date range for email
 */
function nandann_smart_ai_404_format_email_date_range($period, $start_date, $end_date) {
    if ($period === 'daily') {
        return gmdate('F j, Y', strtotime($start_date));
    } else {
        return gmdate('F j', strtotime($start_date)) . ' - ' . gmdate('F j, Y', strtotime($end_date));
    }
}

/**
 * Send test email
 */
function nandann_smart_ai_404_send_test_email($recipient) {
    $stats = nandann_smart_ai_404_get_email_stats('weekly');
    
    // Create sample data if no real data exists
    if ($stats['total_errors'] == 0) {
        $stats = array(
            'total_errors' => 42,
            'unique_urls' => 15,
            'smart_matches' => 28,
            'fallback_redirects' => 14,
            'top_urls' => array(
                array('requested_url' => home_url('/sample-404-url'), 'total_hits' => 12, 'redirect_type' => 'keyword_match', 'match_score' => 85),
                array('requested_url' => home_url('/another-broken-link'), 'total_hits' => 8, 'redirect_type' => 'typo_match', 'match_score' => 92),
            ),
            'start_date' => gmdate('Y-m-d 00:00:00', strtotime('-7 days', current_time('timestamp'))),
            'end_date' => gmdate('Y-m-d 23:59:59', current_time('timestamp'))
        );
    }
    
    $subject = '[TEST] ' . nandann_smart_ai_404_get_email_subject('weekly', $stats);
    $message = nandann_smart_ai_404_generate_email_html('weekly', $stats);
    
    $headers = array(
        'Content-Type: text/html; charset=UTF-8',
        'From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>'
    );
    
    // Capture mail failures to provide a useful error back to AJAX
    add_action('wp_mail_failed', 'nandann_smart_ai_404_capture_mail_error');
    $result = wp_mail($recipient, $subject, $message, $headers);
    remove_action('wp_mail_failed', 'nandann_smart_ai_404_capture_mail_error');
    
    return $result;
}

/**
 * Store the last mail error briefly so AJAX can return it
 */
function nandann_smart_ai_404_capture_mail_error($wp_error) { // phpcs:ignore WordPress.NamingConventions.ValidFunctionName.FunctionNameInvalid
    if (is_wp_error($wp_error)) {
        set_transient('nandann_smart_ai_404_last_mail_error', $wp_error->get_error_message(), MINUTE_IN_SECONDS);
    }
}

