<?php
/**
 * Admin Settings Page
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Enqueue chart rendering script
 */
function nandann_smart_ai_404_enqueue_chart_script($chart_data) {
    // Make sure the main script is enqueued first
    if (!wp_script_is('nandann-smart-ai-404-admin-scripts', 'enqueued')) {
        wp_enqueue_script('nandann-smart-ai-404-admin-scripts');
    }
    
    $chart_js = "
    jQuery(document).ready(function($) {
        var ctx = document.getElementById('nandann-smart-ai-404-chart');
        if (!ctx) return;
        
        ctx = ctx.getContext('2d');
        var chartData = " . wp_json_encode($chart_data) . ";
        
        var labels = chartData.map(function(item) { return item.date; });
        var data = chartData.map(function(item) { return item.count; });
        
        // Simple line chart implementation
        var canvas = document.getElementById('nandann-smart-ai-404-chart');
        var context = canvas.getContext('2d');
        var width = canvas.width;
        var height = canvas.height;
        var padding = 40;
        
        context.clearRect(0, 0, width, height);
        
        // Draw axes
        context.strokeStyle = '#cbd5e0';
        context.lineWidth = 1;
        context.beginPath();
        context.moveTo(padding, padding);
        context.lineTo(padding, height - padding);
        context.lineTo(width - padding, height - padding);
        context.stroke();
        
        // Draw data
        if (data.length > 0) {
            var maxValue = Math.max(...data, 10);
            var xStep = (width - 2 * padding) / (data.length - 1);
            var yScale = (height - 2 * padding) / maxValue;
            
            context.strokeStyle = '#667eea';
            context.lineWidth = 3;
            context.beginPath();
            
            data.forEach(function(value, index) {
                var x = padding + (index * xStep);
                var y = height - padding - (value * yScale);
                
                if (index === 0) {
                    context.moveTo(x, y);
                } else {
                    context.lineTo(x, y);
                }
                
                // Draw point
                context.fillStyle = '#667eea';
                context.fillRect(x - 3, y - 3, 6, 6);
            });
            
            context.stroke();
            
            // Draw labels
            context.fillStyle = '#4a5568';
            context.font = '11px Arial';
            context.textAlign = 'center';
            
            labels.forEach(function(label, index) {
                var x = padding + (index * xStep);
                context.fillText(label, x, height - padding + 20);
            });
        }
    });
    ";
    
    wp_add_inline_script('nandann-smart-ai-404-admin-scripts', $chart_js);
}

// Handle form submissions
if (isset($_POST['nandann_smart_ai_404_save_settings']) && check_admin_referer('nandann_smart_ai_404_settings_nonce')) {
    $options = array(
        'enabled' => isset($_POST['enabled']) ? '1' : '0',
        'redirect_method' => isset($_POST['redirect_method']) ? sanitize_text_field(wp_unslash($_POST['redirect_method'])) : 'smart',
        'fallback_url' => isset($_POST['fallback_url']) ? esc_url_raw(wp_unslash($_POST['fallback_url'])) : home_url(),
        'redirect_status' => isset($_POST['redirect_status']) ? sanitize_text_field(wp_unslash($_POST['redirect_status'])) : '301',
        'smart_matching_enabled' => isset($_POST['smart_matching_enabled']) ? '1' : '0',
        'keyword_threshold' => isset($_POST['keyword_threshold']) ? intval(wp_unslash($_POST['keyword_threshold'])) : 70,
        'typo_detection_enabled' => isset($_POST['typo_detection_enabled']) ? '1' : '0',
        'typo_threshold' => isset($_POST['typo_threshold']) ? intval(wp_unslash($_POST['typo_threshold'])) : 80,
        'exclude_media' => isset($_POST['exclude_media']) ? '1' : '0',
        'excluded_paths' => isset($_POST['excluded_paths']) ? sanitize_textarea_field(wp_unslash($_POST['excluded_paths'])) : '',
        'log_enabled' => isset($_POST['log_enabled']) ? '1' : '0',
        'log_retention_days' => isset($_POST['log_retention_days']) ? intval(wp_unslash($_POST['log_retention_days'])) : 30,
        'email_enabled' => isset($_POST['email_enabled']) ? '1' : '0',
        'email_frequency' => isset($_POST['email_frequency']) ? sanitize_text_field(wp_unslash($_POST['email_frequency'])) : 'weekly',
        'email_recipient' => isset($_POST['email_recipient']) ? sanitize_email(wp_unslash($_POST['email_recipient'])) : get_option('admin_email')
    );
    
    update_option(NANDANN_SMART_AI_404_OPTIONS_KEY, $options);
    
    // Setup email schedules
    nandann_smart_ai_404_setup_email_schedule();
    
    echo '<div class="notice notice-success is-dismissible"><p><strong>' . esc_html__('Settings saved successfully!', 'nandann-ai-smart-404-redirect') . '</strong></p></div>';
}

$options = get_option(NANDANN_SMART_AI_404_OPTIONS_KEY);

// Ensure backward compatibility - add default for new options
$defaults = array(
    'excluded_paths' => ''
);
$options = wp_parse_args($options, $defaults);

$current_tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'settings';

?>

<div class="wrap aismart404-admin-wrap">
    <h1 class="aismart404-main-title">
        <span class="dashicons dashicons-admin-settings"></span>
        <?php esc_html_e('AI Smart 404 Redirect', 'nandann-ai-smart-404-redirect'); ?>
    </h1>
    
    <!-- Navigation Tabs -->
    <nav class="nav-tab-wrapper aismart404-nav-tabs">
        <a href="?page=nandann-ai-smart-404-redirect&tab=settings" class="nav-tab <?php echo esc_attr($current_tab === 'settings' ? 'nav-tab-active' : ''); ?>">
            <span class="dashicons dashicons-admin-generic"></span> <?php esc_html_e('Settings', 'nandann-ai-smart-404-redirect'); ?>
        </a>
        <a href="?page=nandann-ai-smart-404-redirect&tab=dashboard" class="nav-tab <?php echo esc_attr($current_tab === 'dashboard' ? 'nav-tab-active' : ''); ?>">
            <span class="dashicons dashicons-chart-area"></span> <?php esc_html_e('Dashboard', 'nandann-ai-smart-404-redirect'); ?>
        </a>
        <a href="?page=nandann-ai-smart-404-redirect&tab=logs" class="nav-tab <?php echo esc_attr($current_tab === 'logs' ? 'nav-tab-active' : ''); ?>">
            <span class="dashicons dashicons-list-view"></span> <?php esc_html_e('404 Logs', 'nandann-ai-smart-404-redirect'); ?>
        </a>
        <a href="?page=nandann-ai-smart-404-redirect&tab=help" class="nav-tab <?php echo esc_attr($current_tab === 'help' ? 'nav-tab-active' : ''); ?>">
            <span class="dashicons dashicons-editor-help"></span> <?php esc_html_e('Questions / Suggestions', 'nandann-ai-smart-404-redirect'); ?>
        </a>
    </nav>
    
    <div class="aismart404-tab-content">
        <?php
        switch ($current_tab) {
            case 'dashboard':
                nandann_smart_ai_404_render_dashboard_tab($options);
                break;
            case 'logs':
                nandann_smart_ai_404_render_logs_tab();
                break;
            case 'help':
                nandann_smart_ai_404_render_help_tab();
                break;
            case 'settings':
            default:
                nandann_smart_ai_404_render_settings_tab($options);
                break;
        }
        ?>
    </div>
</div>

<?php

/**
 * Render Settings Tab
 */
function nandann_smart_ai_404_render_settings_tab($options) {
    ?>
    <form method="post" action="" class="aismart404-settings-form">
        <?php wp_nonce_field('nandann_smart_ai_404_settings_nonce'); ?>
        
        <div class="aismart404-settings-grid">
            <!-- Basic Settings -->
            <div class="aismart404-card">
                <div class="aismart404-card-header">
                    <h2><span class="dashicons dashicons-admin-settings"></span> <?php esc_html_e('Basic Settings', 'nandann-ai-smart-404-redirect'); ?></h2>
                </div>
                <div class="aismart404-card-body">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('Enable Plugin', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <label class="aismart404-switch">
                                    <input type="checkbox" name="enabled" value="1" <?php checked($options['enabled'], '1'); ?>>
                                    <span class="aismart404-slider"></span>
                                </label>
                                <p class="description"><?php esc_html_e('Enable or disable 404 redirections', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Redirect Method', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <select name="redirect_method" class="regular-text">
                                    <option value="smart" <?php selected($options['redirect_method'], 'smart'); ?>><?php esc_html_e('AI Smart Matching (Recommended)', 'nandann-ai-smart-404-redirect'); ?></option>
                                    <option value="homepage" <?php selected($options['redirect_method'], 'homepage'); ?>><?php esc_html_e('Always redirect to Homepage', 'nandann-ai-smart-404-redirect'); ?></option>
                                    <option value="custom" <?php selected($options['redirect_method'], 'custom'); ?>><?php esc_html_e('Custom URL', 'nandann-ai-smart-404-redirect'); ?></option>
                                </select>
                                <p class="description"><?php esc_html_e('Choose how to handle 404 errors', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Fallback URL', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <input type="url" name="fallback_url" value="<?php echo esc_attr($options['fallback_url']); ?>" class="regular-text">
                                <p class="description"><?php esc_html_e('URL to redirect when no smart match is found', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Redirect Type', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <select name="redirect_status" class="regular-text">
                                    <option value="301" <?php selected($options['redirect_status'], '301'); ?>>301 - <?php esc_html_e('Permanent Redirect (SEO Friendly)', 'nandann-ai-smart-404-redirect'); ?></option>
                                    <option value="302" <?php selected($options['redirect_status'], '302'); ?>>302 - <?php esc_html_e('Temporary Redirect', 'nandann-ai-smart-404-redirect'); ?></option>
                                    <option value="307" <?php selected($options['redirect_status'], '307'); ?>>307 - <?php esc_html_e('Temporary Redirect (HTTP/1.1)', 'nandann-ai-smart-404-redirect'); ?></option>
                                </select>
                                <p class="description"><?php esc_html_e('HTTP status code for redirects', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- AI Smart Matching Settings -->
            <div class="aismart404-card aismart404-featured-card">
                <div class="aismart404-card-header">
                    <h2><span class="dashicons dashicons-admin-site-alt3"></span> <?php esc_html_e('AI Smart Matching', 'nandann-ai-smart-404-redirect'); ?> <span class="aismart404-badge-pro">PRO</span></h2>
                </div>
                <div class="aismart404-card-body">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('Enable Smart Matching', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <label class="aismart404-switch">
                                    <input type="checkbox" name="smart_matching_enabled" value="1" <?php checked($options['smart_matching_enabled'], '1'); ?>>
                                    <span class="aismart404-slider"></span>
                                </label>
                                <p class="description"><?php esc_html_e('Use AI to find the most relevant page based on keywords', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Keyword Match Threshold', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <input type="range" name="keyword_threshold" min="0" max="100" value="<?php echo esc_attr($options['keyword_threshold']); ?>" class="aismart404-range-slider" oninput="this.nextElementSibling.textContent = this.value + '%'">
                                <output><?php echo esc_html($options['keyword_threshold']); ?>%</output>
                                <p class="description"><?php esc_html_e('Minimum similarity percentage for keyword matching (higher = more strict)', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Enable Typo Detection', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <label class="aismart404-switch">
                                    <input type="checkbox" name="typo_detection_enabled" value="1" <?php checked($options['typo_detection_enabled'], '1'); ?>>
                                    <span class="aismart404-slider"></span>
                                </label>
                                <p class="description"><?php esc_html_e('Automatically detect and fix typos in URLs', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Typo Tolerance', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <input type="number" name="typo_threshold" min="1" max="5" value="<?php echo esc_attr($options['typo_threshold']); ?>" class="small-text">
                                <p class="description"><?php esc_html_e('Maximum character differences allowed (1-5, lower = more strict)', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- Logging Settings -->
            <div class="aismart404-card">
                <div class="aismart404-card-header">
                    <h2><span class="dashicons dashicons-database"></span> <?php esc_html_e('Logging Settings', 'nandann-ai-smart-404-redirect'); ?></h2>
                </div>
                <div class="aismart404-card-body">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('Enable Logging', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <label class="aismart404-switch">
                                    <input type="checkbox" name="log_enabled" value="1" <?php checked($options['log_enabled'], '1'); ?>>
                                    <span class="aismart404-slider"></span>
                                </label>
                                <p class="description"><?php esc_html_e('Keep logs of all 404 errors', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Exclude Media Files', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <label class="aismart404-switch">
                                    <input type="checkbox" name="exclude_media" value="1" <?php checked($options['exclude_media'], '1'); ?>>
                                    <span class="aismart404-slider"></span>
                                </label>
                                <p class="description"><?php esc_html_e('Don\'t log requests for images, CSS, JS, etc.', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Excluded Paths', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <input type="text" name="excluded_paths" value="<?php echo esc_attr($options['excluded_paths']); ?>" class="regular-text" placeholder="/go/,/out/,/link/">
                                <p class="description">
                                    <?php esc_html_e('Comma-separated list of URL path prefixes to exclude from redirection (e.g., /go/,/out/,/link/). Useful for affiliate links, pretty links, or custom redirects managed by other plugins.', 'nandann-ai-smart-404-redirect'); ?>
                                </p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Log Retention', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <input type="number" name="log_retention_days" min="1" max="365" value="<?php echo esc_attr($options['log_retention_days']); ?>" class="small-text">
                                <span><?php esc_html_e('days', 'nandann-ai-smart-404-redirect'); ?></span>
                                <p class="description"><?php esc_html_e('Automatically delete logs older than this many days', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <!-- Email Notifications -->
            <div class="aismart404-card">
                <div class="aismart404-card-header">
                    <h2><span class="dashicons dashicons-email"></span> <?php esc_html_e('Email Notifications', 'nandann-ai-smart-404-redirect'); ?></h2>
                </div>
                <div class="aismart404-card-body">
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('Enable Email Reports', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <label class="aismart404-switch">
                                    <input type="checkbox" name="email_enabled" value="1" <?php checked($options['email_enabled'], '1'); ?>>
                                    <span class="aismart404-slider"></span>
                                </label>
                                <p class="description"><?php esc_html_e('Receive periodic email reports about 404 errors', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Email Frequency', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <select name="email_frequency" class="regular-text">
                                    <option value="daily" <?php selected($options['email_frequency'], 'daily'); ?>><?php esc_html_e('Daily', 'nandann-ai-smart-404-redirect'); ?></option>
                                    <option value="weekly" <?php selected($options['email_frequency'], 'weekly'); ?>><?php esc_html_e('Weekly', 'nandann-ai-smart-404-redirect'); ?></option>
                                    <option value="monthly" <?php selected($options['email_frequency'], 'monthly'); ?>><?php esc_html_e('Monthly', 'nandann-ai-smart-404-redirect'); ?></option>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Email Recipient', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <input type="email" name="email_recipient" value="<?php echo esc_attr($options['email_recipient']); ?>" class="regular-text">
                                <p class="description"><?php esc_html_e('Email address to receive reports', 'nandann-ai-smart-404-redirect'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Test Email', 'nandann-ai-smart-404-redirect'); ?></th>
                            <td>
                                <button type="button" class="button button-secondary aismart404-test-email" data-email="<?php echo esc_attr($options['email_recipient']); ?>" style="display: inline-flex; align-items: center; gap: 8px;">
                                    <span class="dashicons dashicons-email-alt"></span>
                                    <span><?php esc_html_e('Send Test Email', 'nandann-ai-smart-404-redirect'); ?></span>
                                </button>
                                <span class="aismart404-test-email-result"></span>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>
        </div>
        
        <div class="aismart404-submit-wrapper">
            <button type="submit" name="nandann_smart_ai_404_save_settings" class="button button-primary button-hero" style="display: inline-flex; align-items: center; gap: 8px;">
                <span class="dashicons dashicons-yes"></span>
                <span><?php esc_html_e('Save All Settings', 'nandann-ai-smart-404-redirect'); ?></span>
            </button>
        </div>
    </form>
    <?php
}

/**
 * Render Dashboard Tab
 */
function nandann_smart_ai_404_render_dashboard_tab($options) {
    $stats_week = nandann_smart_ai_404_get_statistics('week');
    $stats_month = nandann_smart_ai_404_get_statistics('month');
    $stats_all = nandann_smart_ai_404_get_statistics('all');
    $chart_data = nandann_smart_ai_404_get_chart_data(7);
    
    // Add chart rendering script using wp_add_inline_script
    nandann_smart_ai_404_enqueue_chart_script($chart_data);
    ?>
    
    <div class="aismart404-dashboard">
        <!-- Quick Stats -->
        <div class="aismart404-stats-grid">
            <div class="aismart404-stat-card aismart404-stat-danger">
                <div class="aismart404-stat-icon"><span class="dashicons dashicons-warning"></span></div>
                <div class="aismart404-stat-content">
                    <div class="aismart404-stat-value"><?php echo esc_html(nandann_smart_ai_404_format_number($stats_week['total_errors'])); ?></div>
                    <div class="aismart404-stat-label"><?php esc_html_e('Total 404 Errors (7 days)', 'nandann-ai-smart-404-redirect'); ?></div>
                </div>
            </div>
            
            <div class="aismart404-stat-card aismart404-stat-success">
                <div class="aismart404-stat-icon"><span class="dashicons dashicons-yes-alt"></span></div>
                <div class="aismart404-stat-content">
                    <div class="aismart404-stat-value"><?php echo esc_html(nandann_smart_ai_404_format_number($stats_week['smart_matches'])); ?></div>
                    <div class="aismart404-stat-label"><?php esc_html_e('Smart Matches', 'nandann-ai-smart-404-redirect'); ?></div>
                </div>
            </div>
            
            <div class="aismart404-stat-card aismart404-stat-info">
                <div class="aismart404-stat-icon"><span class="dashicons dashicons-admin-links"></span></div>
                <div class="aismart404-stat-content">
                    <div class="aismart404-stat-value"><?php echo esc_html(nandann_smart_ai_404_format_number($stats_week['unique_urls'])); ?></div>
                    <div class="aismart404-stat-label"><?php esc_html_e('Unique URLs', 'nandann-ai-smart-404-redirect'); ?></div>
                </div>
            </div>
            
            <div class="aismart404-stat-card aismart404-stat-warning">
                <div class="aismart404-stat-icon"><span class="dashicons dashicons-backup"></span></div>
                <div class="aismart404-stat-content">
                    <div class="aismart404-stat-value"><?php echo esc_html(nandann_smart_ai_404_format_number($stats_week['fallback_redirects'])); ?></div>
                    <div class="aismart404-stat-label"><?php esc_html_e('Fallback Redirects', 'nandann-ai-smart-404-redirect'); ?></div>
                </div>
            </div>
        </div>
        
        <!-- Chart -->
        <div class="aismart404-card">
            <div class="aismart404-card-header">
                <h2><span class="dashicons dashicons-chart-line"></span> <?php esc_html_e('7-Day Trend', 'nandann-ai-smart-404-redirect'); ?></h2>
            </div>
            <div class="aismart404-card-body">
                <canvas id="nandann-smart-ai-404-chart" width="800" height="300"></canvas>
            </div>
        </div>
        
        <!-- Performance Summary -->
        <div class="aismart404-performance-grid">
            <div class="aismart404-card">
                <div class="aismart404-card-header">
                    <h3><?php esc_html_e('This Month', 'nandann-ai-smart-404-redirect'); ?></h3>
                </div>
                <div class="aismart404-card-body">
                    <div class="aismart404-performance-item">
                        <span class="label"><?php esc_html_e('Total Errors:', 'nandann-ai-smart-404-redirect'); ?></span>
                        <span class="value"><?php echo esc_html(number_format($stats_month['total_errors'])); ?></span>
                    </div>
                    <div class="aismart404-performance-item">
                        <span class="label"><?php esc_html_e('Smart Matches:', 'nandann-ai-smart-404-redirect'); ?></span>
                        <span class="value"><?php echo esc_html(number_format($stats_month['smart_matches'])); ?></span>
                    </div>
                    <div class="aismart404-performance-item">
                        <span class="label"><?php esc_html_e('Success Rate:', 'nandann-ai-smart-404-redirect'); ?></span>
                        <span class="value">
                            <?php
                            $success_rate = $stats_month['total_errors'] > 0 
                                ? round(($stats_month['smart_matches'] / $stats_month['total_errors']) * 100) 
                                : 0;
                            echo esc_html($success_rate) . '%';
                            ?>
                        </span>
                    </div>
                </div>
            </div>
            
            <div class="aismart404-card">
                <div class="aismart404-card-header">
                    <h3><?php esc_html_e('All Time', 'nandann-ai-smart-404-redirect'); ?></h3>
                </div>
                <div class="aismart404-card-body">
                    <div class="aismart404-performance-item">
                        <span class="label"><?php esc_html_e('Total Errors:', 'nandann-ai-smart-404-redirect'); ?></span>
                        <span class="value"><?php echo esc_html(number_format($stats_all['total_errors'])); ?></span>
                    </div>
                    <div class="aismart404-performance-item">
                        <span class="label"><?php esc_html_e('Smart Matches:', 'nandann-ai-smart-404-redirect'); ?></span>
                        <span class="value"><?php echo esc_html(number_format($stats_all['smart_matches'])); ?></span>
                    </div>
                    <div class="aismart404-performance-item">
                        <span class="label"><?php esc_html_e('Unique URLs:', 'nandann-ai-smart-404-redirect'); ?></span>
                        <span class="value"><?php echo esc_html(number_format($stats_all['unique_urls'])); ?></span>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <?php
}

/**
 * Render Logs Tab
 */
function nandann_smart_ai_404_render_logs_tab() {
    $per_page = 20;
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Pagination parameter doesn't require nonce
    $current_page = isset($_GET['paged']) ? max(1, intval(wp_unslash($_GET['paged']))) : 1;
    $offset = ($current_page - 1) * $per_page;
    
    $logs = nandann_smart_ai_404_get_top_404_urls($per_page, $offset);
    $total_logs = nandann_smart_ai_404_get_total_log_count();
    $total_pages = ceil($total_logs / $per_page);
    ?>
    
    <div class="aismart404-logs-container">
        <div class="aismart404-logs-header">
            <h2><?php esc_html_e('404 Error Logs', 'nandann-ai-smart-404-redirect'); ?></h2>
            <div class="aismart404-logs-actions">
                <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=nandann-ai-smart-404-redirect&tab=logs&action=export'), 'nandann_smart_ai_404_export_nonce', 'nonce')); ?>" class="button button-secondary" style="display: inline-flex; align-items: center; gap: 8px;">
                    <span class="dashicons dashicons-download"></span>
                    <span><?php esc_html_e('Export CSV', 'nandann-ai-smart-404-redirect'); ?></span>
                </a>
                <button type="button" class="button button-secondary aismart404-clear-logs" style="display: inline-flex; align-items: center; gap: 8px;">
                    <span class="dashicons dashicons-trash"></span>
                    <span><?php esc_html_e('Clear All Logs', 'nandann-ai-smart-404-redirect'); ?></span>
                </button>
            </div>
        </div>
        
        <?php if (empty($logs)): ?>
            <div class="aismart404-empty-state">
                <span class="dashicons dashicons-thumbs-up"></span>
                <h3><?php esc_html_e('No 404 errors yet!', 'nandann-ai-smart-404-redirect'); ?></h3>
                <p><?php esc_html_e('This is great news - your site has no broken links being tracked.', 'nandann-ai-smart-404-redirect'); ?></p>
            </div>
        <?php else: ?>
            <div class="aismart404-table-wrapper">
                <table class="widefat striped aismart404-logs-table">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Requested URL', 'nandann-ai-smart-404-redirect'); ?></th>
                            <th><?php esc_html_e('Redirected To', 'nandann-ai-smart-404-redirect'); ?></th>
                            <th><?php esc_html_e('Match Type', 'nandann-ai-smart-404-redirect'); ?></th>
                            <th><?php esc_html_e('Score', 'nandann-ai-smart-404-redirect'); ?></th>
                            <th><?php esc_html_e('Hits', 'nandann-ai-smart-404-redirect'); ?></th>
                            <th><?php esc_html_e('Last Occurrence', 'nandann-ai-smart-404-redirect'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($logs as $log): ?>
                        <tr>
                            <td class="aismart404-url-cell">
                                <a href="<?php echo esc_url($log->requested_url); ?>" target="_blank" rel="noopener">
                                    <?php echo esc_html($log->requested_url); ?>
                                </a>
                                <?php if (!empty($log->referrer)): ?>
                                    <div class="aismart404-referrer">
                                        <small><?php esc_html_e('Referrer:', 'nandann-ai-smart-404-redirect'); ?> <?php echo esc_html($log->referrer); ?></small>
                                    </div>
                                <?php endif; ?>
                            </td>
                            <td class="aismart404-url-cell">
                                <a href="<?php echo esc_url($log->redirected_to); ?>" target="_blank" rel="noopener">
                                    <?php echo esc_html($log->redirected_to); ?>
                                </a>
                            </td>
                            <td>
                                <span class="aismart404-badge <?php echo esc_attr($log->redirect_type !== 'fallback' ? 'nandann-smart-ai-404-badge-success' : 'nandann-smart-ai-404-badge-default'); ?>">
                                    <?php echo esc_html(nandann_smart_ai_404_format_redirect_type($log->redirect_type)); ?>
                                </span>
                            </td>
                            <td><?php echo $log->match_score ? wp_kses_post(nandann_smart_ai_404_get_score_badge($log->match_score)) : '-'; ?></td>
                            <td><strong><?php echo esc_html(number_format($log->hit_count)); ?></strong></td>
                            <td><?php echo esc_html(human_time_diff(strtotime($log->last_occurrence), current_time('timestamp')) . ' ' . __('ago', 'nandann-ai-smart-404-redirect')); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <?php if ($total_pages > 1): ?>
            <div class="aismart404-pagination">
                <?php
                echo wp_kses_post(paginate_links(array(
                    'base' => add_query_arg('paged', '%#%'),
                    'format' => '',
                    'current' => $current_page,
                    'total' => $total_pages,
                    'prev_text' => '&laquo; ' . __('Previous', 'nandann-ai-smart-404-redirect'),
                    'next_text' => __('Next', 'nandann-ai-smart-404-redirect') . ' &raquo;'
                )));
                ?>
            </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
    <?php
}

/**
 * Render Questions / Suggestions Tab
 */
function nandann_smart_ai_404_render_help_tab() {
    require_once NANDANN_SMART_AI_404_PLUGIN_DIR . 'includes/admin-page-help.php';
    nandann_smart_ai_404_render_help_tab_content();
}
