<?php
/*
 * Plugin Name:       PlugStudio Installment Calculator for WooCommerce
 * Plugin URI:        https://wordpress.org/plugins/mz-calculate-fees/
 * Description:       Add a visual installment dropdown to WooCommerce product pages and show the monthly payment amount based on months and interest settings.
 * Version:           1.0.3
 * Requires at least: 5.2
 * Requires PHP:      7.2
 * Author:            Miguel Angel Zhañay Guamán
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       mz-calculate-fees
 */

if (!defined('ABSPATH')) {
	exit;
}

// Hooks de activación y desinstalación.
register_activation_hook(__FILE__, 'plst_cf_woo_create_table');
register_uninstall_hook(__FILE__, 'plst_cf_woo_delete_table');

/**
 * Crea la tabla en la base de datos al activar.
 */
function plst_cf_woo_create_table()
{
	global $wpdb;

	$table_name = $wpdb->prefix . 'plst_cf_woo_cuotas';
	$charset_collate = $wpdb->get_charset_collate();

	// Patrón recomendado en la documentación de dbDelta.
	$sql = "CREATE TABLE $table_name (
		id mediumint(9) NOT NULL AUTO_INCREMENT,
		meses int NOT NULL,
		interes float(4,2) NOT NULL,
		PRIMARY KEY  (id)
	) $charset_collate;";

	require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	dbDelta($sql);
}

/**
 * Elimina la tabla al desinstalar.
 */
function plst_cf_woo_delete_table()
{
	global $wpdb;

	$table_name = $wpdb->prefix . 'plst_cf_woo_cuotas';

	// Evitamos interpolación dentro del string.
	$wpdb->query('DROP TABLE IF EXISTS ' . $table_name); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
}

// Carga de estilos y scripts.
add_action('wp_enqueue_scripts', 'plst_cf_enqueue_scripts');
add_action('admin_enqueue_scripts', 'plst_cf_admin_styles');

function plst_cf_admin_styles()
{
	// Lectura de $_GET['page'] solo para condicionar carga de assets (no es envío de formulario).
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended
	$page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';

	if ('mz-calculate-fees' === $page) {
		plst_cf_load_assets();
	}
}

function plst_cf_enqueue_scripts()
{
	if (is_product()) {
		plst_cf_load_assets();

		wp_enqueue_script(
			'plst-calculate-js',
			plugins_url('/assets/js/calculate.js', __FILE__),
			array('jquery'),
			'1.0',
			true
		);

		// Localize script para pasar variables de PHP a JS de forma segura.
		wp_localize_script(
			'plst-calculate-js',
			'plst_cf_vars',
			array(
				'ajaxurl' => admin_url('admin-ajax.php'),
				'nonce' => wp_create_nonce('plst_cf_calc_nonce'), // Nonce de seguridad.
			)
		);
	}
}

function plst_cf_load_assets()
{
	wp_enqueue_style('iconos', plugins_url('/assets/css/icon.css', __FILE__), array(), '1.0.0');
	wp_enqueue_style('plst-estilos', plugins_url('/assets/css/estilos.css', __FILE__), array(), '1.0.0');
}

// AJAX Hooks.
add_action('wp_ajax_nopriv_plst_cf_woo_buttonCalculator', 'plst_cf_woo_button_calculator');
add_action('wp_ajax_plst_cf_woo_buttonCalculator', 'plst_cf_woo_button_calculator');

/**
 * Función AJAX para calcular la cuota.
 */
function plst_cf_woo_button_calculator()
{
	global $wpdb;

	// 1. Verificación de Nonce (Seguridad).
	// 1. Verificación de Nonce (Seguridad).
	$nonce = isset($_GET['nonce']) ? sanitize_text_field(wp_unslash($_GET['nonce'])) : '';

	if (!$nonce || !wp_verify_nonce($nonce, 'plst_cf_calc_nonce')) {
		wp_send_json_error('Error de seguridad');
	}

	// 2. Sanitización de datos.
	// 2. Sanitización de datos.
	$id_raw = isset($_GET['id']) ? sanitize_text_field(wp_unslash($_GET['id'])) : '';
	$precio_raw = isset($_GET['precio']) ? sanitize_text_field(wp_unslash($_GET['precio'])) : '';

	$id = absint($id_raw);
	$precio = floatval($precio_raw);

	if ($id && $precio) {
		$table_name = $wpdb->prefix . 'plst_cf_woo_cuotas';

		// 3. Consulta preparada (evita SQL Injection).
		// 3. Consulta preparada (evita SQL Injection).
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$resultado = $wpdb->get_row(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
				'SELECT * FROM ' . $table_name . ' WHERE id = %d',
				$id
			)
		);

		if ($resultado) {
			$mes = absint($resultado->meses);
			$interes = floatval($resultado->interes);

			// Lógica de cálculo (evitar división por cero).
			if ($mes > 0) {
				$intereses = ($interes / 100) + 1;
				$cal = $precio * $intereses;
				$res = $cal / $mes;

				// wc_price devuelve HTML, lo envolvemos con wp_kses_post.
				echo wp_kses_post(wc_price($res));
			} else {
				echo esc_html__('Error en meses', 'mz-calculate-fees');
			}
		} else {
			echo esc_html__('Cuota no encontrada', 'mz-calculate-fees');
		}
	}

	wp_die();
}

// Menú de Administración.
add_action('admin_menu', 'plst_cf_add_admin_menu');

function plst_cf_add_admin_menu()
{
	add_menu_page(
		esc_html__('Installment Calculator', 'mz-calculate-fees'),
		esc_html__('Installment Calculator', 'mz-calculate-fees'),
		'manage_options',
		'mz-calculate-fees',
		'plst_cf_render_admin_page',
		'dashicons-calculator',
		56
	);
}

// Callback para renderizar la página de admin.
function plst_cf_render_admin_page()
{
	require_once plugin_dir_path(__FILE__) . 'months.php';
}

// Enlace de ajustes en la lista de plugins.
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'plst_cf_add_plugin_page_settings_link');

function plst_cf_add_plugin_page_settings_link($links)
{
	$links[] = '<a href="' . esc_url(admin_url('admin.php?page=mz-calculate-fees')) . '">' . esc_html__('Settings', 'mz-calculate-fees') . '</a>';
	return $links;
}

// Pintar el select en la página del producto.
add_action('woocommerce_before_add_to_cart_form', 'plst_cf_woo_price_by_fees', 15);

function plst_cf_woo_price_by_fees()
{
	global $post, $wpdb;

	$product = wc_get_product($post->ID);
	if (!$product) {
		return;
	}

	$precio = $product->get_price();

	$table_name = $wpdb->prefix . 'plst_cf_woo_cuotas';

	// Consulta segura (sin interpolación dentro del string).
	// Consulta segura (sin interpolación dentro del string).
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery
	$query = $wpdb->get_results('SELECT * FROM ' . $table_name . ' ORDER BY meses ASC'); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared

	if ($query) {
		?>
		<div class="plst-calculator-container" style="margin-bottom: 20px;">
			<h4><?php esc_html_e('Calculate your installments', 'mz-calculate-fees'); ?></h4>

			<p class="form-row form-row-wide">
				<label for="plst_combo">
					<?php esc_html_e('Select number of months', 'mz-calculate-fees'); ?>
				</label>
				<select name="combo" id="plst_combo">
					<option value="0" disabled selected>
						<?php esc_html_e('Select number of months', 'mz-calculate-fees'); ?>
					</option>
					<?php foreach ($query as $imprimir): ?>
						<option data-price="<?php echo esc_attr($precio); ?>" value="<?php echo esc_attr($imprimir->id); ?>">
							<?php echo esc_html($imprimir->meses); ?> 			<?php esc_html_e('months', 'mz-calculate-fees'); ?>
						</option>
					<?php endforeach; ?>
				</select>
			</p>

			<p class="plst-installment-result">
				<?php esc_html_e('Installment cost:', 'mz-calculate-fees'); ?>
				<strong id="plst_txtMessage"></strong>
			</p>
		</div>
		<?php
	}
}