<?php
/**
 * Plugin Name: MyOG Social
 * Plugin URI:  https://myog.social/docs/wordpress
 * Description: Automatic Open Graph image generation via MyOG.social — adds beautiful OG images to your WordPress posts and pages.
 * Version:     1.0.0
 * Author:      MyOG.social
 * Author URI:  https://myog.social
 * License:     GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: myog-social
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'MYOG_SOCIAL_VERSION', '1.0.0' );
define( 'MYOG_SOCIAL_API_URL', 'https://myog.social/og' );

function myog_social_get_template() {
    return get_option( 'myog_social_template', 'screenshot-right' );
}

function myog_social_output_meta_tag() {
    if ( is_singular() ) {
        $page_url = get_permalink();
    } else {
        $page_url = home_url( esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ?? '/' ) ) );
    }

    $page_url  = rawurlencode( $page_url );
    $image_url = MYOG_SOCIAL_API_URL . '?url=' . $page_url;

    $template = myog_social_get_template();
    if ( $template !== '' ) {
        $image_url .= '&template=' . rawurlencode( $template );
    }

    echo '<meta property="og:image" content="' . esc_attr( $image_url ) . '">' . "\n";
}
add_action( 'wp_head', 'myog_social_output_meta_tag', 99 );

function myog_social_register_settings() {
    register_setting( 'myog_social_settings', 'myog_social_template', array(
        'type'              => 'string',
        'sanitize_callback' => 'myog_social_sanitize_template',
        'default'           => 'screenshot-right',
    ) );
}
add_action( 'admin_init', 'myog_social_register_settings' );

function myog_social_sanitize_template( $value ) {
    $allowed = array(
        '',
        'brand',
        'screenshot-right',
        'screenshot-bottom',
        'screenshot-bottom-framed',
        'screenshot',
    );
    return in_array( $value, $allowed, true ) ? $value : '';
}

function myog_social_add_settings_page() {
    add_options_page(
        esc_html__( 'MyOG Social', 'myog-social' ),
        esc_html__( 'MyOG Social', 'myog-social' ),
        'manage_options',
        'myog-social',
        'myog_social_render_settings_page'
    );
}
add_action( 'admin_menu', 'myog_social_add_settings_page' );

function myog_social_render_settings_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $current_template = myog_social_get_template();

    $templates = array(
        ''                         => __( 'Default (use dashboard setting)', 'myog-social' ),
        'brand'                    => __( 'Brand — Logo + colors', 'myog-social' ),
        'screenshot-right'         => __( 'Screenshot Right — Screenshot on right', 'myog-social' ),
        'screenshot-bottom'        => __( 'Screenshot Bottom — Screenshot on bottom', 'myog-social' ),
        'screenshot-bottom-framed' => __( 'Screenshot Bottom, Framed — Framed screenshot', 'myog-social' ),
        'screenshot'               => __( 'Screenshot Only — Full page capture', 'myog-social' ),
    );
    ?>
    <div class="wrap">
        <h1><?php esc_html_e( 'MyOG Social Settings', 'myog-social' ); ?></h1>
        <p>
            <?php
            echo wp_kses(
                sprintf(
                    /* translators: %s: link to MyOG.social */
                    __( 'This plugin adds automatic Open Graph images to your posts and pages using %s.', 'myog-social' ),
                    '<a href="' . esc_url( 'https://myog.social' ) . '" target="_blank" rel="noopener">MyOG.social</a>'
                ),
                array( 'a' => array( 'href' => array(), 'target' => array(), 'rel' => array() ) )
            );
            ?>
        </p>
        <p>
            <strong><?php esc_html_e( 'Setup:', 'myog-social' ); ?></strong>
            <?php
            echo wp_kses(
                sprintf(
                    /* translators: %s: link to MyOG.social dashboard */
                    __( 'Make sure your domain is added to your %s.', 'myog-social' ),
                    '<a href="' . esc_url( 'https://myog.social' ) . '" target="_blank" rel="noopener">' . esc_html__( 'MyOG.social dashboard', 'myog-social' ) . '</a>'
                ),
                array( 'a' => array( 'href' => array(), 'target' => array(), 'rel' => array() ) )
            );
            ?>
        </p>
        <form method="post" action="options.php">
            <?php settings_fields( 'myog_social_settings' ); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label for="myog_social_template"><?php esc_html_e( 'OG Image Template', 'myog-social' ); ?></label>
                    </th>
                    <td>
                        <select name="myog_social_template" id="myog_social_template">
                            <?php foreach ( $templates as $value => $label ) : ?>
                                <option value="<?php echo esc_attr( $value ); ?>" <?php selected( $current_template, $value ); ?>>
                                    <?php echo esc_html( $label ); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description">
                            <?php esc_html_e( 'Choose which template to use for generated OG images. "Default" uses whatever template is configured in your MyOG.social dashboard.', 'myog-social' ); ?>
                        </p>
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
        <hr>
        <h2><?php esc_html_e( 'SEO Plugin Compatibility', 'myog-social' ); ?></h2>
        <p>
            <?php esc_html_e( 'If you use Yoast SEO, RankMath, or All in One SEO, they may set their own OG images. To use MyOG.social instead, disable OG image output in your SEO plugin\'s social settings.', 'myog-social' ); ?>
        </p>
        <h2><?php esc_html_e( 'Need Help?', 'myog-social' ); ?></h2>
        <p>
            <?php
            echo wp_kses(
                sprintf(
                    /* translators: %1$s: link to WordPress guide, %2$s: link to OG Preview tool */
                    __( 'Visit the %1$s or use the %2$s to test your pages.', 'myog-social' ),
                    '<a href="' . esc_url( 'https://myog.social/docs/wordpress' ) . '" target="_blank" rel="noopener">' . esc_html__( 'WordPress integration guide', 'myog-social' ) . '</a>',
                    '<a href="' . esc_url( 'https://myog.social/tools/og-preview' ) . '" target="_blank" rel="noopener">' . esc_html__( 'OG Preview tool', 'myog-social' ) . '</a>'
                ),
                array( 'a' => array( 'href' => array(), 'target' => array(), 'rel' => array() ) )
            );
            ?>
        </p>
    </div>
    <?php
}

function myog_social_add_settings_link( $links ) {
    $settings_link = '<a href="' . esc_url( admin_url( 'options-general.php?page=myog-social' ) ) . '">' . esc_html__( 'Settings', 'myog-social' ) . '</a>';
    array_unshift( $links, $settings_link );
    return $links;
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'myog_social_add_settings_link' );
