/**
 * Admin JavaScript for WPO
 */

(function($) {
    'use strict';

    /**
     * Global Audit functionality
     */
    class GlobalAudit {
        constructor() {
            this.init();
        }

        init() {
            const startButton = document.getElementById('mustwp-start-audit');
            if (startButton) {
                startButton.addEventListener('click', () => this.startAudit());
            }
        }

        async startAudit() {
            const startButton = document.getElementById('mustwp-start-audit');
            const progressText = document.getElementById('mustwp-audit-progress');
            const resultsDiv = document.getElementById('mustwp-audit-results');
            const summaryDiv = document.getElementById('mustwp-summary-stats');
            const warningNotice = document.getElementById('mustwp-audit-warning');

            // Show warning notice
            if (warningNotice) {
                warningNotice.style.display = 'block';
            }

            startButton.disabled = true;
            progressText.textContent = 'Starting audit...';
            resultsDiv.style.display = 'none';

            try {
                // Get all post IDs for selected content types
                const postIds = await this.getAllPostIds();
                
                if (postIds.length === 0) {
                    progressText.textContent = 'No posts found to audit.';
                    startButton.disabled = false;
                    if (warningNotice) warningNotice.style.display = 'none';
                    return;
                }

                progressText.textContent = `Found ${postIds.length} posts to audit. Starting...`;

                // Fetch all post titles upfront
                const postTitleMap = await this.getAllPostTitles(postIds);

                // Process posts in batches
                const results = await this.processBatches(postIds, progressText, postTitleMap);

                // Hide warning notice when done
                if (warningNotice) {
                    warningNotice.style.display = 'none';
                }

                // Show results
                this.displayResults(results, summaryDiv);
                progressText.textContent = '';
                resultsDiv.style.display = 'block';

                // Show admin notice based on results
                const { totalProcessed, totalErrors } = results;
                const totalPosts = postIds.length;
                
                if (totalProcessed === totalPosts && totalErrors === 0) {
                    // Success: All posts audited with no errors
                    this.showAdminNotice('success', `Audit completed successfully! ${totalProcessed} ${totalProcessed === 1 ? 'post' : 'posts'} audited with no errors.`);
                } else if (totalProcessed > 0 && totalErrors > 0) {
                    // Warning: Some posts audited, some errors
                    this.showAdminNotice('warning', `Audit completed with warnings. ${totalProcessed} ${totalProcessed === 1 ? 'post' : 'posts'} audited, ${totalErrors} ${totalErrors === 1 ? 'error' : 'errors'} encountered.`);
                } else if (totalProcessed === 0) {
                    // Error: No posts successfully audited
                    this.showAdminNotice('error', 'Audit failed. No posts were successfully audited. Please check your API key and try again.');
                } else {
                    // Fallback: Show success if we have any processed posts
                    this.showAdminNotice('success', `Audit completed. ${totalProcessed} ${totalProcessed === 1 ? 'post' : 'posts'} audited.`);
                }

            } catch (error) {
                progressText.textContent = `Error: ${error.message}`;
                console.error('Global audit error:', error);
                if (warningNotice) warningNotice.style.display = 'none';
                
                // Show error notice
                this.showAdminNotice('error', `Audit failed: ${error.message}. Please try again or check your API key.`);
            } finally {
                startButton.disabled = false;
            }
        }

        async getAllPostIds() {
            try {
                const response = await fetch(mustwpAdmin.restUrl + 'audit/summary', {
                    method: 'GET',
                    headers: {
                        'X-WP-Nonce': mustwpAdmin.nonce
                    }
                });

                if (!response.ok) {
                    throw new Error('Failed to get post summary');
                }

                const data = await response.json();
                return data.post_ids || [];
            } catch (error) {
                console.error('Error getting post IDs:', error);
                return [];
            }
        }

        async getAllPostTitles(postIds) {
            const titleMap = {};
            
            // Fetch posts in batches to avoid too many simultaneous requests
            const batchSize = 10;
            for (let i = 0; i < postIds.length; i += batchSize) {
                const batch = postIds.slice(i, i + batchSize);
                
                try {
                    const response = await fetch(`${mustwpAdmin.restUrl}posts/titles`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-WP-Nonce': mustwpAdmin.nonce
                        },
                        body: JSON.stringify({ post_ids: batch })
                    });
                    
                    if (response.ok) {
                        const data = await response.json();
                        if (data.titles && typeof data.titles === 'object') {
                            // Merge the titles into the titleMap
                            Object.assign(titleMap, data.titles);
                        }
                    }
                } catch (err) {
                    console.error('Error fetching post titles:', err);
                    // Fallback: set default titles for this batch
                    batch.forEach(postId => {
                        if (!titleMap[postId]) {
                            titleMap[postId] = `Post #${postId}`;
                        }
                    });
                }
            }
            
            // Ensure all post IDs have a title (fallback for any that failed)
            postIds.forEach(postId => {
                if (!titleMap[postId]) {
                    titleMap[postId] = `Post #${postId}`;
                }
            });
            
            return titleMap;
        }

        async processBatches(postIds, progressText, postTitleMap = {}) {
            // Hardcoded batch size to 1 post (2 API calls) to prevent timeout issues (especially with Cloudflare)
            const batchSize = 1;
            const batches = this.chunkArray(postIds, batchSize);
            const allResults = [];
            const allErrors = [];
            const totalPosts = postIds.length;
            const totalApiCalls = totalPosts * 2; // mobile + desktop for each post
            let completedApiCalls = 0;

            // Calculate estimated time (30 seconds per post)
            const estimatedSeconds = totalPosts * 30;
            const formatTime = (seconds) => {
                const hours = Math.floor(seconds / 3600);
                const minutes = Math.floor((seconds % 3600) / 60);
                const secs = seconds % 60;
                
                const parts = [];
                if (hours > 0) parts.push(`${hours} ${hours === 1 ? 'hour' : 'hours'}`);
                if (minutes > 0) parts.push(`${minutes} ${minutes === 1 ? 'minute' : 'minutes'}`);
                if (secs > 0 || parts.length === 0) parts.push(`${secs} ${secs === 1 ? 'second' : 'seconds'}`);
                
                return parts.join(', ');
            };
            const estimatedTimeString = formatTime(estimatedSeconds);

            // Create detailed progress container
            const progressContainer = document.createElement('div');
            progressContainer.style.cssText = 'margin-top: 10px; padding: 15px; background: #f0f0f1; border-radius: 4px; font-size: 13px; line-height: 1.4;';
            progressContainer.innerHTML = `
                <div style="margin-bottom: 12px; padding-bottom: 8px; border-bottom: 1px solid #ddd;">
                    <strong>⏱️ Estimated time to complete audit:</strong> <span style="color: #0073aa; font-weight: bold;">${estimatedTimeString}</span>
                </div>
                <div style="margin-bottom: 8px;"><strong>📊 Audit Progress Details:</strong></div>
                <div style="margin-bottom: 4px;">📝 Total Posts: <strong>${totalPosts}</strong> | 🔄 Total API Calls: <strong>${totalApiCalls}</strong></div>
                <div style="margin-bottom: 4px;">✅ Completed API Calls: <span id="completed-calls" style="font-weight: bold; color: #0073aa;">0</span> / ${totalApiCalls}</div>
                <div style="margin-bottom: 4px;">🔄 Current Batch: <span id="current-batch" style="font-weight: bold;">Starting...</span></div>
                <div style="margin-bottom: 8px;">📈 Progress: <span id="progress-percent" style="font-weight: bold; color: #0073aa;">0%</span></div>
            `;
            progressText.parentNode.insertBefore(progressContainer, progressText.nextSibling);

            // Create post processing status container
            const postStatusContainer = document.createElement('div');
            postStatusContainer.style.cssText = 'margin-top: 15px; padding: 15px; background: #f0f0f1; border-radius: 4px; font-size: 13px;';
            postStatusContainer.innerHTML = `
                <div style="margin-bottom: 10px;"><strong>Post Processing Status:</strong></div>
                <div id="post-status-list" style="max-height: 300px; overflow-y: auto;"></div>
            `;
            progressContainer.parentNode.insertBefore(postStatusContainer, progressContainer.nextSibling);

            // Helper function to get current time
            const getCurrentTime = () => {
                const now = new Date();
                return now.toLocaleTimeString('en-US', { hour12: false, hour: '2-digit', minute: '2-digit', second: '2-digit' });
            };

            // Helper function to add post status
            const addPostStatus = (postTitle, status = 'Processing...') => {
                const statusList = document.getElementById('post-status-list');
                if (!statusList) return;

                const statusItem = document.createElement('div');
                statusItem.style.cssText = 'padding: 8px; margin-bottom: 4px; background: #fff; border-radius: 3px; border-left: 3px solid #2271b1;';
                statusItem.innerHTML = `<span style="color: #646970; font-size: 12px;">[${getCurrentTime()}]</span> ${postTitle} – ${status}`;
                statusList.appendChild(statusItem);

                // Auto-scroll to bottom
                statusList.scrollTop = statusList.scrollHeight;

                // Keep only last 50 items to prevent DOM bloat
                const items = statusList.querySelectorAll('div');
                if (items.length > 50) {
                    items[0].remove();
                }
            };

            const updateProgress = (batchNum, batchSize, completed, total) => {
                const completedElement = document.getElementById('completed-calls');
                const batchElement = document.getElementById('current-batch');
                const percentElement = document.getElementById('progress-percent');
                
                if (completedElement) completedElement.textContent = completed;
                if (batchElement) batchElement.textContent = `Post ${batchNum} of ${totalPosts}`;
                if (percentElement) percentElement.textContent = Math.round((completed / total) * 100) + '%';
                
                // Don't update progressText with batch information - keep it simple
                progressText.textContent = '';
            };

            // Strategy is 'both' which means 2 API calls per post (mobile + desktop)
            const apiCallsPerPost = 2;
            const strategy = 'both';

            for (let i = 0; i < batches.length; i++) {
                const batch = batches[i];
                
                // Update progress at start of batch (shows previous batch completion)
                updateProgress(i + 1, batch.length, completedApiCalls, totalApiCalls);

                // Display post titles from the map
                for (const postId of batch) {
                    const postTitle = postTitleMap[postId] || `Post #${postId}`;
                    addPostStatus(postTitle, 'Processing...');
                }

                try {
                    const data = await this.fetchJsonWithRetry(
                        mustwpAdmin.restUrl + 'audit/batch',
                        {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-WP-Nonce': mustwpAdmin.nonce
                            },
                            body: JSON.stringify({ post_ids: batch, strategy: strategy })
                        }
                    );
                    
                    // CRITICAL: Safely handle response data to prevent "data.results is not iterable" error
                    if (data && Array.isArray(data.results)) {
                        allResults.push(...data.results);
                        
                        // Update post status for successfully processed posts
                        data.results.forEach(result => {
                            const postTitle = result.post_title || postTitleMap[result.post_id] || `Post #${result.post_id}`;
                            addPostStatus(postTitle, 'Completed ✓');
                        });
                    }
                    if (data && Array.isArray(data.errors)) {
                        allErrors.push(...data.errors);
                        
                        // Update post status for posts with errors
                        data.errors.forEach(error => {
                            const postTitle = error.post_title || postTitleMap[error.post_id] || `Post #${error.post_id}`;
                            addPostStatus(postTitle, `Error: ${error.error || 'Unknown error'}`);
                        });
                    }
                    
                    // Count API calls based on actual processed posts from server
                    // Use data.processed if available, otherwise fall back to results length
                    const processedCount = (data && typeof data.processed === 'number') 
                        ? data.processed 
                        : (data && data.results ? data.results.length : 0);
                    
                    // Calculate API calls: each processed post requires apiCallsPerPost calls
                    const batchApiCallsCompleted = processedCount * apiCallsPerPost;
                    completedApiCalls += batchApiCallsCompleted;

                    // Update progress immediately after batch completion
                    updateProgress(i + 1, batch.length, completedApiCalls, totalApiCalls);

                    // Handle timeout warnings (partial batch completion)
                    if (data && data.timeout_warning) {
                        console.warn('Batch timeout warning:', data.message);
                        // Continue processing remaining batches
                        if (data.has_more && data.processed > 0) {
                            // Some posts were processed, continue with remaining
                            progressText.textContent = `Batch ${i + 1} partially completed due to timeout. Continuing...`;
                        }
                    }
                    
                    // Check if we should stop due to quota/403 errors
                    const hasQuotaError = allErrors.some(error => 
                        error.error && (error.error.includes('quota') || error.error.includes('403'))
                    );
                    
                    if (hasQuotaError) {
                        progressText.textContent = 'Stopping due to API quota exceeded.';
                        break;
                    }

                } catch (error) {
                    console.error('Batch error:', error);
                    
                    // Handle timeout errors specifically (including Cloudflare 524)
                    const isTimeoutError = error.message && (
                        error.message.includes('504') || 
                        error.message.includes('524') ||
                        error.message.includes('Cloudflare') ||
                        error.message.includes('Gateway error') ||
                        error.message.includes('timeout')
                    );
                    
                    if (isTimeoutError) {
                        // For timeout errors, try to continue with next batch
                        // Don't break the loop, allow retry or continue
                        const timeoutMsg = error.message.includes('524') || error.message.includes('Cloudflare')
                            ? `Batch ${i + 1} timed out (Cloudflare timeout). The batch size may be too large. Continuing with remaining batches...`
                            : `Batch ${i + 1} timed out. Continuing with remaining batches...`;
                        progressText.textContent = timeoutMsg;
                        allErrors.push({
                            batch: i + 1,
                            error: error.message || 'Request timeout - batch may have partially processed. Continuing with remaining batches.',
                            timeout: true
                        });
                        // Continue to next batch instead of breaking
                    } else {
                        allErrors.push({
                            batch: i + 1,
                            error: error.message
                        });
                    }
                    
                    // Update progress even on error to show current state
                    updateProgress(i + 1, batch.length, completedApiCalls, totalApiCalls);
                }
            }

            // Clean up progress container
            if (progressContainer.parentNode) {
                progressContainer.parentNode.removeChild(progressContainer);
            }

            // Clean up post status container
            if (postStatusContainer.parentNode) {
                postStatusContainer.parentNode.removeChild(postStatusContainer);
            }

            return {
                results: allResults,
                errors: allErrors,
                totalProcessed: allResults.length,
                totalErrors: allErrors.length
            };
        }

        async fetchJsonWithRetry(url, options, retries = 3) {
            let attempt = 0;
            let delay = 2000; // Start with 2 seconds for gateway errors
            while (attempt <= retries) {
                try {
                    const response = await fetch(url, options);
                    // If gateway errors (including Cloudflare 524 timeout), throw to retry with exponential backoff
                    if (response.status === 502 || response.status === 503 || response.status === 504 || response.status === 524) {
                        if (attempt === retries) {
                            // On final attempt, throw a descriptive error
                            const errorMsg = response.status === 524 
                                ? 'Cloudflare timeout (524) - Request took too long. Try reducing batch size or increasing server timeout limits.'
                                : `Gateway error ${response.status}`;
                            throw new Error(errorMsg);
                        }
                        // Wait longer for gateway errors before retrying
                        await this.sleep(delay);
                        delay *= 2; // Exponential backoff: 2s, 4s, 8s
                        attempt++;
                        continue;
                    }
                    // Try parse JSON; check for HTML responses (error pages)
                    const text = await response.text();
                    if (text.trim().startsWith('<')) {
                        // Check if it's a Cloudflare error page
                        if (text.includes('cf-error-details') || text.includes('Cloudflare') || text.includes('524')) {
                            throw new Error('Cloudflare timeout error (524) - The request exceeded Cloudflare\'s timeout limit. The batch size may be too large.');
                        }
                        // Generic HTML response error
                        throw new Error('Non-JSON response from server - Received HTML instead of JSON. This may indicate a timeout or server error.');
                    }
                    return JSON.parse(text);
                } catch (err) {
                    if (attempt === retries) {
                        throw err;
                    }
                    // Only sleep if it's not a gateway error (those are handled above)
                    if (!err.message || (!err.message.includes('Gateway error') && !err.message.includes('Cloudflare') && !err.message.includes('524'))) {
                        await this.sleep(delay);
                        delay *= 2;
                    }
                    attempt++;
                }
            }
        }

        displayResults(results, summaryDiv) {
            const { results: auditResults, errors, totalProcessed, totalErrors } = results;

            let html = `
                <div class="mustwp-summary-stats">
                    <div class="mustwp-stat-card">
                        <div class="mustwp-stat-number">${totalProcessed}</div>
                        <div class="mustwp-stat-label">Posts Audited</div>
                    </div>
                    <div class="mustwp-stat-card">
                        <div class="mustwp-stat-number">${totalErrors}</div>
                        <div class="mustwp-stat-label">Errors</div>
                    </div>
                </div>
            `;

            if (errors.length > 0) {
                html += `
                    <div class="mustwp-audit-errors">
                        <h4>Errors Encountered:</h4>
                        <ul>
                `;
                
                errors.forEach(error => {
                    html += `<li>${error.error || 'Unknown error'}</li>`;
                });
                
                html += `
                        </ul>
                    </div>
                `;
            }

            summaryDiv.innerHTML = html;
        }

        /**
         * Show WordPress-style admin notice
         * 
         * @param {string} type - Notice type: 'success', 'warning', or 'error'
         * @param {string} message - Notice message
         */
        showAdminNotice(type, message) {
            // Remove existing audit notices
            const existingNotices = document.querySelectorAll('.mustwp-audit-notice');
            existingNotices.forEach(notice => notice.remove());

            // Create notice HTML
            const noticeClass = `notice notice-${type} is-dismissible mustwp-audit-notice`;
            const noticeHtml = `
                <div class="${noticeClass}">
                    <p>${message}</p>
                    <button type="button" class="notice-dismiss">
                        <span class="screen-reader-text">Dismiss this notice.</span>
                    </button>
                </div>
            `;

            // Try multiple selectors to find the page title
            // WordPress admin pages use h1 inside .wrap
            let pageTitle = document.querySelector('.wrap h1');
            if (!pageTitle) {
                // Fallback to just h1
                pageTitle = document.querySelector('h1');
            }
            if (!pageTitle) {
                // Fallback to .wp-heading-inline (some WordPress versions)
                pageTitle = document.querySelector('.wp-heading-inline');
            }
            
            if (pageTitle) {
                pageTitle.insertAdjacentHTML('afterend', noticeHtml);
                
                // Add dismiss functionality
                const dismissBtn = document.querySelector('.mustwp-audit-notice .notice-dismiss');
                if (dismissBtn) {
                    dismissBtn.addEventListener('click', function() {
                        this.closest('.mustwp-audit-notice').remove();
                    });
                }
            } else {
                // If no title found, insert at the top of .wrap or body
                const wrap = document.querySelector('.wrap');
                if (wrap) {
                    wrap.insertAdjacentHTML('afterbegin', noticeHtml);
                    
                    // Add dismiss functionality
                    const dismissBtn = document.querySelector('.mustwp-audit-notice .notice-dismiss');
                    if (dismissBtn) {
                        dismissBtn.addEventListener('click', function() {
                            this.closest('.mustwp-audit-notice').remove();
                        });
                    }
                } else {
                    // Last resort: log error
                    console.error('Mustang WPO: Could not find page title or wrap element to insert notice');
                }
            }
        }

        chunkArray(array, size) {
            const chunks = [];
            for (let i = 0; i < array.length; i += size) {
                chunks.push(array.slice(i, i + size));
            }
            return chunks;
        }

        sleep(ms) {
            return new Promise(resolve => setTimeout(resolve, ms));
        }
    }

    /**
     * Settings form enhancements
     */
    class SettingsForm {
        constructor() {
            this.init();
        }

        init() {
            // Add form validation
            this.addFormValidation();
            
            // Add budget presets
            this.addBudgetPresets();
        }

        addFormValidation() {
            const forms = document.querySelectorAll('form[method="post"]');
            
            forms.forEach(form => {
                form.addEventListener('submit', (e) => {
                    if (!this.validateForm(form)) {
                        e.preventDefault();
                    }
                });
            });
        }

        validateForm(form) {
            let isValid = true;
            const errors = [];

            // Validate API key format (if provided)
            const apiKeyInput = form.querySelector('input[name*="api_key"]');
            if (apiKeyInput && apiKeyInput.value.trim()) {
                const apiKey = apiKeyInput.value.trim();
                if (!/^[A-Za-z0-9_-]+$/.test(apiKey)) {
                    errors.push('API key contains invalid characters');
                    isValid = false;
                }
            }

            // Validate rate limit
            const rateLimitInput = form.querySelector('input[name*="audit_rate_limit"]');
            if (rateLimitInput) {
                const rateLimit = parseInt(rateLimitInput.value);
                if (rateLimit < 0 || rateLimit > 10) {
                    errors.push('Rate limit must be between 0 and 10 seconds');
                    isValid = false;
                }
            }

            // Display errors
            if (!isValid) {
                this.showErrors(errors);
            }

            return isValid;
        }

        showErrors(errors) {
            // Remove existing error notices
            const existingNotices = document.querySelectorAll('.mustwp-form-errors');
            existingNotices.forEach(notice => notice.remove());

            // Create error notice
            const errorHtml = `
                <div class="notice notice-error mustwp-form-errors">
                    <p><strong>Please fix the following errors:</strong></p>
                    <ul>
                        ${errors.map(error => `<li>${error}</li>`).join('')}
                    </ul>
                </div>
            `;

            // Insert after page title
            const pageTitle = document.querySelector('.wp-heading-inline');
            if (pageTitle) {
                pageTitle.insertAdjacentHTML('afterend', errorHtml);
            }
        }

        addBudgetPresets() {
            const budgetSections = document.querySelectorAll('h3');
            
            budgetSections.forEach(section => {
                if (section.textContent.includes('Budget')) {
                    this.addPresetButtons(section);
                }
            });
        }

        addPresetButtons(section) {
            const presetHtml = `
                <div class="mustwp-budget-presets">
                    <label>Quick Presets:</label>
                    <button type="button" class="button button-small mustwp-preset-strict">Strict</button>
                    <button type="button" class="button button-small mustwp-preset-moderate">Moderate</button>
                    <button type="button" class="button button-small mustwp-preset-lenient">Lenient</button>
                </div>
            `;

            section.insertAdjacentHTML('afterend', presetHtml);

            // Add preset functionality
            this.addPresetHandlers();
        }

        addPresetHandlers() {
            const presets = {
                strict: { lcp: 2.0, cls: 0.05, inp: 100, weight: 1.0 },
                moderate: { lcp: 3.0, cls: 0.1, inp: 200, weight: 2.0 },
                lenient: { lcp: 4.0, cls: 0.15, inp: 300, weight: 3.0 }
            };

            Object.keys(presets).forEach(preset => {
                const button = document.querySelector(`.mustwp-preset-${preset}`);
                if (button) {
                    button.addEventListener('click', () => {
                        this.applyPreset(presets[preset]);
                    });
                }
            });
        }

        applyPreset(preset) {
            const inputs = document.querySelectorAll('input[name*="budgets"]');
            
            inputs.forEach(input => {
                const name = input.name;
                if (name.includes('[lcp]')) {
                    input.value = preset.lcp;
                } else if (name.includes('[cls]')) {
                    input.value = preset.cls;
                } else if (name.includes('[inp]')) {
                    input.value = preset.inp;
                } else if (name.includes('[weight]')) {
                    input.value = preset.weight;
                }
            });
        }
    }

    /**
     * Initialize when DOM is ready
     */
    $(document).ready(function() {
        new GlobalAudit();
        new SettingsForm();
    });

})(jQuery);
