#!/usr/bin/env node

/**
 * Version Update Script
 * 
 * Updates version numbers in:
 * - mustang-wpo.php (header and constant)
 * - readme.txt (stable tag)
 * 
 * Usage: node scripts/update-version.js 1.0.0
 */

const fs = require('fs');
const path = require('path');

// Get version from command line
const newVersion = process.argv[2];

if (!newVersion) {
    console.error('Error: Version number required');
    console.log('Usage: node scripts/update-version.js <version>');
    console.log('Example: node scripts/update-version.js 1.0.0');
    process.exit(1);
}

// Validate version format (semantic versioning)
if (!/^\d+\.\d+\.\d+$/.test(newVersion)) {
    console.error('Error: Invalid version format. Use semantic versioning (e.g., 1.0.0)');
    process.exit(1);
}

const pluginDir = path.join(__dirname, '..');
const pluginFile = path.join(pluginDir, 'mustang-wpo.php');
const readmeFile = path.join(pluginDir, 'readme.txt');

console.log(`Updating version to ${newVersion}...`);

// Update plugin file
if (fs.existsSync(pluginFile)) {
    let content = fs.readFileSync(pluginFile, 'utf8');
    
    // Update header version
    content = content.replace(
        /(\*\s+Version:\s+)(\d+\.\d+\.\d+)/,
        `$1${newVersion}`
    );
    
    // Update constant version
    content = content.replace(
        /(define\('MUSTWP_VERSION',\s+')(\d+\.\d+\.\d+)('\);)/,
        `$1${newVersion}$3`
    );
    
    fs.writeFileSync(pluginFile, content, 'utf8');
    console.log('✓ Updated mustang-wpo.php');
} else {
    console.error('Error: mustang-wpo.php not found');
    process.exit(1);
}

// Update readme.txt
if (fs.existsSync(readmeFile)) {
    let content = fs.readFileSync(readmeFile, 'utf8');
    
    // Update stable tag
    content = content.replace(
        /(Stable tag:\s+)(\d+\.\d+\.\d+)/,
        `$1${newVersion}`
    );
    
    fs.writeFileSync(readmeFile, content, 'utf8');
    console.log('✓ Updated readme.txt');
} else {
    console.error('Error: readme.txt not found');
    process.exit(1);
}

console.log(`\n✓ Version updated to ${newVersion} successfully!`);
console.log('\nNext steps:');
console.log('1. Update CHANGELOG.md');
console.log('2. Build assets: node build.js');
console.log('3. Commit changes: git commit -m "Bump version to ' + newVersion + '"');
console.log('4. Create tag: git tag -a v' + newVersion + ' -m "Release version ' + newVersion + '"');

