<?php
/**
 * Plugin Name: Mustang WPO
 * Plugin URI: https://mustangwpo.com/
 * Description: Mustang WPO (Web Performance Optimization) helps you audit, view, and manage your site's performance without leaving WordPress.
 * Version: 1.0.2
 * Author: Mustang WPO
 * Author URI: https://mustangwpo.com
 * License: GPLv3
 * License URI: http://www.gnu.org/licenses/gpl.html
 * Text Domain: mustang-wpo
 * Domain Path: /languages
 * Requires at least: 6.2
 * Tested up to: 6.9
 * Requires PHP: 7.4
 */

declare(strict_types=1);

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
// Version management: Update in plugin header, readme.txt, and here
define('MUSTWP_VERSION', '1.0.1');
define('MUSTWP_PLUGIN_FILE', __FILE__);
define('MUSTWP_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MUSTWP_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MUSTWP_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Development mode detection
// Set to true in wp-config.php: define('MUSTWP_DEV_MODE', true);
if (!defined('MUSTWP_DEV_MODE')) {
    define('MUSTWP_DEV_MODE', (defined('WP_DEBUG') && WP_DEBUG));
}

/**
 * Get asset version for cache busting
 * In development mode, uses file modification time
 * In production, uses plugin version
 *
 * @param string $file_path Path to the asset file
 * @return string Version string
 */
function mustwp_get_asset_version(string $file_path): string {
    if (MUSTWP_DEV_MODE && file_exists($file_path)) {
        return MUSTWP_VERSION . '.' . filemtime($file_path);
    }
    return MUSTWP_VERSION;
}

/**
 * Main plugin class
 */
final class MUSTWP {
    
    /**
     * Plugin instance
     */
    private static $instance = null;
    
    /**
     * Get plugin instance
     */
    public static function get_instance(): self {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
        $this->load_classes();
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks(): void {
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
        
        add_action('init', [$this, 'init']);
        add_action('rest_api_init', [$this, 'register_rest_routes']);
    }
    
    /**
     * Plugin initialization
     */
    public function init(): void {
        // Initialize Editor for REST API routes (needed for both admin and frontend)
        new MUSTWP\Editor();
        
        // Initialize admin classes
        if (is_admin()) {
            $this->init_admin();
        }
    }
    
    /**
     * Initialize admin functionality
     */
    private function init_admin(): void {
        new MUSTWP\Admin();
        new MUSTWP\Settings();
        new MUSTWP\Dashboard();
    }
    
    /**
     * Register REST API routes
     */
    public function register_rest_routes(): void {
        $auditor = new MUSTWP\Auditor();
        $auditor->register_rest_routes();
    }
    
    /**
     * Load plugin classes
     */
    private function load_classes(): void {
        require_once MUSTWP_PLUGIN_DIR . 'includes/class-mustwp-api.php';
        require_once MUSTWP_PLUGIN_DIR . 'includes/class-mustwp-admin.php';
        require_once MUSTWP_PLUGIN_DIR . 'includes/class-mustwp-editor.php';
        require_once MUSTWP_PLUGIN_DIR . 'includes/class-mustwp-settings.php';
        require_once MUSTWP_PLUGIN_DIR . 'includes/class-mustwp-budgets.php';
        require_once MUSTWP_PLUGIN_DIR . 'includes/class-mustwp-auditor.php';
        require_once MUSTWP_PLUGIN_DIR . 'includes/class-mustwp-dashboard.php';
    }
    
    /**
     * Plugin activation
     */
    public function activate(): void {
        // Set default options
        $default_options = [
            'api_key' => '',
            'content_types' => ['post', 'page'],
            'budgets' => [
                'post' => [
                    'lcp' => 3.0,
                    'cls' => 0.1,
                    'inp' => 200,
                    'weight' => 2.0
                ],
                'page' => [
                    'lcp' => 3.0,
                    'cls' => 0.1,
                    'inp' => 200,
                    'weight' => 2.0
                ]
            ],
            'audit_rate_limit' => 1
        ];
        
        add_option('mustwp_settings', $default_options);
        
        // Flush rewrite rules for REST API
        flush_rewrite_rules();
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate(): void {
        // Flush rewrite rules
        flush_rewrite_rules();
    }
}

// Initialize the plugin
MUSTWP::get_instance();
