<?php
/**
 * Settings page for WPO
 *
 * @package MUSTWP
 */
 
declare(strict_types=1);
 
namespace MUSTWP;

/**
 * Settings class for plugin configuration
 */
class Settings {
    
    /**
     * Plugin settings
     */
    private array $settings;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->settings = get_option('mustwp_settings', []);
        $this->init_hooks();
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks(): void {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        add_filter('heartbeat_received', [$this, 'handle_heartbeat'], 10, 2);
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu(): void {
        add_menu_page(
            __('Settings', 'mustang-wpo'),
            __('Mustang WPO', 'mustang-wpo'),
            'manage_options',
            'mustwp-settings',
            [$this, 'render_settings_page'],
            'dashicons-performance',
            85
        );
        
        add_submenu_page(
            'mustwp-settings',
            __('Settings', 'mustang-wpo'),
            __('Settings', 'mustang-wpo'),
            'manage_options',
            'mustwp-settings',
            [$this, 'render_settings_page']
        );
        
        add_submenu_page(
            'mustwp-settings',
            __('Integrations', 'mustang-wpo'),
            __('Integrations', 'mustang-wpo'),
            'manage_options',
            'mustwp-integrations',
            [$this, 'render_integrations_page']
        );
        
        add_submenu_page(
            'mustwp-settings',
            __('Performance Budgets', 'mustang-wpo'),
            __('Budgets', 'mustang-wpo'),
            'manage_options',
            'mustwp-budgets',
            [$this, 'render_budgets_page']
        );
        
        add_submenu_page(
            'mustwp-settings',
            __('Global Audit', 'mustang-wpo'),
            __('Audit', 'mustang-wpo'),
            'manage_options',
            'mustwp-audit',
            [$this, 'render_audit_page']
        );
    }
    
    /**
     * Register settings
     */
    public function register_settings(): void {
        // Settings are handled manually in handle_settings_save()
        // No need to register with WordPress settings API
    }
    
    /**
     * Sanitize settings
     *
     * @param array $input Raw settings input
     * @return array Sanitized settings
     */
    public function sanitize_settings(array $input): array {
        $sanitized = [];
        
        // API Key - handle masked input and encrypt for secure storage
        $api_key = sanitize_text_field($input['api_key'] ?? '');
        if (!empty($api_key)) {
            // Check if the input is a masked key (contains asterisks)
            if (strpos($api_key, '*') !== false) {
                // Input is masked, keep existing encrypted key
                $sanitized['api_key'] = $this->settings['api_key'] ?? '';
            } elseif (strpos($api_key, 'AIza') === 0) {
                // Input is a valid Google API key, encrypt it
                $sanitized['api_key'] = $this->encrypt_api_key($api_key);
            } else {
                // Input appears to be already encrypted or invalid, keep as is
                $sanitized['api_key'] = $api_key;
            }
        } else {
            // Empty input - check if this is from a form that doesn't include API key
            // If the input array doesn't have an 'api_key' key, preserve existing value
            if (!array_key_exists('api_key', $input)) {
                // API key field not present in form, preserve existing value
                $sanitized['api_key'] = $this->settings['api_key'] ?? '';
            } else {
                // API key field was present but empty, clear the key
                $sanitized['api_key'] = '';
            }
        }
        
        // Post Types - Free version limitation: only allow 'post' and 'page'
        $sanitized['content_types'] = [];
        $allowed_types = ['post', 'page']; // Free version limitation
        foreach ($input['content_types'] ?? [] as $post_type) {
            $post_type = sanitize_text_field($post_type);
            // Only allow 'post' and 'page' in free version
            if (in_array($post_type, $allowed_types, true)) {
                $sanitized['content_types'][] = $post_type;
            }
        }
        // Ensure at least 'post' and 'page' are selected
        if (empty($sanitized['content_types'])) {
            $sanitized['content_types'] = ['post', 'page'];
        }
        
        // Budgets
        $sanitized['budgets'] = [];
        foreach ($input['budgets'] ?? [] as $post_type => $budgets) {
            $sanitized['budgets'][sanitize_text_field($post_type)] = [
                'lcp' => (float) ($budgets['lcp'] ?? 3.0),
                'cls' => (float) ($budgets['cls'] ?? 0.1),
                'inp' => (int) ($budgets['inp'] ?? 200),
                'weight' => (float) ($budgets['weight'] ?? 2.0)
            ];
        }
        
        return $sanitized;
    }
    
    /**
     * Encrypt API key for secure storage
     *
     * @param string $api_key Raw API key
     * @return string Encrypted API key
     */
    private function encrypt_api_key(string $api_key): string {
        if (empty($api_key)) {
            return '';
        }
        
        // Use WordPress salts for encryption
        $key = wp_salt('auth');
        $iv = wp_salt('secure_auth');
        
        // Simple encryption using WordPress salts
        $encrypted = base64_encode(openssl_encrypt($api_key, 'AES-256-CBC', $key, 0, substr($iv, 0, 16)));
        
        return $encrypted;
    }
    
    /**
     * Decrypt API key for use
     *
     * @param string $encrypted_api_key Encrypted API key
     * @return string Decrypted API key
     */
    public function decrypt_api_key(string $encrypted_api_key): string {
        if (empty($encrypted_api_key)) {
            return '';
        }
        
        // Use WordPress salts for decryption
        $key = wp_salt('auth');
        $iv = wp_salt('secure_auth');
        
        try {
            $decoded = base64_decode($encrypted_api_key);
            if ($decoded === false) {
                return '';
            }
            
            $decrypted = openssl_decrypt($decoded, 'AES-256-CBC', $key, 0, substr($iv, 0, 16));
            if ($decrypted === false) {
                return '';
            }
            
            return $decrypted;
        } catch (Exception $e) {
            return '';
        }
    }
    
    /**
     * Get masked API key for display (first 4 characters + asterisks)
     *
     * @param string $encrypted_api_key Encrypted API key
     * @return string Masked API key for display
     */
    public function get_masked_api_key(string $encrypted_api_key): string {
        if (empty($encrypted_api_key)) {
            return '';
        }
        
        $decrypted_key = $this->decrypt_api_key($encrypted_api_key);
        if (empty($decrypted_key)) {
            return '';
        }
        
        // Check if the decrypted key is a valid Google API key
        if (strpos($decrypted_key, 'AIza') === 0) {
            // Valid Google API key - show first 4 characters
            $length = strlen($decrypted_key);
            if ($length <= 4) {
                return str_repeat('*', $length);
            }
            return substr($decrypted_key, 0, 4) . str_repeat('*', $length - 4);
        } else {
            // Invalid/corrupted key - show generic mask
            return '****' . str_repeat('*', max(0, strlen($decrypted_key) - 4));
        }
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page(): void {
        if (isset($_POST['submit'])) {
            // Verify nonce before processing form data
            if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mustwp_settings_nonce'] ?? '')), 'mustwp_settings')) {
                wp_die(esc_html__('Security check failed.', 'mustang-wpo'));
            }
            $this->handle_settings_save();
        }
        
        $this->render_page_header(__('Settings', 'mustang-wpo'));
        
        // Display admin notices
        settings_errors('mustwp_settings');
        
        // Check if API key is configured and show notice if not
        $api = new API();
        if (!$api->has_valid_api_key()) {
            ?>
            <div class="notice notice-warning">
                <p>
                    <strong><?php esc_html_e('API Key Required:', 'mustang-wpo'); ?></strong>
                    <?php esc_html_e('A valid Google PageSpeed Insights API key is required for the plugin to function.', 'mustang-wpo'); ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=mustwp-integrations')); ?>" class="button button-secondary">
                        <?php esc_html_e('Configure API Key', 'mustang-wpo'); ?>
                    </a>
                </p>
            </div>
            <?php
        }
        ?>
        <form method="post" action="">
            <?php wp_nonce_field('mustwp_settings', 'mustwp_settings_nonce'); ?>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label><?php esc_html_e('Post Types to Audit', 'mustang-wpo'); ?></label>
                    </th>
                    <td>
                        <?php
                        $available_types = get_post_types(['public' => true], 'objects');
                        $selected_types = $this->settings['content_types'] ?? ['post', 'page'];
                        $allowed_types = ['post', 'page']; // Free version limitation
                        
                        foreach ($available_types as $post_type) {
                            // Skip media/attachment post type
                            if ($post_type->name === 'attachment') {
                                continue;
                            }
                            
                            $is_allowed = in_array($post_type->name, $allowed_types, true);
                            $checked = in_array($post_type->name, $selected_types, true) ? 'checked' : '';
                            $disabled = $is_allowed ? '' : 'disabled';
                            $style = $is_allowed ? '' : 'style="opacity: 0.5; cursor: not-allowed;"';
                            
                            printf(
                                '<label %s><input type="checkbox" name="mustwp_settings[content_types][]" value="%s" %s %s> %s%s</label><br>',
                                wp_kses_post($style),
                                esc_attr($post_type->name),
                                esc_attr($checked),
                                esc_attr($disabled),
                                esc_html($post_type->label),
                                $is_allowed ? '' : ' <em style="font-size: 11px; color: #646970;">(' . esc_html__('Premium only', 'mustang-wpo') . ')</em>'
                            );
                        }
                        ?>
                        <p class="description">
                            <?php esc_html_e('Select which post types you would like to audit, view, and manage the performance of. The free version supports Pages and Posts only.', 'mustang-wpo'); ?>
                        </p>
                    </td>
                </tr>
                
            </table>
            
            <?php submit_button(); ?>
        </form>
        <?php
        $this->close_wrap();
    }
    
    /**
     * Render budgets page
     */
    public function render_budgets_page(): void {
        if (isset($_POST['submit'])) {
            // Verify nonce before processing form data
            if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mustwp_budgets_nonce'] ?? '')), 'mustwp_budgets')) {
                wp_die(esc_html__('Security check failed.', 'mustang-wpo'));
            }
            $this->handle_budgets_save();
        }
        
        $this->render_page_header(__('Performance Budgets', 'mustang-wpo'));
        
        // Display admin notices
        settings_errors('mustwp_settings');
        
        // Check if API key is configured and show notice if not
        $api = new API();
        if (!$api->has_valid_api_key()) {
            ?>
            <div class="notice notice-warning">
                <p>
                    <strong><?php esc_html_e('API Key Required:', 'mustang-wpo'); ?></strong>
                    <?php esc_html_e('A valid Google PageSpeed Insights API key is required for the plugin to function.', 'mustang-wpo'); ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=mustwp-integrations')); ?>" class="button button-secondary">
                        <?php esc_html_e('Configure API Key', 'mustang-wpo'); ?>
                    </a>
                </p>
            </div>
            <?php
        }
        
        $content_types = $this->settings['content_types'] ?? ['post', 'page'];
        $allowed_types = ['post', 'page']; // Free version limitation
        
        // Filter to only allow 'post' and 'page' in free version
        $content_types = array_intersect($content_types, $allowed_types);
        
        $budgets = new Budgets();
        ?>
        <form method="post" action="">
            <?php wp_nonce_field('mustwp_budgets', 'mustwp_budgets_nonce'); ?>
            
            <p><?php esc_html_e('Set performance budgets for each content type. Leave fields empty to disable budget checking for that metric.', 'mustang-wpo'); ?></p>
            
            <?php foreach ($content_types as $post_type): ?>
                <?php
                $post_type_obj = get_post_type_object($post_type);
                $post_type_budgets = $this->settings['budgets'][$post_type] ?? [];
                ?>
                <h3><?php echo esc_html($post_type_obj->label); ?></h3>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="lcp_<?php echo esc_attr($post_type); ?>"><?php esc_html_e('Max Load Time (LCP)', 'mustang-wpo'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="lcp_<?php echo esc_attr($post_type); ?>" 
                                   name="mustwp_settings[budgets][<?php echo esc_attr($post_type); ?>][lcp]" 
                                   value="<?php echo esc_attr($post_type_budgets['lcp'] ?? ''); ?>" 
                                   step="0.1" min="0" />
                            <span class="description"><?php esc_html_e('seconds', 'mustang-wpo'); ?></span>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="cls_<?php echo esc_attr($post_type); ?>"><?php esc_html_e('Max CLS', 'mustang-wpo'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="cls_<?php echo esc_attr($post_type); ?>" 
                                   name="mustwp_settings[budgets][<?php echo esc_attr($post_type); ?>][cls]" 
                                   value="<?php echo esc_attr($post_type_budgets['cls'] ?? ''); ?>" 
                                   step="0.001" min="0" />
                            <span class="description"><?php esc_html_e('unitless', 'mustang-wpo'); ?></span>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="inp_<?php echo esc_attr($post_type); ?>"><?php esc_html_e('Max INP', 'mustang-wpo'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="inp_<?php echo esc_attr($post_type); ?>" 
                                   name="mustwp_settings[budgets][<?php echo esc_attr($post_type); ?>][inp]" 
                                   value="<?php echo esc_attr($post_type_budgets['inp'] ?? ''); ?>" 
                                   min="0" />
                            <span class="description"><?php esc_html_e('milliseconds', 'mustang-wpo'); ?></span>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="weight_<?php echo esc_attr($post_type); ?>"><?php esc_html_e('Max Page Weight', 'mustang-wpo'); ?></label>
                        </th>
                        <td>
                            <input type="number" id="weight_<?php echo esc_attr($post_type); ?>" 
                                   name="mustwp_settings[budgets][<?php echo esc_attr($post_type); ?>][weight]" 
                                   value="<?php echo esc_attr($post_type_budgets['weight'] ?? ''); ?>" 
                                   step="0.1" min="0" />
                            <span class="description"><?php esc_html_e('MB', 'mustang-wpo'); ?></span>
                        </td>
                    </tr>
                </table>
            <?php endforeach; ?>
            
            <?php submit_button(); ?>
        </form>
        <?php
        $this->close_wrap();
    }
    
    /**
     * Render integrations page
     */
    public function render_integrations_page(): void {
        if (isset($_POST['submit'])) {
            // Verify nonce before processing form data
            if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mustwp_integrations_nonce'] ?? '')), 'mustwp_integrations')) {
                wp_die(esc_html__('Security check failed.', 'mustang-wpo'));
            }
            $this->handle_integrations_save();
        }
        
        $this->render_page_header(__('Integrations', 'mustang-wpo'));
        
        // Display admin notices
        settings_errors('mustwp_integrations');
        ?>
        <form method="post" action="">
            <?php wp_nonce_field('mustwp_integrations', 'mustwp_integrations_nonce'); ?>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="api_key"><?php esc_html_e('Google PageSpeed Insights API Key', 'mustang-wpo'); ?> <span class="required">*</span></label>
                    </th>
                    <td>
                        <input type="text" id="api_key" name="mustwp_integrations[api_key]" 
                               value="<?php echo esc_attr($this->get_masked_api_key($this->settings['api_key'] ?? '')); ?>" 
                               class="regular-text" 
                               placeholder="<?php esc_attr_e('Enter your Google PageSpeed Insights API key', 'mustang-wpo'); ?>" 
                               required />
                        <p class="description">
                            <strong><?php esc_html_e('Required:', 'mustang-wpo'); ?></strong> <?php esc_html_e('A valid Google PageSpeed Insights API key is required for the plugin to function.', 'mustang-wpo'); ?>
                            <a href="https://developers.google.com/speed/docs/insights/v5/get-started" target="_blank">
                                <?php esc_html_e('Get API Key', 'mustang-wpo'); ?>
                            </a>
                            <?php if (!empty($this->settings['api_key'])): ?>
                                <?php 
                                $decrypted_key = $this->decrypt_api_key($this->settings['api_key']);
                                if (strpos($decrypted_key, 'AIza') === 0): ?>
                                    <br><em style="color: #00a32a;"><?php esc_html_e('✓ Valid API key configured. Plugin is ready to use.', 'mustang-wpo'); ?></em>
                                <?php else: ?>
                                    <br><em style="color: #d63638;"><?php esc_html_e('⚠ Invalid API key detected. Please enter a valid Google PageSpeed Insights API key.', 'mustang-wpo'); ?></em>
                                <?php endif; ?>
                            <?php else: ?>
                                <br><em style="color: #d63638;"><?php esc_html_e('⚠ API key is required. Please enter a valid Google PageSpeed Insights API key to use the plugin.', 'mustang-wpo'); ?></em>
                            <?php endif; ?>
                        </p>
                    </td>
                </tr>
            </table>
            
            <?php submit_button(); ?>
        </form>
        <?php
        $this->close_wrap();
    }
    
    /**
     * Render audit page
     */
    public function render_audit_page(): void {
        $this->render_page_header(__('Global Audit', 'mustang-wpo'));
        
        // Check if API key is configured and show notice if not
        $api = new API();
        if (!$api->has_valid_api_key()) {
            ?>
            <div class="notice notice-warning">
                <p>
                    <strong><?php esc_html_e('API Key Required:', 'mustang-wpo'); ?></strong>
                    <?php esc_html_e('A valid Google PageSpeed Insights API key is required for the plugin to function.', 'mustang-wpo'); ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=mustwp-integrations')); ?>" class="button button-secondary">
                        <?php esc_html_e('Configure API Key', 'mustang-wpo'); ?>
                    </a>
                </p>
            </div>
            <?php
        }
        
        // Check PHP execution time limit and warn if too low
        $max_execution_time = ini_get('max_execution_time');
        // Only show warning if limit is set (not 0/unlimited) and less than 60 seconds
        if ($max_execution_time > 0 && $max_execution_time < 60) {
            ?>
            <div class="notice notice-warning">
                <p>
                    <strong><?php esc_html_e('Low PHP Execution Time Limit:', 'mustang-wpo'); ?></strong>
                    <?php
                    printf(
                        // translators: %1$s is the current execution time limit in seconds
                        esc_html__('Your server\'s PHP execution time limit is currently set to %1$s seconds. This may cause batch processing to timeout during large audits.', 'mustang-wpo'),
                        esc_html($max_execution_time)
                    );
                    ?>
                    <br>
                    <strong><?php esc_html_e('Recommendations:', 'mustang-wpo'); ?></strong>
                    <ul style="list-style: disc; margin-left: 20px; margin-top: 5px;">
                        <li><?php esc_html_e('Contact your hosting provider to increase PHP max_execution_time to at least 60 seconds (recommended: 300 seconds).', 'mustang-wpo'); ?></li>
                        <li><?php esc_html_e('Alternatively, add the following to your wp-config.php file (if your hosting allows):', 'mustang-wpo'); ?>
                            <code style="display: block; margin-top: 5px; padding: 5px; background: #f0f0f1; border-radius: 3px;">ini_set('max_execution_time', 300);</code>
                        </li>
                        <li><?php esc_html_e('Note: The plugin processes audits one post at a time (2 API calls per post) to minimize timeout risk (especially with Cloudflare\'s ~100s timeout limit), but a higher execution time limit is recommended for large sites.', 'mustang-wpo'); ?></li>
                    </ul>
                </p>
            </div>
            <?php
        }
        
        $content_types = $this->settings['content_types'] ?? ['post', 'page'];
        $allowed_types = ['post', 'page']; // Free version limitation
        
        // Filter to only allow 'post' and 'page' in free version
        $content_types = array_intersect($content_types, $allowed_types);
        
        $total_posts = 0;
        
        foreach ($content_types as $post_type) {
            $count = wp_count_posts($post_type);
            $total_posts += $count->publish ?? 0;
        }
        ?>
        
        <div class="mustwp-audit-container">
            <div class="mustwp-audit-controls">
                <button type="button" id="mustwp-start-audit" class="button button-primary">
                    <?php esc_html_e('Run Global Audit', 'mustang-wpo'); ?>
                </button>
                <span id="mustwp-audit-progress" class="mustwp-progress-text"></span>
            </div>
            
            <!-- Warning notice for audit in progress -->
            <div id="mustwp-audit-warning" class="notice notice-warning" style="display: none;">
                <p>
                    <strong><?php esc_html_e('Important:', 'mustang-wpo'); ?></strong>
                    <?php esc_html_e('Please keep this page open while the audit is running. Closing the browser tab or navigating away will interrupt the audit process.', 'mustang-wpo'); ?>
                </p>
            </div>
            
            <div id="mustwp-audit-results" class="mustwp-audit-results" style="display: none;">
                <h3><?php esc_html_e('Audit Results', 'mustang-wpo'); ?></h3>
                <div id="mustwp-summary-stats"></div>
            </div>
        </div>
        <?php
        $this->close_wrap();
    }
    
    /**
     * Handle settings save
     */
    private function handle_settings_save(): void {
        // Verify nonce
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mustwp_settings_nonce'] ?? '')), 'mustwp_settings')) {
            wp_die(esc_html__('Security check failed.', 'mustang-wpo'));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to save settings.', 'mustang-wpo'));
        }
        
        // Unslash and sanitize POST data - separate the variable first
        $post_data = wp_unslash($_POST);
        $posted_raw = isset($post_data['mustwp_settings']) ? $post_data['mustwp_settings'] : array();
        if (!is_array($posted_raw)) {
            $posted_raw = array();
        }
        
        // Merge with existing settings to preserve any fields not in the form
        $merged_input = wp_parse_args($posted_raw, $this->settings);
        
        // Sanitize only the fields that are present in the form
        $sanitized = $this->sanitize_settings($merged_input);
        
        update_option('mustwp_settings', $sanitized);
        $this->settings = $sanitized;
        
        // Clear any existing notices to prevent duplicates
        global $wp_settings_errors;
        if (isset($wp_settings_errors['mustwp_settings'])) {
            unset($wp_settings_errors['mustwp_settings']);
        }
        
        // Settings saved successfully
        add_settings_error('mustwp_settings', 'settings_saved', __('Settings saved successfully.', 'mustang-wpo'), 'updated');
    }
    
    /**
     * Handle budgets save
     */
    private function handle_budgets_save(): void {
        // Verify nonce
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mustwp_budgets_nonce'] ?? '')), 'mustwp_budgets')) {
            wp_die(esc_html__('Security check failed.', 'mustang-wpo'));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to save settings.', 'mustang-wpo'));
        }
        
        // Unslash and sanitize POST data - separate the variable first
        $post_data = wp_unslash($_POST);
        $posted_raw = isset($post_data['mustwp_settings']) ? $post_data['mustwp_settings'] : array();
        if (!is_array($posted_raw)) {
            $posted_raw = array();
        }
        
        // Merge with existing settings to preserve any fields not in the form
        $merged_input = wp_parse_args($posted_raw, $this->settings);
        
        // Sanitize only the fields that are present in the form
        $sanitized = $this->sanitize_settings($merged_input);
        
        update_option('mustwp_settings', $sanitized);
        $this->settings = $sanitized;
        
        // Clear any existing notices to prevent duplicates
        global $wp_settings_errors;
        if (isset($wp_settings_errors['mustwp_settings'])) {
            unset($wp_settings_errors['mustwp_settings']);
        }
        
        add_settings_error('mustwp_settings', 'budgets_saved', __('Budgets saved successfully.', 'mustang-wpo'), 'updated');
    }
    
    /**
     * Handle integrations save
     */
    private function handle_integrations_save(): void {
        // Verify nonce
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mustwp_integrations_nonce'] ?? '')), 'mustwp_integrations')) {
            wp_die(esc_html__('Security check failed.', 'mustang-wpo'));
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to save settings.', 'mustang-wpo'));
        }
        
        // Unslash and sanitize POST data - separate the variable first
        $post_data = wp_unslash($_POST);
        $posted_raw = isset($post_data['mustwp_integrations']) ? $post_data['mustwp_integrations'] : array();
        if (!is_array($posted_raw)) {
            $posted_raw = array();
        }
        
        // Merge with existing settings to preserve any fields not in the form
        $merged_input = wp_parse_args($posted_raw, $this->settings);
        
        // Sanitize only the fields that are present in the form
        $sanitized = $this->sanitize_settings($merged_input);
        
        // Validate that a valid API key is provided
        if (empty($sanitized['api_key'])) {
            add_settings_error('mustwp_integrations', 'api_key_required', __('API key is required. Please enter a valid Google PageSpeed Insights API key.', 'mustang-wpo'), 'error');
            return;
        }
        
        // Validate API key format
        $decrypted_key = $this->decrypt_api_key($sanitized['api_key']);
        if (empty($decrypted_key) || strpos($decrypted_key, 'AIza') !== 0) {
            add_settings_error('mustwp_integrations', 'api_key_invalid', __('Invalid API key format. Please enter a valid Google PageSpeed Insights API key.', 'mustang-wpo'), 'error');
            return;
        }
        
        update_option('mustwp_settings', $sanitized);
        $this->settings = $sanitized;
        
        // Clear any existing notices to prevent duplicates
        global $wp_settings_errors;
        if (isset($wp_settings_errors['mustwp_integrations'])) {
            unset($wp_settings_errors['mustwp_integrations']);
        }
        
        // API key is valid and saved successfully
        add_settings_error('mustwp_integrations', 'api_key_saved', __('API key saved successfully and is valid. Plugin is ready to use.', 'mustang-wpo'), 'updated');
    }
    
    /**
     * Render page header
     *
     * @param string $title Page title
     */
    private function render_page_header(string $title): void {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html($title); ?></h1>
        <?php
    }
    
    /**
     * Close wrap div
     */
    private function close_wrap(): void {
        ?>
        </div>
        <?php
    }
    
    
    /**
     * Enqueue admin assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_admin_assets(string $hook): void {
        if (!str_contains($hook, 'mustwp')) {
            return;
        }
        
        wp_enqueue_style(
            'mustwp-admin',
            MUSTWP_PLUGIN_URL . 'build/css/admin.css',
            [],
            MUSTWP_VERSION
        );
        
        // Add inline styles for audit page
        if (str_contains($hook, 'mustwp-audit')) {
            $audit_css = '
            .mustwp-audit-container {
                max-width: 800px;
            }
            
            .mustwp-audit-controls {
                margin-bottom: 20px;
            }
            
            .mustwp-progress-text {
                margin-left: 10px;
                font-style: italic;
            }
            
            .mustwp-audit-results {
                background: #f9f9f9;
                padding: 20px;
                border: 1px solid #ddd;
                border-radius: 4px;
            }';
            
            wp_add_inline_style('mustwp-admin', $audit_css);
        }
        
        wp_enqueue_script(
            'mustwp-admin',
            MUSTWP_PLUGIN_URL . 'build/js/admin.js',
            ['jquery'],
            MUSTWP_VERSION . '.' . filemtime(MUSTWP_PLUGIN_DIR . 'build/js/admin.js'),
            true
        );
        
        wp_localize_script('mustwp-admin', 'mustwpAdmin', [
            'restUrl' => rest_url('mustwp/v1/'),
            'nonce' => wp_create_nonce('wp_rest'),
            'batchSize' => 1, // Hardcoded batch size to 1 post (2 API calls) to prevent timeout issues (especially with Cloudflare)
            'ajaxUrl' => admin_url('admin-ajax.php')
        ]);
    }
    
    /**
     * Handle heartbeat for audit progress
     */
    public function handle_heartbeat($response, $data): array {
        if (isset($data['mustwp_audit_progress'])) {
            $audit_id = sanitize_text_field($data['mustwp_audit_progress']['audit_id'] ?? '');
            
            if ($audit_id) {
                $progress = get_transient('mustwp_audit_progress_' . $audit_id);
                
                if ($progress) {
                    $response['mustwp_audit_progress'] = $progress;
                } else {
                    $response['mustwp_audit_progress'] = [
                        'status' => 'not_found',
                        'message' => 'No progress data found'
                    ];
                }
            }
        }
        
        return $response;
    }
}

