<?php
/**
 * Gutenberg editor integration for WPO
 *
 * @package MUSTWP
 */
 
declare(strict_types=1);
 
namespace MUSTWP;

/**
 * Editor class for Gutenberg sidebar panel
 */
class Editor {
    
    /**
     * Plugin settings
     */
    private array $settings;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->settings = get_option('mustwp_settings', []);
        $this->init_hooks();
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks(): void {
        add_action('enqueue_block_editor_assets', [$this, 'enqueue_editor_assets']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        add_action('rest_api_init', [$this, 'register_editor_rest_routes']);
        add_action('init', [$this, 'register_block_editor_assets']);
    }
    
    /**
     * Enqueue editor assets
     */
    public function enqueue_editor_assets(): void {
        // Get current post type
        $post_type = '';
        $screen = get_current_screen();
        if ($screen && isset($screen->post_type)) {
            $post_type = $screen->post_type;
        }
        
        // Get allowed content types from settings
        $allowed_types = ['post', 'page']; // Free version limitation
        $content_types = $this->settings['content_types'] ?? ['post', 'page'];
        $content_types = array_intersect($content_types, $allowed_types);
        
        // Only enqueue if current post type is in selected content types
        if (!in_array($post_type, $content_types, true)) {
            return;
        }
        
        $script_url = MUSTWP_PLUGIN_URL . 'build/js/editor.js';
        $script_path = MUSTWP_PLUGIN_DIR . 'build/js/editor.js';
        
        // Get file modification time safely
        $file_time = file_exists($script_path) ? filemtime($script_path) : time();
        $version = MUSTWP_VERSION . '.' . $file_time;
        
        // Enqueue editor script
        wp_enqueue_script(
            'mustwp-editor',
            $script_url,
            ['wp-plugins', 'wp-edit-post', 'wp-element', 'wp-components', 'wp-i18n', 'wp-data'],
            $version,
            true
        );
        
        // Localize script with REST API nonce
        wp_localize_script('mustwp-editor', 'mustwpEditor', [
            'restUrl' => rest_url('mustwp/v1/'),
            'nonce' => wp_create_nonce('wp_rest'),
            'strings' => [
                'panelTitle' => __('Mustang WPO', 'mustang-wpo'),
                'mobileScore' => __('Mobile Score', 'mustang-wpo'),
                'desktopScore' => __('Desktop Score', 'mustang-wpo'),
                'runAudit' => __('Run Audit', 'mustang-wpo'),
                'auditing' => __('Auditing...', 'mustang-wpo'),
                'notAudited' => __('Not yet audited', 'mustang-wpo'),
                'lastAudited' => __('Last audited:', 'mustang-wpo'),
                'error' => __('Error:', 'mustang-wpo'),
                'retry' => __('Retry', 'mustang-wpo')
            ]
        ]);
        
        // Enqueue editor styles
        wp_enqueue_style(
            'mustwp-editor',
            MUSTWP_PLUGIN_URL . 'build/css/admin.css',
            [],
            MUSTWP_VERSION
        );
    }
    
    /**
     * Enqueue admin scripts with localized data
     */
    public function enqueue_admin_scripts($hook): void {
        // Only enqueue on post edit pages
        if ($hook !== 'post.php' && $hook !== 'post-new.php') {
            return;
        }
        
        global $post;
        
        // Get allowed content types from settings
        $allowed_types = ['post', 'page']; // Free version limitation
        $content_types = $this->settings['content_types'] ?? ['post', 'page'];
        $content_types = array_intersect($content_types, $allowed_types);
        
        // Get current post type
        $post_type = '';
        if ($post && isset($post->post_type)) {
            $post_type = $post->post_type;
        } elseif (get_current_screen()) {
            $post_type = get_current_screen()->post_type ?? '';
        }
        
        // Fallback: check URL parameter for new posts (validate against allowed types)
        if (empty($post_type)) {
            // Use filter_input to safely get GET parameter
            $requested_type = filter_input(INPUT_GET, 'post_type', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
            if ($requested_type) {
                // Only use if it's an allowed type (security: whitelist approach)
                if (in_array($requested_type, $allowed_types, true)) {
                    $post_type = $requested_type;
                }
            }
        }
        
        // Only enqueue if current post type is in selected content types
        if (!in_array($post_type, $content_types, true)) {
            return;
        }
        
        // Enqueue the editor script
        $script_url = MUSTWP_PLUGIN_URL . 'build/js/editor.js';
        $script_path = MUSTWP_PLUGIN_DIR . 'build/js/editor.js';
        $file_time = file_exists($script_path) ? filemtime($script_path) : time();
        $version = MUSTWP_VERSION . '.' . $file_time;
        
        wp_enqueue_script(
            'mustwp-editor',
            $script_url,
            ['wp-edit-post', 'wp-plugins', 'wp-element', 'wp-components', 'wp-i18n', 'wp-data'],
            $version,
            true
        );
        
        // Localize script with post data
        $post_status = $post ? $post->post_status : 'draft';
        $post_id = $post ? $post->ID : 0;
        
        // Check if API key is valid
        $api = new API();
        $has_api_key = $api->has_valid_api_key();
        // Ensure boolean value for JavaScript
        $has_api_key = (bool) $has_api_key;
        
        wp_localize_script('mustwp-editor', 'MUSTWP', [
            'postId' => $post_id,
            'postStatus' => $post_status,
            'hasApiKey' => $has_api_key,
            'restUrl' => rest_url('mustwp/v1/'),
            'nonce' => wp_create_nonce('wp_rest'),
            'timezone' => wp_timezone_string(),
            'dateFormat' => get_option('date_format'),
            'timeFormat' => get_option('time_format'),
            'integrationsUrl' => admin_url('admin.php?page=mustwp-integrations'),
            'strings' => [
                'panelTitle' => __('Mustang WPO', 'mustang-wpo'),
                'mobileScore' => __('Mobile Score', 'mustang-wpo'),
                'desktopScore' => __('Desktop Score', 'mustang-wpo'),
                'runAudit' => __('Run Audit', 'mustang-wpo'),
                'auditing' => __('Auditing...', 'mustang-wpo'),
                'notAudited' => __('Not yet audited', 'mustang-wpo'),
                'lastAudited' => __('Last audited:', 'mustang-wpo'),
                'error' => __('Error:', 'mustang-wpo'),
                'retry' => __('Retry', 'mustang-wpo'),
                'apiKeyRequired' => __('A valid Google PageSpeed Insights API key is required for the plugin to function, please add one on the Integrations page.', 'mustang-wpo')
            ]
        ]);
        
        // Enqueue editor styles
        wp_enqueue_style(
            'mustwp-editor',
            MUSTWP_PLUGIN_URL . 'build/css/admin.css',
            [],
            MUSTWP_VERSION
        );
    }
    
    /**
     * Register block editor assets
     */
    public function register_block_editor_assets(): void {
        
        // Register the script
        $script_url = MUSTWP_PLUGIN_URL . 'build/js/editor.js';
        $script_path = MUSTWP_PLUGIN_DIR . 'build/js/editor.js';
        $file_time = file_exists($script_path) ? filemtime($script_path) : time();
        $version = MUSTWP_VERSION . '.' . $file_time;
        
        wp_register_script(
            'mustwp-editor',
            $script_url,
            ['wp-plugins', 'wp-edit-post', 'wp-element', 'wp-components', 'wp-i18n', 'wp-data'],
            $version,
            true
        );
        
        // Localize script
        wp_localize_script('mustwp-editor', 'mustwpEditor', [
            'restUrl' => rest_url('mustwp/v1/'),
            'nonce' => wp_create_nonce('wp_rest'),
            'strings' => [
                'panelTitle' => __('Mustang WPO', 'mustang-wpo'),
                'mobileScore' => __('Mobile Score', 'mustang-wpo'),
                'desktopScore' => __('Desktop Score', 'mustang-wpo'),
                'runAudit' => __('Run Audit', 'mustang-wpo'),
                'auditing' => __('Auditing...', 'mustang-wpo'),
                'notAudited' => __('Not yet audited', 'mustang-wpo'),
                'lastAudited' => __('Last audited:', 'mustang-wpo'),
                'error' => __('Error:', 'mustang-wpo'),
                'retry' => __('Retry', 'mustang-wpo')
            ]
        ]);
    }
    
    
    
    
    /**
     * Register editor-specific REST routes
     */
    public function register_editor_rest_routes(): void {
        
        $audit_route = register_rest_route('mustwp/v1', '/editor/audit', [
            'methods' => 'POST',
            'callback' => [$this, 'handle_editor_audit'],
            'permission_callback' => [$this, 'check_edit_post_permission'],
            'args' => [
                'post_id' => [
                    'required' => true,
                    'type' => 'integer',
                    'sanitize_callback' => 'absint'
                ],
                'strategy' => [
                    'required' => false,
                    'type' => 'string',
                    'enum' => ['mobile', 'desktop', 'both'],
                    'default' => 'both',
                    'sanitize_callback' => 'sanitize_text_field'
                ]
            ]
        ]);
        
        $results_route = register_rest_route('mustwp/v1', '/editor/results', [
            'methods' => 'GET',
            'callback' => [$this, 'get_editor_results'],
            'permission_callback' => [$this, 'check_edit_post_permission'],
            'args' => [
                'post_id' => [
                    'required' => true,
                    'type' => 'integer',
                    'sanitize_callback' => 'absint'
                ]
            ]
        ]);
    }
    
    /**
     * Handle editor audit request
     *
     * @param \WP_REST_Request $request REST request
     * @return \WP_REST_Response|WP_Error Response or error
     */
    public function handle_editor_audit(\WP_REST_Request $request) {
        $post_id = $request->get_param('post_id');
        $strategy = $request->get_param('strategy');
        
        $api = new API();
        $url = $api->get_post_url($post_id);
        
        if (is_wp_error($url)) {
            return new \WP_Error('invalid_url', $url->get_error_message(), ['status' => 400]);
        }
        
        // Run audit
        if ($strategy === 'both') {
            $results = $api->audit_url_both($url);
        } else {
            $results = $api->audit_url($url, $strategy);
        }
        
        if (is_wp_error($results)) {
            return $results;
        }
        
        // Store results
        $api->store_results($post_id, $results);
        
        // Format timestamps using the same PHP function as the admin column
        if (!empty($results)) {
            foreach ($results as $device => $data) {
                if (isset($data['timestamp'])) {
                    // Use WordPress timezone handling
                    $timezone = wp_timezone();
                    $results[$device]['formatted_timestamp'] = wp_date(
                        get_option('date_format') . ', ' . get_option('time_format'),
                        $data['timestamp'],
                        $timezone
                    );
                }
            }
        }
        
        return new \WP_REST_Response([
            'success' => true,
            'results' => $results,
            'timestamp' => time()
        ]);
    }
    
    /**
     * Get editor results
     *
     * @param \WP_REST_Request $request REST request
     * @return \WP_REST_Response Response
     */
    public function get_editor_results(\WP_REST_Request $request) {
        
        $post_id = $request->get_param('post_id');
        
        $api = new API();
        $results = $api->get_results($post_id);
        
        // Format timestamps using the same PHP function as the admin column
        if (!empty($results)) {
            foreach ($results as $device => $data) {
                if (isset($data['timestamp'])) {
                    // Use WordPress timezone handling
                    $timezone = wp_timezone();
                    $results[$device]['formatted_timestamp'] = wp_date(
                        get_option('date_format') . ', ' . get_option('time_format'),
                        $data['timestamp'],
                        $timezone
                    );
                }
            }
        }
        
        return new \WP_REST_Response([
            'success' => true,
            'results' => $results,
            'has_results' => !empty($results)
        ]);
    }
    
    /**
     * Check edit post permission
     *
     * @param \WP_REST_Request $request REST request
     * @return bool True if user can edit post
     */
    public function check_edit_post_permission(\WP_REST_Request $request): bool {
        $post_id = $request->get_param('post_id');
        $can_edit = current_user_can('edit_post', $post_id);
        
        return $can_edit;
    }
}

